package watcher

import (
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/fsnotify/fsnotify"
)

// LocalEvent represents a local filesystem change notification.
type LocalEvent struct {
	Mailbox string
	AbsPath string // Empty = full scan, non-empty = specific file
}

// Watcher monitors local maildir changes using inotify.
type Watcher struct {
	fsWatcher *fsnotify.Watcher // Current fsnotify watcher instance
	localPath string            // Account's local path (base directory)
	queue     chan<- LocalEvent // Output queue for sync tasks
	logger    *slog.Logger
}

// New creates a new filesystem watcher.
func New(localPath string, queue chan<- LocalEvent, logger *slog.Logger) *Watcher {
	return &Watcher{
		localPath: localPath,
		queue:     queue,
		logger:    logger,
	}
}

// Run starts the watcher and monitors filesystem events.
// Automatically retries if fsnotify fails, with exponential backoff.
// Returns only on context cancellation or initial setup failure.
func (w *Watcher) Run(ctx context.Context, ready chan<- error) {
	backoff := time.Second
	maxBackoff := 5 * time.Minute

	for {
		// Set up watchers, scan mailboxes, and enqueue initial tasks.
		fsWatcher, err := w.setupWatchers(ctx)
		if err != nil {
			if ctx.Err() != nil {
				// Clean shutdown requested during setup.
				w.logger.Info("Watcher stopped")
				return
			}
			if ready != nil {
				ready <- fmt.Errorf("failed to set up watcher: %w", err)
				return
			}

			// Setup failed. Log error and retry after backoff.
			w.logger.Error("Watcher setup failed, retrying", "error", err)
			goto Backoff
		}

		// Setup succeeded. Signal readiness.
		if ready != nil {
			w.logger.Info("Watcher ready")
			ready <- nil
			close(ready)
			ready = nil
		}
		backoff = time.Second // Reset after successful setup

		err = w.runEventLoop(ctx, fsWatcher)
		_ = fsWatcher.Close() // Best effort close

		if ctx.Err() != nil {
			// Clean shutdown requested.
			w.logger.Info("Watcher stopped")
			return
		}

		w.logger.Error("Watcher event loop crashed, will retry.", "error", err, "backoff", backoff)

	Backoff:
		select {
		case <-time.After(backoff):
			// Exponential backoff.
			backoff *= 2
			if backoff > maxBackoff {
				backoff = maxBackoff
			}
		case <-ctx.Done():
			w.logger.Info("Watcher stopped")
			return
		}
	}
}

// setupWatchers creates fsnotify watcher, scans mailboxes, sets up watches, and enqueues initial tasks.
// Returns the fsnotify.Watcher on success, error on failure.
// Signals error to ready channel on first run failures.
func (w *Watcher) setupWatchers(ctx context.Context) (*fsnotify.Watcher, error) {
	w.logger.Info("Watcher setting up")

	// Create fresh fsnotify watcher (in case of retry after crash).
	fsWatcher, err := fsnotify.NewWatcher()
	if err != nil {
		return nil, fmt.Errorf("failed to create fsnotify watcher: %w", err)
	}

	w.fsWatcher = fsWatcher

	if err := w.watchRecursive(""); err != nil {
		_ = fsWatcher.Close() // Clean up
		return nil, fmt.Errorf("failed to watch directories: %w", err)
	}

	var localMailboxes []string
	err = w.scanMaildirsRecursive(w.localPath, &localMailboxes)
	if err != nil {
		_ = fsWatcher.Close() // Clean up
		return nil, fmt.Errorf("failed to scan local mailboxes: %w", err)
	}
	w.logger.Info("Scanned local mailboxes", "count", len(localMailboxes))

	// Enqueue full sync with local rescan for all local mailboxes.
	// Ensures that we pick up changes before the watcher started.
	for _, mailbox := range localMailboxes {
		w.logger.Info("Enqueuing full sync with local rescan", "mailbox", mailbox)
		select {
		case w.queue <- LocalEvent{Mailbox: mailbox}:
		case <-ctx.Done():
			_ = fsWatcher.Close() // Cleanup on cancellation
			return nil, ctx.Err()
		}
	}

	return fsWatcher, nil
}

// runEventLoop processes filesystem events until context cancellation or error.
func (w *Watcher) runEventLoop(ctx context.Context, fsWatcher *fsnotify.Watcher) error {
	w.logger.Info("Watcher event loop starting")
	defer w.logger.Info("Watcher event loop stopped")

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()

		case event, ok := <-fsWatcher.Events:
			if !ok {
				return fmt.Errorf("fsnotify events channel closed")
			}
			w.handleEvent(ctx, event)

		case err, ok := <-fsWatcher.Errors:
			if !ok {
				return fmt.Errorf("fsnotify errors channel closed")
			}
			w.logger.Error("Watcher error", "error", err)
		}
	}
}

// handleEvent processes a single fsnotify event.
func (w *Watcher) handleEvent(ctx context.Context, event fsnotify.Event) {
	if !event.Has(fsnotify.Create) && !event.Has(fsnotify.Remove) && !event.Has(fsnotify.Rename) {
		return
	}

	if event.Has(fsnotify.Create) {
		info, err := os.Stat(event.Name)
		if err != nil {
			w.logger.Error("failed to stat newly created file", "name", event.Name, "error", err)
			return
		}
		if info.IsDir() {
			w.handleDirectoryCreation(ctx, event)
			return
		}
	}

	mailbox := w.pathToMailbox(event.Name)
	if mailbox == "" {
		return
	}

	w.logger.Debug(
		"Filesystem event detected",
		"event", event.Op.String(),
		"mailbox", mailbox,
		"path", event.Name,
	)

	select {
	case w.queue <- LocalEvent{
		Mailbox: mailbox,
		AbsPath: event.Name,
	}:
	case <-ctx.Done():
		w.logger.Debug("Context cancelled while queuing file-level sync")
	}
}

// scanMaildirsRecursive recursively scans for maildirs starting from absPath.
func (w *Watcher) scanMaildirsRecursive(absPath string, mailboxes *[]string) error {
	entries, err := os.ReadDir(absPath)
	if err != nil {
		return fmt.Errorf("failed to read directory %s: %w", absPath, err)
	}

	hasCur := false
	hasNew := false
	for _, entry := range entries {
		if !entry.IsDir() {
			continue
		}

		name := entry.Name()
		switch name {
		case "cur":
			hasCur = true
			continue
		case "new":
			hasNew = true
			continue
		case "tmp":
			continue
		}

		subAbsPath := filepath.Join(absPath, name)
		if err := w.scanMaildirsRecursive(subAbsPath, mailboxes); err != nil {
			return err
		}
	}

	if hasCur && hasNew {
		if absPath == w.localPath {
			w.logger.Error("Found maildir at root level, skipping", "path", absPath)
			return nil
		}

		mailbox, err := filepath.Rel(w.localPath, absPath)
		if err != nil {
			return fmt.Errorf("failed to get relative path for mailbox: %w", err)
		}

		*mailboxes = append(*mailboxes, mailbox)
		w.logger.Debug("Found maildir", "mailbox", mailbox)
	}

	return nil
}

// handleDirectoryCreation processes directory creation events.
// Watches the new directory and checks if it completes a maildir.
func (w *Watcher) handleDirectoryCreation(ctx context.Context, event fsnotify.Event) {
	eventBase := filepath.Base(event.Name)

	if eventBase == "cur" || eventBase == "new" {
		if err := w.fsWatcher.Add(event.Name); err != nil {
			w.logger.Warn("Failed to watch new directory", "path", event.Name, "error", err)
			return
		}

		w.checkMaildirCompletion(ctx, filepath.Dir(event.Name))
		return
	}
	if eventBase == "tmp" {
		return
	}

	relPath, err := filepath.Rel(w.localPath, event.Name)
	if err != nil {
		w.logger.Warn("Failed to get relative path", "path", event.Name, "error", err)
		return
	}

	if err := w.watchRecursive(relPath); err != nil {
		w.logger.Warn("Failed to watch new directory tree", "path", event.Name, "error", err)
	}
}

// checkMaildirCompletion checks if a directory is a complete maildir.
// If both cur/ and new/ exist, enqueues a full sync which will catch any files
// created in the race window between directory creation and watch setup.
func (w *Watcher) checkMaildirCompletion(ctx context.Context, maildirPath string) {
	hasCur := false
	if stat, err := os.Stat(filepath.Join(maildirPath, "cur")); err == nil && stat.IsDir() {
		hasCur = true
	}

	hasNew := false
	if stat, err := os.Stat(filepath.Join(maildirPath, "new")); err == nil && stat.IsDir() {
		hasNew = true
	}

	if !hasCur || !hasNew {
		return
	}

	mailbox, err := filepath.Rel(w.localPath, maildirPath)
	if err != nil {
		w.logger.Warn("Failed to get relative path", "path", maildirPath, "error", err)
		return
	}

	w.logger.Info("Complete maildir detected", "mailbox", mailbox)

	// Catch messages created between creation and event handling
	select {
	case w.queue <- LocalEvent{Mailbox: mailbox}:
	case <-ctx.Done():
		w.logger.Debug("Context cancelled while enqueuing maildir sync")
	}
}

// watchRecursive recursively watches all directories starting from relPath.
func (w *Watcher) watchRecursive(relPath string) error {
	absPath := filepath.Join(w.localPath, relPath)

	if err := w.fsWatcher.Add(absPath); err != nil {
		return fmt.Errorf("failed to watch directory %s: %w", absPath, err)
	}

	if relPath == "" {
		w.logger.Debug("Watching base directory")
	} else {
		w.logger.Debug("Watching directory", "path", relPath)
	}

	entries, err := os.ReadDir(absPath)
	if err != nil {
		return fmt.Errorf("failed to read directory %s: %w", absPath, err)
	}

	for _, entry := range entries {
		if !entry.IsDir() {
			continue
		}

		name := entry.Name()
		childRelPath := filepath.Join(relPath, name)

		// Watch cur/new. Other Maildirs will watch themselves in recursive calls.
		if name == "cur" || name == "new" {
			childAbsPath := filepath.Join(w.localPath, childRelPath)
			if err := w.fsWatcher.Add(childAbsPath); err != nil {
				w.logger.Warn("Failed to watch directory", "path", childRelPath, "error", err)
				continue
			}
			continue
		}
		if name == "tmp" {
			continue
		}

		if err := w.watchRecursive(childRelPath); err != nil {
			w.logger.Error("Failed to watch subdirectory", "path", childRelPath, "error", err)
			// Continue with other directories.
		}
	}

	return nil
}

// pathToMailbox maps a message's path to its mailbox's name.
// Returns empty string if path doesn't match mailbox path format.
func (w *Watcher) pathToMailbox(path string) string {
	relPath, err := filepath.Rel(w.localPath, path)
	if err != nil {
		return ""
	}

	// Last component should be filename, second-to-last should be cur/ or new/.
	// Everything before that is the mailbox path.
	parts := strings.Split(relPath, string(filepath.Separator))
	if len(parts) < 3 {
		return ""
	}

	subdir := parts[len(parts)-2]
	if subdir != "cur" && subdir != "new" {
		w.logger.Debug("Bogus path in pathToMailbox", "rel_path", relPath)
		return ""
	}

	// Mailbox is everything except the last two components (subdir and filename).
	mailboxParts := parts[:len(parts)-2]
	mailbox := filepath.Join(mailboxParts...)

	return mailbox
}
