{ inputs, ... }:
let
  sharedLib = inputs.self.sharedLib;
in
{
  perSystem =
    {
      self',
      pkgs,
      lib,
      ...
    }:
    let
      testLib = inputs.self.testLib { inherit self' lib pkgs; };
    in
    {
      checks =
        {
          issueCreateBasic = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              assert issue["title"] == "oh noes, me found a bug", f"issue title incorrect\ngot:{issue}"
              assert issue["user"]["login"] == "${sharedLib.user}", f"issue user incorrect\ngot:{issue}"
              assert issue["body"] == "The bug is bad! 🐛", f"issue description incorrect\ngot:{issue}"
              assert issue["state"] == "open", f"issue state incorrect\ngot:{issue}"
              assert issue["repository"]["full_name"] == "${sharedLib.user}/${sharedLib.repo}", f"issue repository incorrect\ngot:{issue}"
            '';
          };
          issueCreateFull = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              label = ${
                testLib.berg_run_json {
                  cmd = ''label create --name "bug"'';
                  cdTestRepo = true;
                }
              }

              print(label)

              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛" --labels "bug" --assignees "${sharedLib.user}"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              assert issue["title"] == "oh noes, me found a bug", f"issue title incorrect\ngot:{issue}"
              assert issue["user"]["login"] == "${sharedLib.user}", f"issue user incorrect\ngot:{issue}"
              assert issue["body"] == "The bug is bad! 🐛", f"issue description incorrect\ngot:{issue}"
              assert issue["state"] == "open", f"issue state incorrect\ngot:{issue}"
              assert len(issue["labels"]) == 1, f"wrong amount of issue labels\ngot:{issue}"
              assert issue["labels"][0]["name"] == "bug", f"issue labels incorrect\ngot:{issue}"
              assert len(issue["assignees"]) == 1, f"wrong amount of issue assignees\ngot:{issue}"
              assert issue["assignees"][0]["login"] == "${sharedLib.user}", f"issue assignees incorrect\ngot:{issue}"
              assert issue["repository"]["full_name"] == "${sharedLib.user}/${sharedLib.repo}", f"issue repository incorrect\ngot:{issue}"
            '';
          };
          issueCreateMissingLabels = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              ${testLib.berg_run {
                cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛" --labels "bug" --assignees "${sharedLib.user}"'';
                cdTestRepo = true;
                success = false;
              }}
            '';
          };
          issueList = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              issue_list = ${
                testLib.berg_run_json {
                  cmd = ''issue list'';
                  cdTestRepo = true;
                }
              }

              printj(issue_list)

              assert len(issue_list) == 1, f"issue list item incorrect\ngot:{issue_list}"
              assert issue_list[0] == issue, f"issue list item incorrect\ngot:{issue_list}"
            '';
          };
          issueListStateOpen = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              open_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "open issue" --description "This one is open"'';
                  cdTestRepo = true;
                }
              }

              closed_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "closed issue" --description "This one is closed"'';
                  cdTestRepo = true;
                }
              }

              issue_id = closed_issue["id"]
              ${testLib.berg_run {
                cmd = ''issue edit {issue_id} --state "closed"'';
                cdTestRepo = true;
              }}

              open_list = ${
                testLib.berg_run_json {
                  cmd = ''issue list --state open'';
                  cdTestRepo = true;
                }
              }

              printj(open_list)

              assert len(open_list) == 1, f"open issue list should have 1 item\ngot:{open_list}"
              assert open_list[0]["title"] == "open issue", f"open issue list item incorrect\ngot:{open_list}"
            '';
          };
          issueListStateClosed = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              open_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "open issue" --description "This one is open"'';
                  cdTestRepo = true;
                }
              }

              closed_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "closed issue" --description "This one is closed"'';
                  cdTestRepo = true;
                }
              }

              issue_id = closed_issue["id"]
              ${testLib.berg_run {
                cmd = ''issue edit {issue_id} --state "closed"'';
                cdTestRepo = true;
              }}

              closed_list = ${
                testLib.berg_run_json {
                  cmd = ''issue list --state closed'';
                  cdTestRepo = true;
                }
              }

              printj(closed_list)

              assert len(closed_list) == 1, f"closed issue list should have 1 item\ngot:{closed_list}"
              assert closed_list[0]["title"] == "closed issue", f"closed issue list item incorrect\ngot:{closed_list}"
            '';
          };
          issueListStateAll = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              open_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "open issue" --description "This one is open"'';
                  cdTestRepo = true;
                }
              }

              closed_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "closed issue" --description "This one is closed"'';
                  cdTestRepo = true;
                }
              }

              issue_id = closed_issue["id"]
              ${testLib.berg_run {
                cmd = ''issue edit {issue_id} --state "closed"'';
                cdTestRepo = true;
              }}

              all_list = ${
                testLib.berg_run_json {
                  cmd = ''issue list --state all'';
                  cdTestRepo = true;
                }
              }

              printj(all_list)

              assert len(all_list) == 2, f"all issue list should have 2 items\ngot:{all_list}"
            '';
          };
          issueComment = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              comment = ${
                testLib.berg_run_json {
                  cmd = ''issue comment --number 1 --body "Hello, Comment World!"'';
                  cdTestRepo = true;
                }
              }

              printj(comment)

              assert comment["body"] == "Hello, Comment World!", f"issue comment incorrect\ngot:{comment}"
              assert comment["id"] == 1, f"issue comment id incorrect\ngot:{comment}"
              assert comment["user"]["login"] == "${sharedLib.user}", f"issue commenter incorrect\ngot:{comment}"
              assert comment["issue_url"].endswith("/1"), f"issue comment number incorrect\ngot:{comment}"

              comment = ${
                testLib.berg_run_json {
                  cmd = ''issue comment --number 1 --body "Hello, Comment World again!"'';
                  cdTestRepo = true;
                }
              }

              printj(comment)

              assert comment["body"] == "Hello, Comment World again!", f"issue comment incorrect\ngot:{comment}"
              assert comment["id"] == 2, f"issue comment id incorrect\ngot:{comment}"
              assert comment["user"]["login"] == "${sharedLib.user}", f"issue commenter incorrect\ngot:{comment}"
              assert comment["issue_url"].endswith("/1"), f"issue comment number incorrect\ngot:{comment}"
            '';
          };
          issueViewNormal = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              expected_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(expected_issue)

              issue_id = expected_issue["id"]
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue view {issue_id}'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              assert expected_issue == issue, "viewed issue doesn't equal created issue\nexpected:{expected_issue}\ngot:{issue}"
            '';
          };
          issueViewWithRepoFlag = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "test issue" --description "testing repo flag"'';
                  cdTestRepo = true;
                }
              }

              issue_id = issue["id"]

              viewed_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue view {issue_id} --owner-repo ${sharedLib.user}/${sharedLib.repo}'';
                  cdTestRepo = false;
                }
              }

              assert viewed_issue["id"] == issue_id, f"issue id mismatch\ngot:{viewed_issue}"
              assert viewed_issue["title"] == "test issue", f"issue title mismatch\ngot:{viewed_issue}"

              # Test label list with --owner-repo flag
              ${testLib.berg_run {
                cmd = ''label list --owner-repo ${sharedLib.user}/${sharedLib.repo}'';
                cdTestRepo = false;
              }}
            '';
          };
          issueViewComments = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              comment = ${
                testLib.berg_run_json {
                  cmd = ''issue comment --number 1 --body "Hello, Comment World!"'';
                  cdTestRepo = true;
                }
              }

              printj(comment)

              issue_id = issue["id"]
              comments = ${
                testLib.berg_run_json {
                  cmd = ''issue view {issue_id} --comments'';
                  cdTestRepo = true;
                }
              }

              printj(comments)

              assert len(comments) == 1, "There should be exactly one comment!\ngot:{comments}"

              comment = comments[0]

              assert comment["body"] == "Hello, Comment World!", f"issue comment incorrect\ngot:{comment}"
              assert comment["id"] == 1, f"issue comment id incorrect\ngot:{comment}"
              assert comment["user"]["login"] == "${sharedLib.user}", f"issue commenter incorrect\ngot:{comment}"
              assert comment["issue_url"].endswith("/1"), f"issue comment number incorrect\ngot:{comment}"
            '';
          };
          regression-issueViewComments = {
            withRepoSetup = true;
            withAuth = true;
            test = ''
              ${
                builtins.genList (n: ''
                  ${testLib.berg_run {
                    cmd = ''repo create --name ${sharedLib.repo}-${builtins.toString n} --description bar'';
                  }}
                  ${
                    builtins.genList (m: ''
                      ${testLib.berg_run {
                        cmd = ''issue create --title ${sharedLib.repo}-${builtins.toString m} --description ${sharedLib.repo} --owner-repo ${sharedLib.user}/${sharedLib.repo}-${builtins.toString n}'';
                      }}
                    '') 5
                    |> builtins.concatStringsSep ""
                  }
                '') 3
                |> builtins.concatStringsSep ""
              }

              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --description "The bug is bad! 🐛"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              comment = ${
                testLib.berg_run_json {
                  cmd = ''issue comment --number 1 --body "Hello, Comment World!"'';
                  cdTestRepo = true;
                }
              }

              printj(comment)

              assert comment["body"] == "Hello, Comment World!", f"issue comment incorrect\ngot:{comment}"
              assert comment["id"] == 1, f"issue comment id incorrect\ngot:{comment}"
              assert comment["user"]["login"] == "${sharedLib.user}", f"issue commenter incorrect\ngot:{comment}"
              assert comment["issue_url"].endswith("/1"), f"issue comment number incorrect\ngot:{comment}"

              comments = ${
                testLib.berg_run_json {
                  cmd = ''issue view 1 --comments'';
                  cdTestRepo = true;
                }
              }

              printj(comments)

              assert comments[0]["body"] == "Hello, Comment World!", f"issue comments incorrect\ngot:{comments}"
              assert comments[0]["id"] == 1, f"issue comments id incorrect\ngot:{comments}"
              assert comments[0]["user"]["login"] == "${sharedLib.user}", f"issue commentser incorrect\ngot:{comments}"
              assert comments[0]["issue_url"].endswith("/1"), f"issue comments number incorrect\ngot:{comments}"
            '';
          };
          issueEditNoop = {
            withLabels = true;
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --labels "bug,feat,test" --description "The bug is bad! 🐛" --assignees "${sharedLib.user}"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              issue_id = issue["id"]
              updated_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue edit {issue_id}'';
                  cdTestRepo = true;
                }
              }

              printj(updated_issue)

              assert updated_issue["title"] == issue["title"], "issue title changed unexpectedly"
              assert len(issue["labels"]) == 3, f"wrong amount of issue labels\ngot:{issue}"
              assert updated_issue["labels"] == issue["labels"], "issue labels changed unexpectedly"
              assert updated_issue["body"] == issue["body"], "issue body changed unexpectedly"
              assert updated_issue["assignees"] == issue["assignees"], "issue assignees changed unexpectedly"
              assert updated_issue["milestone"] == issue["milestone"], "issue milestone changed unexpectedly"
              assert updated_issue["state"] == issue["state"], "issue state changed unexpectedly"
            '';
          };
          issueEditBasic = {
            withLabels = true;
            withRepoSetup = true;
            withAuth = true;
            test = ''
              issue = ${
                testLib.berg_run_json {
                  cmd = ''issue create --title "oh noes, me found a bug" --labels "bug,feat,test" --description "The bug is bad! 🐛" --assignees "${sharedLib.user}"'';
                  cdTestRepo = true;
                }
              }

              printj(issue)

              issue_id = issue["id"]
              updated_issue = ${
                testLib.berg_run_json {
                  cmd = ''issue edit {issue_id} --title "New Title" --labels "bug" --description "Short body" --assignees "${sharedLib.otherUser}" --state "closed"'';
                  cdTestRepo = true;
                }
              }

              printj(updated_issue)

              assert updated_issue["title"] == "New Title", "issue title didnt chang unexpectedly"
              assert len(updated_issue["labels"]) == 1, f"wrong amount of issue labels\ngot:{issue}"
              assert updated_issue["labels"][0]["name"] == "bug", "issue labels didnt changed unexpectedly"
              assert updated_issue["body"] == "Short body", "issue body didnt change unexpectedly"
              assert updated_issue["assignees"][0]["login"] == "${sharedLib.otherUser}", "issue assignees didnt change unexpectedly"
              assert updated_issue["milestone"] == issue["milestone"], "issue milestone changed unexpectedly"
              assert updated_issue["state"] == "closed", "issue state didnt change unexpectedly"
            '';
          };
        }
        |> testLib.embeddInFlake;
    };
}
