use clap::Parser;

use crate::{
    actions::BergCommand,
    render::json::JsonToStdout,
    types::{config::BergConfig, output::OutputMode},
};

pub async fn berg_main() {
    tracing_subscriber::fmt::init();

    let cmd = BergCommand::parse();
    let output_mode = cmd.output_mode;
    if let Err(err) = cmd.run().await {
        match output_mode {
            // output a pretty, human-readable error to stderr
            OutputMode::Pretty => output_pretty_error(err),
            // This is useful for automation tooling and for testing ... otherwise we wouldn't be
            // able to "catch" the error message and work with it easily
            //
            // note that this outputs to stdout for simplicity
            OutputMode::Json => output_json_error(err),
        }
        std::process::exit(1);
    }
}

fn output_json_error(error: miette::Error) {
    // helper struct to output JSON which is a bit more structured than just a raw
    // string
    #[derive(serde::Serialize)]
    struct BergErrorMessage {
        // the error in text form
        error_message: String,
    }

    BergErrorMessage {
        error_message: error.to_string(),
    }
    .print_json()
    .expect("String to JSON always works");
}

fn output_pretty_error(error: miette::Error) {
    let config = match BergConfig::new() {
        Ok(config) => config,
        Err(e) => {
            eprintln!("{e:?}");
            std::process::exit(1);
        }
    };
    let err = format!("{error:?}");
    let table = config.make_table().add_error(err.trim());
    eprintln!("\n{table}", table = table.show());
}
