/*  $Id: ScriptController.cpp,v 1.12 2011/10/31 01:37:54 sarrazip Exp $

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2007-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#include "ScriptController.h"

using namespace std;

#include <assert.h>
#include <vector>
#include <string>
#include <ctype.h>
#include <stdio.h>


static int
getNonNegInt(const string &line, string::size_type &pos)
{
    string::size_type len = line.length();
    while (pos < len && isspace(line[pos]))
        pos++;
    if (pos >= len)
        return -1;
    string::size_type start = pos;
    int n = 0;
    while (pos < len && isdigit(line[pos]))
    {
        n = n * 10 + static_cast<long>(line[pos] - '0');
        // Note: no check for overflow.
        pos++;
    }
    if (pos == start)
        return -1;
    return n;
}


// Replacement function for getline().
// In the absence of a newline, this function cuts lines at 512 chars.
static bool
safegetline(istream &src, string &line)
{
    line.clear();
    for (size_t i = 512; i != 0; i--)
    {
        int c = src.get();
        if (c == EOF)
            return false;
        if (c == '\n')
            return true;
        line += static_cast<char>(c);
    }
    return true;
}


static bool
readLine(istream &src, vector<int> &integers)
{
    string line;
    if (!safegetline(src, line))
        return false;

    integers.clear();
    string::size_type pos = 0;
    int n;
    while ((n = getNonNegInt(line, pos)) >= 0)
        integers.push_back(n);

    return true;
}


ScriptController::ScriptController(istream &_gameScriptStream,
                                    int requiredFileFormatVersion)
  : gameScriptStream(_gameScriptStream),
    seed(0),
    start(false),
    end(false),
    pause(false),
    resume(false),
    left(false),
    right(false),
    shoot(false),
    hyperspace(false)
{
    vector<int> v;
    if (!readLine(gameScriptStream, v))
        throw -1;
    if (v.size() == 0 || v[0] < 1)
        throw int(-2);
    if (v[0] != requiredFileFormatVersion)
        throw int(v[0]);

    if (readLine(gameScriptStream, v) && v.size() > 0)
        seed = static_cast<unsigned>(v[0]);
}


//virtual
ScriptController::~ScriptController()
{
}


unsigned
ScriptController::getSeed() const
{
    return seed;
}


//virtual
void
ScriptController::startOfTick()
{
    start = end = pause = resume = left = right = shoot = hyperspace = false;

    vector<int> v;
    if (!readLine(gameScriptStream, v))
    {
        end = true;
        return;
    }

    for (vector<int>::const_iterator it = v.begin(); it != v.end(); it++)
    {
        switch (static_cast<Command>(*it))
        {
            case COMMAND_START: start = true; break;
            case COMMAND_END: end = true; break;
            case COMMAND_PAUSE: pause = true; break;
            case COMMAND_RESUME: resume = true; break;
            case COMMAND_MOVE_LEFT: left = true; break;
            case COMMAND_MOVE_RIGHT: right = true; break;
            case COMMAND_SHOOT: shoot = true; break;
            case COMMAND_HYPERSPACE: hyperspace = true; break;
        }
    }
}


//virtual
bool
ScriptController::isStartRequested(bool &/*isExtendedMode*/)
{
    return start;
}


//virtual
bool
ScriptController::isEndRequested()
{
    return end;
}


//virtual
bool
ScriptController::isPauseRequested()
{
    assert(!resume);
    return pause;
}


//virtual
bool
ScriptController::isResumeRequested()
{
    assert(!pause);
    return resume;
}


//virtual
bool
ScriptController::isShootingActive()
{
    return shoot;
}


//virtual
bool
ScriptController::isHyperspaceRequested()
{
    return hyperspace;
}


//virtual
bool
ScriptController::isLeftMoveRequested()
{
    return left;
}


//virtual
bool
ScriptController::isRightMoveRequested()
{
    return right;
}
