/*
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

#ifndef BXX0_BASE64_ENCODE_H
#define BXX0_BASE64_ENCODE_H  1

#include <stddef.h>


/*! Library header for libbasexx Base 64 encode module
 *
 * \file      base64_encode.h
 * \author    Michael Bäuerle
 * \date      2024
 * \copyright BSD-2-Clause
 *
 * Public API (compatible between all versions with same major number).
 *
 * API documentation: \ref BASE64_ENCODE
 */


/*! \defgroup BASE64_ENCODE Base 64 encoder
 *
 * Encode Base 64 (RFC 4648).
 */
/*! \{ */


/*! \name Error codes for comparison with return value of encoder
 *
 * \anchor Encode_Errors
 *
 * Constants for comparison with return value of encoder.
 * <br>
 * Zero means success. Check for success with (0 == ret).
 * <br>
 * All error codes have negative values. Check for error with (0 > ret).
 * <br>
 * Codes with positive value are reserved for future use.
 */
/*! \{ */
#define BXX0_BASE64_ENCODE_ERROR_SIZE  -1  /*!< Output buffer too small */
/*! \} */


/*! \name Flags for encoder (can be ORed together)
 *
 * \anchor Encode_Flags
 */
/*! \{ */
/*! Create output data without padding */
#define BXX0_BASE64_ENCODE_FLAG_NOPAD  0x01
/*! \} */


/*! Calculate required output buffer size with preprocessor
 *
 * \param[in] len_in  Size of input data
 *
 * The encoder creates four output bytes for every three input bytes.
 * <br>
 * If the size of the input data is not a multiple of three, four additional
 * characters are required for tail and padding.
 */
#define BXX0_BASE64_ENCODE_LEN_OUT(len_in)  (((len_in / 3U) * 4U) + 4U)


/*! Base 64 encoder
 *
 * \param[out]     out      Pointer to buffer for output data
 * \param[in,out]  len_out  Pointer to size of output buffer \e out
 * \param[in]      in       Pointer to input buffer
 * \param[in,out]  len_in   Pointer to size of data in input buffer \e in
 * \param[in]      flags    Flags (from \ref Encode_Flags )
 *
 * \attention
 * All memory regions accessed by pointer parameters must not overlap!
 *
 * If the size of the input data is not a multiple of 3, then padding is
 * generated at the end of the output data.
 * <br>
 * The flag \ref BXX0_BASE64_ENCODE_FLAG_NOPAD can be used to calculate the
 * output length without padding.
 *
 * The size of the output buffer must be large enough for the encoded data and
 * potential padding (even if disabled by \ref BXX0_BASE64_ENCODE_FLAG_NOPAD ).
 * <br>
 * The required output buffer size can be calculated with the macro
 * \ref BXX0_BASE64_ENCODE_LEN_OUT .
 *
 * \e len_out is overwritten with the number of bytes still available in
 * \e out .
 *
 * \e len_in is overwritten with the number of input bytes that were not
 * consumed (always zero on success).
 *
 * \return
 * Zero or positive warning code on success. Check for success with (0 <= ret).
 * <br>
 * If the decoder used \ref Encode_Flags to create a modified data format, the
 * corresponding bits for the flags are set in the warning code.
 * Use \ref Encode_Flags constants to check for warnings.
 * <br>
 * Negative value on error. Check for error with (0 > ret).
 * Use \ref Encode_Errors constants to check for errors.
 */
signed char bxx0_base64_encode(unsigned char       *out, size_t *len_out,
                               const unsigned char *in , size_t *len_in,
                               unsigned char        flags);


/*! \} */


#endif  /* BXX0_BASE64_ENCODE_H */
