/*
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

#ifndef BXX0_BASEXX_EBCDIC_H
#define BXX0_BASEXX_EBCDIC_H  1

#include <stddef.h>


/*! Library header for libbasexx EBCDIC conversion module
 *
 * \file      basexx_ebcdic.h
 * \author    Michael Bäuerle
 * \date      2024
 * \copyright BSD-2-Clause
 *
 * Public API (compatible between all versions with same major number).
 *
 * API documentation: \ref EBCDIC
 */


/*! \defgroup EBCDIC EBCDIC conversion
 *
 * Convert US-ASCII to or from EBCDIC.
 *
 * All characters of the Base 64 alphabet as defined by RFC 4648 and the pad
 * character (equal sign) are supported.
 * <br>
 * The alphabets for Base 32 and Base 16 are subsets and supported too.
 *
 * The modified alphabet for URLs from RFC 4648 Section 5 is supported
 * (hyphen-minus and underscore).
 *
 * The modified alphabet for IMAP from RFC 3501 Section 5.1.3 is supported
 * (comma).
 *
 * Control characters for line breaks, as required by RFC 1421, RFC 2045
 * and RFC 4880, but not semantically relevant for the payload, are supported
 * (CR and LF only, NEL is not supported).
 *
 * \note
 * All unsupported characters are converted to question mark in target encoding.
 * <br>
 * Future versions may handle additional characters.
 */
/*! \{ */


/*! Convert data from US-ASCII to EBCDIC
 *
 * \param[out] buf  Pointer to buffer
 * \param[in]  len  Size of data to convert in buffer \e buf
 */
void bxx0_ebcdic_to(unsigned char *buf, size_t len);


/*! Convert data from EBCDIC to US-ASCII
 *
 * \param[out] buf  Pointer to buffer
 * \param[in]  len  Size of data to convert in buffer \e buf
 */
void bxx0_ebcdic_from(unsigned char *buf, size_t len);


/*! \} */


#endif  /* BXX0_BASEXX_EBCDIC_H */
