/*
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

#ifndef UCIC0_I_ICONV_PRIVATE_H
#define UCIC0_I_ICONV_PRIVATE_H  1

#include <errno.h>
#include <stddef.h>


#pragma GCC visibility push(internal)


/*! \cond DO_NOT_DOCUMENT */


/*
 * Error codes
 *
 * On POSIX systems errno is set to the values defined by the Solaris 11 API.
 * For plain C90 the value -1 is used for all errors.
 */
#if defined(EBADF) && defined(EILSEQ) && defined(E2BIG) && defined(EINVAL)
#define UCIC0_I_EBADF   EBADF
#define UCIC0_I_EILSEQ  EILSEQ
#define UCIC0_I_E2BIG   E2BIG
#define UCIC0_I_EINVAL  EINVAL
#else  /* defined(EBADF) && ... && defined(EINVAL) */
#define UCIC0_I_EBADF   -1
#define UCIC0_I_EILSEQ  -1
#define UCIC0_I_E2BIG   -1
#define UCIC0_I_EINVAL  -1
#endif  /* defined(EBADF) && ... && defined(EINVAL) */

/*
 * Unicode replacement codepoint for non-identical conversions
 *
 * Must be an UTF-8 octet-sequence for a single codepoint (max. 4 octets)
 * as string literal.
 */
#define UCIC0_I_NONIDENT  "\xEF\xBF\xBD"  /* U+FFFD  REPLACEMENT CHARACTER */

/* Check for surrogate codepoint (long int) */
#define UCIC0_I_UTF16_SURROGATE(ucp)  ((0xD800L <= ucp) && (0xDFFFL >= ucp))

/* Check for high surrogate codepoint (long int) */
#define UCIC0_I_UTF16_SURROGATE_HIGH(ucp)  ((0xD800L <= ucp) && (0xDBFFL >= ucp))

/* Check for low surrogate codepoint (long int) */
#define UCIC0_I_UTF16_SURROGATE_LOW(ucp)  ((0xDC00L <= ucp) && (0xDFFFL >= ucp))


/* Data type for bool
 *
 * C90 provides no data type bool.
 * Semantics are value 0 for false and true otherwise.
 */
typedef unsigned char  ucic0_i_bool;


/* State of converter */
struct ucic0_i_state
{
    char         *inarray;
    size_t       *inlen;
    char         *outarray;
    size_t       *outlen;
    size_t        inlen_start;   /* Initial value of *inlen */
    size_t        outlen_start;  /* Initial value of *outlen */
    size_t        nonident;      /* Number of on non-identical conversions */
    int           flag;
    ucic0_i_bool  resync;        /* Lost synchronization */
    ucic0_i_bool  abort;         /* Abort conversion before NUL */
    ucic0_i_bool  shift;         /* Shift state of encoding */
};


/* Data type for state */
typedef struct ucic0_i_state  ucic0_i_state;


/*! \endcond */


#pragma GCC visibility pop


#endif  /* UCIC0_I_ICONV_PRIVATE_H */
