// Copyright 2014 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

//go:build ignore

// Generate Windows callback assembly file.
// This was copied from the Go repository and modified to generate non-Windows assembly files.

package main

import (
	"bytes"
	"fmt"
	"os"
)

const maxCallback = 2000

func genasmAmd64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// runtime·callbackasm is called by external code to
// execute Go implemented callback function. It is not
// called from the start, instead runtime·compilecallback
// always returns address into runtime·callbackasm offset
// appropriately so different callbacks start with different
// CALL instruction in runtime·callbackasm. This determines
// which Go callback function is executed later on.
#include "textflag.h"

TEXT callbackasm(SB),NOSPLIT|NOFRAME,$0
`)
	for i := 0; i < maxCallback; i++ {
		buf.WriteString("\tCALL\tcallbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_amd64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmArm64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOV and B instructions.
// The MOV instruction loads R12 with the callback index, and the
// B instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R12 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB),NOSPLIT|NOFRAME,$0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVD\t$%d, R12\n", i)
		buf.WriteString("\tB\tcallbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_arm64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmLoong64() {
        var buf bytes.Buffer

        buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVV and JMP instructions.
// The MOVV instruction loads R12 with the callback index, and the
// JMP instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R12 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB),NOSPLIT|NOFRAME,$0
`)
        for i := 0; i < maxCallback; i++ {
                fmt.Fprintf(&buf, "\tMOVV\t$%d, R13\n", i)
                buf.WriteString("\tJMP\tcallbackasm1(SB)\n")
        }
        if err := os.WriteFile("zcallback_loong64.s", buf.Bytes(), 0644); err != nil {
                fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
                os.Exit(2)
        }
}

func main() {
	genasmAmd64()
	genasmArm64()
	genasmLoong64()
}
