#ifndef HAVE_ADNS_RR_H
#define HAVE_ADNS_RR_H

#include "domain_name.h"
#include "dns_charstring.h"
#include "utility.h"

using namespace System;
using namespace System::Collections::Generic;
using namespace System::Net;

namespace ADNS {

		/**
	 *  The different RR classes.
	 */
	public enum class RR_CLASS
	{
		/** the Internet */
		IN 	= 1,
		/** Chaos class */
		CH	= 3,
		/** Hesiod (Dyer 87) */
		HS	= 4,
		/** None class, dynamic update */
		NONE      = 254,
		/** Any class */
		ANY	= 255,

		FIRST     = 0,
		LAST      = 65535,
		COUNT     = 65536
	};

	/**
	 * The different RR types.
	 */
	public enum class RR_TYPE
	{
		/**  a host address */
		A = 1,
		/**  an authoritative name server */
		NS = 2,
		/**  a mail destination (Obsolete - use MX) */
		MD = 3,
		/**  a mail forwarder (Obsolete - use MX) */
		MF = 4,
		/**  the canonical name for an alias */
		CNAME = 5,
		/**  marks the start of a zone of authority */
		SOA = 6,
		/**  a mailbox domain name (EXPERIMENTAL) */
		MB = 7,
		/**  a mail group member (EXPERIMENTAL) */
		MG = 8,
		/**  a mail rename domain name (EXPERIMENTAL) */
		MR = 9,
		/**  a null RR (EXPERIMENTAL) */
		NULL = 10,
		/**  a well known service description */
		WKS = 11,
		/**  a domain name pointer */
		PTR = 12,
		/**  host information */
		HINFO = 13,
		/**  mailbox or mail list information */
		MINFO = 14,
		/**  mail exchange */
		MX = 15,
		/**  text strings */
		TXT = 16,
		/**  RFC1183 */
		RP = 17,
		/**  RFC1183 */
		AFSDB = 18,
		/**  RFC1183 */
		X25 = 19,
		/**  RFC1183 */
		ISDN = 20,
		/**  RFC1183 */
		RT = 21,
		/**  RFC1706 */
		NSAP = 22,
		/**  RFC1348 */
		NSAP_PTR = 23,
		/**  2535typecode */
		SIG = 24,
		/**  2535typecode */
		KEY = 25,
		/**  RFC2163 */
		PX = 26,
		/**  RFC1712 */
		GPOS = 27,
		/**  ipv6 address */
		AAAA = 28,
		/**  LOC record  RFC1876 */
		LOC = 29,
		/**  2535typecode */
		NXT = 30,
		/**  draft-ietf-nimrod-dns-01.txt */
		EID = 31,
		/**  draft-ietf-nimrod-dns-01.txt */
		NIMLOC = 32,
		/**  SRV record RFC2782 */
		SRV = 33,
		/**  http://www.jhsoft.com/rfc/af-saa-0069.000.rtf */
		ATMA = 34,
		/**  RFC2915 */
		NAPTR = 35,
		/**  RFC2230 */
		KX = 36,
		/**  RFC2538 */
		CERT = 37,
		/**  RFC2874 */
		A6 = 38,
		/**  RFC2672 */
		DNAME = 39,
		/**  dnsind-kitchen-sink-02.txt */
		SINK = 40,
		/**  Pseudo OPT record... */
		OPT = 41,
		/**  RFC3123 */
		APL = 42,
		/**  draft-ietf-dnsext-delegation */
		DS = 43,
		/**  SSH Key Fingerprint */
		SSHFP = 44,
		/**  draft-richardson-ipseckey-rr-11.txt */
		IPSECKEY = 45,
		/**  draft-ietf-dnsext-dnssec-25 */
		RRSIG = 46,
		NSEC = 47,
		DNSKEY = 48,
		DHCID = 49,

		NSEC3 = 50,
		NSEC3PARAMS = 51,

		SPF = 99,

		UINFO = 100,
		UID = 101,
		GID = 102,
		UNSPEC = 103,

		TSIG = 250,
		IXFR = 251,
		AXFR = 252,
		/**  A request for mailbox-related records (MB, MG or MR) */
		MAILB = 253,
		/**  A request for mail agent RRs (Obsolete - see MX) */
		MAILA = 254,
		/**  any type (wildcard) */
		ANY = 255,

		/* RFC 4431, 5074, DNSSEC Lookaside Validation */
		DLV = 32769,

		/* type codes from nsec3 experimental phase
		NSEC3 = 65324,
		NSEC3PARAMS = 65325, */
		FIRST = 0,
		LAST  = 65535,
		COUNT = LAST - FIRST + 1
	};

	/**
	 * Hashing algorithms used in the DS record
	 */

	public enum class HASH_ALGORITHM
	{
		SHA1               = 1,  /* RFC 4034 */
		SHA256             = 2,  /* RFC 4509 */
		GOST94	           = 203 /* not official */
	};

	public enum class CRYPTO_ALGORITHM
	{
		RSAMD5             = 1,   /* RFC 4034,4035 */
		DH                 = 2,
		DSA                = 3,
		ECC                = 4,
		RSASHA1            = 5,
		DSA_NSEC3          = 6,
		RSASHA1_NSEC3      = 7,
		RSASHA256          = 8,   /* RFC 5702 */
		RSASHA512          = 10,  /* RFC 5702 */
		GOST               = 249, /* not official */
		INDIRECT           = 252,
		PRIVATEDNS         = 253,
		PRIVATEOID         = 254
	};

	public enum class SIGN_ALGORITHM 
	{
		RSAMD5             = 1,   /* RFC 4034,4035 */
		RSASHA1            = 5,
		DSA                = 3,
		RSASHA1_NSEC3      = 7,
		RSASHA256          = 8,   /* RFC 5702 */
		RSASHA512          = 10,  /* RFC 5702 */
		DSA_NSEC3          = 6,
		GOST               = 249, /* not official */
		SIGN_HMACMD5	   = 157,	/* not official! This type is for TSIG, not DNSSEC */
		SIGN_HMACSHA1	   = 158,	/* not official! This type is for TSIG, not DNSSEC */
		SIGN_HMACSHA256    = 159  /* ditto */
	};

	/**
	 * algorithms used in CERT rrs
	 */
	public enum class CERT_ALGORITHM
	{
		CERT_PKIX		= 1,
		CERT_SPKI		= 2,
		CERT_PGP		= 3,
		CERT_URI		= 253,
		CERT_OID		= 254
	};

	/**
	 * Resource Record
	 *
	 * This is the basic DNS element that contains actual data
	 *
	 * From RFC1035:
	 * <pre>
	3.2.1. Format

	All RRs have the same top level format shown below:

										1  1  1  1  1  1
		  0  1  2  3  4  5  6  7  8  9  0  1  2  3  4  5
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                                               |
		/                                               /
		/                      NAME                     /
		|                                               |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                      TYPE                     |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                     CLASS                     |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                      TTL                      |
		|                                               |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
		|                   RDLENGTH                    |
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--|
		/                     RDATA                     /
		/                                               /
		+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+

	where:

	NAME            an owner name, i.e., the name of the node to which this
					resource record pertains.

	TYPE            two octets containing one of the RR TYPE codes.

	CLASS           two octets containing one of the RR CLASS codes.

	TTL             a 32 bit signed integer that specifies the time interval
					that the resource record may be cached before the source
					of the information should again be consulted.  Zero
					values are interpreted to mean that the RR can only be
					used for the transaction in progress, and should not be
					cached.  For example, SOA records are always distributed
					with a zero TTL to prohibit caching.  Zero values can
					also be used for extremely volatile data.

	RDLENGTH        an unsigned 16 bit integer that specifies the length in
					octets of the RDATA field.

	RDATA           a variable length string of octets that describes the
					resource.  The format of this information varies
					according to the TYPE and CLASS of the resource record.
	 * </pre>
	 *
	 * The actual amount and type of rdata fields depend on the RR type of the
	 * RR, and can be found by using \ref ldns_rr_descriptor functions.
	 */

		public ref class ResourceRecord
		{
		public:
			/**  Owner name, uncompressed */
			DOMAIN_NAME^ owner;
			/**  Time to live  */
			unsigned int ttl;
			/* Length of the rdata field */
			unsigned short int  rd_length;
			/**  the type of the RR. A, MX etc. */
			RR_TYPE	rr_type;
			/**  Class of the resource record.  */
			RR_CLASS rr_class;
			/* everything in the rdata is in network order */
			bool rr_question;
			/* everything in the rdata is in network order */
		protected:
			array<Byte>^ rdata;

			//Functions
		public:
			array<Byte>^ HeaderToWire();
			String^ PrintHeader();
			array<Byte>^ ToWire();
			bool RdataIsLess(ResourceRecord^ r1, ResourceRecord^ r2);
			static array<int>^ ResourceRecord::SortRRSetCanonicalByIndicies(List<int>^ indicies, List<ResourceRecord^>^ l);

			/* 

			The following methods are prototypes for functions that are required when defining new 
			classes of resource records.  These methods are registered with the Message class to provide support
			for parsing inbound packets and for diagnostic purposes.

			static ResourceRecord^ ParseResourceRecord(array<Byte>^ domainname, UInt16 rr_type, UInt16 rr_class, UInt32 ttl, UInt16 rdata_len, array<Byte>^ packet, int rdata_start);
			static String^ PrintRR(ResourceRecord^ rec);
			static ResourceRecord^ CloneRR(ResourceRecord^ rec);

			*/
			};

		//Other Functions
		String^ PrintResourceRecord(ResourceRecord^ rr);
		
}
#endif
