//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

#include "header.h"
#include "mystring.h"

// 本物の方を使ってしまうと芋づるでリンクが解決できないのでダミーを用意。
class WindrvDevice
{
 public:
	static bool MatchWildcard(const std::string&, const std::string&);
};
class Human68k
{
 public:
	static uint32 Unixtime2DateTime(time_t);
	static time_t DateTime2Unixtime(uint32);
};

#define TEST_WINDRV
#include "windrv.cpp"
#include "test_tool.cpp"

static void
test_MatchWildcard()
{
	struct {
		const char *pname;
		const char *fname;
		bool expected;
	} table[] = {
		// Pattern				Name					Expected
		{ "?.com",				"a.com",				true },
		{ "??.com",				"a.com",				true },
		{ "?.com",				"ab.com",				false },

		{ "??",					"a",					true },
		{ "??",					"ab",					true },
		{ "??",					"abc",					false },

		{ "?",					"a.s",					false },
		{ "???",				"a.s",					false },
		{ "?.?",				"a.s",					true },

		{ "futype.???",			"futype.x",				true },
		{ "futype.???",			"futype.xy",			true },
		{ "futype.???",			"futype.xyz",			true },
		{ "futype.???",			"futype",				true },
		{ "futype.???",			"abcdef.xxx",			false },
	};

	for (int i = 0; i < countof(table); i++) {
		std::string pname(table[i].pname);
		std::string fname(table[i].fname);
		bool expected = table[i].expected;
		char where[256];
		snprintf(where, sizeof(where), "\"%s\", \"%s\"",
			pname.c_str(), fname.c_str());

		bool actual = WindrvDevice::MatchWildcard(fname, pname);
		xp_eq(expected, actual, where);
	}
}

static struct {
	time_t unixtime;
	uint32 dostime;
} table_time[] = {
	{  315500400,	0x00210000 },	// 1980-01-01 00:00:00 (+0900 JST)
	{ 1702990056,	0x5793adf2 },	// 2023-12-19 21:47:36 (+0900 JST)
};

static void
test_Unixtime2DateTime()
{
	for (const auto& a : table_time) {
		time_t utime = a.unixtime;
		uint32 dtime = a.dostime;

		auto dtime_actual = Human68k::Unixtime2DateTime(utime);
		xp_eq(dtime, dtime_actual);
	}

	// 1980 年以前は 1980 年に張り付ける
	xp_eq(0x00210000, Human68k::Unixtime2DateTime(315446400));
}

static void
test_DateTime2Unixtime()
{
	for (const auto& a : table_time) {
		time_t utime = a.unixtime;
		uint32 dtime = a.dostime;

		auto utime_actual = Human68k::DateTime2Unixtime(dtime);
		xp_eq(utime, utime_actual);
	}
}

int
main(int ac, char *av[])
{
	test_MatchWildcard();
	test_Unixtime2DateTime();
	test_DateTime2Unixtime();
	return 0;
}
