//
// nono
// Copyright (C) 2025 nono project
// Licensed under nono-license.txt
//

//
// GVRAM モニタ
//

#include "wxgvrammonitor.h"
#include "wxtextscreen.h"
#include "gvram.h"
#include "videoctlr.h"

enum {
	ID_SCALE = IDGROUP_GVRAM,

	ID_local_end,	// 最後に置く (チェック用)
};
static_assert(ID_local_end - 1 <= (int)IDGROUP_GVRAM_END, "ID exceeded");

//
// グラフィック画面パネル
//

// コンストラクタ
WXGVRAMPanel::WXGVRAMPanel(wxWindow *parent)
	: inherited(parent)
{
	// デバイスを取得。
	gvram = GetGVRAMDevice();
	videoctlr = GetVideoCtlrDevice();

	// サイズはとりあえず 1024x1024。
	Ctor(1024, 1024);
}

// デストラクタ
WXGVRAMPanel::~WXGVRAMPanel()
{
}

// XXX 実際にはこの辺かどこかで更新チェックをしないといけない。
// 今は表示領域を毎回描画しているので無駄。

// ビットマッププレーンの内容を描画する。
void
WXGVRAMPanel::Draw()
{
	if (scale > 1) {
		// 拡大ならインデックス画像から作る。
		const BitmapI8& src = gvram->GetComposite();
		const Color *pal = videoctlr->GetHostPalettePtr(0);
		Rect dr(0, 0, bitmap.GetWidth(), bitmap.GetHeight());
		bitmap.DrawBitmapI8Scale(dr, src, pal,
			view.x, scale, 1,
			view.y, scale, 1);
	} else {
		// 等倍ならビデオコントローラに着色済み画面がある。
		const BitmapRGBX& src = videoctlr->GetGraphicBitmap();
		bitmap.DrawBitmap(0, 0, src, view);
	}
}


//
// グラフィック画面ウィンドウ
//

// イベントテーブル
wxBEGIN_EVENT_TABLE(WXGVRAMWindow, inherited)
	EVT_CHOICE(ID_SCALE, WXGVRAMWindow::OnScale)
wxEND_EVENT_TABLE()

// コンストラクタ
WXGVRAMWindow::WXGVRAMWindow(wxWindow *parent, const wxString& name)
	: inherited(parent, name)
{
	// デバイスを取得。
	gvram = GetGVRAMDevice();
	videoctlr = GetVideoCtlrDevice();

	// コントロールパネル用の横 Sizer
	auto *ctrlbox = new wxBoxSizer(wxHORIZONTAL);

	// 倍率
	wxString scale_choice[] = {
		"x1",
		"x2",
		"x4",
	};
	auto scalesw = new wxChoice(ctrlpanel, ID_SCALE,
		wxDefaultPosition, wxDefaultSize,
		countof(scale_choice), scale_choice);
	ctrlbox->Add(scalesw, 0, wxALIGN_CENTER | wxALL, 3);

	// sizer と下敷きパネルを紐付ける
	ctrlpanel->SetSizer(ctrlbox);
	ctrlbox->SetSizeHints(ctrlpanel);

	// 情報パネル。
	statuspanel = new WXTextScreen(this, nnSize(32, 1));
	statuspanel->SetMinSize(statuspanel->GetBestSize());

	// 表示パネル。
	// 大きさは適当だがフォントサイズを大きくした後で再び小さくすると
	// このパネルはどうしたらいいのか難しいのと、最初から広いほうが見やすい
	// というのもあるしね…ということにして、
	// テキスト画面モニタと同じくらいにしておく。
	constexpr int status_col = 60;
	viewpanel = new WXGVRAMPanel(this);
	viewpanel->SetSize(wxSize(status_col * 12, status_col * 12 / 2));

	// コンストラクタの続き。
	Ctor();

	// 初期値をセット。
	// コントロールへの Set*() ではイベントが飛ばないので、直接呼ぶ。
	scalesw->SetSelection(0);
	DoScale(scalesw->GetSelection());
}

// デストラクタ
WXGVRAMWindow::~WXGVRAMWindow()
{
}

// 情報パネル更新。
void
WXGVRAMWindow::UpdateInfo(TextScreen& screen, int x, int y)
{
	gvram->UpdateInfo(screen, x, y);
}
