##  Perl module which sets up any and all variables that an
##  INN Perl script might need.
##
##  Originally written by James Brister in 1996 as innshellvars.pl, and
##  converted to this INN::Config Perl module by Julien Élie in 2007.
##
##  Various bug fixes, code and documentation improvements since then
##  in 2007-2011, 2013, 2018, 2021, 2022, 2024.

package INN::Config;

use strict;
use warnings;
use Exporter;

# Variables dynamically declared from innconfval, and used in this package.
use vars qw(
    $ovmethod $patharchive $patharticles $pathbin $pathcontrol $pathdb
    $pathetc $pathfilter $pathincoming $pathlog $pathnews $pathoutgoing
    $pathoverview $pathrun $pathspool $pathtmp $runasuser $runasgroup
);

our @ISA = qw(Exporter);
our $VERSION = '2.7.3';

##  First, two necessary variables (listed below in @DIRVAR).
our $prefix = '/usr/pkg';
our $exec_prefix = "${prefix}";

##  Then, process the variables provided by innconfval.
my @INNCONFVAR = ();
my @values = qx(/usr/pkg/libexec/inn/innconfval -p);
foreach my $line (@values) {
    eval 'our ' . $line;
    if ($line =~ /^(.*?) = /m) {
        push(@INNCONFVAR, $1);
    }
}

##  Paths of useful directories.
my @DIRVAR = qw(
    $prefix $exec_prefix $newshome $home $newslib $newsbin $controlprogs
    $newslbin $newsetc $spooldir $archivedir $spool $incoming $spoolnews
    $badnews $batch $overviewdir $most_logs $locks $innddir $tmpdir
    $spooltemp $path
);
our $newshome = $pathnews;
our $home = $pathnews;
our $newslib = "${exec_prefix}/lib";
our $newsbin = $pathbin;
our $controlprogs = $pathcontrol;
our $newslbin = "$pathnews/local";
our $newsetc = $pathetc;
our $spooldir = $pathspool;
our $archivedir = $patharchive;
our $spool = $patharticles;
our $incoming = $pathincoming;
our $spoolnews = $pathincoming;
our $badnews = "$pathincoming/bad";
our $batch = $pathoutgoing;
our $overviewdir = $pathoverview;
our $most_logs = $pathlog;
our $locks = $pathrun;
our $innddir = $pathrun;
our $tmpdir = $pathtmp;
our $spooltemp = $pathtmp;
our $path = $ENV{'PATH'} || '';
$path = "${newslbin}:${newsbin}:${path}:/bin:/usr/bin";

##  Paths of useful files.
my @FILESVAR = qw(
    $active $activetimes $newactive $oldactive $history $newsgroups $ctlfile
    $ctlwatch $localgroups $newsfeeds $path_motd_innd $path_motd_nnrpd
    $path_motd $expirectl $errlog $log
);
our $active = "${pathdb}/active";
our $activetimes = "${pathdb}/active.times";
our $newactive = "${pathdb}/active.tmp";
our $oldactive = "${pathdb}/active.old";
our $history = "${pathdb}/history";
our $newsgroups = "${pathdb}/newsgroups";
our $ctlfile = "${newsetc}/control.ctl";
our $ctlwatch = "${newsetc}/innwatch.ctl";
our $localgroups = "${newsetc}/localgroups";
our $newsfeeds = "${newsetc}/newsfeeds";
our $path_motd_innd = "${newsetc}/motd.innd";
our $path_motd_nnrpd = "${newsetc}/motd.nnrpd";
our $path_motd = "${path_motd_nnrpd}";
our $expirectl = "${newsetc}/expire.ctl";
our $errlog = "${most_logs}/errlog";
our $log = "${most_logs}/news";

##  Paths of useful programs.
my @PROGVAR = qw(
    $inews $innconfval $innd $inndf $innwatch $rnews $perl_startup_innd
    $perl_filter_innd $perl_filter_nnrpd $python_filter_innd
    $path_python_inn_module $path_tcl_startup $path_tcl_filter
);
our $inews = "${newsbin}/inews";
our $innconfval = "${newsbin}/innconfval";
our $innd = "${newsbin}/innd";
our $inndf = "${newsbin}/inndf";
our $innwatch = "${newsbin}/innwatch";
our $rnews = "${newsbin}/rnews";
our $perl_startup_innd = "$pathfilter/startup_innd.pl";
our $perl_filter_innd = "$pathfilter/filter_innd.pl";
our $perl_filter_nnrpd = "$pathfilter/filter_nnrpd.pl";
our $python_filter_innd = "$pathfilter/filter_innd.py";
our $path_python_inn_module = "$pathfilter/INN.py";
our $path_tcl_startup = "$pathfilter/startup.tcl";
our $path_tcl_filter = "$pathfilter/filter.tcl";

##  Paths of lock/PID/status files.
my @LOCKVAR = qw(
    $daily $newscontrol $nntpconnect $serverpid $innwstatus $watchpid
    $tempsock $tempsockdir
);
our $daily = "${locks}/LOCK.news.daily";
our $newscontrol = "${innddir}/control";
our $nntpconnect = "${innddir}/nntpin";
our $serverpid = "${innddir}/innd.pid";
our $innwstatus = "${innddir}/innwatch.status";
our $watchpid = "${innddir}/innwatch.pid";
(our $tempsock = "${innddir}/ctlinndXXXXXX") =~ s!.*/(.*)XXXXXX$!$1*!;
(our $tempsockdir = "${innddir}/ctlinndXXXXXX") =~ s!/[^/]*$!!;

##  Paths of external used programs.
my @EXTPROGVAR = qw(
    $awk $egrep $gpgv $gpg $perl $pgp $sed $sort $getftp $gethttp $uustat
    $uux
);
our $awk = '/usr/bin/awk';
our $egrep = '/usr/bin/egrep';
our $gpgv = '/usr/pkg/bin/gpgv2';
our $gpg = '/usr/pkg/bin/gpg2';
our $perl = '/usr/pkg/bin/perl';
our $pgp = '';
our $sed = '/usr/bin/sed';
our $sort = '/usr/bin/sort';
our $getftp = "/usr/pkg/libexec/inn/simpleftp";
our $gethttp = "/usr/pkg/libexec/inn/simpleftp";
our $uustat = '';
our $uux = 'uux';

##  Paths of external used compressors.
my @EXTCOMPVAR = qw($bzip2 $compress $gzip $uncompress $log_compress $z);
our $bzip2 = '/usr/bin/bzip2';
our $compress = '/usr/bin/compress';
our $gzip = '/usr/bin/gzip';
our $uncompress = '/usr/bin/gzip -d';
our $log_compress = '/usr/bin/gzip';
our $z = '.gz';

##  Set up some useful system file information.
my @SYSVAR = qw(
    $newsmaster $newsuser $newsgroup $filemode $inewsmode $rnewsmode $umask
    $syslog_facility
);
our $newsmaster = 'usenet';
our $newsuser = ${runasuser};
our $newsgroup = ${runasgroup};
#<<V  code skipping because perltidy expects numbers.
our $filemode = 0664;    # It is a number.
our $inewsmode = 02555;
our $rnewsmode = 0500;
our $umask = 02;
#>>V
our $syslog_facility = lc('LOG_NEWS');
$syslog_facility =~ s/log_//;

##  Set up some useful parameters.
my @PARAMVAR = qw($do_dbz_tagged_hash $gpg_has_allow_weak_digest_algos_flag);
our $do_dbz_tagged_hash = 'DONT';
#<<V  code skipping because perltidy expects numbers.
our $gpg_has_allow_weak_digest_algos_flag
  = 1;
#>>V

##  Set up some environment values.
if ($ovmethod && $ovmethod eq "ovdb") {
    $ENV{'DB_HOME'} = $pathoverview;
}
$ENV{'INNDDIR'} = $innddir;
$ENV{'LOCKS'} = $locks;
$ENV{'MOST_LOGS'} = $most_logs;
$ENV{'NEWSBIN'} = $newsbin;
$ENV{'NEWSETC'} = $newsetc;
$ENV{'NEWSHOME'} = $pathnews;
$ENV{'NEWSLBIN'} = ${newslbin};
$ENV{'NEWSLIB'} = $newslib;
$ENV{'PATH'} = $path;
$ENV{'SPOOLDIR'} = $spooldir;
$ENV{'TMPDIR'} = $pathtmp;

##  Set up the locale.
$ENV{'LC_CTYPE'} = "C";

##  Set up umask.
#<<V  code skipping because perltidy expects a number.
umask 02;
#>>V

##  This array will contain what it is possible to export.
our @EXPORT_OK = qw(
    @INNCONFVAR @DIRVAR @FILESVAR @PROGVAR @LOCKVAR @EXTPROGVAR @EXTCOMPVAR
    @SYSVAR @PARAMVAR $VERSION
);

##  Load another script that can override or add variables.
if (-x "${pathetc}/innshellvars.pl.local") {
    do "${pathetc}/innshellvars.pl.local";
}

##  That's all.
1;

__END__

=encoding utf-8

=head1 NAME

INN::Config - Export all the variables an INN Perl script might need

=head1 DESCRIPTION

This Perl module sets up any and all the variables that an INN Perl script
might need.  More particularly, it allows using F<inn.conf> variables:
they are all provided by B<innconfval>, as well as the version of INN
(in the variable C<$INN::Config::VERSION> for its short number form, on in
C<$INN::Config::version> for its complete form).  Other useful variables
are also provided (directories, files, programs, masks, parameters).
The complete list can be obtained with the following script that prints
them out:

    use lib '<pathnews>/lib/perl';
    use INN::Config;
    use Data::Dumper;

    my ($varname, $value);
    foreach my $var (@INN::Config::EXPORT_OK) {
        if ($var =~ /^\$(.*)$/) {
            $varname = "INN::Config::$1";
            $value = Dumper($$varname);
            $value =~ s/^\$VAR1 = //;
            print "\$$varname = $value";
        } elsif ($var =~ /^\@(.*)$/) {
            $varname = "INN::Config::$1";
            $value = Dumper(\@$varname);
            $value =~ s/^\$VAR1 = //;
            print "\@$varname = $value";
        }
    }

A local Perl script named F<innshellvars.pl.local> in I<pathetc> will be
loaded, if present and executable, at the end of the run of this module.
A typical use is to add or override variables.

You only have to declare the module at the beginning of them:

    use lib '<pathnews>/lib/perl';
    use INN::Config;

Then, you can for instance use:

    print $INN::Config::localmaxartsize;

to print the value of I<localmaxartsize> as it is set in F<inn.conf>.

You can also specify a version when you import the module.  If
you write:

    use INN::Config 2.5.0;

only versions of INN superior to 2.5.0 will be able to run the Perl
script.

It is also possible to import the variables directly in your namespace
if you specify what you want to import:

    use INN::Config qw($localmaxartsize $pathbin);

Note that a legacy F<innshellvars.pl> is also provided in I<pathnews>/lib
for compatibility reasons with old Perl scripts not shipped with INN.
It was used by versions of INN anterior to 2.5.0.  The corresponding
scripts for Shell and Tcl are, however, still in use: F<innshellvars>
and F<innshellvars.tcl>.  They offer the same capabilities as this
module.

=head1 HISTORY

F<innshellvars.pl> was written by James Brister <brister@vix.com>
for InterNetNews in 1996.  It was converted to the INN::Config Perl
module by Julien Élie in 2007.

=head1 SEE ALSO

inn.conf(5), innconfval(1), perl(1).

=cut
