#!/bin/sh

# Copyright (C) 2024-2025 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr/pkg"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "XXX-tests-builddir-XXX/src/share/database/scripts/pgsql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "XXX-tests-builddir-XXX/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check only major version to allow for intermediary backported schema changes.
version=$(pgsql_version "${@}" | cut -d '.' -f 1)
if test "${version}" != '23'; then
    printf 'This script upgrades 23.* to 24.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

psql "$@" >/dev/null <<EOF
START TRANSACTION;

-- This line starts the schema upgrade to version 24.0.

SELECT set_config('kea.disable_audit', 'true', false);

DELETE FROM dhcp4_global_parameter WHERE name='control-socket.socket-name';
DELETE FROM dhcp4_global_parameter WHERE name='control-socket.socket-type';

DELETE FROM dhcp6_global_parameter WHERE name='control-socket.socket-name';
DELETE FROM dhcp6_global_parameter WHERE name='control-socket.socket-type';

SELECT set_config('kea.disable_audit', 'false', false);

-- Create a function to conditionally migrate option_def data type
-- values.  If they are updating from 2.6.1 this has been done already
-- and we don't want to do it twice.
DROP FUNCTION IF EXISTS updateOptionDataDef();
CREATE OR REPLACE FUNCTION updateOptionDataDef() RETURNS text AS \$\$
    DECLARE skipper BOOLEAN;
BEGIN
    SELECT exists(SELECT FROM pg_tables WHERE tablename = 'option_def_data_type') INTO skipper;
    IF skipper THEN
        -- Table already exists which means the migration was already done.
        RETURN 'EXISTS';
    END IF;

    -- First we migrate existing OPT_RECORD_TYPE values.
    PERFORM set_config('kea.disable_audit', 'true', false);
    UPDATE dhcp4_option_def SET type = 254 WHERE record_types IS NOT NULL;
    UPDATE dhcp6_option_def SET type = 254 WHERE record_types IS NOT NULL;

    -- Create the table that enumerates option definition data types.
    CREATE TABLE option_def_data_type (
        id smallint NOT NULL,
        name VARCHAR(32) NOT NULL,
        PRIMARY KEY (id)
    );

    -- Now insert supported types.
    -- We skip (9, 'any-address') as it is not externally supported.
    INSERT INTO option_def_data_type VALUES
        (0, 'empty'),
        (1, 'binary'),
        (2, 'boolean'),
        (3, 'int8"'),
        (4, 'int16'),
        (5, 'int32'),
        (6, 'uint8'),
        (7, 'uint16'),
        (8, 'uint32'),
        (10, 'ipv4-address'),
        (11, 'ipv6-address'),
        (12, 'ipv6-prefix'),
        (13, 'psid'),
        (14, 'string'),
        (15, 'tuple'),
        (16, 'fqdn'),
        (17, 'internal'),
        (254, 'record');

    --  Add foreign key constraints to enforce only valid types.
    ALTER TABLE dhcp4_option_def
        ADD CONSTRAINT fk_option_def_data_type4 FOREIGN KEY (type) REFERENCES option_def_data_type(id);

    ALTER TABLE dhcp6_option_def
        ADD CONSTRAINT fk_option_def_data_type6 FOREIGN KEY (type) REFERENCES option_def_data_type(id);

    PERFORM set_config('kea.disable_audit', 'false', false);
    RETURN 'UPDATED';
END;
\$\$ LANGUAGE plpgsql;

SELECT updateOptionDataDef();

-- Get rid of the now obsolete function.
DROP FUNCTION IF EXISTS updateOptionDataDef();

UPDATE schema_version
    SET version = '24', minor = '0';

-- This line concludes the schema upgrade to version 24.0.

-- Commit the script transaction.
COMMIT;

EOF
