<?php

declare(strict_types=1);

/*
 * This file is part of the TYPO3 CMS project.
 *
 * It is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, either version 2
 * of the License, or any later version.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 *
 * The TYPO3 project - inspiring people to share!
 */

namespace TYPO3\CMS\Linkvalidator\Linktype;

use TYPO3\CMS\Core\Localization\LanguageService;

/**
 * This class provides Check Base plugin implementation
 */
abstract class AbstractLinktype implements LinktypeInterface, LabelledLinktypeInterface
{
    /**
     * Contains parameters needed for the rendering of the error message
     *
     * @var array
     */
    protected array $errorParams = [];
    protected string $identifier = '';

    public function getIdentifier(): string
    {
        return $this->identifier;
    }

    /**
     * Function to override config of Linktype. Should be used only
     * if necessary. Add additional configuration to TSconfig.
     */
    public function setAdditionalConfig(array $config): void
    {
        // do nothing by default
    }

    /**
     * Base type fetching method, based on the type that softRefParserObj returns
     *
     * @param array $value Reference properties
     * @param string $type Current type
     * @param string $key Validator hook name
     * @return string Fetched type
     */
    public function fetchType(array $value, string $type, string $key): string
    {
        if (($value['type'] ?? false) == $key) {
            $type = $value['type'];
        }
        return $type;
    }

    /**
     * Set the value of the protected property errorParams
     *
     * @param array $value All parameters needed for the rendering of the error message
     */
    protected function setErrorParams($value)
    {
        $this->errorParams = $value;
    }

    /**
     * Get the value of the private property errorParams
     *
     * @return array All parameters needed for the rendering of the error message
     */
    public function getErrorParams(): array
    {
        return $this->errorParams;
    }

    /**
     * Construct a valid Url for browser output
     *
     * @param array $row Broken link record
     * @return string Parsed broken url
     */
    public function getBrokenUrl(array $row): string
    {
        return $row['url'];
    }

    protected function getLanguageService(): LanguageService
    {
        return $GLOBALS['LANG'];
    }

    /**
     * Get localized label for this linktype to be displayed in Backend user interface.
     * Custom Linktypes should override this and provide language labels for their type.
     */
    public function getReadableName(): string
    {
        $type = $this->getIdentifier();
        return $this->getLanguageService()->sL('LLL:EXT:linkvalidator/Resources/Private/Language/Module/locallang.xlf:hooks.' . $type) ?: $type;
    }
}
