=begin  -*-mode: ruby-*-
  Sytab.rb

  Copyright (c) 2002, Alan Eldridge
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions 
  are met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

  * Neither the name of the copyright owner nor the names of its
  contributors may be used to endorse or promote products derived
  from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.


  2002/12/02 alane@geeksrus.net

=end


######################################################################
# Symbol table that provides for one-pass replacement of all 
# symbols surrounded by delimiters with the symbol values
class SyTab < Hash

  ####################################################################
  # [delims]	pre- and post-name delimiters for symbol dereferencing,
  #		separated by whitespace (e.g., <tt>"$ $"</tt>). 
  #		A symbol not bracketed by these	will not be substituted.
  def initialize(delims)
    darr = delims.split
    @mpos = darr[0].size
    @mend = -(darr[1].size)
    @delims = darr.map { 
      |s| Regexp.escape(s)
    }
    @regex = nil
    @var_restr = [ @delims[0], "[a-zA-Z_][a-zA-Z_0-9]*", @delims[1] ].join
    @var_regex = Regexp.new(@var_restr)
  end # initialize(delims)

  attr_reader :delims, :var_restr

  ####################################################################
  # Call Hash#delete and invalidate match expression.
  def delete(k)
    super(k)
    @regex = nil
  end # delete(k)

  ####################################################################
  # Call Array#[]= and invalidate match expression.
  def []=(k,v)
    super(k,v)
    @regex = nil
  end # []=(k,v)

  ####################################################################
  # Call Array#store and invalidate match expression.
  def store(k,v)
    super(k,v)
    @regex = nil
  end # store(k,v)

  ####################################################################
  # Build a new match expression consisting of each key of the
  # hash surrounded by delimiters, then grouped with parens, and
  # each group separated by "|".
  def compile!()
    @regex = Regexp.new(keys.map { 
      |k| "(" + @delims[0] + k + @delims[1] + ")"
    }.join("|"))
  end # compile!()

  ####################################################################
  # Return a string with each occurrence of a key surrounded by
  # delimiters replaced with the lookup value for the key.
  def replace(s)
    if keys.size < 1
      s
    else
      compile! unless @regex
      s.gsub!(@regex) {
	|match|	fetch(match[@mpos...@mend])
      }
      s.gsub(@var_regex, "*UNDEFINED*")
    end
  end # replace(s)

  ####################################################################
  # Shorthand for replace(s).
  def <<(s)
    replace(s)
  end # <<(s)
end # class SyTab
#EOF
