package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````F!8``#0````PE0````0`!30`(``'
M`"@`'0`<``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````6(\`
M`%B/```%`````!````$```!8CP``6)\``%B?``"T`@``+`,```8`````$```
M`@```&2/``!DGP``9)\``/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&2(B0``B(D``(B)```$`0``!`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$````JY!!!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-FAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"8%@````````,`"P``````8)\````````#`!$`AP``````````````
M$@````$``````````````!(````)```````````````2````<P,``&`A```(
M````$@`+`!$``````````````"(````@```````````````2````_@(`````
M````````$@```(0#``!`H@```````!``%@`F```````````````2````D@,`
M`"0L```8````$@`+`"T``````````````!(````+`P`````````````2````
M-0``````````````$@```*T#``"@-@``6````!(`"P!X```````````````2
M````P0,``(2B````````$``6`#T``````````````!(```"Y`P``#*(`````
M```0`!0`30``````````````$@```%P``````````````!(```!B````````
M```````2````:0``````````````$@```',``````````````!(```!]````
M```````````2````P`,``(2B````````$``6`(8``````````````!(```!%
M`@`````````````2````C```````````````$@```),``````````````!(`
M``"A```````````````2````L@``````````````$0```+\`````````````
M`!(```#,`P``V",``"0````2``L`V0,``/PC``!0````$@`+`.4#``!8+0``
MC`8``!(`"P#-```````````````6````X```````````````$@```!T#````
M`````````!(```#K```````````````2````]````("B```$````$0`6`/P`
M`````````````!(````%`0`````````````2````]`(`````````````$@``
M`!$!`````````````!(````8`0`````````````2````\@,``-A"```$`0``
M$@`+`"L!`````````````!$````]`0`````````````2````30$`````````
M````$@```%L!`````````````!(```!@`0`````````````1````;P$`````
M````````$@```/T#```(H@``!````!$`%`!^`0`````````````2````B@$`
M````````````$@```)\!`````````````!(```"J`0`````````````2````
M;@,``(2B````````$``6`,`"`````````````!(```"]`0`````````````2
M````0`0``)@6````````$@`+`,L!`````````````!(```#3`0``````````
M```B````ZP$`````````````$@```/<!`````````````!(````'!```A*(`
M```````0`!8`#P0``$PD```X!0``$@`+`/T!`````````````"(````3`@``
M```````````2````'@(`````````````$@```!P$``#<0P``0`$``!(`"P`O
M`@`````````````2````)@0``$"B````````$``6`#($``"`(```:````!(`
M"P`Z!```[$4``,0"```2``L`-@(`````````````$@```#\$``"8%@``````
M`!(`"P`]`@`````````````2````1P0``&@A``!P`@``$@`+`%0$```\+```
M'`$``!(`"P!$`@`````````````2````9@0``(0I``"4````$@`+`$T"````
M`````````!(```!>`@`````````````2````N0(`````````````$@```&4"
M`````````````!(```!T!```&"H``&0````2``L`=`(`````````````$@``
M`'T"`````````````!(```"#`@`````````````2````YP(`````````````
M$@```(@"`````````````!(```#:`@``1*(```0````1`!8`D`(`````````
M````$@```!(#`````````````!(```"7`@`````````````2````?`0``'A"
M``!@````$@`+`)X"`````````````!(```"M`@`````````````1````SP(`
M`,2A```$````$0`4`(4$``#D,P``O`(``!(`"P`D`0`````````````2````
ML@(`````````````$@```"0#`````````````!(`````<F5A;&QO8P!M96UM
M;W9E`%]?8WAA7V9I;F%L:7IE`')M9&ER`'-T<F-H<@!S=')N8VUP`'-T<G)C
M:'(`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?8V]N<W1R=6-T`&%B;W)T`'5N;&EN
M:P!?7VQS=&%T-3``<&5R;%]F<F5E`%]?<W1A=#4P`%]E>&ET`&UE;6-P>0!P
M97)L7V1E<W1R=6-T`%]?<W1A8VM?8VAK7V9A:6P`4$Q?9&]?=6YD=6UP`%!E
M<FQ?<W9?<V5T<'8`4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`8VQO
M<V5D:7(`96YV:7)O;@!R96%D;&EN:P!?7V]P96YD:7(S,`!M96US970`4&5R
M;%]S869E<WES;6%L;&]C`%]?<W1A8VM?8VAK7V=U87)D`%!E<FQ?9W9?9F5T
M8VAP=@!?7W-P<FEN=&9?8VAK`')E860`4$Q?;65M;W)Y7W=R87``4&5R;%]S
M>7-?:6YI=#,`4&5R;%]G971?<W8`4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R
M;%]N97=84P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]S>7-?=&5R;0!?7V5R
M<FYO`%]?9&5R96=I<W1E<E]F<F%M95]I;F9O`%!E<FQ?;6=?<V5T`&UK9&ER
M`%]?<F5G:7-T97)?9G)A;65?:6YF;P!P97)L7W!A<G-E`%!E<FQ?;F5W6%-?
M9FQA9W,`86-C97-S`&=E='!I9`!M96UC:'(`=F9P<FEN=&8`4&5R;%]A=&9O
M<FM?;&]C:P!M96UC;7``4&5R;%]S=E]S971P=FX`<&5R;%]R=6X`;'-E96L`
M;W!E;@!S<')I;G1F`&=E='5I9`!S=')L96X`7U]T;'-?9V5T7V%D9'(`7U]S
M1@!S=')C;7``<W1R9'5P`'!T:')E861?871F;W)K`%]?<')O9VYA;64`7U]P
M<U]S=')I;F=S`%]?9V5T<'=U:60U,`!?7W-Y<V-A;&P`7U]S='!C<'E?8VAK
M`&%T97AI=`!?;&EB8U]I;FET`'-T<G1O:P!?7W)E861D:7(S,`!L:6)P97)L
M+G-O`&QI8FTN<V\N,`!L:6)C<GEP="YS;RXQ`&QI8G!T:')E860N<V\N,0!L
M:6)C+G-O+C$R`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,BXP+V5A<FUV-FAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`6)\``!<```!<GP``%P```&BA```7
M````>*$``!<```"(H0``%P```)2A```7````G*$``!<```"LH0``%P```+2A
M```7````O*$``!<```#`H0``%P```,2A```7````R*$``!<```#8H0``%P``
M`-RA```7````Y*$``!<```#HH0``%P```.RA```7````\*$``!<```#TH0``
M%P```/RA```7`````*(``!<````(H@``%P```&RA```5!P``=*$``!43``!\
MH0``%2$``("A```1)@``A*$``!(F``"0H0``%3$``)BA```5-0``H*$``!4[
M``"DH0``%4$``*BA```51@``L*$``!55``"XH0``%64``&"@```6`P``9*``
M`!8$``!HH```%@4``&R@```6!P``<*```!8(``!TH```%@D``'B@```6"P``
M?*```!8-``"`H```%@X``(2@```6#P``B*```!81``",H```%A4``)"@```6
M%@``E*```!87``"8H```%A@``)R@```6&0``H*```!8:``"DH```%AP``*B@
M```6'0``K*```!8>``"PH```%A\``+2@```6(```N*```!8B``"\H```%B<`
M`,"@```6*```Q*```!8I``#(H```%BL``,R@```6+```T*```!8M``#4H```
M%BX``-B@```6+P``W*```!8R``#@H```%C,``.2@```6-```Z*```!8V``#L
MH```%C@``/"@```6.0``]*```!8Z``#XH```%CT``/R@```6/@```*$``!9`
M```$H0``%D$```BA```60@``#*$``!9#```0H0``%D8``!2A```61P``&*$`
M`!9(```<H0``%DH``""A```63@``)*$``!90```HH0``%E,``"RA```65@``
M,*$``!97```TH0``%E@``#BA```66@``/*$``!9;``!`H0``%EP``$2A```6
M70``2*$``!9>``!,H0``%F```%"A```680``5*$``!9B``!8H0``%F0``%RA
M```6:```8*$``!9I``!DH0``%FH```3@+>4$X)_E#N"/X`CPON78C````,:/
MX@C*C.+8_+SE`,:/X@C*C.+0_+SE`,:/X@C*C.+(_+SE`,:/X@C*C.+`_+SE
M`,:/X@C*C.*X_+SE`,:/X@C*C.*P_+SE`,:/X@C*C.*H_+SE`,:/X@C*C.*@
M_+SE`,:/X@C*C.*8_+SE`,:/X@C*C.*0_+SE`,:/X@C*C.*(_+SE`,:/X@C*
MC.*`_+SE`,:/X@C*C.)X_+SE`,:/X@C*C.)P_+SE`,:/X@C*C.)H_+SE`,:/
MX@C*C.)@_+SE`,:/X@C*C.)8_+SE`,:/X@C*C.)0_+SE`,:/X@C*C.)(_+SE
M`,:/X@C*C.)`_+SE`,:/X@C*C.(X_+SE`,:/X@C*C.(P_+SE`,:/X@C*C.(H
M_+SE`,:/X@C*C.(@_+SE`,:/X@C*C.(8_+SE`,:/X@C*C.(0_+SE`,:/X@C*
MC.((_+SE`,:/X@C*C.(`_+SE`,:/X@C*C.+X^[SE`,:/X@C*C.+P^[SE`,:/
MX@C*C.+H^[SE`,:/X@C*C.+@^[SE`,:/X@C*C.+8^[SE`,:/X@C*C.+0^[SE
M`,:/X@C*C.+(^[SE`,:/X@C*C.+`^[SE`,:/X@C*C.*X^[SE`,:/X@C*C.*P
M^[SE`,:/X@C*C.*H^[SE`,:/X@C*C.*@^[SE`,:/X@C*C.*8^[SE`,:/X@C*
MC.*0^[SE`,:/X@C*C.*(^[SE`,:/X@C*C.*`^[SE`,:/X@C*C.)X^[SE`,:/
MX@C*C.)P^[SE`,:/X@C*C.)H^[SE`,:/X@C*C.)@^[SE`,:/X@C*C.)8^[SE
M`,:/X@C*C.)0^[SE`,:/X@C*C.)(^[SE`,:/X@C*C.)`^[SE`,:/X@C*C.(X
M^[SE`,:/X@C*C.(P^[SE`,:/X@C*C.(H^[SE`,:/X@C*C.(@^[SE`,:/X@C*
MC.(8^[SE`,:/X@C*C.(0^[SE`,:/X@C*C.((^[SE`,:/X@C*C.(`^[SE`,:/
MX@C*C.+X^KSE`,:/X@C*C.+P^KSE`,:/X@C*C.+H^KSE`,:/X@C*C.+@^KSE
M`,:/X@C*C.+8^KSE`,:/X@C*C.+0^KSE`!"@X0(`H.$'T,WC-```ZG!`+>E$
M,)_E1""?Y0,PC^!`$)_E`F"3YP$0C^`&`%'A<("]*`!`H.,!4*#A`6!&X@%@
M1N`F8:#A!#"5Y#/_+^$&`%3A`4"$XOK__QIP@+WHF(D``!0!``"4B```\$$M
MZ6`PG^5@$)_E8""?Y0,PC^`!4)/G`G"3YP<`5>'P@;TH`7!'X@5P1^``0*#C
M"%"%XJ=QH.$'``#J"&`5Y0A0A>(`,);E,_\OX0<`5.$``(;E`4"$XO"!O0@$
M,%7EH`!3X_3__PHG___K.(D``#@!```<`0``@$`MZ6R!G^4`4%'B"("/X`C0
M3>)+```*7"&?Y5PQG^4"()CG`)"@X0!0@N4#8)CG`#"5Y0`@D^4(,)7E``!2
MXP`PAN4X,9_E.0``"@,0F.<`(('E`##2Y0``4^,%```*`2""XB\`4^,`(($%
M`3#2Y```4^/Z__\:E?__ZP0QG^4#,)CG``!3XS@```KX,)_E^*"?Y0-PF.<*
MH(_@!P!:X0%P1S(*<$<P`$"@,R=QH#$$```J!#":Y#/_+^$'`%3A`4"$XOK_
M_QH``%GC`0``"@D`H.'B_O_KM`"?Y;1PG^4``(_@WO[_ZZPPG^4'<(_@`X"8
MYP@`5^$!@$@R!X!(,`!`H#,H@:`Q!```*@0PE^0S_R_A!`!8X0%`A.+Z__\:
M`""6Y0`0E>4$`)7E3PL`Z[/^_^MD()_E`S"8YP(@C^``((/ER___ZA/`H.-0
M,)_E`B"@XP0`H.,`$*#C`S"/X`#`C>7Z_O_K`0"@X]?^_^N$___KQ/__ZL2(
M``!@`0``-`$``&@!``!8`0``)`$``#R'``!$_O__Z(8``$`!``"`B0``T"\`
M`,PPG^7P1RWI`(!0X@,PC^`K```*O""?Y0(PD^<`D)/E``!9XR@```H`4-CE
M`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"9Y0``5.,,```*"6"@X00`H.$%
M(*#A"!"@X9+^_^L``%#C`@``&@4PU.<]`%/C"P``"@1`MN4``%3C\___&@0`
MH.'PA[WH"%"@X0$P]>4``%/C/0!3$_O__QH(4$7@YO__Z@E@1N`!4(7B1F&@
MX05`A.`$`*#A`&"'Y?"'O>@(0*#A[?__Z@E`H.'K___J'(<``#0!```00"WI
MC""?Y8PPG^4"((_@H-!-X@,PDN<-$*#A`#"3Y9PPC>4`,*#C`$"@X7W^_^L`
M`%#C``"@$P0``!H(,)WE#SH#X@H*4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,P
MDN<`()/EG#"=Y0(@,^``,*#C!P``&J#0C>(0@+WH`A"@XP0`H.'*_O_K$`]O
MX:`"H.'N___J>/[_ZT"&```\`0``](4``/!`+>E4(9_E5#&?Y0(@C^`#,)+G
MI-!-X@!04.(`,)/EG#"-Y0`PH.,"```*`##5Y0``4^,*```:*"&?Y2`QG^4"
M((_@`S"2YP`@D^6<,)WE`B`SX``PH.,_```:I-"-XO"`O>AM_O_K`$!0XO'_
M_PH$`*#AV_[_ZP``4.,H```*#3#0Y0U@@.(N`%/C`@``&@$PUN4``%/C]/__
M"@TPT.4N`%/C(P``"@4`H.'!_O_K`'"@X08`H.&^_O_K``"'X`(`@.+!_O_K
M!C"@X0!@H.&4$)_E!2"@X0$0C^"L_O_K#1"@X08`H.$K_O_K`0!PXP8`H.$#
M```*"#"=Y0\Z`^(!"5/C$@``"AK^_^L&`*#A#_[_ZP0`H.&R_O_K``!0X];_
M_QH$`*#A-O[_ZP4`H.'U_?_KO___Z@$PUN4N`%/CV/__&@(PUN4``%/CQ___
M"M3__^JI___KZ___ZAW^_^N<A0``/`$``&B%``!`+0``$$`MZ?3`G^7T,)_E
M`=M-X@C03>(,P(_@`S"<YP1`C>(`$*#A`2N@XP0`H.$`,)/E!#2-Y0`PH./:
M_?_K`0!`X@0`4.$'``":`""@XP(``.H$`%#A`2#`Y0(```H!,%#D+P!3X_G_
M_PHO$*#C!`"@X=C]_^L``%#C$```"@0`4.$`,*"#`##`A00`H.%2_O_K<""?
MY6@PG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,1```:`=N-X@C0C>(0@+WH
M1""?Y3@PG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,%```:)`"?Y0``C^`!
MVXWB"-"-XA!`O>@W_O_JV?W_ZR2$```\`0``H(,``'"#``#(*P``\$`MZ?C@
MG^4`<*#A]`"?Y0[@C^```)[G`=M-X@S03>(``)#E!`2-Y0``H./8`)_E!&"-
MX@%0H.'0()_E!A"@X0``C^#1_?_K`$!0X@@``-H!*X3B!R""X@(@C>`$)%+E
M`3!$X@``4N,5```:``!3XQ(``!J8`)_E``"/X"_^_^L%(*#A!Q"@X0``D.6S
M_?_K@""?Y6PPG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,2```:`=N-X@S0
MC>+P@+WH`T"@X00@H.$O$*#C!@"@X?/]_^L``%#CY?__"C@`G^4``(_@%?[_
MZP0PH.$&(*#A!Q"@X0``D.7U_?_KX___ZI/]_^L4@P``/`$``&`K``#_`P``
MV(,``(R"``!P@P``\$<MZ0%`H.$0(I_E$#*?Y0(@C^`#,)+G"-!-X@`PD^4$
M,(WE`#"@XP`PT>4`4*#A/0!3XP%`@0($`*#A]/W_ZPT0H.$`8*#A!0"@X:+^
M_^O0<9_E`)!0X@=PC^!````*Z_W_ZP``5N$W``":N#&?Y0"`G>4#<)?G"(&@
MX0`PU>4``%/C/0!3$P$PH!,`,*`#!0``"@4PH.$!(//E``!2XST`4A/[__\:
M!3!#X`(`AN(#`(#@`&"7Y=O]_^L``%#C"`"&YQ4```H%`*#A`#"7Y0@0D^<!
M,-#D/0!3XP``4Q,`,,'E!@``"@$@@>(!,-#D`A"@X3T`4^,``%,3`3#"Y/G_
M_QH],*#C`2"@X0`PP>4!,-3D``!3XP$PXN7[__\:""&?Y?@PG^4"((_@`S"2
MYP`@D^4$,)WE`B`SX``PH.,U```:"-"-XO"'O>@!D$GB`3#4Y```4^,!,.GE
M^___&N[__^K`,)_E`W"7YP"`E^4`,)CE``!3XR0```H(,*#A!""SY0D0H.$`
M`%+C`9")XOK__QH#$('B`1&@X9`PG^4#,(_@`""3Y0``4N,/```:`2"@XP$`
MH.$`((/EFOW_ZP"@4.+5__\*"!"@X0F!H.$((*#AW?S_ZP"@A^4`,*#C`)"-
MY0&0B>(),8KGHO__Z@@`H.'2_/_K``!0XP"@H.$``(?EQ?__"@F!H.'R___J
M`Y"@X0@0H./?___J!/W_Z_2!```\`0``K($``#0!``#<@```@((``#!`+>E(
M()_E2#"?Y0(@C^!$`)_E`S"2YPS03>(\4)_E/$"?Y0``C^`$,(WE;?W_ZP50
MC^`$0(_@!3"@X000H.$$()WE``"0Y0S0C>(P0+WH&?W_ZL!_```@`0``U(``
M`+QH```H*```,$`MZ5@`G^44T$WB``"/X%K]_^L`0*#A``"0Y=W__^L`P*#C
M/%"?Y3PPG^4%4(_@`S"5YS00G^4,,(WE,#"?Y0``E.4#,(_@#""=Y0`PC>4!
M$(_@!,"-Y1C]_^L4T(WB,("]Z(2````X?P``;`$``,`G``"<*@````"@XQ[_
M+^'P1RWI-,*?Y30BG^5+WDWB"-!-X@S`C^``8*#A)`*?Y0(@G.<8@(WB`4"@
MX0``C^`$$$CB`""2Y;0DC>4`(*#CW?W_ZR\0H.,`4*#A!@"@X8+\_^L``%#C
M$```"N@!G^4&$*#A``"/X!S__^L&`*#AV"&?Y<@QG^4"((_@`S"2YP`@D^6T
M-)WE`B`SX``PH.-I```:2]Z-X@C0C>+PA[WH!`"@X??\_^ND$9_E`1"/X*#\
M_^L`0%#BY___"@`PU.60D9_E``!3XPF0C^"T<(WB/```"@``5>,$```*!!"@
MX04`H.$+_?_K``!0XRP```H$`*#A_OS_ZP`PH.$!($#B`B"$X`(`5.$`H*#A
M`S"$,```H#,!$(0R`P``.@@``.H#`%'A``##Y0(```H!('/E+P!2X_G__PH$
M`*#A[/S_ZP"@H.$&`*#AZ?S_ZPH`@.`!`(#B`0M0X[___RKXP)_E"3"@X0S`
MC^`!*Z#C`!"@XQ`0C>@'`*#A"&"-Y87\_^L($*#A!P"@X5+\_^L``%#C`P``
M&B`PG>4/.@/B`@E3XQ4```JT$)_E``"@XP$0C^!@_/_K`$!0XJ?__PH`,-3E
M``!3X\+__QH``%7C%@``"@`PU>4N`%/C`P``"H!`G^4!H*#C!$"/X-7__^H!
M,-7E``!3X^K__PKW___J`1"@XP<`H.&0_/_K``!0X^3__QI0`)_E!Q"@X0``
MC^"O_O_K!P"@X9?\_^N0___J.$"?Y0&@H.,$0(_@P?__ZC3\_^O0?@``/`$`
M`'PG``!0)P``;'X``!@G````)P``+"D``!@F``!X)0``G"4``"@E```00"WI
M+Q"@XP!`H.$`_/_K``!0XP%`@!($`*#A$$"]Z'S\_^H00"WI`$!0X@D```H`
M,-3E``!3XP8```J0_/_K`0"`X@$+4.,&``"*!`"@X1!`O>CX_?_J$`"?Y1!`
MO>@``(_@:_S_Z@``H.,0@+WHC"0``'!`+>G$))_EQ#2?Y0(@C^#`5)_E`S"2
MY[QDG^4(T$WB!5"/X``PD^4$,(WE`#"@XP9@C^`-0*#A!!"@X08`H.$D_?_K
M``!0XPX```J,-)_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$
M,(/B!""3Y0``4N,$((/D^___&NO__^I49)_E!F"/X`00H.$&`*#A#OW_ZP``
M4.,.```*-#2?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#
MX@0@D^4``%+C!""#Y/O__QKK___J`&2?Y09@C^`$$*#A!@"@X?C\_^L``%#C
M#@``"MPSG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($
M()/E``!2XP0@@^3[__\:Z___ZJQCG^4&8(_@!!"@X08`H.'B_/_K``!0XPX`
M``J$,Y_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3
MY0``4N,$((/D^___&NO__^I88Y_E!F"/X`00H.$&`*#AS/S_ZP``4.,.```*
M+#.?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4`
M`%+C!""#Y/O__QKK___J!&.?Y09@C^`$$*#A!@"@X;;\_^L``%#C#@``"M0R
MG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2
MXP0@@^3[__\:Z___ZK!BG^4&8(_@!!"@X08`H.&@_/_K``!0XPX```I\,I_E
M`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$
M((/D^___&NO__^I<8I_E!F"/X`00H.$&`*#ABOS_ZP``4.,.```*)#*?Y0`0
MG>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#
MY/O__QKK___J"&*?Y09@C^`$$*#A!@"@X73\_^L``%#C#@``"LPQG^4`$)WE
M`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[
M__\:Z___ZK1AG^4&8(_@!!"@X08`H.%>_/_K``!0XPX```IT,9_E`!"=Y0,P
ME><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___
M&NO__^I@89_E!F"/X`00H.$&`*#A2/S_ZP``4.,.```*'#&?Y0`0G>4#,)7G
M`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK
M___J#`&?Y000H.$``(_@,_S_ZP`04.("```*^`"?Y0``C^!W_?_K\`"?Y000
MH.$``(_@*OS_ZP`04.("```*W`"?Y0``C^!N_?_KU`"?Y000H.$``(_@(?S_
MZP`04.(2```*P`"?Y0``C^!E_?_KN""?Y60PG^4"((_@`S"2YP`@D^4$,)WE
M`B`SX``PH.,0```:F!"?Y9@`G^4!$(_@``"/X`C0C>)P0+WH5?W_ZH0`G^4$
M$*#A``"/X`C\_^L`$%#BZ?__"G``G^4``(_@3/W_Z^7__^K7^O_K]'L``#P!
M``#@>P``O"0``#0!``!L)```'"0``-`C``"`(P``.",``(`B``"4(@``2"(`
M`/PA```L(0``5"$``"@A``!$(0``1"$``$`A``!H(```E'<``!PA``"`(```
M["```'`@```$X"WE>""?Y7@PG^4"((_@=`"?Y0S03>(#,)+G#1"@X0``C^``
M,)/E!#"-Y0`PH./:^__K``!0XP4```H``-#E``!0XS``4!,!`*`#``"@$P$`
M(.(T()_E*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP$``!H,T(WB!/"=
MY)SZ_^N\=@``/`$``-@?``!L=@``#P`MZ03@+>5$()_E1#"?Y0(@C^`#,)+G
M#-!-X@`PD^4$,(WE`#"@XRPPG^44((WB*`"?Y0,PC^``((WE``"3YQ`0G>6P
M`(#BW/K_Z_\`H.-$^O_K)'8``#P!````=@``9`$```(0H./P3RWI?.&?Y7S!
MG^4!V$WB'-!-X@[@C^`,P)[G`4B-X@`@H.,`,*#C`,"<Y13`A.4`P*#C`!"-
MY0"0H.'4^O_K`6!0X@"`P>+(+Z#AR#\FX`(P4^!S,/_F`C`CX`(P4^`"(,+@
M`V!6X`*`R.```%CC%$"$XCP``+H0L9_E$'&?Y1"AG^4+L(_@#%"-X@=PC^``
M$*#C!B"@X0@PH.$)`*#A`!"-Y;KZ_^L!`''C`0!P`S````H*(*#A!1"@X0D`
MH.%Q^O_K`0!PXQ\```H'`%#C(```F@@`0.(`0)7@`P``.AP``.H!0$3B!0!4
MX1D``#H((*#C!Q"@X00`H.&8^O_K``!0X_;__QH%0$3@!@"4X,0?J."`()_E
M;#"?Y0(@C^`!R(WB`S"2YQ3`C.(`()/E`#"<Y0(@,^``,*#C$```&@'8C>(<
MT(WB\(^]Z$P`G^4``(_@C___ZP%H5N(`@,CB`0!XXP$(=@/(__\:``#@XP`0
MH.'F___J"P"@X87__^O+___J'/K_Z[AU```\`0``^!X``+AV```(``$`L'0`
M`#P>```00"WIA?K_ZW[Z_^L``%#C``"0%1"`O>CP0"WI\""?Y?!@G^7P,)_E
M`B"/X!303>(#,)+G!F"/X`!0H.$&`*#A"!"-X@`PD^4,,(WE`#"@XRK[_^L`
M0%#B`@``"@`PU.4``%/C#@``&K`@G^6H,)_E`B"/X`,PDN<`()/E##"=Y0(@
M,^``,*#C(```&@40H.&,`)_E``"/X!30C>+P0+WH7_S_Z@4`H.%B^O_K`'"@
MX00`H.%?^O_K``"'X`(`@.)B^O_K7#"?Y5P0G^4%(*#A`$"-Y0,PC^`!$(_@
M`$"@X4OZ_^M$()_E+#"?Y0(@C^`#,)+G`""3Y0PPG>4"(#/@`#"@XP00H`$&
M`*`!X?__"M#Y_^L`=```P!T``#P!``"X<P``8!T``#0<```T'```/',``/!/
M+>E)WTWB.*"-X@K@H.'H%9_EZ,6?Y>@EG^4!$(_@`B"1YPS`C^`,,(SB&$",
MXB!0C>(`()+E'"&-Y0`@H.,`<*#A!P"3Z`<`A>@/`+3H#P"NZ`,`E.@PP(SB
M+)"-X@,`CN@'`)SH!P")Z)@%G^4<8(WB!A"@X0``C^#2^O_KB(6?Y0!`4.((
M@(_@`@``"@`PU.4``%/C_P``&HK__^L`0%#B:@``"A7Z_^N``*#A`0"`XACZ
M_^L`(-3E`)"@X0``4N,4`(WE"```"D"UG^4+L(_@"0"@X0L0H.'_^?_K`2#T
MY0*0B>(``%+C^/__&B`%G^4``(_@`@``Z@0`NN4``%#C#0``"@80H.&N^O_K
M`$!0XOC__PH`,-3E``!3X_7__PK?^O_K``!0X_+__PH$`*#AU_G_ZP!`4.(,
M```:!$"5Y```5.,)```*`##4Y0``4^/.```*!`"@X='Z_^L``%#C\?__&@1`
ME>0``%3C]?__&@0`H.'A^?_K%)"=Y0!0H.$)`*#AW?G_ZP!0A>`!6X7B!%"%
MX@4`H.'>^?_K!)"-Y0"0H.%L-)_E;!2?Y0,PC^``,(WE9#2?Y00@H.$#,(_@
M`1"/X,3Y_^L'':#C"0"@X93Y_^L!`'#CF@``"@D`H.%0$(WB./G_ZP!`4.(L
M```*+"2?Y2P4G^4"`)CG"3"@X0`@E^4!$(_@L`"`XCKY_^L`4*#C$"2?Y>0S
MG^4"((_@`S"2YP`@D^4<,9WE`B`SX``PH./O```:!0"@X4G?C>+PC[WHDOG_
MZP!`4.*4__\:!`"9Y```4.,)```*!A"@X5CZ_^L`,%#B^/__"@`PT^4``%/C
M\O__&@0`F>0``%#C]?__&@T`H..B^?_KG$.?Y5,@H.,`D*#A!$"/X!0`C>6(
M___J6#"=Y0\Z`^(!"5/CSO__&FR@G>6*^?_K``!:X<K__QI8,)WE@SN@X:,[
MH.$'#5/CQ?__&@4`H.&-^?_K`%"@X4@#G^4&$*#A``"/X#3Z_^L`$%#B=```
M"@``E^5!_/_K`(!0XG````H(`*#A>OG_ZP,`4.,`8*#A#```FA`3G^4$`$#B
M``"(X`$0C^![^?_K``!0XP4``!H$`%;CG0``"@9@B.`%,%;E+P!3XYD```HS
M_O_K`&!0X@0``!H(`*#A!A"@X57Y_^L`<%#B6```&L`2G^7``I_E`1"/X```
MC^!8^__K-?G_ZP#`H.&L,I_EK!*?Y0,PC^`$,(WEI#*?Y:2"G^6D<I_EI&*?
MY0D@H.$%`*#A`S"/X`$0C^``P(WE"("/X`=PC^`&8(_@0?G_ZPT``.H)^?_K
M`#"0Y1$`4^,.```:'?G_ZP#`H.$!0(3B"#"@X0D@H.$'$*#A!0"@X5``C>D`
MP(WE,OG_ZP<=H.,%`*#A`OG_ZP$`<./L__\*"0"@X9OX_^LH`I_E!1"@X0``
MC^`K^__K!0"@X:;^_^MR___J[?C_ZP`PD.41`%/C8/__"@!`E^7H^/_K`,"@
MX;@1G^4`P)SE`0"8Y^@1G^4),*#A!""@X0$0C^``P(WEL`"`XIOX_^M?___J
M`T"@X37__^J0_O_KP"&?Y5PQG^4"((_@`S"2YP`@D^4<,9WE`B`SX``PH.--
M```:!`"@X4G?C>+P3[WH\/C_Z@"`E^6,___J$/[_ZP"@4>(`@*#AHO__N@H@
M4.(`,,KB`&"-Y0<`H.'H8(WB[?C_ZP8@H.,&$*#A!P"@X:?X_^L&`%#C`@``
M"D@!G^4``(_@Y?W_ZT`QG^7H()WE`P!2X8___QJT(-;A,#&?Y0,`4N&+__\:
M`!"@XS(@6.(`,,KB!P"@X0`0C>7P0(WBU?C_ZR@@H.,'`*#A!!"@X8_X_^LH
M`%#C`@``"O0`G^4``(_@S?W_ZP#`H./H()_EZ#"?Y>@0G^4"((_@!""-Y04`
MH.$)(*#A`$"-Y0,PC^`!$(_@&,'-Y<CX_^N9___J!&"7Y0``5N-B__\*!@"@
MX<OX_^L#`%#C7O__FJ00G^4$`$#B``"&X`$0C^#-^/_K``!0XP:`H`%6___J
M2/C_Z]AR``!(=```/`$``$`;``!L<@``"!P``'P;```\&P``'!L``.`<``!D
M`0``*!L``-1P```T&@``E!H``%P:```@&0``H!@``-0:``#X&0``Y!H``-@:
M``#\&0``H!H``(`7``!T&```3&X``'P8````0T%#2$4``!P8``#8&```]!@`
M`/@7``"L%P``\$`MZ5PBG^5<,I_E`B"/X%@"G^4#,)+G#-!-X@``C^``,)/E
M!#"-Y0`PH..4^/_K`#"@X0U0H.$`0)/E,`*?Y60PU.4%$*#A``"/X&4PQ.4Z
M^?_K``!0XP(```H`,-#E,0!3XV<```K^^__K!`*?Y0``C^""^/_K`$"0Y?@1
MG^4$`*#A`S"@XP$@H.,!$(_@'OC_ZP!`4.(/```*W`&?Y0PPE.4``(_@`&"3
MY73X_^O,(9_E!A"@X0(@C^``8*#A``"0Y?;W_^L,,)3E`!"3Y0@PD>4!!1/C
M5@``&J0!G^6D09_E``"/X&7X_^L$0(_@!!"@X0,PH.,!(*#C`&"@X0``D.4!
M^/_K`$!0X@D```IX`9_E!1"@X0``C^`'^?_K`"!0XCL```H,,)3E``"6Y0`0
MD^7:]__K5`&?Y51QG^4``(_@3OC_ZTP1G^4`8)#E`2"@XP!`H.$!$(_@!@"@
MX?;W_^LT(9_E`!"@X0(@C^`&`*#ARO?_ZP`@H.,`,)3E'`&?Y040H.$``(_@
M92##Y>KX_^L'<(_@``!0XP@`A^40```*!QV@X_SW_^L!,(#B$0!0XP$`4Q,B
M``"*Y""?Y:0PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,8```:#-"-XO"`
MO>@$`)?EX/W_ZP``4.,(`(?EZ?__&J@`G^4``(_@T/?_ZP$PT.4``%/CF?__
M"I/__^J0,)_E#""4Y0,PC^`$,)/E``"2Y0`0D^7#^?_KO___Z@``EN74]__K
MI?__ZI/W_^MD()_E`#"@X6``G^4"((_@!@"2Z0``C^``$)'EN/?_ZUQL```\
M`0``>&T``#05```D;0``Q!8``/!L``"<%@``L&P``'`6```@%```5&P``/!L
M```(%@``#!8``*03``"D:@``@!4``&1L```P;```2!4``!!`+>D`0*#AM?S_
MZP``5.,``%`3$("]"``PU.4``%/C$("]"`0`H.%+^__K0?O_ZQP0G^4$(*#C
M`1"/X#SW_^L``%#C$("]&`0`H.$00+WHZ_C_ZFP3``#P3RWI6"N?Y5@[G^4"
M((_@`S"2YU/?3>(`,)/E1#&-Y0`PH.,-$*#A`'"@X1P@@.)<`(#B!#"2Y#,_
MO^8``%+A!#"AY?K__QHX`)WE/."=Y4`PG>4$P)WE"%"=Y0@@C>)";XWB````
MZ@%0H.$#0*#A'#"2Y000LN4!,"/@`#`CX`PP(^#C/Z#A`@!6X0X`H.$%P*#A
M!."@X3@P@N7Q__\:Q#J?Y0Q0E^4$$)WE$$"7Y0$0E^D`X)?E!2`LX`,0@>`$
M$('@`"`"X`4@(N`(8)WE[AV!X`(0@>!@(2S@`V"&X`X@`N`%8(;@#"`BX&Z1
MH.$&((+@#&"=Y6"!*>`#8(;@`8`(X`Q@AN`4L)WE8($HX.$M@N`0H)WE!H"(
MX&%A*>#BC8C@`F`&X`.PB^!B(:#A";"+X`.@BN`)8";@89$BX&"ABN`(D`G@
M"F"&X&&1*>`8H)WEZ&V&X`N0B>!HL2+@`Z"*X`:P"^#FG8G@8:&*X&81H.$"
M8"O@"F"&X!R@G>5HL2'@`Z"*X`(@BN`@H)WE";`+X`.@BN#I;8;@:*&*X&B!
M*^!IL2'@`H"(X`:P"^#FC8C@9B&@X0%@*^`*8(;@)*"=Y6FQ(N`#H(K@`1"*
MX"B@G>4(L`O@`Z"*X&FABN!ID2O@Z&V&X`&0B>!H$2+@!A`!X`(0(>`*$('@
M+*"=Y>:=B>`#H(K@9F&@X0(@BN`PH)WE:+$FX`FP"^`#H(K@:*&*X&B!*^#I
M'8'@`H"(X&DA)N`!(`+@!B`BX`H@@N`TH)WEX8V(X`.@BN!A$:#A!F"*X&FA
M(>`(H`K@.+"=Y6FA*N#H+8+@!J"*X&AA(>`#L(O@`F`&X&FQB^`!8";@"V"&
MX#RPG>7BK8K@`["+X&(AH.$!L(O@0!"=Y6B1(N`#$('@"I`)X.IMAN!H$8'@
M:)$IX&J!(N`&@`C@`H`HX`&`B.!$$)WE"Y")X.:=B>`#$('@9F&@X4BPG>4"
M(('@:A$FX`.PB^`)$`'@:K&+X&H1(>!,H)WEZ8V(X`(0@>!I(2;@`Z"*X`@@
M`N`&("+@!F"*X%"@G>7H'8'@:(&@X0,PBN!IH2C@`:`*X`L@@N!IH2K@X2V"
MX`:@BN!A82C@`F`&X&DQ@^`(8";@6+"=Y0-@AN`4.)_E5)"=Y>*MBN`#L(O@
M8B&@X0.0B>!AL8O@81$BX`H0(>`(@(G@ZFV&X`&`B.#FG8C@7("=Y0.`B.`"
M@(C@:B$BX`8@(N`"$(O@8""=Y69AH.$#((+@:B&"X&JA)N`)H"K@"H"(X&2@
MG>7I'8'@`Z"*X`:@BN!I82;@`6`FX.&-B.`&((+@Z&V"X&@@G>5A$:#A`R""
MX&DA@N!ID2'@")`IX`F0BN!LH)WEYIV)X`.@BN`!H(K@:!$AX`80(>`!((+@
MZ1V"X'`@G>5F8:#A`R""X&@A@N!H@2;@"8`HX`B`BN!TH)WEX8V(X`.@BN`&
MH(K@:6$FX`%@)N`&((+@Z&V"X'@@G>5A$:#A`R""X&DA@N!ID2'@")`IX`F0
MBN!\H)WEYIV)X`.@BN`!H(K@:!$AX`80(>`!((+@Z1V"X(`@G>5F8:#A`R""
MX&@A@N!H@2;@"8`HX`B`BN"$H)WEX8V(X`.@BN`&H(K@:6$FX`%@)N`&((+@
MZ&V"X&&QH.&(()WE:1$KX`@0(>`#((+@:2&"X`&0BN!HH2O@!J`JX.:=B>`*
M((+@Z:V"X)`@G>6,$)WE9F&@X0,@@N`#$('@:"&"X&B!)N`)@"C@"Q"!X`@0
M@>#JC8'@E!"=Y:2PG>4#$('@!A"!X&EA)N`*8";@!B""X.AM@N"8()WE:J&@
MX0,@@N!I(8+@:9$JX`B0*>`)$('@G)"=Y>8=@>`#D(G@"I")X&BA*N`&H"K@
M"B""X*"@G>5F8:#A`S"*X&@Q@^!H@2;@`8`HX.$M@N`(@(G@89$FX`*0*>#B
MC8C@"3"#X.B=@^"T-9_EJ*"=Y6(AH.$#L(O@!K"+X`.@BN!A88+A8:&*X`A@
M!N!A$0+@`1"&X0L0@>!HL8+AZ1V!X&EAH.$)L`O@:)$"X`F0B^&LL)WE"I")
MX+"@G>4#L(O@`K"+X`.@BN!H(8;A:*&*X`$@`N!H@0;@X9V)X`@@@N%A@8;A
M"R""X`F`".!AL0;@"X"(X;2PG>7I+8+@`["+X`:PB^"X8)WE:9&@X0J`B.`#
M8(;@8:&)X0*@"N!A88;@81$)X.*-B.`!$(KA8J&)X0L0@>`(H`K@8K$)X`N@
MBN&\L)WEZ!V!X`.PB^`)L(O@P)"=Y6B!H.$&8(K@`Y")X&*AB.$!H`K@8I&)
MX&(A".#A;8;@`B"*X6&AB.$+((+@!J`*X&&Q".`+H(KAQ+"=Y>8M@N`#L(O@
M"+"+X,B`G>5F8:#A"9"*X`.`B.!AH8;A`J`*X&&!B.!A$0;@XIV)X`$0BN%B
MH8;A"Q"!X`F@"N!BL0;@"Z"*X<RPG>7I'8'@`["+X`:PB^#08)WE:9&@X0B`
MBN`#8(;@8J&)X0&@"N!B88;@8B$)X.&-B.`"((KA8:&)X0L@@N`(H`K@8;$)
MX`N@BN'4L)WEZ"V"X`.PB^`)L(O@V)"=Y6B!H.$&8(K@`Y")X&&AB.$"H`K@
M89&)X&$1".#B;8;@`1"*X6*AB.$+$('@!J`*X&*Q".`+H(KAW+"=Y0F0BN#@
MH)WEYAV!X`.PB^!F8:#A"+"+X`.@BN!B@8;A8J&*X`&`".!B(0;@X9V)X`*`
MB.%A(8;A"X"(X`D@`N!AL0;@"R""X>2PG>4*((+@Z*"=Y>F-B.`#L(O@:9&@
MX0:PB^`#H(K@86&)X6&ABN`(8`;@81$)X.@M@N`!8(;A:!&)X0M@AN`"$`'@
M:+$)X`L0@>'LL)WE"A"!X/"@G>7B;8;@`["+X&(AH.$)L(O@`Z"*X&B1@N%H
MH8K@!I`)X&B!`N#H,I_E")")X?2`G>7F'8'@"Y")X`.`B.!FL8+A`;`+X`*`
MB.!F(0+@`B"+X0H@@N#XH)WEX9V)X`.@BN!A$:#A9J&*X&9A(>`)8";@!H"(
MX/Q@G>4`L9WE`V"&X.DM@N`!8(;@:1$AX.*-B.`"$"'@`["+X&(AH.$!$(K@
M:;&+X`2AG>5ID2+@")`IX`E@AN`#H(K@")&=Y>@=@>`"H(K@:"$BX.%MAN`!
M("+@`Y")X&$1H.%HD8G@:($AX`:`*.`(@(K@#*&=Y0(@B^`#H(K@YBV"X`&@
MBN!F$2'@`A`AX`$0B>`0D9WEXHV(X`.0B>!B(:#A9I&)X&9A(N`(8";@!F"*
MX!2AG>7H'8'@`Z"*X`*@BN!H(2+@`2`BX`(@B>`8D9WEX6V&X`.0B>!A$:#A
M:)&)X&B!(>`&@"C@"("*X!RAG>7F+8+@`Z"*X`&@BN!F$2'@`A`AX`$0B>`@
MD9WEXHV(X`.0B>!B(:#A9I&)X&9A(N`(8";@!F"*X"2AG>7H'8'@`Z"*X`*@
MBN!H(2+@`2`BX`(@B>`HD9WEX6V&X`.0B>!A$:#A:)&)X&B!(>`&@"C@"("*
MX"RAG>7F+8+@`Z"*X`&@BN!F$2'@`A`AX`$0B>`PD9WEXHV(X`.0B>!B(:#A
M9I&)X&9A(N`(8";@!F"*X#2AG>7H'8'@`Z"*X`*@BN!H(2+@`2`BX`(@B>`X
MD9WEX6V&X`.0B>!A$:#A:)&)X&B!(>`&@"C@YBV"X`B`BN#BC8C@:%&%X`Q0
MA^4\49WE`U"%X`%0A>!F$2'@`A`AX`$0B>#H'8'@8<&,X`C`A^5`P9WE8B&@
MX0,PC.!FP2+@`D"$X`XP@^`(P"S@:"$BX`$@(N!F,8/@#%"%X.$=A>`",(/@
MX3V#X%`@G^4`,(?E`0"`X#`PG^4"((_@$$"'Y00`A^4#,)+G`""3Y40QG>4"
M(#/@`#"@XP$``!I3WXWB\(^]Z(?T_^M(:0``/`$``)EY@EJAZ]ENW+P;C];!
M8LH@7@``$$`MZ6``H./[]/_K`""@XPQ[G^TTP)_E-!"?Y31`G^4TX)_E!'N`
M[0A`@.4,X(#E`,"`Y000@.48((#E7""`Y1"`O>@`\"#C\.'2PP`````!(T5G
MB:O-[_[<NIAV5#(0\$<MZ0*0H.$`4*#A%#"0Y8(AH.$",)/@&""0Y10P@.5<
M`)#E`2""(JDN@N```%#C`4"@X1@@A>4<@(7B#@``"D!P8.()`%?A"7"@H0``
MB.`'(*#A3/3_ZUPPE>4#,(?@0`!3XUPPA>7PA[T8!0"@X0>02>`'0(3@Z/S_
MZS\`6>,=``#:0)!)XBECH.$!<(;B!W.$X`@PH.%`X(3B`,"4Y00`E.4($)3E
M#""4Y1!`A.(.`%3A`,"#Y00`@^4($(/E#""#Y1`P@^+S__\:!0"@X=+\_^L'
M`%3A[?__&@9M9N`&DXG@"2"@X0<0H.$(`*#A)?3_ZUR0A>7PA[WH!'"@X??_
M_^I_P.#C\$$MZ10PD>488)'E,W^_YJ,QH.$_,`/B`2"#XAR`@>(X`%+C`%"@
MX0%`H.$V;[_F`\#(YP(`B.`[``#:`!"@XT`@8N(L]/_K!`"@X;'\_^L(`*#A
M.""@XP`0H.,F]/_K!`"@X?1EQ.&J_/_K`S#4Y00`H.$`,,7ELC#4X0$PQ>4`
M,)3E(S2@X0(PQ>4`,)3E`S#%Y0<PU.4$,,7EMC#4X04PQ>4$,)3E(S2@X08P
MQ>4$,)3E!S#%Y0LPU.4(,,7ENC#4X0DPQ>4(,)3E(S2@X0HPQ>4(,)3E"S#%
MY0\PU.4,,,7EOC#4X0TPQ>4,,)3E(S2@X0XPQ>4,,)3E#S#%Y1,PU.40,,7E
MLC'4X1$PQ>40,)3E(S2@X1(PQ>40,)3E$S#%Y?!!O>BZ\__J`!"@XS@@8N+P
M\__KR/__ZA!`+>E8,)_E6$"?Y0,PC^`D(-/E!$"/X```4N,0@+T8`1"@XT`@
MG^4D$,/E`C"4YP``4^,"```*,#"?Y0,`G^>0\__K*#"?Y0,PE.<``%/C$("]
M"!P`G^400+WH``"/X/KS_^H870``'%L``!@!``!H7```4`$``'A%```\$)_E
M/#"?Y0$0C^``(-'E`S"/X```4N,>_R\1`0"@XR0@G^4``,'E`C"3YP``4^,>
M_R\!%`"?Y000@>(``(_@[//_ZJ!<``"D6@``5`$``!Q%``#P0"WI8,*?Y2S0
M3>)<,I_E'`"-Y1@0C>44((WE#,"/X`,PG.<8$(WB%""-XAP`C>(\0I_E`#"3
MY20PC>4`,*#CN?/_ZRPBG^4$0(_@`B"4YR0RG^4,((WE`S"4YP(0H.$#`*#A
M$#"-Y;OS_^L,,I_E`S"4YP!0T^4``%7C40``"OP!G^4``(_@^_/_ZP`@D.4<
M4)WE1C/2Y000A>(#,(/C!P%QXP-0A>)&,\+E8@``B@$!H.&0\__K`#"@X1@0
MG>7`89_E``"1Y;SAG^4&8(_@00"#Z+3!G^4.X(_@!#".Y0S`E.<<()WEI'&?
MY0``G.4'<(_@`0!2XPQP@^4(`(/E5```V@0`0>(,P(/B`@&`X`3@L>4!`%#A
M!."LY?O__QH#P(+B`."@XVA!G^5H$9_E!$"/X`P`E.4%(*#A`."-Y0$0C^`,
MX8/GF//_ZP``4.,I```*1%&?Y2!@C>(%4(_@#`"5Y4;S_^L`0*#A,`&?Y080
MH.$``(_@;O3_ZP``4.,@```*'#&?Y0,PC^`,`)/E+//_ZW/S_^L,(9_ES#"?
MY0(@C^`#,)+G`""3Y20PG>4"(#/@`#"@XRH``!H$`*#A+-"-XO"`O>@V\__K
MW""?Y0``4.,"((_@#`""Y1\```H,\__KR`"?Y0``C^"B\__K`#"0Y>Y5P^6@
M___J#`"4Y87S_^O2___J"`"5Y0``4.,!```*IOO_Z]G__^J4`)_E!A"@X0``
MC^!"]/_K``!0XP@`A>72__\*]?__ZG@PG^5X`)_E`S"4YP``C^`#$*#A##"-
MY3KS_^L$P*#CL/__Z@$`H./*\O_K"//_ZT1:```\`0``&%H``$P!``!<`0``
M*`$```A;```4!0``I%L``$@!``#T!```3%L``+39__\@6P``$`(``/1:``#`
M6```J%H``*19`````0``1`$``-@"``!P<U]S=')I;F=S(&UI<W-I;F<*`"5S
M+R5S````+@```"]P<F]C+V-U<G!R;V,O97AE````1'EN84QO861E<CHZ8F]O
M=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````%!!4E]414U0
M`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S``!015),-4Q)0@````!0
M15),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!
M4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#04-(10``
M`%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]435!$25(```!005)?
M5$U01$E2``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@`````Q
M````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?4$%4
M2`!365-414T``"4P,G@`````)7,E<R5S)7,`````<&%R+0`````E<SH@8W)E
M871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N
M;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E
M("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*
M``!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R96%D```E<R5S8V%C
M:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````
M&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)324].`````#$N,#8S
M````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z(&-R
M96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE
M9"`H97)R;F\])6DI"@`````M90``+2T``%5315(`````55-%4DY!344`````
M5$5-4$1)4@!435``+W1M<"\```!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@
M("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S
M;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#+"`D
M1DE,15]O9F9S971?<VEZ92P@)&-A8VAE7VYA;65?<VEZ92D["D)%1TE.('L*
M)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*)$9)3$5?;V9F<V5T7W-I>F4@
M/2`T.R`@(",@<&%C:R@B3B(I"B1C86-H95]N86UE7W-I>F4@/2`T,#L*)%!+
M6DE07TU!1TE#(#T@(E!+7#`P,UPP,#0B.PI]"G-U8B!F:6YD7W!A<E]M86=I
M8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T
M.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@)'!O<R`]("@D<VEZ
M92TQ*2`M("@D<VEZ92TQ*2`E("1C:'5N:U]S:7IE.R`@("`@(",@3D]413H@
M)'!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D<&]S
M(#X](#`I('L*<V5E:R`D9F@L("1P;W,L(#`["G)E860@)&9H+"`D8G5F+"`D
M8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`]
M(')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@>PIR971U<FX@)'!O
M<R`K("1I.PI]"B1P;W,@+3T@)&-H=6YK7W-I>F4["GT*<F5T=7)N("TQ.PI]
M"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U;&5#
M86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&
M:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I
M;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M
M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R
M7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@
M/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E
M;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM
M>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A8
M6"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*
M4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE;F%M
M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@],3L@
M=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP)PII
M9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE97`@
M,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN"F-L
M;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]("(D
M9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D["F]U
M=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C
M;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z
M.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM0
M05)?1$5"54=].PIM>2`E<WES(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^
M("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]
M?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E
M;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G
M*2P**3L*7W-E=%]P<F]G;F%M92@I.PIO=71S*'%Q6UPD<')O9VYA;64@/2`B
M)'!R;V=N86UE(ETI.PI?<V5T7W!A<E]T96UP*"D["F]U=',H<7%;7"1%3E9[
M4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](ETI.PIM>2`H)'-T87)T7W!O
M<RP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD["DU!1TE#.B!["FQO8V%L("13
M24=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P96X@7T9(+"`G/#IR
M87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R96%D(&9R;VT@9FEL
M92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A<R!I="!R
M97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY("1M86=I8U]P;W,@
M/2!F:6YD7W!A<E]M86=I8R@J7T9(*3L*:68@*"1M86=I8U]P;W,@/"`P*2![
M"F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'24,B
M(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IO=71S*")&
M;W5N9"!005(@;6%G:6,@870@<&]S:71I;VX@)&UA9VEC7W!O<R(I.PIS965K
M(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@;&5N9W1H
M*")<,$-!0TA%(BDL(#`["G)E860@7T9(+"`D8G5F+"!L96YG=&@H(EPP0T%#
M2$4B*3L*:68@*"1B=68@;F4@(EPP0T%#2$4B("8F("1B=68@;F4@(EPP0TQ%
M04XB*2!["F]U=',H(DYO(&-A8VAE(&UA<FME<B!F;W5N9"(I.PIL87-T($U!
M1TE#.PI]"F]U=',H<7%;0V%C:&4@;6%R:V5R("(D8G5F(B!F;W5N9%TI.PIS
M965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE+"`P.PIR
M96%D(%]&2"P@)&)U9BP@)$9)3$5?;V9F<V5T7W-I>F4["FUY("1O9F9S970@
M/2!U;G!A8VLH(DXB+"`D8G5F*3L*;W5T<R@B3V9F<V5T(&9R;VT@<W1A<G0@
M;V8@1DE,17,@:7,@)&]F9G-E="(I.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M
M("1&24Q%7V]F9G-E=%]S:7IE("T@)&]F9G-E="P@,#L*)'-T87)T7V]F7T9)
M3$5?<V5C=&EO;B`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T.PIR96%D
M(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@("`@("`C(')E
M860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,12(I('L*
M<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB
M+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5;G!A8VMI
M;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D9G5L;&YA
M;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F.PIM>2`H
M)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H+BHI*%PN
M+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP
M86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^
M("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA;64@/2!?<V%V
M95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A=GDZ.D9U
M;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V>3HZ
M1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64["GT*96QS92!["B1-
M;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^
M("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T>R1F
M=6QL;F%M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#
M(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF
M(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U;&5#
M86-H97LD;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E("1R
M97%U:7)E7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO861E
M<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F
M:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=U]T;7!F:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B
M.PHD9F@M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD
M9F@M/G-E96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE
M;F%M92`](%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U
M9GTI.PHD:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F:"P@
M)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL
M96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG
M(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`24Y#
M*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U
M:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*4$%2
M.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&:6QE
M.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER95]L
M:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL96YA
M;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_)"]I
M*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA;64@
M=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*?0I]
M"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*:68@*"1B=68@
M;F4@)%!+6DE07TU!1TE#*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R
M($9)3$4@<V5C=&EO;B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'
M24,@.PI]"B1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@
M("`@("`@(R!S=&%R="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T
M+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[
M4$%27T%04%]2155317T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E
M8W5R:71Y('!R;V)L96TI"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@
M97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@
M/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI
M;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@
M*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%2
M1U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E
M('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H
M:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V
M72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T
M;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R
M(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D
M,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H
M)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q
M("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@
M>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G
M9F@L("(^/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D
M,B(Z("0A73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@
M)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD
M,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E
M<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S
M<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*
M97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H
M)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ
M6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`
M05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D
M9F@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I
M;&4@(B1P87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@
M/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD
M;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D
M:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FUY
M("5M971A7W!A<B`](&1O('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II
M<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z
M)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>
M.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N
M(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A9"!P
M87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B=68L
M(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S<R`D
M8G5F(&5Q("102UI)4%]-04=)0SL*8VQO<V4@)'!H.PI]"D-R96%T95!A=&@H
M)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL
M93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?4D174B@I('P@24\Z.D9I;&4Z
M.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E(&9I
M;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@,"P@
M,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@)'-T87)T7V]F7T9)3$5?<V5C
M=&EO;BD@>PIR96%D(%]&2"P@)&QO861E<BP@)'-T87)T7V]F7T9)3$5?<V5C
M=&EO;CL*?2!E;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?
M1D@^.PI]"FEF("@A)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^
M("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13
M=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@
M)#`I.PI]"B1F:"T^<')I;G0H)&QO861E<BD*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@;&]A9&5R('1O("(D;W5T(CH@)"%=.PII9B`H)&)U;F1L92D@>PIR
M97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@
M/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E
M)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*
M)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS
M?"\K?"]\9RP@<WPO)'Q\(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL
M97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T<RP@
M=F%L=65S("5)3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI
M8E]E>'1].R`@("`@("`@(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E
M=W)I='1E;CL*9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S*2![
M"FUY("@D9FEL92P@)&YA;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2
M.CI(96%V>3HZ1G5L;$-A8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE*2`]
M("@D:V5Y+"`D9F,I.PI]"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I
M('L*:68@*"1K97D@/7X@;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L
M("1N86UE*2`]("@D:V5Y+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@;7Q>
M+VQO861E<B];7B]=*R\H+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D=6QE
M0V%C:&5[)#%]*2!["B@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S
M=#L*?0II9B`H+68@(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D
M9&ER+R0Q(BP@)#$I.PIL87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN
M960@)&YA;64["FYE>'0@:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@
M(7)E9B@D9FEL92D@86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*
M;W5T<RAS<')I;G1F*'%Q6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F("1F
M:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT.PII
M9B`H<F5F*"1F:6QE*2D@>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE
M;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*;W!E;B!M>2`D=&@L("<\.G)A=R<L
M("1F:6QE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*)&-O
M;G1E;G0@/2`\)'1H/CL*8VQO<V4@)'1H.PI005(Z.D9I;'1E<CHZ4&]D4W1R
M:7`M/F%P<&QY*%PD8V]N=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A
M)$5.5GM005)?5D520D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA
M;"DD+VD["E!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH
M7"1C;VYT96YT+"`D9FEL92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B
M+`IP86-K*"=.)RP@;&5N9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX
M+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D
M;F%M92DL"G!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I
M"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&5M8F5D9&5D($9)3$4@=&\@(B1O
M=70B.B`D(5T["F]U=',H<7%;5W)I='1E;B!A<R`B)&YA;64B72D["GT*?0II
M9B`H)'II<"D@>PHD>FEP+3YW<FET951O1FEL94AA;F1L92@D9F@I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TL*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@>FEP
M('!A<G0@;V8@(B1O=70B73L*?0II9B`H)&UE=&%?<&%R>V-L96%N?2D@>PHD
M9F@M/G!R:6YT*")<,$-,14%.(BD["GT*96QS92!["FEF("@A9&5F:6YE9"`D
M8V%C:&5?;F%M92D@>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*
M<V5E:R`D9F@L(#`L(#`["B1C='@M/F%D9&9I;&4H)&9H*3L*<V5E:R`D9F@L
M(#`L(#(["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H
M95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P+"`D8V%C:&5?;F%M95]S
M:7IE.PHD8V%C:&5?;F%M92`N/2`B7#`B('@@*"1C86-H95]N86UE7W-I>F4@
M+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT*"1C86-H95]N86UE
M+`HB7#!#04-(12(I.PI]"FUY("1O9F9S970@/2`D9F@M/G1E;&P@+2!L96YG
M=&@H)&QO861E<BD["B1F:"T^<')I;G0H<&%C:R@G3B<L("1O9F9S970I+`HD
M4$%27TU!1TE#*3L*)&9H+3YC;&]S92!O<B!D:64@<7%;17)R;W(@=W)I=&EN
M9R!T<F%I;&5R(&]F("(D;W5T(CH@)"%=.PIC:&UO9"`P-S4U+"`D;W5T.PIE
M>&ET.PI]"GL*;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?9FEX
M7W!R;V=N86UE*"D["G)E<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI=%]D
M>6YA;&]A9&5R*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E($%R
M8VAI=F4Z.EII<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T97,@
M;VX@86X@24\Z.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@)W(G
M*2!O<B!D:64@<7%;9F1O<&5N*"D@9F%I;&5D.B`D(5T["D%R8VAI=F4Z.EII
M<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D["FUY("1Z:7`@/2!!<F-H:79E.CI:
M:7`M/FYE=SL**"1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G
M;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R
M;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z
M.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB
M0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP
M.PIO=71S*")%>'1R86-T:6YG('II<"XN+B(I.PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*
M9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I
M9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE
M;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M
M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO*3\*
M*%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE(#T@)#$["FUY("1D97-T
M7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[4$%27U1%35!]+"`D
M97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA;64@)B8@+7,@7R`]/2`D
M;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*;W5T<RAQ<5M3:VEP<&EN
M9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E861Y(&5X:7-T<R!A="`B
M)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T:6YG("(D
M;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE;6)E<BT^
M97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I(#T]($%R8VAI=F4Z.EII
M<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@>FEP(&UE
M;6)E<B!T;R`B)&1E<W1?;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE
M*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB
M0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*
M57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@
M72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/
M;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G
M;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*
M;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H
M)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL
M97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*
M1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C
M(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S
M('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I
M;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q
M=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD[
M"G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U
M:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E
M;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q
M=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ
M1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E
M.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U
M:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T
M8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E
M($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER
M92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ
M:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#
M87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L
M24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S
M971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@
M)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR
M971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W
M*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@
M0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A
M=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM
M>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E
M9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I
M('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E
M9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*
M)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G
M4UE35$5-)SL*?0IM>2`D<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM?7!A
M<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@
M,#<U-3L*;7D@)&-A8VAE7V1I<CL*:68@*"1%3E9[4$%27T-,14%.?2D@>PHD
M8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PI]"F5L<V4@>PIO<&5N(&UY("1F:"P@
M(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O
M9VYA;64B.B`D(5T["FEF("@H;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA
M9VEC*"1F:"DI(#X](#`I('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,
M15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I+"`P.PIM>2`D8G5F
M.PIR96%D("1F:"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F
M(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%
M7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BD*+2`D8V%C:&5?;F%M
M95]S:7IE+"`P.PIR96%D("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ93L*
M)&)U9B`]?B!S+UPP+R]G.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B.PI]
M"GT*8VQO<V4@)&9H.PI]"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?9&ER
M(#T@(G1E;7`M)"0B.PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP9&ER
M("X]("(D<WES>U]D96QI;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D:7(L
M(#`W-34["B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*)'!A
M<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%
M35!](#U^("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C
M;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E
M;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA
M;64@/2`B)&9U;&QN86UE+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^.G)A
M=R<L("1T96UP;F%M92D*)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F("AC
M;&]S92`D9F@I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA;64B
M.B`D(5T["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D
M93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E
M;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N
M86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6
M>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]
M"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD
M97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA
M;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D<WES
M>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@
M:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@*&]P
M96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H
M.PI]"FEF("@M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N86UE
M("X]("1S>7-[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H
M<W!L:70@+UQ1)'-Y<WMP871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X
M="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM0
M05)?5$5-4'T["B1D:7(@/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH*"1P
M<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE
M?2(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S
M>7-[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R
M;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N
M86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@
M?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@
M)'-Y<WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D96QI
M;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C
M=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W
M9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD
M<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D<'=D
M)'-Y<WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE
M.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,
M25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.
M251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.
M($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)005)?
M)%\B?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2![
M"B1%3E9[(E!!4E\D7R)](#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF(&5X
M:7-T<R`D14Y6>R)005)?1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%27U1%
M35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T<R![
M"G)E='5R;B!I9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H
M(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?
M:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D
M+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I
M=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@
M=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*
M=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P
M87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I
M<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*
M=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET
M*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@
M:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET
M*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD[
M"F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*`'!E
M<FQX<VDN8P````!#04-(10```1L#.P`!```?````((W__Q@!``!\C?__-`$`
M`/2-__]4`0``G(___Z0!``!XD/__R`$``!R1___L`0``C)+__Q0"``"@D___
M2`(``,"4__]X`@``^);__Z@"``!@E___T`(``-B7___P`@``X)?__P0#``!0
MFO__.`,``'2:__]8`P``Q)K__X`#``#\G___L`,``)"@___0`P``]*#___0#
M``"<HO__+`0``+2B__]$!```T*/__W0$``!<JO__O`0``!BM___D!```<*W_
M_RP%``#PN/__8`4``%"Y__]X!0``5+K__YP%``"4N___A`$``!"\__]P`0``
M9+S__P0%```0``````````%Z4@`"?`X!&PP-`!@````8`````(S__UP`````
M0@X0A`2%`X8"C@$<````-````$",__]X`````$(.&(0&A06&!(<#B`*.`1@`
M``!4````F(S__Z@!````0@X(AP*.`4@.$``0````<````)BZ__]4````````
M`!P```"$````"+K__WP`````0@X(A`*.`6S.Q`X`````(````*0```#PC?__
MW`````!$#B"$"(4'A@:'!8@$B0.*`HX!(````,@```"HCO__I`````!"#@B$
M`HX!2`ZH`7(*#@A""P``)````.P````HC___<`$```!"#A2$!84$A@.'`HX!
M2@ZX`60*#A1""S`````4`0``<)#__Q0!````0@X(A`*.`48.B`A"#I`(`E0*
M#@A""UH*#@A"SL0.`$(+```L````2`$``%"1__\@`0```$(.%(0%A02&`X<"
MC@%,#I0(0@Z@"`)."@X40@L````L````>`$``$"2__\X`@```$(.((0(A0>&
M!H<%B`2)`XH"C@%,#B@"E@H.($(+```D````J`$``$B4__]H`````$(.#(0#
MA0*.`4P.&%@.#$+.Q<0.````'````-`!``"(E/__>`````!"#@R$`X4"C@%$
M#B!J#@P0````\`$``."4__\(`````````#`````$`@``U)3__W`"````0@X@
MA`B%!X8&AP6(!(D#B@*.`48.T`E"#M@)`D0*#B!""P`<````.`(``!"7__\D
M`````$(."(0"C@%.SL0.`````"0```!8`@``%)?__U``````0@X(A`*.`58*
MSL0.`$(+1`K.Q`X`1`LL````@`(``#R7__\X!0```$(.$(0$A0.&`HX!3@X8
M`SX""@X00L[&Q<0.`$(+```<````L`(``$2<__^4`````$(.!(X!2@X0<@H.
M!$(+`"````#0`@``N)S__V0`````0@X0@`2!`X("@P%"#A2.!4H.(#0```#T
M`@``^)S__Z@!````1`XDA`F%"(8'AP:(!8D$B@.+`HX!1@ZD@`1"#L"`!`*:
M"@XD0@L`%````"P#``!HGO__&`````!"#@B$`HX!+````$0#``!HGO__'`$`
M``!"#A2$!84$A@.'`HX!2@XH=@H.%$+.Q\;%Q`X`0@L`1````'0#``!4G___
MC`8```!"#B2$"84(A@>'!H@%B02*`XL"C@%"#L@"`QX!"@XD0@L#0@$*#B1"
MSLO*R<C'QL7$#@!""P``)````+P#``"8I?__O`(```!"#A2$!84$A@.'`HX!
M3`X@`N0*#A1""QP```#D`P``+*C__U@`````0@X(A`*.`6;.Q`X`````)```
M``0$``!8M___Q`(```!"#A2$!84$A@.'`HX!1`Y``MP*#A1""S`````L!```
M/*C__X`+````0@XDA`F%"(8'AP:(!8D$B@.+`HX!2@[P`@.B!0H.)$(+```4
M````8`0``(BS__]@`````$(."(0"C@$@````>`0``-"S__\$`0```$(.((0(
MA0>&!H<%B`2)`XH"C@$L````G`0``+"T__]``0```$0.&(0&A06&!(<#B`*.
M`0*2"L[(Q\;%Q`X`0@L`````````F$4``!Q%`````````0```#`#```!````
M.P,```$```!%`P```0```%,#```!````8P,```\```";!```&0```%B?```;
M````!````!H```!<GP``'`````0````$````>`$```4```!@"P``!@```+`$
M```*````Y`0```L````0````%0`````````#````5*````(````0`@``%```
M`!$````7````7!$``!$```!$$```$@```!@!```3````"````/O__V\````(
M^O__;Q<`````````````````````````````````````````````````````
M````9)\`````````````;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``
M;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L
M$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``8)\``````````````````%B?``````````````````"`H@``
M``````````!<GP````````BB``````````````````!DGP```````$2B````
M````Q*$``.0S``!`H@``R*$```I005(N<&T*`````/Q+``#,2````````-!)
M``#420``\$L``!Q)``#X2P```````-Q+``#D2P````````1,``!'0T,Z("A.
M971"4T0@;F(Q(#(P,C4P-S(Q*2`Q,BXU+C``1T-#.B`H;F(Q(#(P,C4P-S(Q
M*2`Q,BXU+C``)$YE=$)31#H@8W)T,"Y3+'8@,2XT(#(P,3@O,3$O,C8@,3<Z
M,S<Z-#0@:F]E<F<@17AP("0`)$YE=$)31#H@8W)T,"UC;VUM;VXN8RQV(#$N
M,S`@,C`R-2\P-2\P,B`R,SHP-#HP-B!R:6%S=')A9&@@17AP("0`)$YE=$)3
M1#H@8W)T:2Y3+'8@,2XQ(#(P,3(O,#@O,3,@,#(Z-#DZ,#0@;6%T="!%>'`@
M)``D3F5T0E-$.B!C<G1B96=I;BYC+'8@,2XQ-R`R,#$X+S$R+S(X(#$X.C$W
M.C$Q(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R=&5N9"Y3+'8@,2XR(#(P
M,3,O,3(O,C`@,34Z,S4Z-#@@:F]E<F<@17AP("0`\<\#````````````````
M``@````(````"`````@````;`````0``````````*BHJ($YO($QA8F5L(%!R
M;W9I9&5D("HJ*@``\<\#``````````````````@````(````"`````@````;
M`````0``````````*BHJ($YO($QA8F5L(%!R;W9I9&5D("HJ*@!!+@```&%E
M86)I``$D````!39+6@`&!P@!"0$*`A($%`$5`1<#&`$:`AP!(@%$`0`N<VAS
M=')T86(`+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D
M+G!A>``N;F]T92YN971B<V0N;6%R8V@`+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R
M`"YR96PN9'EN`"YR96PN<&QT`"YT97AT`"YR;V1A=&$`+F5H7V9R86UE7VAD
M<@`N96A?9G)A;64`+FEN:71?87)R87D`+F9I;FE?87)R87D`+FIC<@`N9'EN
M86UI8P`N9V]T`"YD871A`"YP97)S:7-T96YT`"YB<W,`+FYO:6YI=``N8V]M
M;65N=``N4U5.5U]C=&8`+D%232YA='1R:6)U=&5S````````````````````
M````````````````````````````````````"P````$````"````%`$``!0!
M```7```````````````!`````````!,````'`````@```"P!```L`0``&```
M````````````!``````````F````!P````(```!$`0``1`$``!0`````````
M``````0`````````-P````<````"````6`$``%@!```@```````````````$
M`````````$H````%`````@```'@!``!X`0``.`,```8`````````!`````0`
M``!0````"P````(```"P!```L`0``+`&```'`````P````0````0````6```
M``,````"````8`L``&`+``#D!``````````````!`````````&`````)````
M`@```$00``!$$```&`$```8`````````!`````@```!I````"0```$(```!<
M$0``7!$``!`"```&````$P````0````(````;0````$````&````;!,``&P3
M```L`P`````````````$````!````'(````!````!@```)@6``"8%@``&#(`
M````````````"`````````!X`````0````(```"P2```L$@``-=`````````
M``````0`````````@`````$````"````B(D``(B)```$`0`````````````$
M`````````(X````!`````@```(R*``",B@``S`0`````````````!```````
M``"8````#@````,```!8GP``6(\```0```````````````0````$````I```
M``\````#````7)\``%R/```$```````````````$````!````+`````!````
M`P```&"?``!@CP``!```````````````!`````````"U````!@````,```!D
MGP``9(\``/`````'``````````0````(````O@````$````#````5*```%20
M``!P`0`````````````$````!````,,````!`````P```,2A``#$D0``2```
M````````````!`````````#)`````0````,````,H@``#)(`````````````
M``````0`````````U0````@````#````0*(```R2``!$``````````````!`
M`````````-H````(`````P```(2B````````````````````````!```````
M``#B`````0```#``````````#)(``#T```````````````$````!````'P``
M``$````P`````````$F2```C`0`````````````!`````0```.L````!````
M``````````!LDP``CP``````````````!`````````#U`````P``<```````
M````^Y,``"\```````````````$``````````0````,``````````````"J4
6```%`0`````````````!````````````
