/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.trogdor.agent;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.internals.KafkaFutureImpl;
import org.apache.kafka.common.utils.Scheduler;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.trogdor.common.Platform;
import org.apache.kafka.trogdor.common.ThreadUtils;
import org.apache.kafka.trogdor.rest.WorkerDone;
import org.apache.kafka.trogdor.rest.WorkerRunning;
import org.apache.kafka.trogdor.rest.WorkerStarting;
import org.apache.kafka.trogdor.rest.WorkerState;
import org.apache.kafka.trogdor.rest.WorkerStopping;
import org.apache.kafka.trogdor.task.TaskSpec;
import org.apache.kafka.trogdor.task.TaskWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class WorkerManager {
    private static final Logger log = LoggerFactory.getLogger(WorkerManager.class);
    private final Platform platform;
    private final String nodeName;
    private final Scheduler scheduler;
    private final Time time;
    private final Map<String, Worker> workers;
    private final ScheduledExecutorService stateChangeExecutor;
    private final ScheduledExecutorService workerCleanupExecutor;
    private final ScheduledExecutorService shutdownExecutor;
    private final ShutdownManager shutdownManager = new ShutdownManager();

    WorkerManager(Platform platform, Scheduler scheduler) {
        this.platform = platform;
        this.nodeName = platform.curNode().name();
        this.scheduler = scheduler;
        this.time = scheduler.time();
        this.workers = new HashMap<String, Worker>();
        this.stateChangeExecutor = Executors.newSingleThreadScheduledExecutor(ThreadUtils.createThreadFactory("WorkerManagerStateThread", false));
        this.workerCleanupExecutor = Executors.newScheduledThreadPool(1, ThreadUtils.createThreadFactory("WorkerCleanupThread%d", false));
        this.shutdownExecutor = Executors.newScheduledThreadPool(0, ThreadUtils.createThreadFactory("WorkerManagerShutdownThread%d", false));
    }

    public void createWorker(final String id, TaskSpec spec) throws Exception {
        try (ShutdownManager.Reference ref = this.shutdownManager.takeReference();){
            final Worker worker = this.stateChangeExecutor.submit(new CreateWorker(id, spec, this.time.milliseconds())).get();
            if (worker == null) {
                log.info("{}: Ignoring request to create worker {}, because there is already a worker with that id.", (Object)this.nodeName, (Object)id);
                return;
            }
            KafkaFutureImpl haltFuture = new KafkaFutureImpl();
            haltFuture.thenApply((KafkaFuture.BaseFunction)new KafkaFuture.BaseFunction<String, Void>(){

                public Void apply(String errorString) {
                    if (errorString.isEmpty()) {
                        log.info("{}: Worker {} is halting.", (Object)WorkerManager.this.nodeName, (Object)id);
                    } else {
                        log.info("{}: Worker {} is halting with error {}", new Object[]{WorkerManager.this.nodeName, id, errorString});
                    }
                    WorkerManager.this.stateChangeExecutor.submit(new HandleWorkerHalting(worker, errorString, false));
                    return null;
                }
            });
            try {
                worker.taskWorker.start(this.platform, worker.status, (KafkaFutureImpl<String>)haltFuture);
            }
            catch (Exception e) {
                this.stateChangeExecutor.submit(new HandleWorkerHalting(worker, "worker.start() exception: " + e.getMessage(), true));
            }
            this.stateChangeExecutor.submit(new FinishCreatingWorker(worker));
        }
    }

    public TaskSpec stopWorker(String id) throws Exception {
        try (ShutdownManager.Reference ref = this.shutdownManager.takeReference();){
            TaskSpec taskSpec = this.stateChangeExecutor.submit(new StopWorker(id)).get();
            if (taskSpec == null) {
                throw new KafkaException("No task found with id " + id);
            }
            TaskSpec taskSpec2 = taskSpec;
            return taskSpec2;
        }
    }

    public TreeMap<String, WorkerState> workerStates() throws Exception {
        try (ShutdownManager.Reference ref = this.shutdownManager.takeReference();){
            TreeMap<String, WorkerState> treeMap = this.stateChangeExecutor.submit(new GetWorkerStates()).get();
            return treeMap;
        }
    }

    public void beginShutdown() throws Exception {
        if (this.shutdownManager.shutdown()) {
            this.shutdownExecutor.submit(new Shutdown());
        }
    }

    public void waitForShutdown() throws Exception {
        while (!this.shutdownExecutor.isShutdown()) {
            this.shutdownExecutor.awaitTermination(1L, TimeUnit.DAYS);
        }
    }

    class Shutdown
    implements Callable<Void> {
        Shutdown() {
        }

        @Override
        public Void call() throws Exception {
            log.info("{}: Shutting down WorkerManager.", (Object)WorkerManager.this.platform.curNode().name());
            for (Worker worker : WorkerManager.this.workers.values()) {
                WorkerManager.this.stateChangeExecutor.submit(new StopWorker(worker.id));
            }
            WorkerManager.this.shutdownManager.waitForQuiescence();
            WorkerManager.this.workerCleanupExecutor.shutdownNow();
            WorkerManager.this.stateChangeExecutor.shutdownNow();
            WorkerManager.this.workerCleanupExecutor.awaitTermination(1L, TimeUnit.DAYS);
            WorkerManager.this.stateChangeExecutor.awaitTermination(1L, TimeUnit.DAYS);
            WorkerManager.this.shutdownExecutor.shutdown();
            return null;
        }
    }

    class GetWorkerStates
    implements Callable<TreeMap<String, WorkerState>> {
        GetWorkerStates() {
        }

        @Override
        public TreeMap<String, WorkerState> call() throws Exception {
            TreeMap<String, WorkerState> workerMap = new TreeMap<String, WorkerState>();
            for (Worker worker : WorkerManager.this.workers.values()) {
                workerMap.put(worker.id(), worker.state());
            }
            return workerMap;
        }
    }

    class CleanupWorker
    implements Callable<Void> {
        private final Worker worker;

        CleanupWorker(Worker worker) {
            this.worker = worker;
        }

        @Override
        public Void call() throws Exception {
            String failure = "";
            try {
                this.worker.taskWorker.stop(WorkerManager.this.platform);
            }
            catch (Exception exception) {
                log.error("{}: worker.stop() exception", (Object)WorkerManager.this.nodeName, (Object)exception);
                failure = exception.getMessage();
            }
            WorkerManager.this.stateChangeExecutor.submit(new CompleteWorker(this.worker, failure));
            return null;
        }
    }

    class StopWorker
    implements Callable<TaskSpec> {
        private final String id;

        StopWorker(String id) {
            this.id = id;
        }

        @Override
        public TaskSpec call() throws Exception {
            Worker worker = (Worker)WorkerManager.this.workers.get(this.id);
            if (worker == null) {
                return null;
            }
            switch (worker.state) {
                case STARTING: {
                    log.info("{}: Cancelling worker {} during its startup process.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                    worker.state = State.CANCELLING;
                    break;
                }
                case CANCELLING: {
                    log.info("{}: Can't stop worker {}, because it is already being cancelled.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                    break;
                }
                case RUNNING: {
                    log.info("{}: Stopping running worker {}.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                    worker.transitionToStopping();
                    break;
                }
                case STOPPING: {
                    log.info("{}: Can't stop worker {}, because it is already stopping.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                    break;
                }
                case DONE: {
                    log.debug("{}: Can't stop worker {}, because it is already done.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                }
            }
            return worker.spec();
        }
    }

    static class CompleteWorker
    implements Callable<Void> {
        private final Worker worker;
        private final String failure;

        CompleteWorker(Worker worker, String failure) {
            this.worker = worker;
            this.failure = failure;
        }

        @Override
        public Void call() throws Exception {
            if (this.worker.error.isEmpty() && !this.failure.isEmpty()) {
                this.worker.error = this.failure;
            }
            this.worker.transitionToDone();
            return null;
        }
    }

    class HandleWorkerHalting
    implements Callable<Void> {
        private final Worker worker;
        private final String failure;
        private final boolean startupHalt;

        HandleWorkerHalting(Worker worker, String failure, boolean startupHalt) {
            this.worker = worker;
            this.failure = failure;
            this.startupHalt = startupHalt;
        }

        @Override
        public Void call() throws Exception {
            if (this.worker.error.isEmpty()) {
                this.worker.error = this.failure;
            }
            String verb = this.worker.error.isEmpty() ? "halting" : "halting with error [" + this.worker.error + "]";
            switch (this.worker.state) {
                case STARTING: {
                    if (this.startupHalt) {
                        log.info("{}: Worker {} {} during startup.  Transitioning to DONE.", new Object[]{WorkerManager.this.nodeName, this.worker.id, verb});
                        this.worker.transitionToDone();
                        break;
                    }
                    log.info("{}: Worker {} {} during startup.  Transitioning to CANCELLING.", new Object[]{WorkerManager.this.nodeName, this.worker.id, verb});
                    this.worker.state = State.CANCELLING;
                    break;
                }
                case CANCELLING: {
                    log.info("{}: Cancelling worker {} {}.  ", new Object[]{WorkerManager.this.nodeName, this.worker.id, verb});
                    break;
                }
                case RUNNING: {
                    log.info("{}: Running worker {} {}.  Transitioning to STOPPING.", new Object[]{WorkerManager.this.nodeName, this.worker.id, verb});
                    this.worker.transitionToStopping();
                    break;
                }
                case STOPPING: {
                    log.info("{}: Stopping worker {} {}.", new Object[]{WorkerManager.this.nodeName, this.worker.id, verb});
                    break;
                }
                case DONE: {
                    log.info("{}: Can't halt worker {} because it is already DONE.", (Object)WorkerManager.this.nodeName, (Object)this.worker.id);
                }
            }
            return null;
        }
    }

    class FinishCreatingWorker
    implements Callable<Void> {
        private final Worker worker;

        FinishCreatingWorker(Worker worker) {
            this.worker = worker;
        }

        @Override
        public Void call() throws Exception {
            switch (this.worker.state) {
                case CANCELLING: {
                    log.info("{}: Worker {} was cancelled while it was starting up.  Transitioning to STOPPING.", (Object)WorkerManager.this.nodeName, (Object)this.worker.id);
                    this.worker.transitionToStopping();
                    break;
                }
                case STARTING: {
                    log.info("{}: Worker {} is now RUNNING.  Scheduled to stop in {} ms.", new Object[]{WorkerManager.this.nodeName, this.worker.id, this.worker.spec.durationMs()});
                    this.worker.transitionToRunning();
                    break;
                }
            }
            return null;
        }
    }

    class CreateWorker
    implements Callable<Worker> {
        private final String id;
        private final TaskSpec spec;
        private final long now;

        CreateWorker(String id, TaskSpec spec, long now) {
            this.id = id;
            this.spec = spec;
            this.now = now;
        }

        @Override
        public Worker call() throws Exception {
            Worker worker = (Worker)WorkerManager.this.workers.get(this.id);
            if (worker != null) {
                log.info("{}: Task ID {} is already in use.", (Object)WorkerManager.this.nodeName, (Object)this.id);
                return null;
            }
            worker = new Worker(this.id, this.spec, this.now);
            WorkerManager.this.workers.put(this.id, worker);
            log.info("{}: Created a new worker for task {} with spec {}", new Object[]{WorkerManager.this.nodeName, this.id, this.spec});
            return worker;
        }
    }

    class Worker {
        private final String id;
        private final TaskSpec spec;
        private final TaskWorker taskWorker;
        private final AtomicReference<String> status = new AtomicReference<String>("");
        private final long startedMs;
        private State state = State.STARTING;
        private long doneMs = -1L;
        private String error = "";
        private Future<TaskSpec> timeoutFuture = null;
        private ShutdownManager.Reference reference;

        Worker(String id, TaskSpec spec, long now) {
            this.id = id;
            this.spec = spec;
            this.taskWorker = spec.newTaskWorker(id);
            this.startedMs = now;
            this.reference = WorkerManager.this.shutdownManager.takeReference();
        }

        String id() {
            return this.id;
        }

        TaskSpec spec() {
            return this.spec;
        }

        WorkerState state() {
            switch (this.state) {
                case STARTING: {
                    return new WorkerStarting(this.spec);
                }
                case RUNNING: {
                    return new WorkerRunning(this.spec, this.startedMs, this.status.get());
                }
                case CANCELLING: 
                case STOPPING: {
                    return new WorkerStopping(this.spec, this.startedMs, this.status.get());
                }
                case DONE: {
                    return new WorkerDone(this.spec, this.startedMs, this.doneMs, this.status.get(), this.error);
                }
            }
            throw new RuntimeException("unreachable");
        }

        void transitionToRunning() {
            this.state = State.RUNNING;
            this.timeoutFuture = WorkerManager.this.scheduler.schedule(WorkerManager.this.stateChangeExecutor, (Callable)new StopWorker(this.id), this.spec.durationMs());
        }

        void transitionToStopping() {
            this.state = State.STOPPING;
            if (this.timeoutFuture != null) {
                this.timeoutFuture.cancel(false);
                this.timeoutFuture = null;
            }
            WorkerManager.this.workerCleanupExecutor.submit(new CleanupWorker(this));
        }

        void transitionToDone() {
            this.state = State.DONE;
            this.doneMs = WorkerManager.this.time.milliseconds();
            if (this.reference != null) {
                this.reference.close();
                this.reference = null;
            }
        }
    }

    static enum State {
        STARTING,
        CANCELLING,
        RUNNING,
        STOPPING,
        DONE;

    }

    static class ShutdownManager {
        private boolean shutdown = false;
        private long refCount = 0L;

        ShutdownManager() {
        }

        synchronized Reference takeReference() {
            if (this.shutdown) {
                throw new KafkaException("WorkerManager is shut down.");
            }
            ++this.refCount;
            return new Reference();
        }

        synchronized boolean shutdown() {
            if (this.shutdown) {
                return false;
            }
            this.shutdown = true;
            return true;
        }

        synchronized void waitForQuiescence() throws InterruptedException {
            while (!this.shutdown || this.refCount > 0L) {
                this.wait();
            }
        }

        class Reference
        implements AutoCloseable {
            AtomicBoolean closed = new AtomicBoolean(false);

            Reference() {
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void close() {
                if (this.closed.compareAndSet(false, true)) {
                    ShutdownManager shutdownManager = ShutdownManager.this;
                    synchronized (shutdownManager) {
                        ShutdownManager.this.refCount--;
                        if (ShutdownManager.this.shutdown && ShutdownManager.this.refCount == 0L) {
                            ShutdownManager.this.notifyAll();
                        }
                    }
                }
            }
        }
    }
}

