package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````J!8``#0````HE`````0`!30`(``'
M`"@`&P`:``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````7(X`
M`%R.```%```````!``$```!<C@``7(X!`%R.`0"T`@``*`,```8```````$`
M`@```&B.``!HC@$`:(X!`/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&1DAP``9(<``&2'```,`0``#`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$````\N$U!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-FAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"H%@````````,`"P``````9(X!```````#`!$`D@``````````````
M$@````P``````````````!(````4```````````````2````<P,```0B```(
M````$@`+`!P``````````````"(````K```````````````2````/`,`````
M````````$@```(0#```0D0$``````!``%0`Q```````````````2````D@,`
M`*0I```@````$@`+`#@``````````````!(```!)`P`````````````2````
M0```````````````$@```*T#``#$,P``9````!(`"P"#```````````````2
M````P0,``(21`0``````$``5`$@``````````````!(```"Y`P``$)$!````
M```0`!0`6```````````````$@```&<``````````````!(```!M````````
M```````2````=```````````````$@```'X``````````````!(```"(````
M```````````2````P`,``(21`0``````$``5`)$``````````````!(```!0
M`@`````````````2````EP``````````````$@```)X``````````````!(`
M``"L```````````````2````O0``````````````$0```,H`````````````
M`!(```#,`P``C"0``#`````2``L`V0,``+PD``!@````$@`+`.4#``"L*@``
MB`8``!(`"P#8```````````````6````ZP``````````````$@```%L#````
M`````````!(```#V```````````````2````_P```("1`0`$````$0`5``<!
M`````````````!(````0`0`````````````2````,@,`````````````$@``
M`!P!`````````````!(````C`0`````````````2````\@,``(A````8`0``
M$@`+`#8!`````````````!$```!(`0`````````````2````6`$`````````
M````$@```&8!`````````````!(```!K`0`````````````1````>@$`````
M````````$@```/T#```,D0$`!````!$`%`")`0`````````````2````E0$`
M````````````$@```*H!`````````````!(```"U`0`````````````2````
M;@,``(21`0``````$``5`.P"`````````````!(```#(`0`````````````2
M````0`0``*@6````````$@`+`-8!`````````````!(```#>`0``````````
M```B````]@$`````````````$@````("`````````````!(````'!```A)$!
M```````0`!4`#P0``!PE``#,`0``$@`+``@"`````````````"(````>`@``
M```````````2````*0(`````````````$@```!P$``"@00``>`$``!(`"P`Z
M`@`````````````2````)@0``!"1`0``````$``5`#($```D(0``:````!(`
M"P`Z!```]$,``)0"```2``L`00(`````````````$@```#\$``"H%@``````
M`!(`"P!(`@`````````````2````1P0```PB``"``@``$@`+`%0$``#$*0``
MZ````!(`"P!/`@`````````````2````9@0``.@F``"`````$@`+`%@"````
M`````````!(```!I`@`````````````2````^P(`````````````$@```'`"
M`````````````!(```!T!```:"<``%P````2``L`?P(`````````````$@``
M`(@"`````````````!(```".`@`````````````2````)0,`````````````
M$@```),"`````````````!(````8`P``1)$!``0````1`!4`FP(`````````
M````$@```%`#`````````````!(```"B`@`````````````2````?`0``"A`
M``!@````$@`+`*D"`````````````!(```"X`@`````````````1````#0,`
M`,B0`0`$````$0`4`(4$```T,0``D`(``!(`"P`O`0`````````````2````
MO0(`````````````$@```&(#`````````````!(`````;&EB<&5R;"YS;P!R
M96%L;&]C`&UE;6UO=F4`7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T
M<FYC;7``<W1R<F-H<@!B;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`
M86)O<G0`=6YL:6YK`%]?;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X
M:70`;65M8W!Y`'!E<FQ?9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D
M;U]U;F1U;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R
M;%]A;&QO8P!C;&]S961I<@!E;G9I<F]N`')E861L:6YK`%]?;W!E;F1I<C,P
M`&UE;7-E=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`
M4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A9`!03%]M96UO<GE?
M=W)A<`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!097)L7V-R;V%K7VYO
M8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y
M<U]T97)M`%]?97)R;F\`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`4&5R;%]M
M9U]S970`;6MD:7(`7U]R96=I<W1E<E]F<F%M95]I;F9O`'!E<FQ?<&%R<V4`
M4&5R;%]N97=84U]F;&%G<P!A8V-E<W,`9V5T<&ED`&UE;6-H<@!V9G!R:6YT
M9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!097)L7W-V7W-E='!V;@!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W1L<U]G
M971?861D<@!?7W-&`'-T<F-M<`!L:6)M+G-O+C``;&EB8W)Y<'0N<V\N,0!L
M:6)P=&AR96%D+G-O+C$`<'1H<F5A9%]A=&9O<FL`<W1R9'5P`&QI8F,N<V\N
M,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`7U]G971P=W5I9#4P`%]?<WES
M8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L:6)C7VEN:70`<W1R=&]K`%]?
M<F5A9&1I<C,P`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,"XP+V5A<FUV-FAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`7(X!`!<```!@C@$`%P```&R0`0`7
M````=)`!`!<```!\D`$`%P```(R0`0`7````D)`!`!<```"8D`$`%P```*"0
M`0`7````L)`!`!<```"XD`$`%P```,"0`0`7````Q)`!`!<```#(D`$`%P``
M`,R0`0`7````W)`!`!<```#@D`$`%P```.B0`0`7````[)`!`!<```#PD`$`
M%P```/20`0`7````^)`!`!<`````D0$`%P````21`0`7````#)$!`!<```!P
MD`$`%0<``'B0`0`5$P``@)`!`!4A``"$D`$`$28``(B0`0`2)@``E)`!`!4Q
M``"<D`$`%34``*20`0`5.P``J)`!`!5!``"LD`$`%48``+20`0`550``O)`!
M`!5E``!DCP$`%@,``&B/`0`6!```;(\!`!8%``!PCP$`%@<``'2/`0`6"```
M>(\!`!8)``!\CP$`%@L``("/`0`6#0``A(\!`!8.``"(CP$`%@\``(R/`0`6
M$0``D(\!`!85``"4CP$`%A8``)B/`0`6%P``G(\!`!88``"@CP$`%AD``*2/
M`0`6&@``J(\!`!8<``"LCP$`%AT``+"/`0`6'@``M(\!`!8?``"XCP$`%B``
M`+R/`0`6(@``P(\!`!8G``#$CP$`%B@``,B/`0`6*0``S(\!`!8K``#0CP$`
M%BP``-2/`0`6+0``V(\!`!8N``#<CP$`%B\``."/`0`6,@``Y(\!`!8S``#H
MCP$`%C0``.R/`0`6-@``\(\!`!8X``#TCP$`%CD``/B/`0`6.@``_(\!`!8]
M````D`$`%CX```20`0`60```")`!`!9!```,D`$`%D(``!"0`0`60P``%)`!
M`!9&```8D`$`%D<``!R0`0`62```()`!`!9*```DD`$`%DX``"B0`0`64```
M+)`!`!93```PD`$`%E8``#20`0`65P``.)`!`!98```\D`$`%EH``$"0`0`6
M6P``1)`!`!9<``!(D`$`%ET``$R0`0`67@``4)`!`!9@``!4D`$`%F$``%B0
M`0`68@``7)`!`!9D``!@D`$`%F@``&20`0`6:0``:)`!`!9J```$X"WE!."?
MY0[@C^`(\+[ES'L!``#&C^(7RHSBS/N\Y0#&C^(7RHSBQ/N\Y0#&C^(7RHSB
MO/N\Y0#&C^(7RHSBM/N\Y0#&C^(7RHSBK/N\Y0#&C^(7RHSBI/N\Y0#&C^(7
MRHSBG/N\Y0#&C^(7RHSBE/N\Y0#&C^(7RHSBC/N\Y0#&C^(7RHSBA/N\Y0#&
MC^(7RHSB?/N\Y0#&C^(7RHSB=/N\Y0#&C^(7RHSB;/N\Y0#&C^(7RHSB9/N\
MY0#&C^(7RHSB7/N\Y0#&C^(7RHSB5/N\Y0#&C^(7RHSB3/N\Y0#&C^(7RHSB
M1/N\Y0#&C^(7RHSB//N\Y0#&C^(7RHSB-/N\Y0#&C^(7RHSB+/N\Y0#&C^(7
MRHSB)/N\Y0#&C^(7RHSB'/N\Y0#&C^(7RHSB%/N\Y0#&C^(7RHSB#/N\Y0#&
MC^(7RHSB!/N\Y0#&C^(7RHSB_/J\Y0#&C^(7RHSB]/J\Y0#&C^(7RHSB[/J\
MY0#&C^(7RHSBY/J\Y0#&C^(7RHSBW/J\Y0#&C^(7RHSBU/J\Y0#&C^(7RHSB
MS/J\Y0#&C^(7RHSBQ/J\Y0#&C^(7RHSBO/J\Y0#&C^(7RHSBM/J\Y0#&C^(7
MRHSBK/J\Y0#&C^(7RHSBI/J\Y0#&C^(7RHSBG/J\Y0#&C^(7RHSBE/J\Y0#&
MC^(7RHSBC/J\Y0#&C^(7RHSBA/J\Y0#&C^(7RHSB?/J\Y0#&C^(7RHSB=/J\
MY0#&C^(7RHSB;/J\Y0#&C^(7RHSB9/J\Y0#&C^(7RHSB7/J\Y0#&C^(7RHSB
M5/J\Y0#&C^(7RHSB3/J\Y0#&C^(7RHSB1/J\Y0#&C^(7RHSB//J\Y0#&C^(7
MRHSB-/J\Y0#&C^(7RHSB+/J\Y0#&C^(7RHSB)/J\Y0#&C^(7RHSB'/J\Y0#&
MC^(7RHSB%/J\Y0#&C^(7RHSB#/J\Y0#&C^(7RHSB!/J\Y0#&C^(7RHSB_/F\
MY0#&C^(7RHSB]/F\Y0#&C^(7RHSB[/F\Y0#&C^(7RHSBY/F\Y0#&C^(7RHSB
MW/F\Y0#&C^(7RHSBU/F\Y0#&C^(7RHSBS/F\Y0#&C^(7RHSBQ/F\Y0`0H.$"
M`*#A!]#-XSD``.H-P*#A,-@MZ40@G^5$,)_E`B"/X$!`G^4#,)+G!$"/X`,`
M5.$$L$SB,*B=*`-0@^($,(3B`U!%X`-0Q>,#4(7@!#"4Y#/_+^$%`%3A^___
M&C"HG>B(>`$`%`$``(1W`0`-P*#A\-@MZ6P@G^5L$)_E;#"?Y0(@C^`!0)+G
M`S"2YP2P3.(#`%3A\*B=*`0@U.6@`%+C#P``&@A`A.('<(/B!'!'X`=PQ^,$
M<(?@"%`4Y01@H.$`,)7E,_\OX0<`5.$``(7E\*B="`0PUN4(0(3BH`!3X_3_
M_PHB___K)'@!`#@!```<`0``#<"@X8#8+>EP@9_E!+!,X@S03>(`8%'B"("/
MX$L```I<,9_E`""6Y0,0F.=4,9_E`,"2Y0!@@>4#<)CG"#"6Y0``7.,`,(?E
M.@``"C@QG^4#$)CG`,"!Y0`@DN4`,-+E``!3XP4```H!((+B+P!3XP`@@04!
M,-+D``!3X_K__QH``%#C````"O#^_^N+___K^#"?Y0,PF.<``%/C-0``"NPP
MG^7L0)_E`S"8YP1`C^`#`%3A"```*@-0@^($,(3B`U!%X`-0Q>,#4(7@!#"4
MY#/_+^$%`%3A^___&K@`G^6X0)_E``"/X-C^_^NP,)_E!$"/X`,PF.<#`%3A
M"```*@-0@^($,(3B`U!%X`-0Q>,#4(7@!#"4Y#/_+^$$`%7A^___&@`@E^4`
M$);E!`"6Y<<*`.NM_O_K3""?Y60PG^4"()CG`S"/X``P@N7+___J$R"@XU`P
MG^4`((WE`S"/X`(@H.,$`*#C`!"@X_/^_^L!`*#CT/[_ZW[__^O'___JF'<!
M`&`!```T`0``:`$``%@!```D`0``"'8!`"S^___$=0$`0`$``%1X`0"$+0``
M#<"@X=0PG^7PVRWI`(!0X@2P3.(!<*#A`S"/X"L```J\()_E`C"3YP"0D^4`
M`%GC*```"@`PV.4]4%/B`5"@$P``4^,`4*`#``!5XQ$``!H`0)GE``!4XPP`
M``H)8*#A!2"@X0@0H.$$`*#AB/[_ZP``4.,"```:!3#4YST`4^,+```*!$"V
MY0``5./S__\:!`"@X?"KG>@(4*#A`3#UY0``4^,]`%,3^___&@A01>#F___J
M"6!&X`%0A>)&8:#A!4"$X`!@A^4$`*#A\*N=Z`A`H.'M___J"4"@X>O__^KH
M=0$`-`$```W`H.'PV"WI!+!,X@C03>(`4*#A@'"?Y8`PG^4'<(_@`V"7YR1`
M2^(`,);E(#`+Y000H.$%`*#AMO__ZP``4.,.```*6#"?Y200&^4#,)?G`""3
MY0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J
M("`;Y0`PEN4#`%+A`0``&AS02^+PJ)WH;?[_Z_QT`0`\`0``-`$```W`H.'P
MWRWI!+!,XJS03>(T,9_E-"&?Y0,PC^`"0)/G`&!0X@`PE.4P,`OE`@``"@`P
MUN4``%/C!0``&C`@&^4`,)3E`P!2X3X``!HHT$OB\*^=Z&;^_^L`<%#B]O__
M"NPPG^7L@)_E`S"/X.B0G^70,`OES#!+X@B`C^`)D(_@U#`+Y0<`H.'+_O_K
M``!0XR4```H-4(#B!0"@X0@0H.'"_O_K``!0X_7__PH)$*#A!0"@X;W^_^L`
M`%#C\/__"@8`H.&P_O_K`*"@X04`H.&M_O_K``"*X`(`@.*P_O_K`*"@X04P
MH.$&(*#AT!`;Y9S^_^O4$!OE"@"@X1O^_^L!`'#C"@"@X0,```K$,!OE#SH#
MX@$)4^,(```*"O[_ZPH`H.'__?_KU?__Z@<`H.$I_O_K!@"@X>C]_^N^___J
MKO__Z_7__^H7_O_K6'0!`#P!``!H*P``4"L``%`K```-P*#A\-\MZ02P3.(4
MT$WB`4"@X0R2G^4,,I_E`"#1Y0F0C^`#8)GG/0!2XP%`@0(`,);E`%"@X00`
MH.$P,`OE>?[_ZS002^(`<*#A!0"@X2[__^L`@%#B/@``"G+^_^L'`%#A-0``
M*L`QG^4T@!OE`Y"9YPB!H.$`,-7E/0!3X@$`H!,``%/C``"@`P``4.,%```*
M!0"@X0$P\.4``%/C/0!3$_O__QH%`$#@`G"'X@``A^``<)GE8?[_ZP``4.,(
M`(?G5```"@`PU>4`()GE/0!3XP``4Q,T$!OE`0"%X@$1DN<`,,'E!@``"@$@
M@>(!,-#D`A"@X3T`4^,``%,3`3#"Y/G__QH],*#C`2"@X0`PP>4!,-3D``!3
MXP$PXN7[__\:``"@XS`@&^4`,);E`P!2X3H``!HHT$OB\*^=Z`$P2.(!(-3D
M``!2XP$@X^7[__\:\O__ZM`PG^4#D)GG`""9Y0"@DN4``%KC"!"@`PD```H(
M,*#A`A"@X0```.H*,*#A!`"QY0&@@^(``%#C^O__&@,P@^(#$:#AD#"?Y0,P
MC^```)/E``!0XQ,``!H!P*#C`0"@X0#`@^4\(`OE'_[_ZP`P4.(X,`OE$@``
M"CP@&^4*@:#A`A"@X0@@H.%@_?_K.#`;Y0`PB>4`$*#C`2"*X@(1@^<TH`OE
MG?__Z@(`H.%4_?_K``!0XP`PH.$``(GE"H&@$?/__QH``.#CP/__ZHC]_^OP
M<@$`/`$``#0!``"(<P$`#<"@X3#8+>D$L$SBH-!-XG`PG^5P()_E`S"/X`)`
MD^>T$$OB`#"4Y0!0H.$8,`OE8OW_ZP``4.,``*`3!```&JPP&^4/.@/B"@I3
MXP$)4Q,%```*&"`;Y0`PE.4#`%+A!P``&A302^(PJ)WH!0"@X0(0H..T_?_K
M$`]OX:`"H.'S___J8OW_Z\!P`0`\`0``#<"@X?#;+>D$L$SB0=Y-XO0PG^5"
MSDOB",!,XNP@G^4#,(_@,,0+Y0)0D^<`<*#AW`"?Y0`PE>4!D*#AU""?Y0P0
MH.$``(_@*#`+Y5O]_^L`@%#B(@``VKP`G^4!0$CB``"/X,']_^LD,$OB!#"#
MX`0T4^4`8)#E``!3XQ$```H(0*#A!""@X2\0H.,P!!OEC_W_ZP``4.,,```*
M!#"@X0<0H.$&`*#A,"0;Y93]_^LH(!OE`#"5Y0,`4N$1```:)-!+XO"KG>@`
M`%3C[/__&@D@H.$'$*#A!@"@X2O]_^OR___J-`"?Y0``C^"?_?_K`&"0Y0D@
MH.$'$*#A!@"@X2+]_^OI___J'?W_ZRAP`0`\`0``;"<``/\#```4<0$`C'`!
M``W`H.$PV"WI`=M-X@2P3.((T$WBT#"?Y=`@G^4#,(_@`E"3YT%.2^((0$3B
M`#"5Y0`0H.$!*Z#C!`"@X1@P"^76_/_K!`!`X`$`0.(`,(3@!`!3X0H``)H`
M(-3G+P!2XP<``!H`$*#C`@``Z@`@T^4O`%+C`@``&@$00^0$`%/A^?__&B\0
MH.,$`*#AS_S_ZP``4.,*```*!`!0X0`PH(,`,,"%!`"@X4G]_^L8(!OE`#"5
MY0,`4N$*```:%-!+XC"HG>@8(!OE`#"5Y0,`4N$$```:&`"?Y0``C^`4T$OB
M,&B=Z#K]_^K<_/_K"&\!`#P!``"D)0``#<"@X?#8+>E$()_E1`"?Y40PG^4$
ML$SB`B"/X```C^`X4)_E.$"?Y0-@DN=(_?_K!5"/X`1`C^`%,*#A!B"@X000
MH.$``)#E'-!+XO!HG>CT_/_J%&X!`#QO`0`@`0``0"4``'0E```-P*#A,-@M
MZ50`G^4(T$WB!+!,X@``C^`S_?_K`%"@X0``D.7;___K`!"@XS1`G^4T,)_E
M!$"/X`,@E.<L,)_E``"5Y000C>4D$)_E`S"/X``PC>4!$(_@\_S_ZQ302^(P
MJ)WHW&X!`)!M`0!L`0``X"<```@E`````*#C'O\OX0W`H.'PWRWI$]U-X@2P
M3.(,T$WB-#*?Y30BG^4#,(_@`B"3YP!`H.$3?4OB)`*?Y0`PDN4,<$?B`6"@
MX0``C^`$$$?BX"0+Y3`P"^6\_?_K+Q"@XP!0H.$$`*#A6OS_ZP``4.,+```*
M[`&?Y000H.$``(_@;O[_ZP0`H.'@-!OE,"`;Y0`PD^4#`%+A;@``&BC02^+P
MKYWH!@"@X=3\_^NX$9_E`1"/X'W\_^L`D%#B[/__"J@QG^6HH9_E`S"/X-PT
M"^6@,9_EH(&?Y0,PC^#D-`OEF#&?Y0J@C^`#,(_@V#0+Y0`PV>4``%/C/0``
M"@``5>,$```*"1"@X04`H.'?_/_K``!0XRT```H)`*#ATOS_ZP$00.(!,(G@
M`P!9X0!@H.$-```J`1#9YR\`4>,*```:``"@XP(``.H`(-/E+P!2XP(``!H!
M`$/D`P!9X?G__QH)`*#AP/S_ZP!@H.$$`*#AO?S_ZP8`@.`!`(#B"`!0X;O_
M_XK8-!OE0VY+X@0PC>4!*Z#C"C"@X0`0H.,(0(WE`)"-Y08`H.%8_/_K!Q"@
MX08`H.$E_/_K``!0XP,``!K$-!OE#SH#X@()4^,2```*K!"?Y0``H.,!$(_@
M,_S_ZP"04.*B__\*`##9Y0``4^/!__\:``!5XQ,```K<%!OE!0"@X:'\_^L`
M`%#C`6"@$]R4&Q7M__\*T___Z@$0H.,&`*#A9OS_ZP``4./G__\:4`"?Y080
MH.$``(_@__W_ZP8`H.%M_/_KCO__Z@%@H./DE!OEQ/__Z@O\_^LH;0$`/`$`
M`+@D``"0)```;"0``.`C```\)```T",``/\#````)P``1",``-0B```-P*#A
M+Q"@XS#8+>D$L$SB`$"@X=7[_^L``%#C`4"`$@0`H.$4T$OB,&B=Z%#\_^H-
MP*#A,-@MZ0!`4.($L$SB"@``"@`PU.4``%/C!P``"F+\_^L!`(#B`0M0XP@`
M`(H$`*#A%-!+XC!HG>C+_O_J%`"?Y0``C^`4T$OB,&B=Z#O\_^H``*#C,*B=
MZ*@A```-P*#A<-@MZ02P3.(,T$WB7#&?Y5PAG^4#,(_@`E"3YU0!G^4`,)7E
M``"/X"`P"^4Y_?_K1`&?Y41AG^4``(_@-?W_ZSP!G^4&8(_@``"/X#']_^LP
M`9_E)$!+X@``C^`M_?_K)`&?Y0``C^`J_?_K'`&?Y0``C^`G_?_K%`&?Y0``
MC^`D_?_K#`&?Y0``C^`A_?_K!@"@X1_]_^O\`)_E``"/X!S]_^OT`)_E``"/
MX!G]_^OL`)_E!!"@X0``C^#;_/_K`!!0X@$```H&`*#AD_W_Z]``G^4$$*#A
M``"/X-/\_^L`$%#B`@``"KP`G^4``(_@BOW_Z[0`G^4$$*#A``"/X,K\_^L`
M$%#B#0``"J``G^4``(_@@?W_ZY@0G^68`)_E`1"/X```C^!\_?_K("`;Y0`P
ME>4#`%+A"P``&AC02^)PJ)WH=`"?Y000H.$``(_@MOS_ZP`04.+N__\*8`"?
MY0``C^!M_?_KZO__ZG[[_^L<:@$`/`$``.0A``#<(0``%"(``-0A``#0(0``
MS"$``-`A``!D(0``Q"$``,@A``!$(0``N"$``*PA``"L(0``J"$``-`@``"H
M(0``#"$``&@A``#L(```#<"@X1#8+>D$L$SB#-!-XEPPG^5<()_E`S"/X`)`
MD^=4`)_E`""4Y1P02^(``(_@&"`+Y8O\_^L``%#C!0``"@``T.4``%#C,`!0
M$P$`H`,``*`3`0`@XA@@&^4`,)3E`P!2X0$``!H0T$OB$*B=Z$O[_^M0:`$`
M/`$``%@@```-P*#A#P`MZ0#8+>D4L$SB"-!-XC0PG^4T$)_E`S"/X`$`D^<(
M((OB``"0Y200G^44(`OE$``+Y0$`D^<$$)OEL`"`XHW[_^O_`*#C]?K_Z\QG
M`0`\`0``9`$```W`H.'PWRWI`=A-X@2P3.(DT$WB`N"@XZ01G^6DP9_E`1"/
MX`P0D><!R$OB1,!,X@`0C.4`X(WE`!"1Y0`@H.,`,*#C`("@X3`0"^6#^__K
M`4!0X@!0P>+%;Z#A!@"@X0!PH./&'Z#A!F`DX``@5N!R8/_F`"`FX`$P)^``
M(%+@`3##X`)`5.`#4,7@``!4XP`PU>($8*#A!7"@X3\``+HD,9_E`2A+XB!1
MG^4#,(_@0"!"X@%(2^(4D9_E`#""Y050C^`X0$3B`!"@XP8@H.$',*#A`!"-
MY0@`H.%?^__K`##@XP`@X.,#`%'A`@!0`2T```H)(*#A!!"@X0@`H.$4^__K
M`0!PXQH```H'`%#C&P``F@@`0.(`H)3@`P``.A<``.H!H$KB!`!:X10``#H(
M(*#C!1"@X0H`H.$[^__K``!0X_;__QH$0$K@!`"6X,0?I^`!.$OB1#!#X@`P
MD^4P(!OE`#"3Y0,`4N$3```:*-!+XO"OG>A8`)_E``"/X(?__^L`,.#C`6A6
MX@!PQ^)$()_E`P!7X0(`5@')__\:``#@XP`0X./I___J`3A+XD`P0^(``)/E
M>?__Z\S__^K`^O_K;&<!`#P!``!\'P``1&@!``@``0"\'@````#__PW`H.$`
MV"WI!+!,XB?[_^L@^__K``!0XP``D!4`J)WH#<"@X?#9+>D$L$SB%-!-XK@P
MG^6X()_E`S"/X`)`D^>P8)_E`#"4Y09@C^``<*#A+!!+X@8`H.$H,`OETOO_
MZP!04.("```*`##5Y0``4^,)```:@`"?Y0<0H.$``(_@A?S_ZR@@&^4`,)3E
M`P!2X14``!H@T$OB\*F=Z`<`H.$(^__K`("@X04`H.$%^__K``"(X`(`@.((
M^__K`("@X3@PG^4X$)_E`%"-Y0<@H.$#,(_@`1"/X/'Z_^L($*#A!@"@X6S\
M_^OE___J??K_ZW1E`0`\`0``'!X``.@=``"H'```J!P```W`H.'PWRWI!+!,
MXDG?3>+XQ9_E^(6?Y0S`C^#T-9_E.`$+Y0B`C^`,((SB&$",XDMO2^)%[TOB
M`U"8YP<`DN@'`(;H#P"TZ`\`KN@#`)3H,,",XA).2^(#`([H!P"<Z`<`A.BP
M!9_E`#"5Y3Q1"^437DOB!1"@X0``C^`P,`OEA_O_ZP!P4.("```*`##7Y0``
M4^.+```:EO__ZP"@4.*-```*"@"@X<3Z_^N``*#A`0"`XL?Z_^L`(-KE`)"@
MX0``4N,)```*"4"@X4QUG^4'<(_@!`"@X0<0H.&N^O_K`2#ZY0)`A.(``%+C
M^/__&A0!&^4``%#C$@``"A%.2^("``#J!`"4Y```4.,-```*!1"@X6+[_^L`
M<%#B^/__"@`PU^4``%/C]?__"J;\_^L``%#C\O__"@<`H.&$^O_K`*!0XA<`
M`!H`<*#C+#!+X@<Q@^``01/E``!4XPH```H`,-3E``!3XP<```H$`*#AE?S_
MZP``4.,!<(?B!```&@=!EN<``%3C]/__&@"@H.,$``#J!`"@X6WZ_^L``%#C
MZ?__"@"@H.$*`*#A@_K_ZP!@H.$)`*#A@/K_ZP``AN`!:X#B!&"&X@8`H.&!
M^O_K`$"@X4@TG^5(%)_E`S"/X``PC>5`-)_E"B"@X0,PC^`!$(_@!)"-Y6?Z
M_^L$`*#A!QV@XS?Z_^L!`'#C&@``"OP02^($`*#AV_G_ZP!P4.(#```:]#`;
MY0\Z`^(!"5/C00``"O0CG^7T$Y_E`@"8YS@A&^4$,*#A`""2Y;``@.(!$(_@
MV/G_ZP!@H.,\,1OE,"`;Y0`PD^4&`*#A`P!2X>4``!HHT$OB\*^=Z!#Z_^L`
M,)#E$0!3X^#__PHX,1OE`&"@XP!0D^4)^O_KC#.?Y0`0D.4#`)CG`!"-Y803
MG^4%(*#A!#"@X;``@.(!$(_@O?G_Z^3__^H2___K!P"@X1_Z_^L`8*#AW___
MZ@IPH.$'`93G``!0XPP```H%$*#AZ_K_ZP`P4.(!<(?B]___"@`PT^4``%/C
M]/__"A#Z_^L``%#C\?__"@"@H.%@___J#0"@XROZ_^L,HY_E`)"@X0J@C^!3
M(*#C8O__ZN"0&^48^O_K``!9X;G__QKT,!OE@SN@X:,[H.$'#5/CM/__&@8`
MH.$;^O_K`&"@X<P"G^4%$*#A``"/X,GZ_^L`$%#B4```"C@Q&^4``)/E\_S_
MZP!04.)+```*!0"@X0?Z_^L#`%#C`("@X0P``)J0$I_E!`!`X@``A>`!$(_@
M"/K_ZP``4.,%```:!`!8XWX```H(@(7@!3!8Y2\`4^-Z```*%?[_ZP"@4.($
M```:!0"@X0H0H.'B^?_K`%!0XC0``!I`$I_E0`*?Y0$0C^```(_@7_O_Z\+Y
M_^LP,I_E,!*?Y0,PC^`)`(WH*#*?Y2B2G^4H@I_E*%*?Y0,PC^`$(*#A`1"/
MX`8`H.'3^?_K"9"/X`B`C^`%4(_@#```ZICY_^L`,)#E$0!3XPT``!JL^?_K
M`7"'XH$`C>@(4(WE"3"@X00@H.$($*#A!@"@X<+Y_^L'':#C!@"@X9+Y_^L!
M`'#C[?__"@0`H.$K^?_KM`&?Y080H.$``(_@-?O_ZP8`H.&4_O_K9/__ZC@Q
M&^4`4)/EL/__Z@_^_^L``%#C`##1X@"`H.$!D*#AQ/__N@H@4.(`,,'B`*"-
MY04`H.%DH$OBG/G_ZPH0H.$&(*#C!0"@X5;Y_^L&`%#C`@``"D@!G^4``(_@
MY/W_ZT`QG^5D$!OE`S"/X`P@D^4,,(/B`@!1X:[__QJT(-KAM##3X0,`4N&J
M__\:`!"@XS(@6.(`,,GB`!"-Y04`H.%<<$OB@?G_ZP4`H.$'$*#A*""@XSOY
M_^LH`%#C`@``"N0`G^4``(_@R?W_ZP#`H./8,)_EV!"?Y0,PC^`$,(WET#"?
MY0!PC>4#,(_@!""@X0$0C^`&`*#A-,!+Y73Y_^NU___J.#$;Y02`D^4``%CC
M@/__"@@`H.%V^?_K`P!0XWS__YJ0$)_E!`!`X@``B.`!$(_@>/G_ZP``4.,(
M4*`!=/__ZO/X_^L$9@$`@&0!`#P!``#<&P``I!P``*P;``",&P``4!T``&0!
M``!\&P``U!H``#P:``"L&@``<!H``#09``"T&```[!H``!0:```$&P``\!H`
M`!0:``"X&@``H!<``!@9``"X%@``K!@``&P9``",&```C!D``#@8```-P*#A
M\-DMZ02P3.(,T$WB+#*?Y2PBG^4#,(_@`E"3YR0"G^4`()7E``"/X"@@"^5#
M^?_K`#"0Y1`"G^5D(-/E+&!+X@80H.$``(_@92##Y?'Y_^L``%#C`@``"@`P
MT.4Q`%/C7@``"M[\_^O@`9_EX$&?Y0``C^`Q^?_K!$"/X`00H.$`<*#A`S"@
MXP``D.4!(*#CS?C_ZP!`4.(*```*##"4Y;`AG^4`$)/E`B"/X```E^6J^/_K
M##"4Y0`0D^4(,)'E`043XU$``!J,`9_EC$&?Y0``C^`9^?_K!$"/X`00H.$`
M<*#A`S"@XP``D.4!(*#CM?C_ZP!`4.()```*8`&?Y080H.$``(_@POG_ZP`@
M4.(V```*##"4Y0``E^4`$)/ECOC_ZSP!G^4\@9_E``"/X`+Y_^LT$9_E`'"0
MY0!`H.$!(*#C`1"/X`<`H.&J^/_K'"&?Y0`0H.$"((_@!P"@X7[X_^L`(*#C
M`#"4Y00!G^4&$*#A92##Y0``C^"E^?_K"("/X```4.,(`(CE"P``"@<=H..P
M^/_K`3"`XA$`4.,!`%,3'0``BB@@&^4`,)7E`P!2X1@``!H@T$OB\*F=Z`0`
MF.7J_?_K``!0XP@`B.7N__\:H`"?Y0``C^")^/_K`3#0Y0``4^.>__\*G/__
MZH@PG^4,()3E`S"/X`0PD^4``)+E`!"3Y?#Z_^O$___J``"7Y8WX_^NJ___J
M3/C_ZUP@G^4`,*#A6`"?Y0(@C^`&`)+I``"/X``0D>5Q^/_K!%X!`#P!```@
M7P$`U!4``-1>`0!T%P``2!<``'1>`0`@%P``T!0``!A>`0"P7@$`N!8``+P6
M``!0%```1!8``#A>`0`$7@$`#!8```W`H.$PV"WI!+!,X@!`H.'#_/_K``!0
MXP``5!,PJ)T(`##4Y0``4^,PJ)T(!`"@X3#\_^LC_/_K(!"?Y00@H.,!$(_@
M]??_ZP``4.,PJ)T8!`"@X1302^(P:)WHKOG_ZC`4```-P*#A\-\MZ02P3.)5
MWTWBT#N?Y=#KG^4#,(_@#N"3YUW/2^(.0*#A#!"@X0!`E.5XX0OE'""`XES@
M@.(P0`OE!#"2Y#,_O^8"`%[A!#"AY?K__QITX$OB.#"<Y21`G.4,$)SE!#`C
MX`0@O.4!,"/@`C`CX.,_H.$.`%SA0#",Y?3__QH$$)#E8#N?Y0C`D.5P(1OE
M`$"0Y6%AH.$0@)#E;!$;Y0S@D.4&4"S@`R""X`@@@N`$4`7@!("0Y0,0@>`.
M<"S@#%`EX`X0@>`!D(7@"'`'X&@1&^5D@:#A#G`GX.0M@N`'((+@9*$;Y0-P
M@>`($";@8E&@X0(0`>`,<(?@XBV)X`.@BN`%D"C@!A`AX`<0@>`&8(K@8G&@
MX0*0">!@H1OEXBV!X`B0*>`'$"7@!I")X`.@BN`"$`'@8F&@X>*=B>`(H(K@
M!2`AX%P1&^4&@"?@"B""X.DM@N`#H('@"8`(X&D1H.%8D1OE!X`HX`6@BN`!
M4";@"J"(X`.0B>`"@`7@!Y")X`:`*.!B4:#AXBV*X`F@B.!4@1OE!7`AX`.0
MB.`"@`?@!I")X`&`*.!B<:#AXBV*X`F@B.!0@1OE!V`EX`.0B.`"@`;@`9")
MX`6`*.!B8:#AXBV*X`F@B.!,@1OE!A`GX`.0B.`"@`'@!9")X`>`*.!B$:#A
MXBV*X`F@B.!(@1OE`5`FX`.0B.`"@`7@!Y")X`:`*.!B4:#AXBV*X`F@B.!$
M@1OE!7`AX`.0B.`"<`?@!I")X`%P)^!B@:#AXBV*X`F@A^!`<1OE"&`EX`.0
MA^`"<`;@8F&@X05P)^#B+8K@`1")X#RA&^4&D"C@`1"'X`*0">!B<:#A`Z"*
MX.(M@>`($"G@.)$;Y06@BN`'4";@"A"!X`.@B>`"D`7@"("*X`:@*>!B4:#A
M"*"*X#2!&^7B'8'@!2`GX`.0B.`!(`+@88&@X>&MBN`&8(G@!Q`BX`80@>`P
M81OE""`EX`.0AN#J'8'@:F&@X0H@`N`LH1OE!W")X`4@(N`&D"C@!R""X`&0
M">`#<(K@**$;Y>$M@N`%4(?@81&@X0B0*>`#H(K@!9")X`%0)N!B<:#AXIV)
MX`)0!>`(((K@)*$;Y0>`(>`&4"7@"8`(X`)0A>`#,(K@E"B?Y2"A&^4&,(/@
M`6`HX&F!H.'IG87@`U"&X`)@BN`(,"?@'*$;Y0$0AN`)8"/@:3&@X>E=A>`&
M8('@&)$;Y0(0BN`#H"C@!:`JX`<0@>!E<:#AY6V&X`H0@>`"4(G@%*$;Y0>0
M(^`&D"G@"%"%X&:!H.'F'8'@"5"%X`*@BN`(D"?@X6V%X`,PBN`!4"G@!5"#
MX!`Q&^5A$:#A#)$;Y0*@@^`!,"C@!Z"*X`8P(^!F<:#AYEV%X`,PBN`"8(G@
M"*$;Y0>0(>`%D"G@"&"&X&6!H.'E/8/@"6"&X`A0)^`"D(K@!*$;Y>-MAN`!
M$(G@`U`EX&,QH.$`D1OE!5"!X`*@BN`#$"C@!Z"*X`80(>!F<:#AYEV%X`$0
MBN`"8(G@_*`;Y0>0(^`%D"G@"&"&X&6!H.'E'8'@"6"&X`A0)^`"D(K@^*`;
MY>%MAN`#,(G@`5`EX&$1H.'TD!OE!5"#X`*@BN`!,"C@!Z"*X`8P(^!F<:#A
MYEV%X`,PBN`"8(G@\*`;Y0>0(>`%D"G@"&"&X&6!H.'E/8/@"6"&X`A0)^`"
MD(K@[*`;Y>-MAN`!$(G@`U`EX&,QH.'HD!OE!5"!X`*@BN`#$"C@!Z"*X`80
M(>!F<:#AYEV%X`$0BN`"8(G@Y*`;Y0>0(^`(8(;@!8`IX&61H.'E'8'@"("&
MX`)@BN`)H"?@`S"&X`%@*N`&8(/@X#`;Y>%=B.!A$:#AW(`;Y0*@@^`!,"G@
M!Z"*X`4P(^!E<:#AY6V&X`,PBN`"4(C@V*`;Y0>`(>`)4(7@!H`HX&:1H.$"
MH(K@YFV#X`B`A>`),"?@U%`;Y>:-B.`!$(K@!C`CX&9AH.$"H(7@`S"!X`80
M*>!H4:#AZ"V#X`=PBN`!@"C@!#:?Y0B@A^#0<!OE!1"&X0.`A^`"$`'@8G&@
MX>(MBN`%H`;@"9"(X`J@@>',$!OE!X"%X0F@BN`"@`C@`Y"!X&(1H.'B+8K@
M!Z`%X`9@B>`*H(CAR)`;Y0&`A^$&H(K@`Y")X&)AH.$"@`C@XBV*X`&@!^`%
M4(G@"J"(X<20&^4&@('A!:"*X`.0B>!B4:#A`H`(X.(MBN`&H`'@!W")X`J@
MB.'`D!OE!8"&X0>@BN`#D(G@8G&@X0*`".#B+8K@!:`&X`$0B>`*H(CAO)`;
MY0>`A>$!H(K@`Y")X&(1H.$"@`C@XBV*X`>@!>`&8(G@"J"(X;B0&^4!@(?A
M!J"*X`.0B>!B8:#A`H`(X.(MBN`!H`?@!5")X`J@B.&TD!OE!H"!X06@BN`#
MD(G@8E&@X0*`".#B+8K@!J`!X`=PB>`*H(CAL)`;Y06`AN$'H(K@`Y")X&)Q
MH.$"@`C@XBV*X`6@!N`!$(G@"J"(X:R0&^4'@(7A`:"*X`.0B>!B$:#A`H`(
MX.(MBN`'H`7@!F")X`J@B.&HD!OE`8"'X0:@BN`#D(G@8F&@X0*`".#B+8K@
M`:`'X`50B>`*H(CAI)`;Y0:`@>$%H(K@`Y")X&)1H.$"@`C@XBV*X`:@`>`'
M<(G@"J"(X:"0&^4%@(;A!Z"*X`.0B>!B<:#A`H`(X.(MBN`%H`;@`1")X`J@
MB.&<D!OE!X"%X0&@BN`#D(G@8A&@X0*`".#B+8K@!Z`%X`9@B>`*H(CAF)`;
MY0&`A^$&H(K@`Y")X&)AH.$"@`C@XBV*X`&@!^`%4(G@"J"(X920&^4&@('A
M!:"*X`.0B>!B4:#A`H`(X.(MBN`&H`'@!W")X`J@B.&0D!OE!8"&X0>@BN`#
MD(G@8G&@X0*`".#B+8K@!:`&X`$0B>`*H(CAC)`;Y0>`A>$!H(K@`Y")X&(1
MH.$"@`C@XBV*X`>@!>`&8(G@"J"(X8B0&^4!@(?A!J"*X`.0B>!B8:#A`H`(
MX.(MBN`!H`?@!9")X`J`B.$)@(C@A)`;Y090@>$#,(G@!W"#X`:0`>#T,I_E
M`E`%X("@&^7BC8C@"5"%X6(AH.%\D!OE!U"%X`-PBN`"H";@"*`JX`$0A^!H
M<:#AZ%V%X`H0@>`#@(G@>*`;Y0>0(N`%D"G@!H"(X&5AH.'E'8'@"8"(X`90
M)^`#D(K@=*`;Y>&-B.`"((G@`5`EX&$1H.%PD!OE!5""X`.@BN`!(";@!Z"*
MX`@@(N!H<:#AZ%V%X`(@BN`#@(G@;*`;Y0>0(>`%D"G@!F"(X&6!H.'E+8+@
M"6"&X`A0)^`#D(K@:*`;Y>)MAN`!$(G@`E`EX&(AH.%DD!OE!5"!X`.@BN`"
M$"C@!Z"*X`80(>!F<:#AYEV%X`$0BN`#8(G@8*`;Y0>0(N`%D"G@"&"&X&6!
MH.'E'8'@"6"&X`A0)^`#D(K@7*`;Y>%MAN`"((G@`5`EX&$1H.%8D!OE!5""
MX`.@BN`!("C@!Z"*X`8@(N!F<:#AYEV%X`(@BN`#8(G@5*`;Y0>0(>`%D"G@
M"&"&X&6!H.'E+8+@"6"&X`A0)^`#D(K@4*`;Y>)MAN`!$(G@`E`EX&(AH.%,
MD!OE!5"!X`.@BN`"$"C@!Z"*X`80(>!F<:#AYEV%X`$0BN`#8(G@2*`;Y0>0
M(N`(8(;@!8`IX&61H.'E'8'@"("&X`-@BN`)H"?@`B"&X`%@*N`&8(+@1"`;
MY>%=B.!A$:#A`Z""X$"`&^4!("G@!Z"*X`4@(N!E<:#AY6V&X`(@BN`#4(C@
M/*`;Y0>`(>`&@"C@"5"%X&:1H.'F+8+@"%"%X`.@BN`)@"?@XFV%X`%0BN`"
M$"C@-*`;Y0$0A>`X4!OE8B&@X0.`A>`#,(K@`E`IX&:AH.'F'8'@!W"(X`0P
M@^`*@"+@!E`EX`&`*.`),(/@!5"'X.%=A>`(@(/@Y3V(X&$1C.!X01OE$,"0
MY0`P@.4"((S@,#`;Y03`D.4`0)3E!5",X`K@CN`$`%/A"!"`Y1`@@.4$4(#E
M#."`Y0$``!HHT$OB\*^=Z![U_^L06P$`/`$``)EY@EJAZ]ENW+P;C];!8LH-
MP*#A8`"@XS#8+>D$L$SBD?7_ZP`@H.,L$)_E+$"?Y2S@G^4LP)_E`!"`Y2@0
MG^400(#I#,"`Y1`0@.44((#E&""`Y5P@@.4PJ)WH`2-%9XFKS>_^W+J8=E0R
M$/#ATL,-P*#A\-LMZ10PD.4"@*#A@B&@X0(PD^``(*#C`2"@(P!@H.$``%+C
M&""0Y5P`D.4!(((2J"Z"X```4.,$L$SB`4"@X1@@AN44,(;E')"&X@X```I`
M4&#B!0!8X0A0H+$``(G@!2"@X=_T_^M<,);E`S"%X$``4^-<,(;E\*N=&`8`
MH.$%@$C@!4"$X,/\_^L_`%CC&0``VD"`2.(H4Z#A`7"%X@=SA.`$$*#A"3"@
MX4`@A.(`0)'E!."1Y0C`D>4,`)'E$!"!X@(`4>$`0(/E!."#Y0C`@^4,`(/E
M$#"#XO/__QH!0*#A!@"@X:O\_^L'`%3AZ___&@5=9>`%@XC@!!"@X0D`H.$(
M(*#AMO3_ZUR`AN7PJYWH#<"@X?#;+>D!0*#A?Q#@XQ1@E.4<@(3BIC&@X3\P
M`^(!((/B.`!2XP!0H.$8<)3E!+!,X@,0R.<"`(C@20``VD`@8N(`$*#CO_3_
MZP0`H.&,_/_K"`"@X3@@H.,`$*#CN?3_ZR<\H.$F+*#A)PB@X2<4H.%;8,3E
M5##$Y28XH.$F9*#A50#$Y5DPQ.4$`*#A5W#$Y580Q.58(,3E6F#$Y7C\_^L#
M,-3E!`"@X0`PQ>6R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00P
MQ>6V,-3A!3#%Y00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%
MY0@PE.4C-*#A"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A
M#C#%Y0PPE.4/,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43
M,,7E)-!+XO!KG>@_]/_J."!BX@`0H.-U]/_KNO__Z@W`H.$PV"WI8#"?Y6!`
MG^4#,(_@)"#3Y01`C^```%+C!+!,XC"HG1@!$*#C1""?Y200P^4",)3G``!3
MXP(```HT,)_E`P"?YQ/T_^LL,)_E`S"4YP``4^,PJ)T((`"?Y0``C^`4T$OB
M,&B=Z'ST_^H83@$`($P!`!@!``!H30$`4`$``'1%```\$)_E/#"?Y0$0C^``
M(-'E`S"/X```4N,>_R\1`0"@XR0@G^4``,'E`C"3YP``4^,>_R\!%`"?Y000
M@>(``(_@;O3_ZIA-`0"@2P$`5`$``!!%```-P*#A\-@MZ02P3.(@T$WB+%*?
MY2PRG^4%4(_@*``+Y2P0"^4P(`OE`T"5YS`@2^(`,)3E+!!+XB@`2^(@,`OE
M._3_ZP`BG^4`,I_E`B"5YP,`E><"$*#A0?3_Z_`QG^4#,)7G`&#3Y0``5N-/
M```*X`&?Y0``C^"!]/_K`""0Y2A@&^5&,]+E!`"&X@,P@^,'`7#C1C/"Y0-@
MAN)8``"*``&@X1;T_^L`P*#A+#`;Y:0AG^6D$9_E``"3Y0(@C^`$P(+E`1"/
MX)0AG^4``(SE!!",Y0(0E><H(!OE``"1Y8`1G^4!`%+C`1"/X/@`S.$0$*#3
M"```V@3@0^("X8[@#!",X@0`L^4.`%/A!`"AY?O__QH#((+B`A&@X0#@H.-$
M49_E`>",YT`1G^4%4(_@#`"5Y08@H.$`X(WE##"@X0$0C^`;]/_K``!0XRD`
M``H<49_E)&!+X@50C^`,`)7ER?/_ZP!PH.$(`9_E!A"@X0``C^#X]/_K``!0
MXQ@```KT,)_E`S"/X`P`D^6O\__K]O/_ZR`@&^4`,)3E!P"@X0,`4N$C```:
M'-!+XO"HG>B^\__KQ#"?Y0``4.,#,(_@#`"#Y1P```J4\__KL`"?Y0``C^`J
M]/_K`#"0Y>YEP^6B___J"`"5Y0``4.,)```*=OO_Z^'__^H,`)7E"/3_Z]+_
M_^I\,)_E?`"?Y0,0E><``(_@S//_ZW``G^4&$*#A``"/X,ST_^L``%#C"`"%
MY=+__PKM___JEO/_ZP$`H.-5\__K1$L!`#P!``!,`0``7`$``"@!```43`$`
MN$P!``0%``!(`0``Y`0``$A,`0!0W/__'$P!`/P!``#P2P$`N$L!`+A*`0!$
M`0``^`(``.P```!P<U]S=')I;F=S(&UI<W-I;F<*`'!E<FQX<VDN8P````!#
M04-(10``+@```"XN```E<R\E<P```"]P<F]C+V-U<G!R;V,O97AE````1'EN
M84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4
M`````%!!4E]414U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S``!0
M15),-4Q)0@````!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.
M251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P``
M`%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]4
M35!$25(```!005)?5$U01$E2``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!
M3%]#3$5!3@`````Q````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D`$Q$
M7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````)7,E<R5S)7,`````<&%R
M+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@
M9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@
M)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R
M(&]P97)A=&EO;BD*``!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R
M96%D```E<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP
M+25U+25U)7,`````&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)3
M24].`````#$N,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O
M<GD`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T
M;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`````M90``+2T``%5315(`````
M55-%4DY!344`````5$5-4$1)4@!435``+W1M<"\```!E=F%L("=E>&5C("]U
M<W(O8FEN+W!E<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N
M;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D
M4$%27TU!1TE#+"`D1DE,15]O9F9S971?<VEZ92P@)&-A8VAE7VYA;65?<VEZ
M92D["D)%1TE.('L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*)$9)3$5?
M;V9F<V5T7W-I>F4@/2`T.R`@(",@<&%C:R@B3B(I"B1C86-H95]N86UE7W-I
M>F4@/2`T,#L*)%!+6DE07TU!1TE#(#T@(E!+7#`P,UPP,#0B.PI]"G-U8B!F
M:6YD7W!A<E]M86=I8PI["FUY("@D9F@I(#T@0%\["FUY("1C:'5N:U]S:7IE
M(#T@-C0@*B`Q,#(T.PIM>2`D8G5F.PIM>2`D<VEZ92`]("US("1F:#L*;7D@
M)'!O<R`]("@D<VEZ92TQ*2`M("@D<VEZ92TQ*2`E("1C:'5N:U]S:7IE.R`@
M("`@(",@3D]413H@)'!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE
M"G=H:6QE("@D<&]S(#X](#`I('L*<V5E:R`D9F@L("1P;W,L(#`["G)E860@
M)&9H+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*
M:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@
M>PIR971U<FX@)'!O<R`K("1I.PI]"B1P;W,@+3T@)&-H=6YK7W-I>F4["GT*
M<F5T=7)N("TQ.PI]"FUY("@D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I
M;&4L("5-;V1U;&5#86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI
M('L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE
M.PIR97%U:7)E($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E
M;F%M93HZ9&ER;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I
M;&5S(&EN("(D<&%R7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS
M=6(@>R`H("UD("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR
M;61I<B`D<&%R7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-
M4U=I;C,R)RD@>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%
M(#T^("=T;7!86%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M
M92@D=&]P9&ER*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD[
M"FUY("1F:6QE;F%M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\
M(BXN+B(["G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G
M)'!A<E]T96UP)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E
M86L*9FD*<VQE97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE
M;F%M92<*+BXN"F-L;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*
M;7D@)&-M9"`]("(D9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T
M96TH)&-M9"D["F]U=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@
M=&\@<&5R9F]R;2!C;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@
M>PI);G1E<FYA;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L
M<SHZ4$%2.CI"3T]4.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D<75I
M970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E<WES(#T@*`IP871H7W-E<"`@
M("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X92`@("`@
M("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_("<N97AE
M)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN?$]3,B\@
M/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PIO=71S*'%Q6UPD
M<')O9VYA;64@/2`B)'!R;V=N86UE(ETI.PI?<V5T7W!A<E]T96UP*"D["F]U
M=',H<7%;7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](ETI.PIM
M>2`H)'-T87)T7W!O<RP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD["DU!1TE#
M.B!["FQO8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIU;FQE<W,@*&]P
M96X@7T9(+"`G/#IR87<G+"`D<')O9VYA;64I('L*;W5T<RAQ<5M#86XG="!R
M96%D(&9R;VT@9FEL92`B)'!R;V=N86UE(ETI.R`@(R!D;VXG="!U<V4@)"$@
M:&5R92!A<R!I="!R97%U:7)E<R!%<G)N;RYP;0IL87-T($U!1TE#.PI]"FUY
M("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M86=I8R@J7T9(*3L*:68@*"1M86=I
M8U]P;W,@/"`P*2!["F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@
M(B1005)?34%'24,B(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)
M0SL*?0IO=71S*")&;W5N9"!005(@;6%G:6,@870@<&]S:71I;VX@)&UA9VEC
M7W!O<R(I.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S
M:7IE("T@;&5N9W1H*")<,$-!0TA%(BDL(#`["G)E860@7T9(+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@;F4@(EPP0T%#2$4B("8F("1B
M=68@;F4@(EPP0TQ%04XB*2!["F]U=',H(DYO(&-A8VAE(&UA<FME<B!F;W5N
M9"(I.PIL87-T($U!1TE#.PI]"F]U=',H<7%;0V%C:&4@;6%R:V5R("(D8G5F
M(B!F;W5N9%TI.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E
M=%]S:7IE+"`P.PIR96%D(%]&2"P@)&)U9BP@)$9)3$5?;V9F<V5T7W-I>F4[
M"FUY("1O9F9S970@/2!U;G!A8VLH(DXB+"`D8G5F*3L*;W5T<R@B3V9F<V5T
M(&9R;VT@<W1A<G0@;V8@1DE,17,@:7,@)&]F9G-E="(I.PIS965K(%]&2"P@
M)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@)&]F9G-E="P@,#L*
M)'-T87)T7V]F7T9)3$5?<V5C=&EO;B`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER
M95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@
M("`@("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E
M<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F
M+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T
M<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C
M(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z
M('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z
M+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&
M2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I
M(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL
M96YA;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD
M4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64[
M"GT*96QS92!["B1-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@
M)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PHD<F5Q
M=6ER95]L:7-T>R1F=6QL;F%M97TK*SL*?0IR96%D(%]&2"P@)&)U9BP@-#L*
M?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D<V5L9BP@)&UO9'5L92D@/2!`
M7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF
M;R`]("1-;V1U;&5#86-H97LD;6]D=6QE?3L*<F5T=7)N('5N;&5S<R`D:6YF
M;SL*9&5L971E("1R97%U:7)E7VQI<W1[)&UO9'5L97T["B1)3D-[)&UO9'5L
M97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,
M14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@
M24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I92`B0V%N)W0@8W)E871E('1E
M;7`@9FEL93H@)"$B.PHD9F@M/F)I;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF
M;RT^>V)U9GTI.PHD9F@M/G-E96LH,"P@,"D["G)E='5R;B`D9F@["GT*96QS
M92!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB
M+"`D:6YF;RT^>V)U9GTI.PHD:6YF;RT^>V9I;&5](#T@)&9I;&5N86UE.PIO
M<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")"
M;V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L
M92$B.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z
M.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U
M:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y
M<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE
M?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]
M?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID
M;R`D9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ
M95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T[
M"GT*:68@*"1B=68@;F4@)%!+6DE07TU!1TE#*2!["F]U=',H<7%;3F\@>FEP
M(&9O=6YD(&%F=&5R($9)3$4@<V5C=&EO;B!I;B!F:6QE("(D<')O9VYA;64B
M72D["FQA<W0@34%'24,@.PI]"B1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@
M-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O9B!Z:7`*?0IM>2!`<&%R7V%R
M9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*
M9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@<V%N:71I>F4@*%)%55-%
M(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"FEF("@A)'-T87)T7W!O<R!O
M<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM
M>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL
M7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF
M>5]P87(**3L*:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I
M('L*<VAI9G0@0$%21U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`
M05)'5CL*?0IE;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D
M9%]T;U]I;F,["G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)
M34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2![
M"G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*
M979A;"`B=7-E("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@
M0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D
M,CL*?0IE;'-I9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*
M96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@
M*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I
M('L*;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92!Q<5M#86XG="!O<&5N
M(&QO9R!F:6QE("(D,B(Z("0A73L*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C
M86-H95]N86UE(#T@)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]
M("1D:7-T7V-M9'LD,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN
M:71?:6YC*"D["G)E<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C
M;60B?2@I('5N;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I
M(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I
M;F,["GT*?0II9B`H)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I
M<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P
M87(@/2!S:&EF="A`05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I
M('L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T
M(&9I;F0@<&%R(&9I;&4@(B1P87(B.B`D(5T["F)L97-S*"1F:"P@)TE/.CI&
M:6QE)RD["B1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D
M1G)O;49I;&5(86YD;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:
M7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B
M)'!A<B)=.PI]"FUY("5M971A7W!A<B`](&1O('L*:68@*"1Z:7`@86YD(&UY
M("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@
M/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T
M82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A
M9"`D<&@L("1B=68L(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I
M;&5=('5N;&5S<R`D8G5F(&5Q("102UI)4%]-04=)0SL*8VQO<V4@)'!H.PI]
M"D-R96%T95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD
M;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?4D174B@I
M('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@<7%;0V%N
M)W0@8W)E871E(&9I;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D92@I.PIS
M965K(%]&2"P@,"P@,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@)'-T87)T
M7V]F7T9)3$5?<V5C=&EO;BD@>PIR96%D(%]&2"P@)&QO861E<BP@)'-T87)T
M7V]F7T9)3$5?<V5C=&EO;CL*?2!E;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*
M)&QO861E<B`](#Q?1D@^.PI]"FEF("@A)$5.5GM005)?5D520D%424U](&%N
M9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E(%!!4CHZ
M1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L
M>2A<)&QO861E<BP@)#`I.PI]"B1F:"T^<')I;G0H)&QO861E<BD*;W(@9&EE
M('%Q6T5R<F]R('=R:71I;F<@;&]A9&5R('1O("(D;W5T(CH@)"%=.PII9B`H
M)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S
M*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N
M9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H
M;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]
M*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO)'Q\(&9O<F5A8V@@0&EN8SL*;7D@
M)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R
M961?;V)J96-T<RP@=F%L=65S("5)3D,["FUY("1L:6)?97AT(#T@)$-O;F9I
M9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@("`@(",@6%A8(&QI8E]E>'0@=G,@
M9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME
M>7,@)69I;&5S*2!["FUY("@D9FEL92P@)&YA;64I.PII9B`H9&5F:6YE9"AM
M>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1K97E]*2D@>PHH)&9I
M;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I.PI]"F5L<V4@>PIF;W)E86-H(&UY
M("1D:7(@*$!I;F,I('L*:68@*"1K97D@/7X@;7Q>7%$D9&ER7$4O*"XJ*21\
M:2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D,2D["FQA<W0["GT*:68@
M*"1K97D@/7X@;7Q>+VQO861E<B];7B]=*R\H+BHI)'PI('L*:68@*&UY("1R
M968@/2`D36]D=6QE0V%C:&5[)#%]*2!["B@D9FEL92P@)&YA;64I(#T@*"1R
M968L("0Q*3L*;&%S=#L*?0II9B`H+68@(B1D:7(O)#$B*2!["B@D9FEL92P@
M)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PIL87-T.PI]"GT*?0I]"FYE>'0@
M=6YL97-S(&1E9FEN960@)&YA;64["FYE>'0@:68@)'=R:71T96Y[)&YA;65]
M*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD("1F:6QE(#U^("]<+EQ1)&QI
M8E]E>'1<120O:3L*;W5T<RAS<')I;G1F*'%Q6U!A8VMI;F<@1DE,12`B)7,B
M+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F:6QE*2D["FUY
M("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@>PHD8V]N=&5N="`]("1F:6QE
M+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@)"\@/2!U;F1E9CL*;W!E;B!M>2`D
M=&@L("<\.G)A=R<L("1F:6QE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL
M92(Z("0A73L*)&-O;G1E;G0@/2`\)'1H/CL*8VQO<V4@)'1H.PI005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N=&5N="P@(CQE;6)E9&1E9#XO
M)&YA;64B*0II9B`A)$5.5GM005)?5D520D%424U](&%N9"`D;F%M92`]?B`O
M7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M
M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL92P@)&YA;64I.PI]"B1F:"T^
M<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@;&5N9W1H*"1N86UE*2`K(#DI+`IS
M<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H
M)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I
M*2P*)&-O;G1E;G0I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&5M8F5D9&5D
M($9)3$4@=&\@(B1O=70B.B`D(5T["F]U=',H<7%;5W)I='1E;B!A<R`B)&YA
M;64B72D["GT*?0II9B`H)'II<"D@>PHD>FEP+3YW<FET951O1FEL94AA;F1L
M92@D9F@I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TL*;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@>FEP('!A<G0@;V8@(B1O=70B73L*?0II9B`H)&UE=&%?<&%R
M>V-L96%N?2D@>PHD9F@M/G!R:6YT*")<,$-,14%.(BD["GT*96QS92!["FEF
M("@A9&5F:6YE9"`D8V%C:&5?;F%M92D@>PIM>2`D8W1X(#T@1&EG97-T.CI3
M2$$M/FYE=R@Q*3L*<V5E:R`D9F@L(#`L(#`["B1C='@M/F%D9&9I;&4H)&9H
M*3L*<V5E:R`D9F@L(#`L(#(["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG
M97-T.PI]"B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N86UE+"`P+"`D
M8V%C:&5?;F%M95]S:7IE.PHD8V%C:&5?;F%M92`N/2`B7#`B('@@*"1C86-H
M95]N86UE7W-I>F4@+2!L96YG=&@@)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT
M*"1C86-H95]N86UE+`HB7#!#04-(12(I.PI]"FUY("1O9F9S970@/2`D9F@M
M/G1E;&P@+2!L96YG=&@H)&QO861E<BD["B1F:"T^<')I;G0H<&%C:R@G3B<L
M("1O9F9S970I+`HD4$%27TU!1TE#*3L*)&9H+3YC;&]S92!O<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!T<F%I;&5R(&]F("(D;W5T(CH@)"%=.PIC:&UO9"`P
M-S4U+"`D;W5T.PIE>&ET.PI]"GL*;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A
M<G1?<&]S.PI?9FEX7W!R;V=N86UE*"D["G)E<75I<F4@4$%2.PI005(Z.DAE
M879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD
M.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^
M;F5W.R`@("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!!<F-H:79E.CI:
M:7`@;W!E<F%T97,@;VX@86X@24\Z.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE
M;F\H7T9(*2P@)W(G*2!O<B!D:64@<7%;9F1O<&5N*"D@9F%I;&5D.B`D(5T[
M"D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D["FUY("1Z:7`@
M/2!!<F-H:79E.CI::7`M/FYE=SL**"1Z:7`M/G)E861&<F]M1FEL94AA;F1L
M92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO
M<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE
M(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U
M<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G
M;F%M97T@/2`D>FEP.PIO=71S*")%>'1R86-T:6YG('II<"XN+B(I.PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@
M=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R
M<R`I('L*;F5X="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E
M<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB
M97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMA
M<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N86UE(#T@
M)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE*"1%3E9[
M4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T7VYA;64@
M)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI('L*;W5T
M<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A;')E861Y
M(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*;W5T<RAQ<5M%
M>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE(BXN+ETI
M.PHH)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(&5X=')A
M8W1I;F<@>FEP(&UE;6)E<B!T;R`B)&1E<W1?;F%M92)=.PIC:&UO9"@P-34U
M+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S
M<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N
M;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=
M(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;
M("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.
M04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#
M<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A
M<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"
M87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@
M1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H
M+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q
M=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E
M<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W
M87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@
M0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z
M.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U
M:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!8
M4TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*
M<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q
M=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I
M<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z
M.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V
M86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE
M=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@
M07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ
M2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@
M<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X
M('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!
M4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?
M=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@
M)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@
M5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S
M(&1E9FEN960@)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@
M)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U
M:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ
M3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64[
M"GT*96QS:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I
M9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.
M5GM54T52?2!\?"`G4UE35$5-)SL*?0IM>2`D<W1M<&1I<B`]("(D<&%T:"1S
M>7-[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB+"`D=7-E<FYA;64I.PIM:V1I
M<B`D<W1M<&1I<BP@,#<U-3L*;7D@)&-A8VAE7V1I<CL*:68@*"1%3E9[4$%2
M7T-,14%.?2D@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PI]"F5L<V4@>PIO
M<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG
M="!R96%D("(D<')O9VYA;64B.B`D(5T["FEF("@H;7D@)&UA9VEC7W!O<R`]
M(&9I;F1?<&%R7VUA9VEC*"1F:"DI(#X](#`I('L*<V5E:R`D9F@L("1M86=I
M8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(12(I
M+"`P.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%
M(BD["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@)&UA9VEC
M7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BD*
M+2`D8V%C:&5?;F%M95]S:7IE+"`P.PIR96%D("1F:"P@)&)U9BP@)&-A8VAE
M7VYA;65?<VEZ93L*)&)U9B`]?B!S+UPP+R]G.PHD8V%C:&5?9&ER(#T@(F-A
M8VAE+21B=68B.PI]"GT*8VQO<V4@)&9H.PI]"FEF("@A)&-A8VAE7V1I<BD@
M>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PHD14Y6>U!!4E]#3$5!3GT@/2`Q
M.PI]"B1S=&UP9&ER("X]("(D<WES>U]D96QI;7TD8V%C:&5?9&ER(CL*;6MD
M:7(@)'-T;7!D:7(L(#`W-34["B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7([
M"FQA<W0["GT*)'!A<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD
M("1%3E9[4$%27U1%35!](#U^("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM
M>2`H)&YA;64L("1C;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M
M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@
M>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN86UE+B0D(CL*;7D@)&9H.PHH;W!E
M;B`D9F@L("<^.G)A=R<L("1T96UP;F%M92D*)B8@*'!R:6YT("1F:"`D8V]N
M=&5N=',I"B8F("AC;&]S92`D9F@I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG
M("(D=&5M<&YA;64B.B`D(5T["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@
M9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O
M<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U
M8B!?<V5T7W!R;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!
M345](&%N9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G
M;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]4
M14U0?2!A;F0@:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@
M,"D@>PHD<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R
M;V=N86UE+"`D<WES>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]0
M4D]'3D%-17T@;W(@:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#X]
M(#`I('L*:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U
M<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)'-Y<WM?97AE?2(I
M('L*)'!R;V=N86UE("X]("1S>7-[7V5X97T["G)E='5R;CL*?0I]"F9O<F5A
M8V@@;7D@)&1I<B`H<W!L:70@+UQ1)'-Y<WMP871H7W-E<'U<12\L("1%3E9[
M4$%42'TI('L*;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-4'T@86YD("1D
M:7(@97$@)$5.5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421S>7-[7V1E;&EM
M?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N
M86UE)'-Y<WM?97AE?2(I+"!L87-T*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM
M?21P<F]G;F%M921S>7-[7V5X97TB.PHH*"1P<F]G;F%M92`]("(D9&ER)'-Y
M<WM?9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?
M9&5L:6U])'!R;V=N86UE(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@
M/2`D<')O9VYA;64@?'P]("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X
M*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@
M(BXD<WES>U]D96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE
M9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F
M5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H
M;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421S>7-[7V1E
M;&EM?5Q%*2\D<'=D)'-Y<WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345]
M(#T@)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6
M>U!!4E])3DE424%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92![
M"B1%3E9[4$%27TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%
M1"!414U0($-,14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T
M92`D14Y6>R)005)?)%\B?3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!
M3B!$14)51R`O*2!["B1%3E9[(E!!4E\D7R)](#T@)$5.5GLB4$%27T=,3T)!
M3%\D7R)](&EF(&5X:7-T<R`D14Y6>R)005)?1TQ/0D%,7R1?(GT["GT*:68@
M*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*
M?0IS=6(@;W5T<R!["G)E='5R;B!I9B`D<75I970["FEF("@D;&]G9F@I('L*
M<')I;G0@)&QO9V9H(")`7UQN(CL*?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]
M"GT*<W5B(&EN:71?:6YC('L*<F5Q=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@
M9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R
M8VAL:6)E>'`@<')I=FQI8F5X<"!S:71E87)C:&5X<"!S:71E;&EB97AP"G9E
M;F1O<F%R8VAE>'`@=F5N9&]R;&EB97AP"BD["GT*<&%C:V%G92!M86EN.PIR
M97%U:7)E(%!!4CL*=6YS:&EF="!`24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!
M4BT^:6UP;W)T*$!P87)?87)G<RD["F1I92!Q<5MP87(N<&PZ($-A;B=T(&]P
M96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M92(Z($YO('-U8V@@9FEL92!O<B!D
M:7)E8W1O<GE<;ET*=6YL97-S("UE("1P<F]G;F%M93L*9&\@)'!R;V=N86UE
M.PI#3U)%.CIE>&ET*"0Q*2!I9B`H)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<
M*2\I.PID:64@)$`@:68@)$`["GT["B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*
M?0I#3U)%.CIE>&ET*"0Q*2!I9B`H)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?
M7"@H7&0K*5PI+RD["F1I92`D.CI?7T524D]2(&EF("0Z.E]?15)23U(["C$[
M"E]?14Y$7U\*``$;`SL(`0``(````%2/__\@`0``M(___T0!```\D/__;`$`
M`.R1___4`0``U)+__P`"``!\D___,`(``-R4__]D`@``%)?__YP"``"DE___
MR`(``,B8___\`@``P)G__S@#```HFO__<`,``*":__^8`P``J)K__ZP#```H
MG?__Y`,``%B=__\4!```N)W__U@$``"$G___B`0```2@__^P!```8*#__]@$
M``!`HO__$`4``&"B__\P!0``2*/__V`%``#0J?__F`4``&"L___,!0``Q*S_
M_RP&``#$N/__9`8``"2Y__^(!@``/+K__[0&``"TN___I`$``#R\__^0`0``
MD+S___P%```0``````````%Z4@`"?`X!&PP-`"`````8````+([__V``````
M0@T,2(0&A06+!(T#C@)*#`L$`"0````\````:([__X@`````0@T,2H0(A0>&
M!H<%BP2-`XX"1@P+!``@````9````,B.__^P`0```$(-#$2'!8L$C0..`D(,
M"P0````0````B````*2Z__]4`````````"P```"<````"+K__X@`````0@T,
M1(0&A06+!(T#C@),#`L$8@P-&$+.S<O%Q`X$`"@```#,````$)#__^@`````
M0@T,1(0*A0F&"(<'B`:)!8L$C0..`D0,"P0`+````/@```#,D/__J`````!"
M#0Q"A`B%!X8&AP6+!(T#C@)"#`L$`D0*#`T@0@L`,````"@!``!$D?__8`$`
M``!"#0Q"A`N%"H8)APB(!XD&B@6+!(T#C@)"#`L$8@H,#2Q""S0```!<`0``
M<)+__S@"````0@T,0H0+A0J&"8<(B`>)!HH%BP2-`XX"0@P+!`*8"@P-+$(+
M````*````)0!``!PE/__D`````!"#0Q"A`:%!8L$C0..`D(,"P1N"@P-&$(+
M```P````P`$``-24__\D`0```$(-#$*$"H4)A@B'!X@&B06+!(T#C@)"#`L$
M`EP*#`TH0@L`.````/0!``#$E?__^`````!"#0Q"A`:%!8L$C0..`D0,"P0"
M6`H,#1A""TX*#`T80L[-R\7$#@1""P``-````#`"``"`EO__:`````!"#0Q(
MA`B%!X8&AP6+!(T#C@)"#`L$6@P-($+.S<O'QL7$#@0````D````:`(``+"6
M__]X`````$(-#$2$!H4%BP2-`XX"1`P+!&8,#1@`$````)`"````E___"```
M```````T````I`(``/26__^``@```$(-#$*$"X4*A@F'"(@'B0:*!8L$C0..
M`D0,"P0"0@H,#2Q""P```"P```#<`@``/)G__S``````0@T,1(0&A06+!(T#
MC@)"#`L$3`P-&$+.S<O%Q`X$`$`````,`P``/)G__V``````0@T,0H0&A06+
M!(T#C@)$#`L$5`H,#1A"SLW+Q<0.!$(+1@H,#1A"SLW+Q<0.!$(+````+```
M`%`#``!8F?__S`$```!"#0Q"A`>%!H8%BP2-`XX"0@P+!`*<"@P-'$(+````
M)````(`#``#TFO__@`````!"#0Q"A`6+!(T#C@)"#`L$<`H,#11""R0```"H
M`P``3)O__UP`````0@T,1(`$@0."`H,!BPB-!XX&0@P+%``T````T`,``(";
M___@`0```$(-#$*$"X4*A@F'"(@'B0:*!8L$C0..`D0,"P0"L@H,#2Q""P``
M`!P````(!```*)W__R``````0@T,0HL$C0..`D(,"P0`+````"@$```HG?__
MZ`````!"#0Q"A`F%"(8'AP:(!8L$C0..`D(,"P1V"@P-)$(+-````%@$``#@
MG?__B`8```!"#0Q"A`N%"H8)APB(!XD&B@6+!(T#C@)"#`L$`S8!"@P-+$(+
M```P````D`0``#"D__^0`@```$(-#$*$"84(A@>'!H@%BP2-`XX"0@P+!`+:
M"@P-)$(+````+````,0$``",IO__9`````!"#0Q"A`:%!8L$C0..`D(,"P1F
M#`T80L[-R\7$#@0`+````/0$``",MO__E`(```!"#0Q"A`B%!X8&AP6+!(T#
MC@)"#`L$`M`*#`T@0@L`-````"0%``"0IO__``P```!"#0Q"A`N%"H8)APB(
M!XD&B@6+!(T#C@)"#`L$`^H%"@P-+$(+```@````7`4``%BR__]@`````$(-
M#$2$!H4%BP2-`XX"0@P+!``H````@`4``)2R__\8`0```$(-#$2$"H4)A@B'
M!X@&B06+!(T#C@):#`L$`#P```"L!0``@+/__W@!````0@T,0H0*A0F&"(<'
MB`:)!8L$C0..`E8,"P0"E@H,#2A"SLW+R<C'QL7$#@1""P``````H$,``!A#
M`````````0````$````!````Q`(```$```#.`@```0```-P"```!`````@,`
M``\```";!```&0```%R.`0`;````!````!H```!@C@$`'`````0````$````
M>`$```4```!@"P``!@```+`$```*````Y`0```L````0````%0`````````#
M````6(\!``(````0`@``%````!$````7````;!$``!$```!$$```$@```"@!
M```3````"````/O__V\````(^O__;QD`````````````````````````````
M````````````````````````````:(X!````````````?!,``'P3``!\$P``
M?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\
M$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3
M``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,`
M`'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``
M?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\
M$P``?!,``'P3``!\$P``?!,``'P3``!\$P``9(X!``````!<$0```````%R.
M`0````````````````"`D0$`7!$```````!@C@$```````R1`0``````````
M``````!HC@$``````$21`0``````R)`!`#0Q``!`D0$`S)`!``I005(N<&T*
M`````.Q)``"P1@```````,!'``#$1P``X$D```Q'``#H20```````,Q)``#4
M20```````/1)``!'0T,Z("A.971"4T0@;F(T(#(P,C`P.#$P*2`W+C4N,`!'
M0T,Z("AN8C0@,C`R,#`X,3`I(#<N-2XP`"1.971"4T0Z(&-R=#`N4RQV(#$N
M-"`R,#$X+S$Q+S(V(#$W.C,W.C0T(&IO97)G($5X<"`D`"1.971"4T0Z(&-R
M=#`M8V]M;6]N+F,L=B`Q+C(S(#(P,3@O,3(O,C@@,C`Z,3(Z,S4@8VAR:7-T
M;W,@17AP("0`)$YE=$)31#H@8W)T:2Y3+'8@,2XQ(#(P,3(O,#@O,3,@,#(Z
M-#DZ,#0@;6%T="!%>'`@)``D3F5T0E-$.B!C<G1B96=I;BYC+'8@,2XQ-R`R
M,#$X+S$R+S(X(#$X.C$W.C$Q(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R
M=&5N9"Y3+'8@,2XR(#(P,3,O,3(O,C`@,34Z,S4Z-#@@:F]E<F<@17AP("0`
M````\<\"``````````````````@````(````"`````@````;`````0``````
M````*BHJ($YO($QA8F5L(%!R;W9I9&5D("HJ*@``\<\"````````````````
M``@````(````"`````@````;`````0``````````*BHJ($YO($QA8F5L(%!R
M;W9I9&5D("HJ*@!!-P```&%E86)I``$M````!4%233$Q-S9*6D8M4P`&!P@!
M"0$*`A($%`$5`1<#&`$:`AP!(@%$`0`N<VAS=')T86(`+FEN=&5R<``N;F]T
M92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A>``N;F]T92YN971B<V0N
M;6%R8V@`+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YR96PN9'EN`"YR96PN<&QT
M`"YT97AT`"YR;V1A=&$`+F5H7V9R86UE7VAD<@`N96A?9G)A;64`+FEN:71?
M87)R87D`+F9I;FE?87)R87D`+FIC<@`N9'EN86UI8P`N9V]T`"YD871A`"YB
M<W,`+F-O;6UE;G0`+E-53E=?8W1F`"Y!4DTN871T<FEB=71E<P``````````
M````````````````````````````````````````````"P````$````"````
M%`$``!0!```7```````````````!`````````!,````'`````@```"P!```L
M`0``&```````````````!``````````F````!P````(```!$`0``1`$``!0`
M``````````````0`````````-P````<````"````6`$``%@!```@````````
M```````$`````````$H````%`````@```'@!``!X`0``.`,```8`````````
M!`````0```!0````"P````(```"P!```L`0``+`&```'`````P````0````0
M````6`````,````"````8`L``&`+``#D!``````````````!`````````&``
M```)`````@```$00``!$$```*`$```8`````````!`````@```!I````"0``
M`$(```!L$0``;!$``!`"```&````$P````0````(````;0````$````&````
M?!,``'P3```L`P`````````````$````!````'(````!````!@```*@6``"H
M%@``X"\`````````````!`````````!X`````0````(```"(1@``B$8``-Q`
M``````````````0`````````@`````$````"````9(<``&2'```,`0``````
M```````$`````````(X````!`````@```'"(``!PB```[`4`````````````
M!`````````"8````#@````,```!<C@$`7(X```0```````````````0````$
M````I`````\````#````8(X!`&".```$```````````````$````!````+``
M```!`````P```&2.`0!DC@``!```````````````!`````````"U````!@``
M``,```!HC@$`:(X``/`````'``````````0````(````O@````$````#````
M6(\!`%B/``!P`0`````````````$````!````,,````!`````P```,B0`0#(
MD```2```````````````!`````````#)````"`````,```!`D0$`$)$``$0`
M`````````````$``````````S@````$````P`````````!"1```[````````
M```````!`````0```!\````!``````````````!+D0``(@$`````````````
M`0````````#7`````0``````````````<)(``(\```````````````0`````
M````X0````,``'```````````/^2```X```````````````!``````````$`
F```#```````````````WDP``\0```````````````0``````````
