/* 
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2006 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2015)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/

package GoGrinder.ui;

import java.io.*;
import java.awt.*;
import java.util.zip.*;
import java.util.*;
import java.text.*;
import javax.swing.*;
import java.awt.event.*;
import javax.swing.tree.*;
import javax.swing.event.*;

import GoGrinder.*;

/**
 *
 * @author  tkington
 * @author  Ruediger Klehn
 */
public class SelectionDialog extends JDialog implements TreeSelectionListener, TagListener {
    static NumberFormat format;
    
    {
        format = DecimalFormat.getNumberInstance();
        format.setMaximumFractionDigits(2);
    }
    
    private JTree tree;
    private DefaultTreeModel model;
    private ProbCollection probs;
    private ArrayList selectedSets; // ArrayList<E> 
    private ArrayList selectedTags; // ArrayList<E> 
    private int numSelected;
    private boolean cancelled;
    private int order;
    private int matchType;
    
    private JLabel countLabel;
    private JLabel pctRight;
    private JLabel avgTime;
    
    private DefaultListModel listModel;
    private JList tagList;
    
    private TagCB tagCB; // is a JComboBox
    private JComboBox orderCB;
    private JComboBox matchCB;
    private JButton hsButton;
    private String question = "";
    private String caption = "";
    private SelectionDialog selDlg;
    private static final String [] ORDERS = {Messages.getString("order_number"),  //$NON-NLS-1$
                                            Messages.getString("order_random"), //$NON-NLS-1$
                                            Messages.getString("order_easiest"), //$NON-NLS-1$
                                            Messages.getString("order_hardest"), //$NON-NLS-1$
                                            Messages.getString("order_least_recent")}; //$NON-NLS-1$
    private static final String [] MATCHTYPES = {Messages.getString("match_all_tags"), //$NON-NLS-1$
                                                Messages.getString("match_any_tags"), //$NON-NLS-1$
                                                Messages.getString("match_no_tags")}; //$NON-NLS-1$

    
    /** Creates a new instance of SelectionDialog */ // why do we handle probs (= comlete collection)? (DO WE REALLY?)
    public SelectionDialog(JFrame owner, ProbCollection probs, ArrayList selSets, ArrayList selTags, int order, int matchType) {
        super(owner, Messages.getString("current_sel"), true); //$NON-NLS-1$
        selDlg = this;
        this.probs = probs;
    //if (probs.getSize()<2)d.b.g("probs.getSize()<2: " + probs.getSize());
    //else d.b.g("probs.getSize(): " + probs.getSize());
        selectedSets = selSets;
        selectedTags = selTags;
        cancelled = false;
        numSelected = 0;
        
        Container cp = getContentPane();
        cp.setLayout(new BorderLayout());
        
        Box tlPanel = new Box(BoxLayout.Y_AXIS);
        tlPanel.setBorder(BorderFactory.createEmptyBorder(0, 5, 5, 5));
        
        tlPanel.add(new JLabel(Messages.getString("collections"))); //$NON-NLS-1$
        
        
        model = new DefaultTreeModel(probs);
        tree = new JTree(model);
        tree.setToolTipText("At least one entry needs to be marked!");
        tree.addTreeSelectionListener(this);
        
        JScrollPane sp = new JScrollPane(tree);
        sp.setAlignmentX(Component.LEFT_ALIGNMENT);
        tlPanel.add(sp);
        
        Box butPanel = new Box(BoxLayout.X_AXIS);
        butPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
        
        tagCB = new TagCB(this);
        tagCB.setToolTipText(Messages.getString("sel_dlg_tip_applyTag")); //$NON-NLS-1$
        Dimension dim2 = new Dimension(300,30); // the "300" doesn't what I expected, but the "30" does it
        tagCB.setMaximumSize(dim2);
        butPanel.add(tagCB);
        
        // a gap
        butPanel.add(Box.createHorizontalGlue());
        
        hsButton = new JButton(Messages.getString("high_scores")); //$NON-NLS-1$
        hsButton.setToolTipText(Messages.getString("sel_dlg_tip_bestTimes")); //$NON-NLS-1$
        hsButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onHighScores();
            }
        });
        butPanel.add(hsButton);
        
        tlPanel.add(Box.createRigidArea(new Dimension(0, 3)));
        tlPanel.add(butPanel);
        
        JPanel trPanel = new JPanel(new BorderLayout());
        trPanel.setBorder(BorderFactory.createEmptyBorder(0, 0, 5, 5));
        
        trPanel.add(new JLabel(Messages.getString("tags")), BorderLayout.NORTH); //$NON-NLS-1$
        
        listModel = new DefaultListModel();
        tagList = new JList(listModel);
        tagList.addListSelectionListener(new ListSelectionListener() {
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel lsm = tagList.getSelectionModel();
                if(lsm.getMinSelectionIndex() == 0 && lsm.getMaxSelectionIndex() != 0)
                    lsm.setSelectionInterval(0, 0);
                updateStats();
            }
        });
        
        fillTagList();
        
        
        sp = new JScrollPane(tagList);
        sp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        trPanel.add(sp, BorderLayout.CENTER);
        
        Box sortPanel = new Box(BoxLayout.Y_AXIS);
        sortPanel.add(Box.createRigidArea(new Dimension(0, 3)));
        
        Box delPanel = new Box(BoxLayout.X_AXIS);
        delPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
      // delete button
        JButton b = new JButton(Messages.getString("delete")); //$NON-NLS-1$
        b.setToolTipText(Messages.getString("sel_dlg_tip_deleteTag")); //$NON-NLS-1$
        b.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onDeleteTags();
            }
        });
        delPanel.add(Box.createHorizontalGlue());
        delPanel.add(b);
        sortPanel.add(delPanel);
        
        sortPanel.add(new JLabel(Messages.getString("match"))); //$NON-NLS-1$
        matchCB = new JComboBox(MATCHTYPES);
        matchCB.setAlignmentX(Component.LEFT_ALIGNMENT);
//        matchCB.setToolTipText(Messages.getString("sel_dlg_tip_matchTags"));
        matchCB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                updateStats();
            }
        });
        sortPanel.add(matchCB);
        
        sortPanel.add(new JLabel(Messages.getString("sort_order"))); //$NON-NLS-1$
        orderCB = new JComboBox(ORDERS);
        orderCB.setAlignmentX(Component.LEFT_ALIGNMENT);
        orderCB.setToolTipText(Messages.getString("sel_dlg_tip_order")); //$NON-NLS-1$
        sortPanel.add(orderCB);
        
        trPanel.add(sortPanel, BorderLayout.SOUTH);
        
        JPanel topPanel = new JPanel(new BorderLayout());
        topPanel.setBorder(BorderFactory.createTitledBorder(Messages.getString("select_problems_colon"))); //$NON-NLS-1$
        topPanel.add(tlPanel, BorderLayout.CENTER);
        topPanel.add(trPanel, BorderLayout.EAST);
        
        Box leftStats = new Box(BoxLayout.Y_AXIS);
        Box rightStats = new Box(BoxLayout.Y_AXIS);
        
        leftStats.add(new JLabel(Messages.getString("num_selected"))); //$NON-NLS-1$
        
        countLabel = new JLabel(Messages.getString("zero_of") + " " + probs.getSize()); //$NON-NLS-1$ //$NON-NLS-2$
        rightStats.add(countLabel);
        
        leftStats.add(new JLabel(Messages.getString("pct_right_colon"))); //$NON-NLS-1$
        
        pctRight = new JLabel();
        rightStats.add(pctRight);
        
        leftStats.add(new JLabel(Messages.getString("avg_time_colon"))); //$NON-NLS-1$
        
        avgTime = new JLabel();
        rightStats.add(avgTime);
        
        JPanel statPanel = new JPanel();
        statPanel.add(leftStats);
        statPanel.add(rightStats);
        
        JPanel brPanel = new JPanel(new BorderLayout());
        
      // button export selection (with tags)
        JButton exp = new JButton(Messages.getString("export_dot_dot_dot")); //$NON-NLS-1$
        exp.setToolTipText(Messages.getString("sel_dlg_tip_export")); //$NON-NLS-1$
        exp.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onExport();
            }
        });
        brPanel.add(exp, BorderLayout.NORTH);

        JButton reset = new JButton("< < <   " + Messages.getString("reset_stats")); //$NON-NLS-1$
        reset.setToolTipText(Messages.getString("sel_dlg_tip_resetStats")); //$NON-NLS-1$
        reset.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onResetStats();
            }
        });
        brPanel.add(reset, BorderLayout.SOUTH);
        
        Box bottomPanel = new Box(BoxLayout.X_AXIS);
        bottomPanel.setBorder(BorderFactory.createTitledBorder(Messages.getString("current_sel_colon"))); //$NON-NLS-1$
        
        JPanel dumb = new JPanel(new BorderLayout());
        dumb.add(brPanel, BorderLayout.WEST);
        
        bottomPanel.add(Box.createRigidArea(new Dimension(5, 0)));
        bottomPanel.add(statPanel);
        bottomPanel.add(Box.createRigidArea(new Dimension(10, 0)));

        // a gap
        bottomPanel.add(Box.createHorizontalStrut(5));
        
        bottomPanel.add(dumb);

        // a gap
        bottomPanel.add(Box.createHorizontalStrut(5));
        

      // #### BEGIN IMPORT + EXPORT TAGS BUTTONS ####
        JPanel rightOfDumb = new JPanel(new BorderLayout());
        
      // button import tags
        JButton btnImpTags = new JButton(Messages.getString("btn_import_tags")); //$NON-NLS-1$
        btnImpTags.setToolTipText("<HTML>" + Messages.getString("btn_tip_import_tags") 
                                + "<BR>You can adjust the decoding (try e.g. \"UTF-8\").</HTML>");
         // toolTip, button: double tags, leading whitespace etc.
        btnImpTags.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              TagsExpImp tei = new TagsExpImp(1, selDlg); // 2 = export, 1 = import
            }
        });
        rightOfDumb.add(btnImpTags, BorderLayout.NORTH); 

      // button export tags
        JButton btnExpTags = new JButton(Messages.getString("btn_export_tags")); //$NON-NLS-1$
        btnExpTags.setToolTipText("<HTML>" + Messages.getString("btn_tip_export_tags") + "<BR>Export is always done with UTF-8 encoding.</HTML>");
        btnExpTags.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
               TagsExpImp tei = new TagsExpImp(2, selDlg); // 2 = export, 1 = import
            }
        });
        rightOfDumb.add(btnExpTags, BorderLayout.SOUTH);
        
        bottomPanel.add(rightOfDumb);
      // #### END IMPORT + EXPORT TAGS BUTTONS ####

        Box buttonPanel = new Box(BoxLayout.X_AXIS);
        
      // Button Exit
        JButton exitButton = new JButton("Exit"); //$NON-NLS-1$ // Messages.getString("ok")
        exitButton.setToolTipText("If something goes really wrong"); // Messages.getString("sel_dlg_tip_ok")
        ActionListener exitListener = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                Main.onExit();
            }
        };
        exitButton.addActionListener(exitListener);
        buttonPanel.add(exitButton);
        
      // a variable gap
        buttonPanel.add(Box.createHorizontalGlue());
        // and then a reload button and a fixed gap to the ok-button
        
      // button Refresh
        JButton refreshButton = new JButton("Refresh"); // Messages.getString("refresh");
        refreshButton.setToolTipText("<html>Reloads the problems directory<br>"
                                   + "(when ready, this window closes - you need to reopen it)</html>"); 
                                   // Messages.getString("sel_dlg_tip_refresh")
        ActionListener refreshListener = new ActionListener() {
          public void actionPerformed(ActionEvent e) {
            onRefresh();
          }
        };
        refreshButton.addActionListener(refreshListener);
        buttonPanel.add(refreshButton);

      // a small fixed gap
        buttonPanel.add(Box.createHorizontalStrut(50));
        
      // button OK (use selection)
        JButton okButton = new JButton(Messages.getString("ok")); //$NON-NLS-1$
        okButton.setToolTipText(Messages.getString("sel_dlg_tip_ok"));

        ActionListener okListener = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onOK();
            }
        };
        okButton.registerKeyboardAction(okListener, 
                                        KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0), 
                                        JComponent.WHEN_FOCUSED); // WHEN_IN_FOCUSED_WINDOW doesn't work
        okButton.addActionListener(okListener);

        buttonPanel.add(okButton);
      
      // .registerKeyboardAction:
      // "This method is now obsolete, please use a combination of getActionMap() and getInputMap() for similiar behavior."
      
      // button Cancel (continue with previous selection)
        JButton cancelButton = new JButton(Messages.getString("cancel")); //$NON-NLS-1$
        cancelButton.setToolTipText(Messages.getString("sel_dlg_tip_cancel"));

        ActionListener cancelListener = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                onCancel();
            }
        };
        cancelButton.addActionListener(cancelListener);
        cancelButton.registerKeyboardAction(cancelListener,
                                            KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), 
                                            JComponent.WHEN_IN_FOCUSED_WINDOW);

        buttonPanel.add(cancelButton);
        
        Box southPanel = new Box(BoxLayout.Y_AXIS);
        southPanel.add(Box.createRigidArea(new Dimension(0, 5)));
        southPanel.add(bottomPanel);
        southPanel.add(Box.createRigidArea(new Dimension(0, 5)));
        southPanel.add(buttonPanel);
        
        cp.add(topPanel, BorderLayout.CENTER);
        cp.add(southPanel, BorderLayout.SOUTH);
        
        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                onCancel();
            }
        });
        
        TreePath [] paths = new TreePath[selectedSets.size()];
        for(int i = 0; i < selectedSets.size(); i++) {
            ProbCollection c = (ProbCollection)selectedSets.get(i);
            paths[i] = new TreePath(c.getPath());
        }
        tree.setSelectionPaths(paths);
        
        if(selectedTags.isEmpty())
            tagList.setSelectedIndex(0);
        else {
            int [] sel = new int[selectedTags.size()];
            int index = 0;
            for(int i = 0; i < sel.length; i++) {
                for(int j = 0; j < listModel.size(); j++) {
                    if(listModel.get(j).equals(selectedTags.get(i))) {
                        sel[index++] = j;
                        break;
                    }
                }
            }
            tagList.setSelectedIndices(sel);
        }
        
        orderCB.setSelectedIndex(order);
        matchCB.setSelectedIndex(matchType);
        
        pack();
        setSize(getSize().width, 500);
        setLocationRelativeTo(owner);
        cancelButton.requestFocusInWindow();
        getRootPane().setDefaultButton(cancelButton);
        if (paths.length == 0){
          d.b.g("Select at least one entry in the \"Collections\" pane! \n"
              + "If this is your first use of MyGoGrinder, you need to \n"
              + "  fill the \"problems\" folder with problems in \n"
              + "  sgf format and: only one problem per file. Then restart \n"
              + "  Grinder or click the refresh button!");
        }
        setVisible(true);
    }
    
    // common ok-cancel message with default to cancel
    public static boolean OKCancelMsg(String question, String caption){
      return OKCancelMsg(question, caption, 1);
    }
    
    // common ok-cancel message
    public static boolean OKCancelMsg(String question, String caption, int defBtn){
      Object[] options = {"OK", "Cancel"};
      Component parent = null;
      boolean answer = false;
      int btnClicked = JOptionPane.showOptionDialog(parent, question, 
                                                   caption, JOptionPane.OK_CANCEL_OPTION, 
                                                   0, null, 
                                                   options, options[defBtn]);
      if (btnClicked == 0) {answer = true;}
      // 2=CANCEL_OPTION, -1=CLOSED_OPTION, 0=OK_OPTION, // ESC/ALT-F4 (is CLOSED_OPTION
      return answer;
    }
    
    public void fillTagList() { // anywhere here was a worm hidden; an ArrayIndexOutOfBounds came up, but 
                                // only sometimes (and anywhere here), conditions were unclear.
                                // I assumed, it was connected to task.java
                                // since 2x Thread.sleep(100) in Task this is gone (but we still 
                                // have it sometimes with ProgressDialog)
        listModel.clear();
        listModel.addElement(Messages.getString("none")); //$NON-NLS-1$
        String [] tags = GS.getTagList().getTags();
        for(int i = 0; i < tags.length; i++)
            listModel.addElement(tags[i]);
    }
    
    public void valueChanged(TreeSelectionEvent e) {
        TreePath[] paths = tree.getSelectionPaths();
        
        if(paths != null) {
            for(int i = 0; i < paths.length; i++) {
                for(int j = 0; j < paths.length; j++) {
                    if(i == j)
                        continue;

                    if(paths[i].isDescendant(paths[j]))
                        tree.removeSelectionPath(paths[j]);
                }
            }
        }
        
        updateStats();
    }
    
    public void updateStats() {
        SelectionStats s = getSelectionStats();
        numSelected = s.getNum();
        
        countLabel.setText(s.getNum() + " " + Messages.getString("of") + " " + probs.getSize()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        
        int tried = s.getNumTried();
        int right = s.getNumRight();
        long time = s.getTotalTime();
        if(tried == 0)
            pctRight.setText(Messages.getString("none_right")); //$NON-NLS-1$
        else pctRight.setText(format.format((double)right / tried * 100)
                              + Messages.getString("pct_right_paren") + right  //$NON-NLS-1$
                              + " " + Messages.getString("of") + " " + tried //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                              + Messages.getString("rparen")); //$NON-NLS-1$
            
        if(right == 0)
            avgTime.setText(Messages.getString("avg_time_zero")); //$NON-NLS-1$
        else avgTime.setText(format.format((double) time / right / 1000)
                            + Messages.getString("seconds")); //$NON-NLS-1$
        
        TreePath[] paths = tree.getSelectionPaths();
        if(paths != null && paths.length == 1)
            hsButton.setEnabled(true);
        else hsButton.setEnabled(false);
    }
    
    private SelectionStats getSelectionStats() {
        ArrayList tags = new ArrayList(); // ArrayList<E>
        ListSelectionModel lsm = tagList.getSelectionModel();
        for(int i = lsm.getMinSelectionIndex(); i <= lsm.getMaxSelectionIndex(); i++) {
            if(lsm.isSelectedIndex(i)) {
                //  If NONE is selected, no tags
                if(i == 0)
                    break;
                
                tags.add(listModel.getElementAt(i)); // add(Object) belongs to the raw type ArrayList -> <E>
            } 
        }
        
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        
        SelectionStats s = new SelectionStats();
        TreePath[] paths = tree.getSelectionPaths();
        if(paths != null) {
            for(int i = 0; i < paths.length; i++) {
                ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
                c.getSelectionStats(s, tags, matchCB.getSelectedIndex());
            }
        }
        
        setCursor(null);
        
        return s;
    }
    
    public void onOK() {
        TreePath[] paths = tree.getSelectionPaths();
        
        if(numSelected == 0 || paths == null) {
            JOptionPane.showMessageDialog(this, Messages.getString("no_probs_match_colls_and_tags")); //$NON-NLS-1$
            return;
        }
        
        selectedSets.clear();
        for(int i = 0; i < paths.length; i++) {
            ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
            selectedSets.add(c); // add(Object) belongs to the raw type ArrayList -> <E>
        }
        
        selectedTags.clear();
        int [] sel = tagList.getSelectedIndices();
        for(int i = 0; i < sel.length; i++) {
            if(sel[i] == 0)
                break;
            
            selectedTags.add(listModel.getElementAt(sel[i])); // add(Object) belongs to the raw type ArrayList -> <E>
        }
        
        order = orderCB.getSelectedIndex();
        matchType = matchCB.getSelectedIndex();
        
        setVisible(false);
        dispose();
    }
    
    public void onCancel() {
        setVisible(false);
        cancelled = true;
        dispose();
    }
    
    private void onRefresh(){
      ReloadDialog.reloadProblems(this);
      JOptionPane.showMessageDialog(this, "Reopen the selection window to see the refreshed folder view.");
      onCancel();
    }
    
    public void onResetStats() {
        if (!OKCancelMsg(Messages.getString("reset_stats_for_sel"), "Reset current highscores")) {
          return;
        }
        Cursor cur = getCursor();
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        new ProgressDialog(this, Messages.getString("resetting_stats"), new Task() { //$NON-NLS-1$
            public void doTask() {
                resetStats(progDlg);
            }
        });
        updateStats();
        setCursor(cur);
    }
    
    public void resetStats(ProgressDialog progDlg) {
        TreePath[] paths = tree.getSelectionPaths();
        if(paths == null) {
            JOptionPane.showMessageDialog(SelectionDialog.this, Messages.getString("no_sets_selected")); //$NON-NLS-1$
            return;
        }

        int num = 0;
        for(int i = 0; i < paths.length; i++) {
            ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
            num += c.getSize();
        }
        progDlg.setMaximum(num);

        for(int i = 0; i < paths.length; i++) {
            ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
            c.resetStats(progDlg);
        }
    }
    
    public void addTag(final String t) {
        question = Messages.getString("apply_the_tag") + t
                  + Messages.getString("to_sel_colls");
        caption = "Apply tag";
        if (!OKCancelMsg(question, caption)) {
          return;
        }
        Cursor cur = getCursor();
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        
        new ProgressDialog(this, Messages.getString("applying") + t + Messages.getString("tag"), new Task() { //$NON-NLS-1$ //$NON-NLS-2$
            public void doTask() {
                addTag(t, progDlg);
            }
        });
        
        setCursor(cur);
        
        updateStats();
    }
    
    private void addTag(String t, ProgressDialog progDlg) {
        TreePath[] paths = tree.getSelectionPaths();
        if(paths == null) {
            JOptionPane.showMessageDialog(SelectionDialog.this, Messages.getString("no_sets_selected")); //$NON-NLS-1$
            return;
        }

        int numProbs = 0;
        for(int i = 0; i < paths.length; i++) {
            ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
            numProbs += c.getSize();
        }
        progDlg.setMaximum(numProbs);

        for(int i = 0; i < paths.length; i++) {
            ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
            c.addTag(t, progDlg);
        }       
    }
    
    public void onDeleteTags() {
        if(!OKCancelMsg(Messages.getString("perm_del_tag"), "Delete tag from collection")){
          return;
        }
        
        new ProgressDialog(this, Messages.getString("removing_tags"), new Task() { //$NON-NLS-1$
            public void doTask() {
                deleteTags(progDlg); // ###### event. hier refresh einbauen (nach doTask)
            }
        });
        updateTagsLists();
    }
    
    private void deleteTags(ProgressDialog progDlg) { 
              // ? check, if the tag (or tags) are still in the list (else abort)?
              // anywhere here happens an ArrayIndexOutOfBoundsException:
              // "Exception in thread "AWT-EventQueue-0" java.lang.ArrayIndexOutOfBoundsException: 63" 
              // other time "...119", "121", "53", "144" (I have around 150 tags in my list) (the number is the wrong index)
              // oh - everytime a ProgressDialog is involved
        int [] sel = tagList.getSelectedIndices();
        if(sel.length == 1 && sel[0] == 0 || sel.length == 0) { 
        // the first part doesn't match, if nothing is selected (happens after deleting a tag)
            JOptionPane.showMessageDialog(this, Messages.getString("no_tags_selected")); //$NON-NLS-1$
            return;
        }
        
        ArrayList tags = new ArrayList(); // ArrayList<E>
        for(int i = 0; i < sel.length; i++) {
            if(sel[i] == 0)
                continue;
            
            tags.add(listModel.getElementAt(sel[i])); // fill a list "tags" from the positions (int)sel of the listModel
        } // add(Object) belongs to the raw type ArrayList -> <E>
        
        ProbCollection c = GS.getCollections(); // all files of the Collection
        progDlg.setMaximum(c.getSize());
        c.removeTags(tags, progDlg);
        
        TagList list = GS.getTagList();
        list.removeTags(tags);
        
    }
    
    public void onHighScores() {
        TreePath[] paths = tree.getSelectionPaths();
        if(paths == null)
            return;
        
        ProbCollection c = (ProbCollection)paths[0].getLastPathComponent();
        new HighScoreDialog(this, (String)c.getUserObject(), c.getHighScores(), null);
    }
    
    public void onExport() {
        Cursor cur = getCursor();  // Why do I need this? Mouse cursor shape!! a) remember old, b) set new (and reset later)
                                   // possibly this is for Mac systems
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR)); // HOW DOES THE WAIT_CURSOR LOOK LIKE?
        
        new ProgressDialog(this, Messages.getString("exporting_probs"), new Task() { //$NON-NLS-1$
            public void doTask() {
                doExport(progDlg);
            }
        });
        
        setCursor(cur);
    }
    
    private void doExport(ProgressDialog progDlg) {
        TreePath[] paths = tree.getSelectionPaths();
        if(paths == null) {
            JOptionPane.showMessageDialog(SelectionDialog.this,
                        Messages.getString("no_probs_match_colls_and_tags")); //$NON-NLS-1$
            return;
        }

        String filename = FileWorks.selectFile(SelectionDialog.this, FileFilters.gxpFilter,
                                               Messages.getString("export_problems"), true,
                                               Messages.getString("export"));
        if(filename == null)
            return;
        if (new File(filename).exists() && !OKCancelMsg("Export: overwrite existing file?", "Overwrite?")) return;

        try {
            
            ZipOutputStream zout = new ZipOutputStream(
                                   new BufferedOutputStream(
                                   new FileOutputStream(filename)));
            zout.putNextEntry(new ZipEntry("Exported Data")); //$NON-NLS-1$

            ObjectOutputStream out = new ObjectOutputStream(zout);

            ArrayList tags = new ArrayList(); // ArrayList<E>
            int [] sel = tagList.getSelectedIndices();
            for(int i = 0; i < sel.length; i++) {
                if(sel[i] == 0)
                    break;

                tags.add(listModel.getElementAt(sel[i])); // add(Object) belongs to the raw type ArrayList
            }

            int matchType = matchCB.getSelectedIndex();

            ArrayList probs = new ArrayList(); // ArrayList<E>
            for(int i = 0; i < paths.length; i++) {
                ProbCollection c = (ProbCollection)paths[i].getLastPathComponent();
                c.getProblems(probs, tags, matchType);
            }
             // anywhere here we need to change the paths to relative - this happens in ProbData.export()
            progDlg.setMaximum(probs.size());

            out.writeInt(Main.EXPORTREVISION);
            out.writeInt(probs.size());
            for(int i = 0; i < probs.size(); i++) {
                ((ProbData)probs.get(i)).export(out); // <-- HERE!!
                progDlg.bump();
            }

            out.close();
        }
        catch(Exception e) {
            JOptionPane.showMessageDialog(null, Messages.getString("ex_while_exporting")); //$NON-NLS-1$
            Main.logSilent(e);
        }     
    }
    public void updateTagsLists(){
      fillTagList();
      tagCB.fillCB();
    }
    
    public int getNumSelected() { return numSelected; }
    public ArrayList getSelectedSets() { return selectedSets; } // ArrayList<E> 
    public ArrayList getSelectedTags() { return selectedTags; } // ArrayList<E> 
    public void newTagCreated() { fillTagList(); } // possibly the plan to create an abbrevation for "fillTagList(); tagCB.fillCB();"
    public boolean wasCancelled() { return cancelled; }
    public int getOrder() { return order; }
    public int getMatchType() { return matchType; }
}
