package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````F!8``#0```#@E0````0`!30`(``'
M`"@`'0`<``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````!)``
M``20```%`````!````$````$D```!*````2@``"T`@``0`,```8`````$```
M`@```!"0```0H```$*```/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&0HB@``*(H``"B*```$`0``!`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$````JY!!!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-VAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"8%@````````,`"P``````#*`````````#`!$`AP``````````````
M$@````$``````````````!(````)```````````````2````<P,``%@A```(
M````$@`+`!$``````````````"(````@```````````````2````_@(`````
M````````$@```(0#``#`H@```````!``%@`F```````````````2````D@,`
M`!PL```8````$@`+`"T``````````````!(````+`P`````````````2````
M-0``````````````$@```*T#``"(-@``6````!(`"P!X```````````````2
M````P0,``$2C````````$``6`#T``````````````!(```"Y`P``N*(`````
M```0`!0`30``````````````$@```%P``````````````!(```!B````````
M```````2````:0``````````````$@```',``````````````!(```!]````
M```````````2````P`,``$2C````````$``6`(8``````````````!(```!%
M`@`````````````2````C```````````````$@```),``````````````!(`
M``"A```````````````2````L@``````````````$0```+\`````````````
M`!(```#,`P``T",``"0````2``L`V0,``/0C``!0````$@`+`.4#``!0+0``
M?`8``!(`"P#-```````````````6````X```````````````$@```!T#````
M`````````!(```#K```````````````2````]````$"C```$````$0`6`/P`
M`````````````!(````%`0`````````````2````]`(`````````````$@``
M`!$!`````````````!(````8`0`````````````2````\@,``%!#```(`0``
M$@`+`"L!`````````````!$````]`0`````````````2````30$`````````
M````$@```%L!`````````````!(```!@`0`````````````1````;P$`````
M````````$@```/T#``"TH@``!````!$`%`!^`0`````````````2````B@$`
M````````````$@```)\!`````````````!(```"J`0`````````````2````
M;@,``$2C````````$``6`,`"`````````````!(```"]`0`````````````2
M````0`0``)@6````````$@`+`,L!`````````````!(```#3`0``````````
M```B````ZP$`````````````$@```/<!`````````````!(````'!```1*,`
M```````0`!8`#P0``$0D```X!0``$@`+`/T!`````````````"(````3`@``
M```````````2````'@(`````````````$@```!P$``!81```8`$``!(`"P`O
M`@`````````````2````)@0``,"B````````$``6`#($``!\(```:````!(`
M"P`Z!```B$8``,@"```2``L`-@(`````````````$@```#\$``"8%@``````
M`!(`"P`]`@`````````````2````1P0``&`A``!P`@``$@`+`%0$```T+```
M'`$``!(`"P!$`@`````````````2````9@0``'PI``"4````$@`+`$T"````
M`````````!(```!>`@`````````````2````N0(`````````````$@```&4"
M`````````````!(```!T!```$"H``&0````2``L`=`(`````````````$@``
M`'T"`````````````!(```"#`@`````````````2````YP(`````````````
M$@```(@"`````````````!(```#:`@``Q*(```0````1`!8`D`(`````````
M````$@```!(#`````````````!(```"7`@`````````````2````?`0```A#
M``!(````$@`+`)X"`````````````!(```"M`@`````````````1````SP(`
M`'"B```$````$0`4`(4$``#,,P``O`(``!(`"P`D`0`````````````2````
ML@(`````````````$@```"0#`````````````!(`````<F5A;&QO8P!M96UM
M;W9E`%]?8WAA7V9I;F%L:7IE`')M9&ER`'-T<F-H<@!S=')N8VUP`'-T<G)C
M:'(`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?8V]N<W1R=6-T`&%B;W)T`'5N;&EN
M:P!?7VQS=&%T-3``<&5R;%]F<F5E`%]?<W1A=#4P`%]E>&ET`&UE;6-P>0!P
M97)L7V1E<W1R=6-T`%]?<W1A8VM?8VAK7V9A:6P`4$Q?9&]?=6YD=6UP`%!E
M<FQ?<W9?<V5T<'8`4$Q?8W5R<F5N=%]C;VYT97AT`'!E<FQ?86QL;V,`8VQO
M<V5D:7(`96YV:7)O;@!R96%D;&EN:P!?7V]P96YD:7(S,`!M96US970`4&5R
M;%]S869E<WES;6%L;&]C`%]?<W1A8VM?8VAK7V=U87)D`%!E<FQ?9W9?9F5T
M8VAP=@!?7W-P<FEN=&9?8VAK`')E860`4$Q?;65M;W)Y7W=R87``4&5R;%]S
M>7-?:6YI=#,`4&5R;%]G971?<W8`4&5R;%]C<F]A:U]N;V-O;G1E>'0`4&5R
M;%]N97=84P!097)L7V%T9F]R:U]U;FQO8VL`4&5R;%]S>7-?=&5R;0!?7V5R
M<FYO`%]?9&5R96=I<W1E<E]F<F%M95]I;F9O`%!E<FQ?;6=?<V5T`&UK9&ER
M`%]?<F5G:7-T97)?9G)A;65?:6YF;P!P97)L7W!A<G-E`%!E<FQ?;F5W6%-?
M9FQA9W,`86-C97-S`&=E='!I9`!M96UC:'(`=F9P<FEN=&8`4&5R;%]A=&9O
M<FM?;&]C:P!M96UC;7``4&5R;%]S=E]S971P=FX`<&5R;%]R=6X`;'-E96L`
M;W!E;@!S<')I;G1F`&=E='5I9`!S=')L96X`7U]T;'-?9V5T7V%D9'(`7U]S
M1@!S=')C;7``<W1R9'5P`'!T:')E861?871F;W)K`%]?<')O9VYA;64`7U]P
M<U]S=')I;F=S`%]?9V5T<'=U:60U,`!?7W-Y<V-A;&P`7U]S='!C<'E?8VAK
M`&%T97AI=`!?;&EB8U]I;FET`'-T<G1O:P!?7W)E861D:7(S,`!L:6)P97)L
M+G-O`&QI8FTN<V\N,`!L:6)C<GEP="YS;RXQ`&QI8G!T:')E860N<V\N,0!L
M:6)C+G-O+C$R`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,BXP+V5A<FUV-VAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`!*```!<````(H```%P```!2B```7
M````)*(``!<````TH@``%P```$"B```7````2*(``!<```!8H@``%P```&"B
M```7````:*(``!<```!LH@``%P```'"B```7````=*(``!<```"$H@``%P``
M`(BB```7````D*(``!<```"4H@``%P```)BB```7````G*(``!<```"@H@``
M%P```*BB```7````K*(``!<```"TH@``%P```!BB```5!P``(*(``!43```H
MH@``%2$``"RB```1)@``,*(``!(F```\H@``%3$``$2B```5-0``3*(``!4[
M``!0H@``%4$``%2B```51@``7*(``!55``!DH@``%64```RA```6`P``$*$`
M`!8$```4H0``%@4``!BA```6!P``'*$``!8(```@H0``%@D``"2A```6"P``
M**$``!8-```LH0``%@X``#"A```6#P``-*$``!81```XH0``%A4``#RA```6
M%@``0*$``!87``!$H0``%A@``$BA```6&0``3*$``!8:``!0H0``%AP``%2A
M```6'0``6*$``!8>``!<H0``%A\``&"A```6(```9*$``!8B``!HH0``%B<`
M`&RA```6*```<*$``!8I``!TH0``%BL``'BA```6+```?*$``!8M``"`H0``
M%BX``(2A```6+P``B*$``!8R``",H0``%C,``)"A```6-```E*$``!8V``"8
MH0``%C@``)RA```6.0``H*$``!8Z``"DH0``%CT``*BA```6/@``K*$``!9`
M``"PH0``%D$``+2A```60@``N*$``!9#``"\H0``%D8``,"A```61P``Q*$`
M`!9(``#(H0``%DH``,RA```63@``T*$``!90``#4H0``%E,``-BA```65@``
MW*$``!97``#@H0``%E@``.2A```66@``Z*$``!9;``#LH0``%EP``/"A```6
M70``]*$``!9>``#XH0``%F```/RA```680```*(``!9B```$H@``%F0```BB
M```6:```#*(``!9I```0H@``%FH```3@+>4$X)_E#N"/X`CPON6$C0```,:/
MX@C*C.*$_;SE`,:/X@C*C.)\_;SE`,:/X@C*C.)T_;SE`,:/X@C*C.)L_;SE
M`,:/X@C*C.)D_;SE`,:/X@C*C.)<_;SE`,:/X@C*C.)4_;SE`,:/X@C*C.),
M_;SE`,:/X@C*C.)$_;SE`,:/X@C*C.(\_;SE`,:/X@C*C.(T_;SE`,:/X@C*
MC.(L_;SE`,:/X@C*C.(D_;SE`,:/X@C*C.(<_;SE`,:/X@C*C.(4_;SE`,:/
MX@C*C.(,_;SE`,:/X@C*C.($_;SE`,:/X@C*C.+\_+SE`,:/X@C*C.+T_+SE
M`,:/X@C*C.+L_+SE`,:/X@C*C.+D_+SE`,:/X@C*C.+<_+SE`,:/X@C*C.+4
M_+SE`,:/X@C*C.+,_+SE`,:/X@C*C.+$_+SE`,:/X@C*C.*\_+SE`,:/X@C*
MC.*T_+SE`,:/X@C*C.*L_+SE`,:/X@C*C.*D_+SE`,:/X@C*C.*<_+SE`,:/
MX@C*C.*4_+SE`,:/X@C*C.*,_+SE`,:/X@C*C.*$_+SE`,:/X@C*C.)\_+SE
M`,:/X@C*C.)T_+SE`,:/X@C*C.)L_+SE`,:/X@C*C.)D_+SE`,:/X@C*C.)<
M_+SE`,:/X@C*C.)4_+SE`,:/X@C*C.),_+SE`,:/X@C*C.)$_+SE`,:/X@C*
MC.(\_+SE`,:/X@C*C.(T_+SE`,:/X@C*C.(L_+SE`,:/X@C*C.(D_+SE`,:/
MX@C*C.(<_+SE`,:/X@C*C.(4_+SE`,:/X@C*C.(,_+SE`,:/X@C*C.($_+SE
M`,:/X@C*C.+\^[SE`,:/X@C*C.+T^[SE`,:/X@C*C.+L^[SE`,:/X@C*C.+D
M^[SE`,:/X@C*C.+<^[SE`,:/X@C*C.+4^[SE`,:/X@C*C.+,^[SE`,:/X@C*
MC.+$^[SE`,:/X@C*C.*\^[SE`,:/X@C*C.*T^[SE`,:/X@C*C.*L^[SE`,:/
MX@C*C.*D^[SE`,:/X@C*C.*<^[SE`,:/X@C*C.*4^[SE`,:/X@C*C.*,^[SE
M`,:/X@C*C.*$^[SE`,:/X@C*C.)\^[SE`!"@X0(`H.$'T,WC,P``ZD0PG^5$
M()_E`S"/X'!`+>D\4)_E`F"3YP50C^`&`%7A<("]*`%@1N(`0*#C!6!&X"9A
MH.$$,)7D,_\OX08`5.$!0(3B^O__&G"`O>A(B@``%`$``$")``!D,)_E9!"?
MY60@G^4#,(_@\$$MZ0$0D^<"<)/G!P!1X?"!O2@!<$?B"%"!X@%P1^``0*#C
MIW&@X0<``.H(8!7E"%"%X@`PEN4S_R_A!P!4X0%`A.(``(;E\(&]"`0P5>6@
M`%/C]/__"BC__^OLB0``.`$``!P!``!P@9_E`&!1XH!`+>D(@(_@"-!-XDL`
M``I<(9_E`)"@X5@QG^4"()CG`&""Y0-PF.<`,);E`""3Y0@PEN4``%+C`#"'
MY3@QG^4Y```*`Q"8YP`@@>4`,-+E``!3XP4```H!((+B+P!3XP`@@04!,-+D
M``!3X_K__QJ6___K!#&?Y0,PF.<``%/C.```"O@PG^7X4)_E`Z"8YP50C^`*
M`%7A`:!*,@6@2C``0*`S*J&@,00``"H$,)7D,_\OX0H`5.$!0(3B^O__&@``
M6>,!```*"0"@X>/^_^NT`)_EM%"?Y0``C^#?_O_KK#"?Y050C^`#@)CG"`!5
MX0&`2#(%@$@P`$"@,RB!H#$$```J!#"5Y#/_+^$$`%CA`4"$XOK__QH`()?E
M`!"6Y00`EN5W"P#KM/[_ZP,PF.=@()_E`B"/X``@@^7+___J5#"?Y1/`H.,$
M`*#C`!"@XP,PC^`"(*#C`,"-Y?O^_^L!`*#CV/[_ZX3__^O$___J=(D``&`!
M```T`0``:`$``%@!```D`0``[(<``$C^__^8AP``0`$```2*``!X,```\$<M
MZ0"`4.+$,)_E`S"/X"L```J\()_E`C"3YP"0D^4``%GC*```"@!0V.4!<*#A
M``!5XST`51,!4*`3`%"@`Q$``!H`0)GE``!4XPP```H)8*#A!`"@X04@H.$(
M$*#AD_[_ZP``4.,"```:!3#4YST`4^,+```*!$"VY0``5./S__\:!`"@X?"'
MO>@(4*#A`3#UY0``4^,]`%,3^___&@A01>#F___J`5"%X@E@1N`%0(3@1F&@
MX0!@A^4$`*#A\(>]Z`A`H.'M___J"4"@X>O__^K,AP``-`$``)`@G^60,)_E
M`B"/X!!`+>F@T$WB`S"2YP!`H.$-$*#A`#"3Y9PPC>4`,*#C?O[_ZP``4.,`
M`*`3!```&@@PG>4/.@/B"@I3XP$)4Q,*```*2""?Y4`PG^4"((_@`S"2YP`@
MD^6<,)WE`B`SX``PH.,'```:H-"-XA"`O>@"$*#C!`"@X<O^_^L0#V_AH`*@
MX>[__^IY_O_K](8``#P!``"DA@``6"&?Y5@QG^4"((_@\$`MZ0!04.(#,)+G
MI-!-X@`PD^6<,(WE`#"@XP(```H`,-7E``!3XPH``!HH(9_E(#&?Y0(@C^`#
M,)+G`""3Y9PPG>4"(#/@`#"@XS\``!JDT(WB\("]Z&[^_^L`0%#B\?__"@0`
MH.'<_O_K``!0XR@```H-,-#E#6"`XBX`4^,"```:`3#6Y0``4^/T__\*#3#0
MY2X`4^,C```*!0"@X<+^_^L`<*#A!@"@X;_^_^L``(?@`@"`XL+^_^N<$)_E
M!C"@X04@H.$!$(_@`'"@X:W^_^L-$*#A!P"@X2S^_^L!`'#C!P"@X0,```H(
M,)WE#SH#X@$)4^,2```*&_[_ZP<`H.$0_O_K!`"@X;/^_^L``%#CUO__&@0`
MH.$W_O_K!0"@X?;]_^N____J`3#6Y2X`4^/8__\:`C#6Y0``4^/'__\*U/__
MZJG__^OK___J'O[_ZU"&```\`0``&(8``.@M````$*#A^`"?Y3!`+>D!VTWB
M\#"?Y0S03>(``(_@!%"-X@$KH.,#,)#G!0"@X0`PD^4$-(WE`#"@X]O]_^L!
M,$#B`P!5X0@``"H%((WB`!"@XP(``.H"`%#A`!#`Y0(```H!,'#E+P!3X_G_
M_PHO$*#C!0"@X=C]_^L``%#C$```"@4`4.$`,*"#`##`A04`H.%2_O_K<""?
MY6@PG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,1```:`=N-X@S0C>(P@+WH
M1""?Y3@PG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,%```:)`"?Y0``C^`!
MVXWB#-"-XC!`O>@W_O_JV?W_Z]"$```\`0``3(0``!R$``!H+```\$`MZ0!P
MH.'T`)_E`=M-XO`PG^4,T$WB``"/X`1@C>(!4*#A_R,`XP,PD.<&$*#AU`"?
MY0`PD^4$-(WE`#"@XP``C^#1_?_K`$!0X@@``-H!*X3B`3!$X@<@@N("((W@
M!"12Y0``4N,5```:``!3XQ(``!J4`)_E``"/X"_^_^L%(*#A!Q"@X0``D.6S
M_?_K?""?Y6PPG^4"((_@`S"2YP`@D^4$-)WE`B`SX``PH.,2```:`=N-X@S0
MC>+P@+WH`T"@X00@H.$O$*#C!@"@X?/]_^L``%#CY?__"C0`G^4``(_@%?[_
MZP0PH.$&(*#A!Q"@X0``D.7U_?_KX___ZI/]_^NX@P``/`$````L``"$A```
M.(,``!R$``#P1RWI`4"@X0`PT>4(T$WB"!*?Y0!0H.$$(I_E/0!3XP$0C^`!
M0(0"^(&?Y0(@D><$`*#A"("/X``@DN4$((WE`""@X_/]_^L-$*#A`&"@X04`
MH.&@_O_K`)!0XD````KL_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5
MY0``4^,]`%,3`3"@$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@
M`@"&X@!@F.4#`(#@W/W_ZP``4.,'`(;G%0``"@`@F.4%`*#A`3#0Y`<0DN<]
M`%/C``!3$P`PP>4&```*`2"!X@$PT.0"$*#A/0!3XP``4Q,!,,+D^?__&@$@
MH.$],*#C`##!Y0$PU.0``%/C`3#BY?O__QH((9_E^#"?Y0(@C^`#,)+G`""3
MY00PG>4"(#/@`#"@XS4``!H(T(WB\(>]Z`&02>(!,-3D``!3XP$PZ>7[__\:
M[O__ZL`PG^4#@)CG`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(`
M`%+C^O__&@,0@>(!$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@
M@^6;_?_K`*!0XM7__PH'$*#A"7&@X0<@H.'>_/_K`*"(Y0"0C>4!D(GB`#"@
MXPDQBN>B___J!P"@X=/\_^L``%#C`*"@X0``B.7%__\*"7&@X?+__^H#D*#A
M"!"@X]___^H%_?_KE((``#P!``"`@@``-`$``(R!```(@P``3""?Y4PPG^4"
M((_@2`"?Y3!`+>D,T$WB`S"2YP``C^`X4)_E.$"?Y00PC>5N_?_K!5"/X`1`
MC^`$()WE!3"@X000H.$``)#E#-"-XC!`O>@:_?_J=(```"`!``",@0``8&D`
M`,PH``!8`)_E,$`MZ0``C^`4T$WB3%"?Y5K]_^L%4(_@`$"@X0``D.7;___K
M.""?Y0``E.4`$*#C,#"?Y0(@E><$$(WE`S"/X"00G^4`,(WE`1"/X`P@C>4:
M_?_K%-"-XC"`O>@X@0``_'\``&P!``!$*P``:"@`````H.,>_R_A."*?Y3@R
MG^7P1RWI`B"/X$O>3>(`8*#A"-!-XB0"G^4#,)+G&("-X@``C^`!0*#A!!!(
MX@`PD^6T-(WE`#"@X][]_^LO$*#C`%"@X08`H.&$_/_K``!0XQ````KH`9_E
M!A"@X0``C^`=___K!@"@X=@AG^7(,9_E`B"/X`,PDN<`()/EM#2=Y0(@,^``
M,*#C:0``&DO>C>((T(WB\(>]Z`0`H.'Y_/_KI!&?Y0$0C^"B_/_K`$!0XN?_
M_PH`,-3EM'"-XHR1G^4``%/C"9"/X#P```H``%7C!```"@00H.$%`*#A#?W_
MZP``4.,L```*!`"@X0#]_^L!($#B`#"@X0(@A.``H*#A`@!4X0`PA#`!`(0R
M`!"@,P,``#H(``#J`P!0X0`0P^4"```*`2!SY2\`4N/Y__\*!`"@X>[\_^L`
MH*#A!@"@X>O\_^L*`(#@`0"`X@$+4..___\J^,"?Y0DPH.$!*Z#C`!"@XPS`
MC^`'`*#A$!"-Z`A@C>6'_/_K"!"@X0<`H.%4_/_K``!0XP,``!H@,)WE#SH#
MX@()4^,5```*M!"?Y0``H.,!$(_@8OS_ZP!`4.*G__\*`##4Y0``4^/"__\:
M``!5XQ8```H`,-7E+@!3XP,```J`0)_E`:"@XP1`C^#5___J`3#5Y0``4^/J
M__\*]___Z@$0H.,'`*#ADOS_ZP``4./D__\:4`"?Y0<0H.$``(_@L/[_ZP<`
MH.&9_/_KD/__ZCA`G^4!H*#C!$"/X,'__^HV_/_KC'\``#P!```H*```^"<`
M`"!_``#`)P``I"<``,PI``#`)@``("8``$0F``#0)0``$$`MZ2\0H.,`0*#A
M`OS_ZP``4.,!0(`2!`"@X1!`O>A^_/_J$$`MZ0!`4.()```*`##4Y0``4^,&
M```*DOS_ZP$`@.(!"U#C!@``B@0`H.$00+WH^?W_ZA``G^400+WH``"/X&W\
M_^H``*#C$("]Z#0E``#())_ER#2?Y0(@C^!P0"WI"-!-XKQDG^4#,)+G#4"@
MX;14G^4&8(_@`#"3Y00PC>4`,*#C!5"/X`00H.$&`*#A)?W_ZP``4.,.```*
MC#2?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4`
M`%+C!""#Y/O__QKK___J5&2?Y09@C^`$$*#A!@"@X0_]_^L``%#C#@``"C0T
MG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2
MXP0@@^3[__\:Z___Z@!DG^4&8(_@!!"@X08`H.'Y_/_K``!0XPX```K<,Y_E
M`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$
M((/D^___&NO__^JL8Y_E!F"/X`00H.$&`*#AX_S_ZP``4.,.```*A#.?Y0`0
MG>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#
MY/O__QKK___J6&.?Y09@C^`$$*#A!@"@X<W\_^L``%#C#@``"BPSG^4`$)WE
M`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[
M__\:Z___Z@1CG^4&8(_@!!"@X08`H.&W_/_K``!0XPX```K4,I_E`!"=Y0,P
ME><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___
M&NO__^JP8I_E!F"/X`00H.$&`*#AH?S_ZP``4.,.```*?#*?Y0`0G>4#,)7G
M`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK
M___J7&*?Y09@C^`$$*#A!@"@X8O\_^L``%#C#@``"B0RG^4`$)WE`S"5YP`@
MD^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___
MZ@ABG^4&8(_@!!"@X08`H.%U_/_K``!0XPX```K,,9_E`!"=Y0,PE><`()/E
M`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^JT
M89_E!F"/X`00H.$&`*#A7_S_ZP``4.,.```*=#&?Y0`0G>4#,)7G`""3Y0$Q
M@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J8&&?
MY09@C^`$$*#A!@"@X4G\_^L``%#C#@``"APQG^4`$)WE`S"5YP`@D^4!,8+@
M!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___Z@P!G^4$
M$*#A``"/X#3\_^L`$%#B`@``"O@`G^4``(_@>/W_Z_``G^4$$*#A``"/X"O\
M_^L`$%#B`@``"MP`G^4``(_@;_W_Z]0`G^4$$*#A``"/X"+\_^L`$%#B$@``
M"L``G^4``(_@9OW_Z[@@G^5D,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C
M$```&I@0G^68`)_E`1"/X```C^`(T(WB<$"]Z%;]_^J$`)_E!!"@X0``C^`)
M_/_K`!!0XNG__PIP`)_E``"/X$W]_^OE___JV?K_ZZQ\```\`0``<"4``(!\
M```T`0``%"4``,0D``!X)```*"0``.`C```H(P``/",``/`B``"D(@``U"$`
M`/PA``#0(0``["$``.PA``#H(0``$"$``$AX``#$(0``*"$``)0A```8(0``
M?""?Y7PPG^4"((_@>`"?Y03@+>4,T$WB`S"2YP``C^`-$*#A`#"3Y00PC>4`
M,*#CV_O_ZP``4.,%```*``#0Y0``4.,P`%`3`0"@`P``H!,!`"#B-""?Y2@P
MG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,!```:#-"-X@3PG>2>^O_K='<`
M`#P!``"$(```('<``$S`G^4/`"WI#,"/X$00G^4$X"WE#-!-X@$0G.<4((WB
M-#"?Y30`G^4`$)'E!!"-Y0`0H.,#,(_@`""-Y1`0G>4``)/GL`"`XM[Z_^O_
M`*#C1OK_Z^!V```\`0``M'8``&0!``#P3RWI`=A-XAS03>)\P9_E`A"@XP'H
MC>(`$(WE#,"/X&P1G^44X([B`""@XP`PH.,`D*#A`1"<YP`0D>4`$([E`!"@
MX]7Z_^L!8%#B`(#!XL@OH.$",";@`C!3X',P_^8","/@`C!3X`(@PN`#8%;@
M`H#(X```6.,]``"Z%+&?Y0B@H.,0<9_E#%"-X@NPC^`!H$#C!W"/X``0H.,&
M(*#A"#"@X0D`H.$`$(WEN_K_ZP$`<>,!`'`#,```"@H@H.$%$*#A"0"@X7+Z
M_^L!`'#C'P``"@<`4.,@``":"`!`X@!`E>`#```Z'```Z@%`1.(%`%3A&0``
M.@@@H.,'$*#A!`"@X9GZ_^L``%#C]O__&@5`1.`&`)3@Q!^HX'P@G^4!R(WB
M:#"?Y13`C.("((_@`S"2YP`@D^4`,)SE`B`SX``PH.,0```:`=B-XAS0C>+P
MC[WH2`"?Y0``C^".___K`6A6X@"`R.(!`'CC`0AV`\C__QH``.#C`!"@X>;_
M_^H+`*#AA/__Z\O__^H=^O_K:'8``#P!``"<'P``:'<``%AU``#@'@``$$`M
MZ8?Z_^N`^O_K``!0XP``D!40@+WH]""?Y?!`+>D"((_@[&"?Y1303>+H,)_E
M"!"-X@9@C^``4*#A`S"2YP8`H.$`,)/E##"-Y0`PH.,K^__K`$!0X@(```H`
M,-3E``!3XPX``!JP()_EJ#"?Y0(@C^`#,)+G`""3Y0PPG>4"(#/@`#"@XR``
M`!J0`)_E!1"@X0``C^`4T(WB\$"]Z&#\_^H%`*#A9/K_ZP!PH.$$`*#A8?K_
MZP``A^`"`(#B9/K_ZUPPG^5<$)_E!2"@X0,PC^``0(WE`1"/X`!`H.%-^O_K
M1""?Y2PPG^4"((_@`S"2YP`@D^4,,)WE`B`SX``PH.,$$*`!!@"@`>'__PK2
M^?_KO'0``&@>```\`0``;'0```@>``#@'```W!P``/!S``#P%9_E\,6?Y?`E
MG^4!$(_@#,"/X/!/+>D,,(SB`B"1YTG?3>(`<*#A&$",X@`@DN4<(8WE`""@
MXR!0C>('`)/H.*"-XC#`C.(LD(WB"N"@X1Q@C>*HA9_E!P"%Z`B`C^`/`+3H
M#P"NZ`,`E.@#`([H!P"<Z`<`B>@&$*#AA`6?Y0``C^#1^O_K`$!0X@(```H`
M,-3E``!3X_P``!J*___K`$!0XFH```H7^O_K@`"@X0$`@.(:^O_K`"#4Y0``
M4N,`D*#A%`"-Y0@```HXM9_E"["/X`D`H.$+$*#A`?K_ZP$@].4"D(GB``!2
MX_C__QH8!9_E``"/X`(``.H$`+KE``!0XPT```H&$*#AK_K_ZP!`4.+X__\*
M`##4Y0``4^/U__\*X/K_ZP``4./R__\*!`"@X=GY_^L`0%#B#```&@1`E>0`
M`%3C"0``"@`PU.4``%/CRP``"@0`H.'2^O_K``!0X_'__QH$0)7D``!4X_7_
M_QH$`*#AX_G_ZQ20G>4`4*#A"0"@X=_Y_^L`4(7@`5N%X@10A>(%`*#AX/G_
MZVPTG^5L%)_E!""@X0,PC^``,(WE8#2?Y0$0C^`$D(WE`S"/X`"0H.'&^?_K
M!QV@XPD`H.&6^?_K`0!PXY@```I0$(WB"0"@X3KY_^L`0%#B+```"B0$G^4)
M,*#A(!2?Y0`@E^4``)CG`1"/X+``@.(\^?_K`%"@XP@DG^7<,Y_E`B"/X`,P
MDN<`()/E'#&=Y0(@,^``,*#C[0``&@4`H.%)WXWB\(^]Z)3Y_^L`0%#BE/__
M&@0`F>0``%#C"0``"@80H.%9^O_K`#!0XOC__PH`,-/E``!3X_+__QH$`)GD
M``!0X_7__QH-`*#CF$.?Y:/Y_^M3(*#C!$"/X`"0H.$4`(WEB/__ZE@PG>4/
M.@/B`0E3X\[__QILH)WEC/G_ZP``6N'*__\:6#"=Y5,PZ.<'#5/CQO__&@4`
MH.&0^?_K!A"@X0!0H.%``Y_E``"/X#;Z_^L`$%#B<@``"@``E^5"_/_K`(!0
MXFX```H(`*#A??G_ZP,`4.,`8*#A#```F@P3G^4$`$#B``"(X`$0C^!^^?_K
M``!0XP4``!H$`%;CG```"@9@B.`%,%;E+P!3XY@```HT_O_K`&!0X@0``!H(
M`*#A!A"@X5CY_^L`<%#B5@``&KP2G^6\`I_E`1"/X+B"G^4``(_@M'*?Y5C[
M_^NP8I_E-?G_ZZPBG^6L,I_E"("/X*@2G^4"((_@`S"/X`0@C>4!$(_@"2"@
MX0=PC^`&8(_@``"-Y04`H.%%^?_K#0``Z@WY_^L`,)#E$0!3XPX``!HA^?_K
M`4"$X@@PH.$)(*#A!Q"@X5``C>D`P*#A!0"@X0#`C>4V^?_K!QV@XP4`H.$&
M^?_K`0!PX^S__PH)`*#AG_C_ZR@"G^4%$*#A``"/X"[[_^L%`*#AJ/[_ZW3_
M_^KQ^/_K`#"0Y1$`4^-B__\*`$"7Y>SX_^N\$9_E"3"@X00@H.$`P)#E`0"8
MY^01G^4`P(WEL`"`X@$0C^"@^/_K8O__Z@-`H.$X___JD_[_Z\0AG^5@,9_E
M`B"/X`,PDN<`()/E'#&=Y0(@,^``,*#C3@``&@0`H.%)WXWB\$^]Z/7X_^H`
M@)?ECO__ZA/^_^L`H%'B`("@X:3__[H*(%#B`&"-Y>A@C>(`,,KB!P"@X?+X
M_^L&(*#C!A"@X0<`H.&L^/_K!@!0XP(```I,`9_E``"/X.C]_^OH()WE0SR@
MXT$S1.,#`%+AD/__&K0@UN%(-03C`P!2X8S__QHR(%CB\$"-X@`PRN(`$*#C
M!P"@X0`0C>79^/_K*""@XP<`H.$$$*#AD_C_ZR@`4.,"```*[`"?Y0``C^#/
M_?_KY`"?Y0`@H./@,)_EX!"?Y0``C^`8(<WE`S"/X`0`C>4!$(_@"2"@X04`
MH.$`0(WES/C_ZYG__^H$8)?E``!6XV/__PH&`*#AS_C_ZP,`4.-?__^:G!"?
MY00`0.(``(;@`1"/X-'X_^L``%#C!H"@`5?__^I,^/_KG',``!!U```\`0``
M3',``.`;``"P'```)!P``.@;``#0&P``B!T``&0!``#0&P``B'$``.`:``!`
M&P``"!L``,P9``!(&0``J!L``+0:``!8&P``<!L``)P;``"L&@``,!@``"`9
M```,;P``,!D``,P8``"(&0``L!D``+`8``!<&```8"*?Y6`RG^4"((_@7`*?
MY?!`+>D,T$WB`S"2YP``C^`-4*#A`#"3Y00PC>4`,*#CF?C_ZP40H.$`0)#E
M,`*?Y60PU.4``(_@93#$Y4#Y_^L``%#C`@``"@`PT.4Q`%/C:0``"@/\_^L(
M`I_E``"/X(GX_^L`0)#E_!&?Y00`H.$#,*#C`2"@XP$0C^`E^/_K`$!0XA``
M``H,,)3EW`&?Y=QAG^4``(_@`'"3Y7KX_^L&8(_@!Q"@X08@H.$`8*#A``"0
MY?SW_^L,,)3E`!"3Y0@PD>4!!1/C5P``&J0!G^6D09_E``"/X&OX_^L$0(_@
M`S"@XP$@H.,$$*#A`&"@X0``D.4'^/_K`$!0X@D```IX`9_E!1"@X0``C^`,
M^?_K`"!0XCP```H,,)3E``"6Y0`0D^7@]__K5`&?Y51QG^4``(_@5/C_ZTP1
MG^4!(*#C!W"/X`$0C^``8)#E`$"@X08`H.'[]__K,"&?Y0(@C^``$*#A!@"@
MX<_W_^L`,)3E'`&?Y0`@H.,%$*#A``"/X&4@P^7N^/_K``!0XP@`A^41```*
M!QV@XP+X_^L!((#B`#"@X1$`4.,!`%(3(@``BN`@G^6@,)_E`B"/X`,PDN<`
M()/E!#"=Y0(@,^``,*#C&```&@S0C>+P@+WH!`"7Y>/]_^L``%#C"`"'Y>C_
M_QJD`)_E``"/X-7W_^L!,-#E``!3XY?__PJ1___JC#"?Y0P@E.4#,(_@!#"3
MY0``DN4`$)/ER/G_Z[[__^H``);EV??_ZZ3__^J8]__K8""?Y6``G^4"((_@
M``"/X`8`DND`$)'EOO?_ZR1M```\`0``/&X``/`5``#L;0``@!<``+1M``!<
M%P``=&T``"@7``#8%```&&T``-!M``#$%@``Q!8``%@4``!D:P``-!8``/QL
M``#,;```!!8``!!`+>D`0*#AN?S_ZP``5.,``%`3$("]"``PU.4``%/C$("]
M"`0`H.%/^__K1?O_ZQP0G^4$(*#C`1"/X$+W_^L``%#C$("]&`0`H.$00+WH
M\/C_ZB04``#P3RWI5]]-X@Q,G^40,(WB".R?Y1P@@.($0(_@7,"`X@,0H.$.
MX)3G`.">Y53AC>4`X*#CC0IB].``\/,"`%SAW0I!]/K__QH1'HWB!DO=[0@[
MW>T*&]WM````ZK(18O((*]/M-""#X@@P@^(""U/M`0!3X;`!0O./5V+TM0%`
M\[0!0/.S06/RL3%A\J`88/(P`.'SL`%A\@X+P^WN__\:($"0Z9DY!^,,0)#E
M@CI%XQ#`G>5ED:#A`&"0Y000+N`0<)#E`\",X`40`>`4()WE!\",X`00(>#F
MS8S@`R""X`'`C.`)$"[@!A`!X&:!H.$$((+@&*"=Y0X0(>`"$('@""`IX`.@
MBN`,(`+@#J"*X`D@(N`*((+@'*"=Y>P=@>!LP:#A`Z"*X`F@BN`,D"C@`9`)
MX.$M@N`(D"G@81&@X0J0B>`@H)WEXIV)X`.@BN`(H(K@`8`LX`*`".!B(:#A
M#(`HX`*P(>`*@(C@)*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A`9`KX`RP(N`*
MD(G@**"=Y0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*@(C@+*"=Y0FP
M"^#IC8C@`Z"*X`*@BN!I(:#A#)`KX`*P(>`*D(G@,*"=Y0BP"^#HG8G@`Z"*
MX`R@BN!HP:#A`8`KX`RP(N`*@(C@-*"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A
M`I`KX`&P+.`*D(G@.*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A#(`KX`*P(>`*
M@(C@/*"=Y0FP"^#IC8C@`Z"*X`R@BN!IP:#A`9`KX`RP(N`*D(G@0*"=Y0BP
M"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P+.`*@(C@1*"=Y0FP"^#IC8C@`Z"*
MX`*@BN!I(:#A#)`KX`*P(>`*D(G@2*"=Y0BP"^#HG8G@`Z"*X`R@BN!HP:#A
M`8`KX`RP(N`*@(C@3*"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A`I`KX`&P+.`*
MD(G@4*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A#(`KX`J`B.!4H)WEZ8V(X`.@
MBN`,P(K@`J`AX`F@"N!ID:#A`:`JX`R@BN!8P)WEZ*V*X`/`C.`!P(S@"1`B
MX`@0`>!H@:#A`A`AX`P0@>!<P)WEZAV!X`,PC.`(P"G@"L`,X`(P@^`)P"S@
M8""=Y0/`C."A.P[CV3Y&XVJAH.$#((+@X<V,X`F0@N!D()WE`R""X`@@@N`*
M@"C@`8`HX`B`B>!AD:#A:!"=Y>R-B.`#$('@"A"!X`F@*N`,H"K@;,&@X0H@
M@N#HK8+@;""=Y0,@@N`)((+@#)`IX`B0*>!H@:#A"1"!X.J=@>!P$)WE`Q"!
MX`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"X'0@G>4#((+@""""X`J`*.`)@"C@
M:9&@X0@0@>#LC8'@>!"=Y0,0@>`*$('@":`JX`R@*N!LP:#A"B""X.BM@N!\
M()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X(`0G>4#$('@#!"!X`C`
M+.`*P"S@:J&@X0P@@N#IS8+@A""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A"!"!
MX.R-@>"($)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X(P@G>4#((+@
M:K&@X0D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X)`0G>4#$('@#!"!X`C`+.`*
MP"S@#"""X.FM@N"4()WE`R""X`@@@N`+@"C@"<`HX&F1H.$,$('@F,"=Y>J-
M@>`)$"O@`\",X`H0(>`+P(S@G+"=Y6JAH.$!((+@`["+X.@=@N`)L(O@"I`I
MX`@@*>!H@:#A`L",X*`@G>7AG8S@`R""X`H@@N`(H"K@`<`JX*2@G>5A$:#A
M#,"+X`.@BN#IS8S@"*"*X`&`*.`)@"C@:9&@X0@@@N"PL)WE[(V"X*@@G>4#
M((+@`2""X`D0(>`,$"'@;,&@X0&@BN`,$"G@Z*V*X`@0(>`!((+@:(&@X>H=
M@N"L()WE`R""X`@P+.`*,"/@"2""X`,@@N#</`OC&S](XVJ1H.$#L(O@":"(
MX0RPB^"TP)WEX2V"X`&@"N`#P(S@81&@X0C`C.`)@`C@"*"*X0&`B>$+H(K@
M`H`(X`&P">#BK8K@"X"(X;RPG>4,@(C@N,"=Y6(AH.$#L(O@`\",X`&PB^`)
MD(S@`L"!X0K`#.`"$`'@`<",X>J-B.`)P(S@P)"=Y6JAH.'HS8S@"A""X0.0
MB>`"D(G@"!`!X`H@`N!H@:#A`A"!X0@@"N`+L('@"!"*X0P0`>#LO8O@`A"!
MX6S!H.$)$('@Q)"=Y0P@B.'K'8'@`Y")X`L@`N`*D(G@R*"=Y6NQH.$#H(K@
M"*"*X`R`".`(((+A88&@X0F0@N`+((SAX9V)X`$@`N#,$)WE`Q"!X`P0@>`+
MP`S@#,""X0@@B^$*P(S@"2`"X`B@"^#IS8S@"B""X="@G>5ID:#A`1""X`.@
MBN`)((CA"Z"*X-2PG>4,(`+@[!V!X`.PB^!LP:#A"+"+X`F`".`(((+A88&@
MX0J@@N`,((GAX:V*X`$@`N#8$)WE`Q"!X`D0@>`,D`G@"9""X0@@C.$+D(G@
M"B`"X`BP#.#JG8G@"R""X>"PG>4!$(+@W""=Y6JAH.$#L(O@`R""X`BPB^`,
MP(+@"B"(X0D@`N`*@`C@Z1V!X`B`@N%ID:#A#("(X`D@BN%AP:#AX8V(X`$@
M`N#D$)WE`Q"!X`H0@>`)H`K@"J""X0P@B>$+H(K@""`"X`RP">#HK8K@"R""
MX>RPG>4!$(+@Z""=Y6B!H.$#L(O@`R""X`RPB^`)D(+@""",X0H@`N`(P`S@
MZAV!X`P@@N%JH:#A"9""X`H@B.%AP:#AX9V)X`$@`N#P$)WE`Q"!X`@0@>`*
M@`C@"(""X0P@BN$+@(C@"2`"X`RP"N#IC8C@"R""X?BPG>4!((+@]!"=Y6F1
MH.$#L(O@`Q"!X`RPB^`*H('@"1",X0@0`>`)P`S@#,"!X>@M@N`*P(S@_*"=
MY6B!H.'BS8S@"!")X0,PBN`),(/@`A`!X`B0">!B(:#A`J"(X0D0@>$+$('@
M#*`*X`*P".#L'8'@;)&@X0N@BN$`P9WE`Z"*X-8Q#.-B.DSC`\",X.&MBN`(
M@(S@!,&=Y6&QH.$#P(S@`L",X`D@(N`!("+@"!&=Y0*`B.`+("G@ZHV(X`H@
M(N`#$('@:J&@X0D0@>`"P(S@"I`KX.C-C.`(D"G@#"&=Y0D0@>!H@:#A[)V!
MX!`1G>4#((+@`Q"!X`L@@N`*$('@"*`JX`R@*N!LP:#A"B""X.FM@N`4(9WE
M`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X!@1G>4#$('@#!"!X`G`+.`*
MP"S@:J&@X0P@@N#HS8+@'"&=Y0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!X.R=
M@>`@$9WE`Q"!X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X"0AG>4#((+@""""
MX`R`*.`)@"C@:9&@X0@0@>#JC8'@*!&=Y0,0@>`,$('@"<`LX`K`+.!JH:#A
M#"""X.C-@N`L(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@[)V!X#`1G>4#
M$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@-"&=Y0,@@N`(((+@#(`HX`F`
M*.!ID:#A"!"!X.J-@>`X$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"
MX#PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>!`D9WE[!V!X`.0B>`*D(G@
M"*`JX`R@*N!LP:#A"B""X$2AG>7A+8+@`Z"*X`B@BN`,@"C@`8`HX&$1H.$(
MD(G@2(&=Y>*=B>`#@(C@#("(X`'`+.`"P"S@8B&@X0S`BN`"<(?@Z<V,X!!P
M@.5LX8[@"."-Y4SAG>4#,([@!C"#X`$P@^`"$"'@"1`AX&F1H.$!$(C@"2`B
MX.P=@>`,("+@`C"#X%`@G^7A/8/@"4"$X``PC>4!4(7@#$"-Y0(@C^`$4(WE
M+#"?Y=\*;?2/"D#T`S"2YP`@D^54,9WE`B`SX``PH.,!```:5]^-XO"/O>A?
M]/_K`&H``#P!```T7@``$$`MZ6``H./7]/_K!RO?[0@[W^T`(*#C"`O?[=\J
M0/08((#E!`O`[5P@@.40@+WH`2-%9XFKS>_^W+J8=E0R$/#ATL,`````\$<M
MZ0)@H.$4,)#E@B&@X0!0H.$!0*#A`C"3X!@@D.44,(#E'("%XEP`D.4!(((B
MIBZ"X!@@A>4``%#C#@``"D!P8.(``(C@!@!7X09PH*$'(*#A+O3_ZUPPE>4#
M,(?@7#"%Y4``4^/PA[T8!0"@X0=@1N`'0(3@Q/S_ZS\`5N,>``#:0)!&XBEC
MH.$!<(;B!W.$X`00H.%`0(3B"#"@X0#@D>40$('B#,`1Y1`P@^((`!'E!"`1
MY00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@4`H.$!0*#AK/S_ZP<`5.'K__\:
M!F-)X`8@H.$'$*#A"`"@X0;T_^M<8(7E\(>]Z`1PH.'W___J<$`MZ0%`H.$"
MBRWM`%"@X1Q@A.(4$)'E&`"4Y3$_O^84$)3E,"^_YM$1Y><8*T/L`2"!XC@`
M4N-_,.#C`@"&X`$PQN=!``#:0"!BX@`0H.,+]/_K!`"@X8K\_^M0`,#R+#"$
MX@\*1O0/"D/T/#"$X@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#]'[\_^L#,-3E
M!`"@X0`PQ>6R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V
M,-3A!3#%Y00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@P
ME.4C-*#A"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%
MY0PPE.4/,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E
M`HN]['!`O>B3\__J."!BX@`0H./)\__KQO__ZEPPG^400"WI`S"/X%1`G^4H
M(-/E!$"/X```4N,0@+T81""?Y0$0H.,H$,/E`C"4YP``4^,"```*,#"?Y0,`
MG^=I\__K*#"?Y0,PE.<``%/C$("]"!P`G^400+WH``"/X-/S_^H`70``+%L`
M`!@!``!X7```4`$``'Q%```\$)_E/#"?Y0$0C^`#,(_@`"#1Y0``4N,>_R\1
M*""?Y0$`H.,``,'E`C"3YP``4^,>_R\!%`"?Y0@0@>(``(_@Q?/_ZH1<``"X
M6@``5`$``"!%``!HPI_E:#*?Y7!`+>D,P(_@*-!-XEQ2G^4<`(WE'`"-XA@0
MC>48$(WB%""-Y10@C>(#,)SG!5"/X``PD^4D,(WE`#"@XY'S_^LL(I_E+#*?
MY0(@E><,((WE`A"@X0,PE><#`*#A$#"-Y93S_^L0,I_E`S"5YP!`T^4``%3C
M4@``"@`"G^4``(_@U//_ZQQ@G>4`()#E!`"&X@<!<.,#8(;B1C/2Y0,P@^-&
M,\+E8P``B@`!H.%I\__KS"&?Y1A`G>4"((_@Q!&?Y0$0C^`$`(+E`#"@X0`@
ME.4`((#EL"&?Y0(@C^`$((#EJ"&?Y1P`G>4"()7G`0!0XPP0@^4`()+E"""#
MY50``-H$$(3B#,"#X@!!A.`$X)'D`0!4X03@K.7[__\:`P"`XFQ!G^4`$*#C
M`!"-Y08@H.$`$8/G!$"/X%@1G^4,`)3E`1"/X'#S_^L``%#C*0``"D11G^4@
M8(WB!5"/X`P`E>4>\__K!A"@X0!`H.$L`9_E``"/X$7T_^L``%#C(```"APQ
MG^4#,(_@#`"3Y03S_^M+\__K#"&?Y<PPG^4"((_@`S"2YP`@D^4D,)WE`B`S
MX``PH.,J```:!`"@X2C0C>)P@+WH#O/_Z]P@G^4"((_@``!0XPP`@N4?```*
MY/+_Z\@`G^4``(_@>O/_ZP`PD.7N1</EG___Z@P`E.5=\__KTO__Z@@`E>4`
M`%#C`0``"GC[_^O9___JE`"?Y080H.$``(_@&?3_ZP``4.,(`(7ETO__"O7_
M_^IX,)_E>`"?Y0,PE><``(_@`Q"@X0PPC>42\__K!`"@X[#__^H!`*#CHO+_
MZ^#R_^MD6@``/`$``#Q:``!,`0``7`$``"@!```86P``I%L``"`%```$!0``
M2`$``"A;```,V?__!%L``!`"``#86@``S%@``)!:``"P60````$``$0!``#8
M`@``<'-?<W1R:6YG<R!M:7-S:6YG"@`E<R\E<P```"X````O<')O8R]C=7)P
M<F]C+V5X90```$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!);G1E<FYA
M;',Z.E!!4CHZ0D]/5`````!005)?5$5-4`````!005)?4%)/1TY!344`````
M.@```"5S)7,E<P``4$523#5,24(`````4$523$Q)0@!015),-4]05`````!0
M15),24\``%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X`
M``!005)?1$5"54<```!005)?0T%#2$4```!005)?1TQ/0D%,7T1%0E5'````
M`%!!4E]'3$]"04Q?5$U01$E2````4$%27U1-4$1)4@``4$%27T=,3T)!3%]4
M14U0`%!!4E]'3$]"04Q?0TQ%04X`````,0```&QS965K(&9A:6QE9`````!R
M96%D(&9A:6QE9`!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````"5S
M)7,E<R5S`````'!A<BT`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B
M9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@``)7,Z('!R:79A=&4@
M<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I="!A
M;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@``4$%42`````!P87)L`````"YP
M87(`````<VAO<G0@<F5A9```)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U
M)7,````E<R5S=&5M<"TE=2TE=25S`````!@```!P97)L`````#````!005(Z
M.E!A8VME<CHZ5D524TE/3@`````Q+C`V,P```%5N86)L92!T;R!C<F5A=&4@
M8V%C:&4@9&ER96-T;W)Y`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A
M8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````+64`
M`"TM``!54T52`````%5315).04U%`````%1%35!$25(`5$U0`"]T;7`O````
M979A;"`G97AE8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF
M(#`[(",@;F]T(')U;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?
M<&%R7W!L.PIM>2`H)%!!4E]-04=)0RP@)$9)3$5?;V9F<V5T7W-I>F4L("1C
M86-H95]N86UE7W-I>F4I.PI"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N
M<&U<;B(["B1&24Q%7V]F9G-E=%]S:7IE(#T@-#L@("`C('!A8VLH(DXB*0HD
M8V%C:&5?;F%M95]S:7IE(#T@-#`["B102UI)4%]-04=)0R`](")02UPP,#-<
M,#`T(CL*?0IS=6(@9FEN9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM
M>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@
M/2`M<R`D9F@["FUY("1P;W,@/2`H)'-I>F4M,2D@+2`H)'-I>F4M,2D@)2`D
M8VAU;FM?<VEZ93L@("`@("`C($Y/5$4Z("1P;W,@:7,@82!M=6QT:7!L92!O
M9B`D8VAU;FM?<VEZ90IW:&EL92`H)'!O<R`^/2`P*2!["G-E96L@)&9H+"`D
M<&]S+"`P.PIR96%D("1F:"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG=&@H
M)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-
M04=)0RDI(#X](#`I('L*<F5T=7)N("1P;W,@*R`D:3L*?0HD<&]S("T]("1C
M:'5N:U]S:7IE.PI]"G)E='5R;B`M,3L*?0IM>2`H)'!A<E]T96UP+"`D<')O
M9VYA;64L($!T;7!F:6QE+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.
M5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&
M:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER
M(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ
M<5M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD
M.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@
M)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP
M("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI4
M96UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S
M96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.
M3$E.2R`]/B`P+`HI.PIM>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*
M<')I;G0@)'1M<"`\/"(N+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=
M.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP
M)R!=.R!T:&5N"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID
M;VYE"G)M("<D9FEL96YA;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P
M+"`D9FEL96YA;64["FUY("1C;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@
M,CXF,2`F(CL*<WES=&5M*"1C;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R
M;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D[
M"GT*?2!]"D)%1TE.('L*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I
M;F5D("9);G1E<FYA;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E
M;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]
M("@*<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z
M)RDL"E]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG
M=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@
M+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H
M*3L*;W5T<RAQ<5M<)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P
M87)?=&5M<"@I.PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!
M4E]414U0?2)=*3L*;7D@*"1S=&%R=%]P;W,L("1S=&%R=%]O9E]&24Q%7W-E
M8W1I;VXI.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![
M?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U
M=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@
M9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S
M="!-04=)0SL*?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H*E]&
M2"D["FEF("@D;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@
M;6%G:6,@<W1R:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B
M72D["FQA<W0@34%'24,["GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC(&%T('!O
M<VET:6]N("1M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D
M1DE,15]O9F9S971?<VEZ92`M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIR96%D
M(%]&2"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&YE(")<
M,$-!0TA%(B`F)B`D8G5F(&YE(")<,$-,14%.(BD@>PIO=71S*").;R!C86-H
M92!M87)K97(@9F]U;F0B*3L*;&%S="!-04=)0SL*?0IO=71S*'%Q6T-A8VAE
M(&UA<FME<B`B)&)U9B(@9F]U;F1=*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@
M+2`D1DE,15]O9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L("1&24Q%
M7V]F9G-E=%]S:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U9BD[
M"F]U=',H(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S970B
M*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M
M("1O9F9S970L(#`["B1S=&%R=%]O9E]&24Q%7W-E8W1I;VX@/2!T96QL(%]&
M2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@
M("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*
M=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR
M96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M
M92`]("1B=68["F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N
M+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]
M*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H
M)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U
M9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H
M9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD
M+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U
M9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]
M("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65]
M(#T@)&9U;&QN86UE.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE
M?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL
M;F%M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?
M1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L
M("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO
M9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R
M;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5]
M.PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF
M("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I
M*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A
M;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F
M:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U
M<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN
M9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]
M("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O
M<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@
M)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@
M;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97([
M"G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L
M;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M
M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D
M24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF
M("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M
M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@
M+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES
M='LD9FEL96YA;65].PI]"FEF("@D8G5F(&YE("102UI)4%]-04=)0RD@>PIO
M=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL
M92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@
M*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP
M"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D
M8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A
M;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0II9B`H
M(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F
M)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A
M<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?
M<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@
M97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D55
M4T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H
M879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;
M,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@
M*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H
M)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!
M)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/
M)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE
M(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A
M;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I
M9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@
M<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q
M(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*
M:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=0
M05)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB
M4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES
M=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)
M3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/
M.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H
M9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R
M(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S
M<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<[
M"B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@
M>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E;65T85]P87(@/2!D;R!["FEF
M("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM
M;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<
M4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT[
M"FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P
M87(@;W(@9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T[
M"FUY("1B=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES
M(&YO="!A('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D4$M:25!?34%'24,[
M"F-L;W-E("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z
M.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&
M:6QE.CI/7U)$5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@
M;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M
M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD
M969I;F5D("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*<F5A9"!?1D@L("1L
M;V%D97(L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@96QS92!["FQO8V%L
M("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%2
M7U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*
M<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0
M;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0HD9F@M/G!R:6YT*"1L
M;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO861E<B!T;R`B)&]U
M="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI0
M05(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E
M<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]
M"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG
M.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I
M9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E
M86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO
M861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB
M7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A8
M6"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@
M)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*
M:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD
M:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E
M('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1
M)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I
M.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\
M*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L
M("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q
M(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*
M?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W
M<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL
M92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K
M:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@
M.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O
M;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD
M968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N
M)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1T:#X["F-L;W-E
M("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L
M("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A
M;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ
M4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N
M86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D
M;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ
M8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N
M9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@<7%;17)R;W(@=W)I
M=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=.PIO=71S*'%Q6U=R
M:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I('L*)'II<"T^=W)I
M=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+"F]R
M(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F("(D;W5T(ET["GT*
M:68@*"1M971A7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN="@B7#!#3$5!3B(I
M.PI]"F5L<V4@>PII9B`H(61E9FEN960@)&-A8VAE7VYA;64I('L*;7D@)&-T
M>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H+"`P+"`P.PHD8W1X
M+3YA9&1F:6QE*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD8V%C:&5?;F%M92`]
M("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C
M:&5?;F%M92P@,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A8VAE7VYA;64@+CT@
M(EPP(B!X("@D8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H("1C86-H95]N86UE
M*3L*)&9H+3YP<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#2$4B*3L*?0IM>2`D
M;V9F<V5T(#T@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R
M:6YT*'!A8VLH)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)0RD["B1F:"T^8VQO
M<V4@;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z
M("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S
M(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E
M(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I
M<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@
M(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H
M+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I
M(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?
M1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D
M1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII
M<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI
M=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V
M-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2.CI,
M:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT<F%C=&EN9R!Z
M:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U
M;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H
M("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O
M<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT
M('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N
M9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X
M=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^
M8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M
M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D
M4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN
M8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS
M92!["F]U=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E
M<W1?;F%M92(N+BY=*3L**"1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D
M9&5S=%]N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q
M<5M%<G)O<B!E>'1R86-T:6YG('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B
M73L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*
M?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*
M9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A
M<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G
M<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!
M4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q
M=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E
M>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ
M)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D
M<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R
M<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U
M:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I
M8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#
M87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U
M:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER
M92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P
M96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E
M($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E
M<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S
M=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U
M:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`[
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A
M='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I
M<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V
M86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@
M<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q
M=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![
M(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E
M9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H
M+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@
M)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@
M5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@
M>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@
M+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W
M=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F
M:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS
M,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E
M<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%
M4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*;7D@)'-T;7!D
M:7(@/2`B)'!A=&@D<WES>U]D96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S
M97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L(#`W-34["FUY("1C86-H95]D:7([
M"FEF("@D14Y6>U!!4E]#3$5!3GTI('L*)&-A8VAE7V1I<B`](")T96UP+20D
M(CL*?0IE;'-E('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O
M<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PII9B`H*&UY
M("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M86=I8R@D9F@I*2`^/2`P*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*2P@,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(&QE
M;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E
M:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T
M:"@B7#!#04-(12(I"BT@)&-A8VAE7VYA;65?<VEZ92P@,#L*<F5A9"`D9F@L
M("1B=68L("1C86-H95]N86UE7W-I>F4["B1B=68@/7X@<R]<,"\O9SL*)&-A
M8VAE7V1I<B`](")C86-H92TD8G5F(CL*?0I]"F-L;W-E("1F:#L*?0II9B`H
M(21C86-H95]D:7(I('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*)$5.5GM0
M05)?0TQ%04Y](#T@,3L*?0HD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U])&-A
M8VAE7V1I<B(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PHD14Y6>U!!4E]414U0
M?2`]("1S=&UP9&ER.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B
M(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?
M.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M
M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"([
M"FUY("1F:#L**&]P96X@)&9H+"`G/CIR87<G+"`D=&5M<&YA;64I"B8F("AP
M<FEN="`D9F@@)&-O;G1E;G1S*0HF)B`H8VQO<V4@)&9H*0IO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R`B)'1E;7!N86UE(CH@)"%=.PIC:&UO9"`D;6]D92P@
M)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L
M("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F
M=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%
M3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H
M+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII
M9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM0
M05)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M
M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`K(#$I.PI]"FEF
M("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)'-Y
M<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O
M9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M
M921S>7-[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D<WES>U]E>&5].PIR971U
M<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421S>7-[<&%T:%]S
M97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%2
M7U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O
M7%$D<WES>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[
M7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](CL**"@D<')O9VYA
M;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M
M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P
M<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-
M17T["FEF("AI;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/"`P*2![
M"B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D
M<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*
M.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I
M(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN
M7"X_7%$D<WES>U]D96QI;7U<12DO)'!W9"1S>7-[7V1E;&EM?2\["B1%3E9[
M4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N
M=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T
M=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O
M<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-
M12`I("D@>PID96QE=&4@)$5.5GLB4$%27R1?(GT["GT*9F]R("AQ=R\@5$U0
M1$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R)005)?)%\B?2`]("1%
M3E9[(E!!4E]'3$]"04Q?)%\B?2!I9B!E>&ES=',@)$5.5GLB4$%27T=,3T)!
M3%\D7R)].PI]"FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM0
M05)?0TQ%04Y].PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII
M9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R
M:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG
M.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I
M9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@
M<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A
M8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z
M.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R
M+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S
M=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64[
M"F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?
M15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2
M(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@
M/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D
M.CI?7T524D]2.PHQ.PI?7T5.1%]?"@!P97)L>'-I+F,`````0T%#2$4```$;
M`SL``0``'P```(",__\8`0``V(S__S0!``!0C?__5`$``/B.__^D`0``U(__
M_\@!``!XD/__[`$``.B1__\4`@```)/__TP"```<E/__?`(``%26__^L`@``
MO);__]0"```PE___]`(``#B7__\(`P``J)G__SP#``#,F?__7`,``!R:__^$
M`P``5)___[0#``#HG___U`,``$R@___X`P``]*'__S`$```,HO__2`0``"BC
M__]X!```I*G__\`$``!@K/__Z`0``+BL__\P!0``X+C__V0%```HN?__?`4`
M`#"Z__^@!0``D+O__X0!```,O/__<`$``&"\__\(!0``$``````````!>E(`
M`GP.`1L,#0`8````&````&"+__]8`````$@.$(0$A0.&`HX!'````#0```"<
MB___>`````!*#AB$!H4%A@2'`X@"C@$8````5````/2+__^H`0```$8."(<"
MC@%$#A``$````'````"4NO__5``````````<````A`````2Z__]\`````$0.
M"(0"C@%JSL0.`````"````"D````3(W__]P`````0@X@A`B%!X8&AP6(!(D#
MB@*.`2````#(````!([__Z0`````2`X(A`*.`4(.J`%R"@X(0@L``"0```#L
M````A([__W`!````2`X4A`6%!(8#AP*.`48.N`%B"@X40@LT````%`$``,R/
M__\8`0```$8.#(0#A0*.`4(.C`A$#I@(`E0*#@Q""UH*#@Q"SL7$#@!""P``
M`"P```!,`0``K)#__QP!````0@X4A`6%!(8#AP*.`48.E`A$#J`(`E(*#A1"
M"P```"P```!\`0``F)'__S@"````0@X@A`B%!X8&AP6(!(D#B@*.`48.*`*<
M"@X@0@L``"0```"L`0``H)/__V@`````2@X,A`.%`HX!0@X86@X,0L[%Q`X`
M```<````U`$``."3__]T`````$0.#(0#A0*.`40.(&8.#!````#T`0``-)3_
M_P@`````````,`````@"```HE/__<`(```!&#B"$"(4'A@:'!8@$B0.*`HX!
M1`[0"40.V`D"0`H.($(+`!P````\`@``9);__R0`````0@X(A`*.`4[.Q`X`
M````)````%P"``!HEO__4`````!"#@B$`HX!5@K.Q`X`0@M$"L[$#@!$"RP`
M``"$`@``D);__S@%````2`X0A`2%`X8"C@%"#A@#1`(*#A!"SL;%Q`X`0@L`
M`!P```"T`@``F)O__Y0`````2@X$C@%"#A!R"@X$0@L`(````-0"```,G/__
M9`````!$#A"`!($#@@*#`48.%(X%0@X@-````/@"``!,G/__J`$```!"#B2$
M"84(A@>'!H@%B02*`XL"C@%"#J2`!$(.P(`$`J(*#B1""P`4````,`,``+R=
M__\8`````$(."(0"C@$L````2`,``+R=__\<`0```$0.%(0%A02&`X<"C@%&
M#BAX"@X40L['QL7$#@!""P!$````>`,``*B>__]\!@```$P.)(0)A0B&!X<&
MB`6)!(H#BP*.`48.R`(#$`$*#B1""P,\`0H.)$+.R\K)R,?&Q<0.`$(+```D
M````P`,``-RD__^\`@```$H.%(0%A02&`X<"C@%"#B`"Z`H.%$(+'````.@#
M``!PI___6`````!"#@B$`HX!9L[$#@`````D````"`0``%"W___(`@```$8.
M$(0$A0.&`HX!1`XX`MH*#A!""P``,````#`$``"`I___*`P```!"#B2$"84(
MA@>'!H@%B02*`XL"C@%"#H`#`P8&"@XD0@L``!0```!D!```=+/__T@`````
M0@X(A`*.`2````!\!```I+/__P@!````0@X@A`B%!X8&AP6(!(D#B@*.`30`
M``"@!```B+3__V`!````0@X0A`2%`X8"C@%$#A@%4`8%404"G@H&4`91#A!"
MSL;%Q`X`0@L``````#1&``"X10````````$````P`P```0```#L#```!````
M10,```$```!3`P```0```&,#```/````FP0``!D````$H```&P````0````:
M````"*```!P````$````!````'@!```%````8`L```8```"P!```"@```.0$
M```+````$````!4``````````P````"A```"````$`(``!0````1````%P``
M`%P1```1````1!```!(````8`0``$P````@```#[__]O````"/K__V\7````
M`````````````````````````````````````````````````````!"@````
M`````````&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``
M;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L
M$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M``R@```````````````````$H```````````````````0*,`````````````
M"*````````"TH@``````````````````$*````````#$H@```````'"B``#,
M,P``P*(``'2B```*4$%2+G!M"@````"<3```;$D```````!P2@``=$H``)!,
M``"\20``F$P```````!\3```A$P```````"D3```1T-#.B`H3F5T0E-$(&YB
M,2`R,#(U,#<R,2D@,3(N-2XP`$=#0SH@*&YB,2`R,#(U,#<R,2D@,3(N-2XP
M`"1.971"4T0Z(&-R=#`N4RQV(#$N-"`R,#$X+S$Q+S(V(#$W.C,W.C0T(&IO
M97)G($5X<"`D`"1.971"4T0Z(&-R=#`M8V]M;6]N+F,L=B`Q+C,P(#(P,C4O
M,#4O,#(@,C,Z,#0Z,#8@<FEA<W1R861H($5X<"`D`"1.971"4T0Z(&-R=&DN
M4RQV(#$N,2`R,#$R+S`X+S$S(#`R.C0Y.C`T(&UA='0@17AP("0`)$YE=$)3
M1#H@8W)T8F5G:6XN8RQV(#$N,3<@,C`Q."\Q,B\R."`Q.#HQ-SHQ,2!C:')I
M<W1O<R!%>'`@)``D3F5T0E-$.B!C<G1E;F0N4RQV(#$N,B`R,#$S+S$R+S(P
M(#$U.C,U.C0X(&IO97)G($5X<"`D`/'/`P`````````````````(````"```
M``@````(````&P````$``````````"HJ*B!.;R!,86)E;"!0<F]V:61E9"`J
M*BH``/'/`P`````````````````(````"`````@````(````&P````$`````
M`````"HJ*B!.;R!,86)E;"!0<F]V:61E9"`J*BH`03(```!A96%B:0`!*```
M``4W+4$`!@H'00@!"0(*`PP!$@04`14!%P,8`1H"'`$B`40!`"YS:'-T<G1A
M8@`N:6YT97)P`"YN;W1E+FYE=&)S9"YI9&5N=``N;F]T92YN971B<V0N<&%X
M`"YN;W1E+FYE=&)S9"YM87)C:``N:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E
M;"YD>6X`+G)E;"YP;'0`+G1E>'0`+G)O9&%T80`N96A?9G)A;65?:&1R`"YE
M:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N:F-R`"YD>6YA;6EC
M`"YG;W0`+F1A=&$`+G!E<G-I<W1E;G0`+F)S<P`N;F]I;FET`"YC;VUM96YT
M`"Y354Y77V-T9@`N05)-+F%T=')I8G5T97,`````````````````````````
M```````````````````````````````+`````0````(````4`0``%`$``!<`
M``````````````$`````````$P````<````"````+`$``"P!```8````````
M```````$`````````"8````'`````@```$0!``!$`0``%```````````````
M!``````````W````!P````(```!8`0``6`$``"````````````````0`````
M````2@````4````"````>`$``'@!```X`P``!@`````````$````!````%``
M```+`````@```+`$``"P!```L`8```<````#````!````!````!8`````P``
M``(```!@"P``8`L``.0$``````````````$`````````8`````D````"````
M1!```$00```8`0``!@`````````$````"````&D````)````0@```%P1``!<
M$0``$`(```8````3````!`````@```!M`````0````8```!L$P``;!,``"P#
M``````````````0````$````<@````$````&````F!8``)@6``"X,@``````
M```````(`````````'@````!`````@```%!)``!020``UT``````````````
M!`````````"``````0````(````HB@``*(H```0!``````````````0`````
M````C@````$````"````+(L``"R+``#8!``````````````$`````````)@`
M```.`````P````2@```$D```!```````````````!`````0```"D````#P``
M``,````(H```")````0```````````````0````$````L`````$````#````
M#*````R0```$```````````````$`````````+4````&`````P```!"@```0
MD```\`````<`````````!`````@```"^`````0````,`````H0```)$``'`!
M``````````````0````$````PP````$````#````<*(``'"2``!(````````
M```````$`````````,D````!`````P```+BB``"XD@``````````````````
M!`````````#5````"`````,```#`H@``N)(``(0``````````````$``````
M````V@````@````#````1*,````````````````````````$`````````.(`
M```!````,`````````"XD@``/0```````````````0````$````?`````0``
M`#``````````]9(``",!``````````````$````!````ZP````$`````````
M`````!B4``"/```````````````$`````````/4````#``!P``````````"G
ME```,P```````````````0`````````!`````P``````````````VI0```4!
2``````````````$`````````
