/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.start;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;
import java.util.regex.Pattern;
import org.eclipse.jetty.start.BaseHome;
import org.eclipse.jetty.start.FS;
import org.eclipse.jetty.start.Main;
import org.eclipse.jetty.start.Module;
import org.eclipse.jetty.start.Props;
import org.eclipse.jetty.start.StartArgs;
import org.eclipse.jetty.start.StartLog;
import org.eclipse.jetty.start.UsageException;

public class Modules
implements Iterable<Module> {
    private final BaseHome baseHome;
    private final StartArgs args;
    private Map<String, Module> modules = new HashMap<String, Module>();
    private Set<String> missingModules = new HashSet<String>();
    private int maxDepth = -1;

    public Modules(BaseHome basehome, StartArgs args) {
        this.baseHome = basehome;
        this.args = args;
    }

    private Set<String> asNameSet(Set<Module> moduleSet) {
        HashSet<String> ret = new HashSet<String>();
        for (Module module : moduleSet) {
            ret.add(module.getName());
        }
        return ret;
    }

    private void assertNoCycle(Module module, Stack<String> refs) {
        for (Module parent : module.getParentEdges()) {
            if (refs.contains(parent.getName())) {
                StringBuilder err = new StringBuilder();
                err.append("A cyclic reference in the modules has been detected: ");
                for (int i = 0; i < refs.size(); ++i) {
                    if (i > 0) {
                        err.append(" -> ");
                    }
                    err.append((String)refs.get(i));
                }
                err.append(" -> ").append(parent.getName());
                throw new IllegalStateException(err.toString());
            }
            refs.push(parent.getName());
            this.assertNoCycle(parent, refs);
            refs.pop();
        }
    }

    private void bfsCalculateDepth(Module module, int depthNow) {
        int depth = depthNow + 1;
        for (Module child : module.getChildEdges()) {
            child.setDepth(Math.max(depth, child.getDepth()));
            this.maxDepth = Math.max(this.maxDepth, child.getDepth());
        }
        for (Module child : module.getChildEdges()) {
            this.bfsCalculateDepth(child, depth);
        }
    }

    public void buildGraph() throws FileNotFoundException, IOException {
        this.normalizeDependencies();
        for (Module module : this.modules.values()) {
            for (String parentName : module.getParentNames()) {
                Module parent = this.get(parentName);
                if (parent == null) {
                    if (Props.hasPropertyKey(parentName)) {
                        StartLog.debug("Module property not expandable (yet) [%s]", parentName);
                        continue;
                    }
                    StartLog.warn("Module not found [%s]", parentName);
                    continue;
                }
                module.addParentEdge(parent);
                parent.addChildEdge(module);
            }
            for (String optionalParentName : module.getOptionalParentNames()) {
                Module optional = this.get(optionalParentName);
                if (optional == null) {
                    StartLog.debug("Optional module not found [%s]", optionalParentName);
                    continue;
                }
                if (!optional.isEnabled()) continue;
                module.addParentEdge(optional);
                optional.addChildEdge(module);
            }
        }
        Stack<String> refs = new Stack<String>();
        for (Module module : this.modules.values()) {
            refs.push(module.getName());
            this.assertNoCycle(module, refs);
            refs.pop();
        }
        for (Module module : this.modules.values()) {
            if (!module.getParentEdges().isEmpty()) continue;
            this.bfsCalculateDepth(module, 0);
        }
    }

    public void clearMissing() {
        this.missingModules.clear();
    }

    public Integer count() {
        return this.modules.size();
    }

    public void dump() {
        ArrayList<Module> ordered = new ArrayList<Module>();
        ordered.addAll(this.modules.values());
        Collections.sort(ordered, new Module.NameComparator());
        List<Module> active = this.resolveEnabled();
        for (Module module : ordered) {
            boolean activated = active.contains(module);
            boolean enabled = module.isEnabled();
            boolean transitive = activated && !enabled;
            char status = '-';
            if (enabled) {
                status = '*';
            } else if (transitive) {
                status = '+';
            }
            System.out.printf("%n %s Module: %s%n", Character.valueOf(status), module.getName());
            if (!module.getName().equals(module.getFilesystemRef())) {
                System.out.printf("      Ref: %s%n", module.getFilesystemRef());
            }
            for (String parent : module.getParentNames()) {
                System.out.printf("   Depend: %s%n", parent);
            }
            for (String lib : module.getLibs()) {
                System.out.printf("      LIB: %s%n", lib);
            }
            for (String xml : module.getXmls()) {
                System.out.printf("      XML: %s%n", xml);
            }
            if (StartLog.isDebugEnabled()) {
                System.out.printf("    depth: %d%n", module.getDepth());
            }
            if (activated) {
                for (String source : module.getSources()) {
                    System.out.printf("  Enabled: <via> %s%n", source);
                }
                if (!transitive) continue;
                System.out.printf("  Enabled: <via transitive reference>%n", new Object[0]);
                continue;
            }
            System.out.printf("  Enabled: <not enabled in this configuration>%n", new Object[0]);
        }
    }

    public void dumpEnabledTree() {
        ArrayList<Module> ordered = new ArrayList<Module>();
        ordered.addAll(this.modules.values());
        Collections.sort(ordered, new Module.DepthComparator());
        List<Module> active = this.resolveEnabled();
        for (Module module : ordered) {
            if (!active.contains(module)) continue;
            String indent = this.toIndent(module.getDepth());
            System.out.printf("%s + Module: %s [%s]%n", indent, module.getName(), module.isEnabled() ? "enabled" : "transitive");
        }
    }

    public void enable(String name) throws IOException {
        List<String> empty = Collections.emptyList();
        this.enable(name, empty);
    }

    public void enable(String name, List<String> sources) throws IOException {
        if (name.contains("*")) {
            Pattern pat = Pattern.compile(name);
            ArrayList<Module> matching = new ArrayList<Module>();
            do {
                matching.clear();
                for (Map.Entry<String, Module> entry : this.modules.entrySet()) {
                    if (!pat.matcher(entry.getKey()).matches() || entry.getValue().isEnabled()) continue;
                    matching.add(entry.getValue());
                }
                for (Module module : matching) {
                    this.enableModule(module, sources);
                }
            } while (!matching.isEmpty());
        } else {
            Module module = this.modules.get(name);
            if (module == null) {
                System.err.printf("WARNING: Cannot enable requested module [%s]: not a valid module name.%n", name);
                return;
            }
            this.enableModule(module, sources);
        }
    }

    private void enableModule(Module module, List<String> sources) throws IOException {
        String via = "<transitive>";
        if (sources != null) {
            module.addSources(sources);
            via = Main.join(sources, ", ");
        }
        if (module.isEnabled()) {
            StartLog.debug("Enabled module: %s (via %s)", module.getName(), via);
            return;
        }
        StartLog.debug("Enabling module: %s (via %s)", module.getName(), via);
        module.setEnabled(true);
        this.args.parseModule(module);
        module.expandProperties(this.args.getProperties());
        HashSet<String> parentNames = new HashSet<String>();
        parentNames.addAll(module.getParentNames());
        for (String name : parentNames) {
            StartLog.debug("Enable parent '%s' of module: %s", name, module.getName());
            Module parent = this.modules.get(name);
            if (parent == null) {
                Path file = this.baseHome.getPath("modules/" + name + ".mod");
                if (FS.canReadFile(file)) {
                    parent = this.registerModule(file);
                    parent.expandProperties(this.args.getProperties());
                    this.updateParentReferencesTo(parent);
                } else if (!Props.hasPropertyKey(name)) {
                    StartLog.debug("Missing module definition: [ Mod: %s | File: %s ]", name, file);
                    this.missingModules.add(name);
                }
            }
            if (parent == null) continue;
            this.enableModule(parent, null);
        }
    }

    private void findChildren(Module module, Set<Module> ret) {
        ret.add(module);
        for (Module child : module.getChildEdges()) {
            ret.add(child);
        }
    }

    private void findParents(Module module, Map<String, Module> ret) {
        ret.put(module.getName(), module);
        for (Module parent : module.getParentEdges()) {
            ret.put(parent.getName(), parent);
            this.findParents(parent, ret);
        }
    }

    public Module get(String name) {
        return this.modules.get(name);
    }

    public int getMaxDepth() {
        return this.maxDepth;
    }

    public Set<Module> getModulesAtDepth(int depth) {
        HashSet<Module> ret = new HashSet<Module>();
        for (Module module : this.modules.values()) {
            if (module.getDepth() != depth) continue;
            ret.add(module);
        }
        return ret;
    }

    @Override
    public Iterator<Module> iterator() {
        return this.modules.values().iterator();
    }

    public List<String> normalizeLibs(List<Module> active) {
        ArrayList<String> libs = new ArrayList<String>();
        for (Module module : active) {
            for (String lib : module.getLibs()) {
                if (libs.contains(lib)) continue;
                libs.add(lib);
            }
        }
        return libs;
    }

    public List<String> normalizeXmls(List<Module> active) {
        ArrayList<String> xmls = new ArrayList<String>();
        for (Module module : active) {
            for (String xml : module.getXmls()) {
                if (xmls.contains(xml)) continue;
                xmls.add(xml);
            }
        }
        return xmls;
    }

    public Module register(Module module) {
        this.modules.put(module.getName(), module);
        return module;
    }

    public void registerParentsIfMissing(Module module) throws IOException {
        HashSet<String> parents = new HashSet<String>(module.getParentNames());
        for (String name : parents) {
            Path file;
            if (this.modules.containsKey(name) || !FS.canReadFile(file = this.baseHome.getPath("modules/" + name + ".mod"))) continue;
            Module parent = this.registerModule(file);
            this.updateParentReferencesTo(parent);
            this.registerParentsIfMissing(parent);
        }
    }

    public void registerAll() throws IOException {
        for (Path path : this.baseHome.getPaths("modules/*.mod")) {
            this.registerModule(path);
        }
    }

    private void normalizeDependencies() throws FileNotFoundException, IOException {
        HashSet<String> expandedModules = new HashSet<String>();
        boolean done = false;
        while (!done) {
            done = true;
            HashSet<String> missingParents = new HashSet<String>();
            for (Module m : this.modules.values()) {
                for (String parent : m.getParentNames()) {
                    String expanded = this.args.getProperties().expand(parent);
                    if (this.modules.containsKey(expanded) || this.missingModules.contains(parent) || expandedModules.contains(parent)) continue;
                    done = false;
                    StartLog.debug("Missing parent module %s == %s for %s", parent, expanded, m);
                    missingParents.add(parent);
                }
            }
            for (String missingParent : missingParents) {
                String expanded = this.args.getProperties().expand(missingParent);
                Path file = this.baseHome.getPath("modules/" + expanded + ".mod");
                if (FS.canReadFile(file)) {
                    Module module = this.registerModule(file);
                    this.updateParentReferencesTo(module);
                    if (expanded.equals(missingParent)) continue;
                    expandedModules.add(missingParent);
                    continue;
                }
                if (Props.hasPropertyKey(expanded)) {
                    StartLog.debug("Module property not expandable (yet) [%s]", expanded);
                    expandedModules.add(missingParent);
                    continue;
                }
                StartLog.debug("Missing module definition: %s expanded to %s", missingParent, expanded);
                this.missingModules.add(missingParent);
            }
        }
    }

    private Module registerModule(Path file) throws FileNotFoundException, IOException {
        if (!FS.canReadFile(file)) {
            throw new IOException("Cannot read file: " + file);
        }
        StartLog.debug("Registering Module: %s", this.baseHome.toShortForm(file));
        Module module = new Module(this.baseHome, file);
        return this.register(module);
    }

    public Set<String> resolveChildModulesOf(String moduleName) {
        HashSet<Module> ret = new HashSet<Module>();
        Module module = this.get(moduleName);
        this.findChildren(module, ret);
        return this.asNameSet(ret);
    }

    public List<Module> resolveEnabled() {
        HashMap<String, Module> active = new HashMap<String, Module>();
        for (Module module : this.modules.values()) {
            if (!module.isEnabled()) continue;
            this.findParents(module, active);
        }
        for (String missing : this.missingModules) {
            for (String activeModule : active.keySet()) {
                if (!missing.startsWith(activeModule)) continue;
                StartLog.warn("** Unable to continue, required dependency missing. [%s]", missing);
                StartLog.warn("** As configured, Jetty is unable to start due to a missing enabled module dependency.", new Object[0]);
                StartLog.warn("** This may be due to a transitive dependency akin to spdy on npn, which resolves based on the JDK in use.", new Object[0]);
                throw new UsageException(-6, "Missing referenced dependency: " + missing, new Object[0]);
            }
        }
        ArrayList<Module> ordered = new ArrayList<Module>();
        ordered.addAll(active.values());
        Collections.sort(ordered, new Module.DepthComparator());
        return ordered;
    }

    public Set<String> resolveParentModulesOf(String moduleName) {
        HashMap<String, Module> ret = new HashMap<String, Module>();
        Module module = this.get(moduleName);
        this.findParents(module, ret);
        return ret.keySet();
    }

    private String toIndent(int depth) {
        char[] indent = new char[depth * 2];
        Arrays.fill(indent, ' ');
        return new String(indent);
    }

    private void updateParentReferencesTo(Module module) {
        if (module.getName().equals(module.getFilesystemRef())) {
            return;
        }
        for (Module m : this.modules.values()) {
            HashSet<String> resolvedParents = new HashSet<String>();
            for (String parent : m.getParentNames()) {
                if (parent.equals(module.getFilesystemRef())) {
                    resolvedParents.add(module.getName());
                    continue;
                }
                resolvedParents.add(parent);
            }
            m.setParentNames(resolvedParents);
        }
    }

    public String toString() {
        StringBuilder str = new StringBuilder();
        str.append("Modules[");
        str.append("count=").append(this.modules.size());
        str.append(",<");
        boolean delim = false;
        for (String name : this.modules.keySet()) {
            if (delim) {
                str.append(',');
            }
            str.append(name);
            delim = true;
        }
        str.append(">");
        str.append("]");
        return str.toString();
    }
}

