//
// $Id: ZlibInStream.h,v 1.13 2007/03/06 20:42:20 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(ZLIBINSTREAM_OL_GUARD)
#define ZLIBINSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredInStream.h>

struct z_stream_s;

/**
 * @class OLZlibInStream ZlibInStream.h ObjectiveLib/ZlibInStream.h
 *
 * A stream for zlib decompression. This is a concrete layered stream that decompresses all
 * data read using the library zlib. It is suitable for reading data compressed with
 * OLZlibOutStream.
 *
 * @sa OLZlibOutStream
 *
 * @ingroup Streams
 */
@interface OLZlibInStream : OLLayeredInStream
{
@protected
    /**
     * The opaque data type used by zlib to represent the stream
     */
    struct z_stream_s*  zstream;

    /**
     * The buffer required by zlib
     */
    uint8_t*            buffer;

    /**
     * The size of the zlib buffer
     */
    unsigned            bufferSize;
}

/**
 * Create and return a newly created stream. The stream is created with a buffer size
 * of 1,024. The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @return a reference to a newly created stream
 */
+ (id) streamWithInStream: (OLInStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The stream is created with a buffer size
 * of 1,024. The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream;

/**
 * Initialize the stream. The stream is created with the given buffer size.
 * The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @param size the buffer size
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size;

/**
 * Initialize the stream. The stream is created with the given buffer size.
 * The given stream @a underStream will become
 * the next layer down in the stack of streams. If @a zlibHeader is YES, then
 * the stream will attempt to read the zlib header.
 *
 * @attention You probably do not want to use this initializer. This initializer exists so that
 * subclasses will have maximum control over how the zstream is created. Do not use this
 * initializer unless you know what you are doing.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @param size the buffer size
 * @param zlibHeader should be YES if the zlib header should be read from the stream,
 * otherwise NO
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size readZlibHeader: (BOOL)zlibHeader;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Read a sequence of bytes from the stream. Bytes are decompressed up to the value
 * of @a max and the number decompressed is returned. If no more bytes are available
 * from the stream, then UINT_MAX is returned.
 *
 * @exception OLInputOutputException if there is an error reading from the stream
 * @param dest the address to which the bytes should be read
 * @param max the maximum number bytes that should be read
 * @return the number of bytes read or UINT_MAX if no bytes could be read
 */
- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max;

@end

#endif
