#!/usr/bin/perl -w

# This file is part of the t1toinf project version 0.1.
#
# T1TOINF is a perl script which retrieves from a PostScript Type 1
# font afm file and (optionally) also from the corresponding
# PostScript font program information, necessary to construct
# a Windows INF file, which may be used later to install that
# font under ATM. 
#
# Although the INF file format is not documented, all necessary
# information about it is available here:
# http://homepages.fbmev.de/bm134751/inf_fmt_en.html
# Thanks to Johannes Schmidt-Fischer for providing that information.
#
# Copyright (C) 2004 Alexios Kryukov
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation  either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# You can contact the author via e-mail at: basileia@yandex.ru

use Cwd;
use File::Basename;

sub make_path {
	my $PATH = $_[0];
	my $i = 0;
	
	for ($i = 1; $i <= $#_; $i++) {
		$EXTENSION[$i - 1] = $_[$i];
	}

        if ($^O eq 'MSWin32' || $^O eq 'MSDOS') {
	        $DIR_SEP = '\\';
        } elsif ($^O eq 'MacOS') {
	        $DIR_SEP = ':';
        } else {
	        $DIR_SEP = '/';
        }

	my $CURPATH = getcwd ();		# current directory

	($FILE_NAME, $FILE_PATH, $FILE_EXT) = fileparse ($PATH, @EXTENSION);
	
	unless (defined ($FILE_EXT)) {
		$FILE_EXT = '';
	}
	
	if ($FILE_PATH eq '.' . $DIR_SEP) {
		$PATH = $CURPATH . $DIR_SEP . $FILE_NAME . $FILE_EXT;
	}
	
	if ($FILE_EXT eq '') {
		$PATH = $PATH . $EXTENSION[0];
	}
	
	return ($PATH);
}

sub test_numeric {
	my $NUM_STRING = $_[0];

	if ($NUM_STRING =~ m/^-?\d*$/) {
		return True;
	} else {
		return False;
	}
}

sub test_boolean {
	my $NUM_STRING = $_[0];

	if ($NUM_STRING =~ m/^true|false$/) {
		return True;
	} else {
		return False;
	}
}

sub get_inf_data {
	my $AFM_STRING = shift;
	my $T1A_STRING = shift;
        my $BASENAME = shift;
        my $REGULAR_SUFFIX = shift;
        my $serif = shift || True;
	
        $INF_STRING = '';
        
        my %INF_DATA = ();
        
        # Get FontName from AFM
        if ($AFM_STRING =~ m/FontName\s(\S*)/) {
                $INF_DATA{'FontName'} = $1;
                
                # FontName suffix determines font's style
                if ($INF_DATA{'FontName'} =~ m/.*-([^-]*)/) {
                        $STYLE_SPEC = $1;
                }
        }

        # Get FullName from AFM
        if ($AFM_STRING =~ m/FullName\s(.*)/) {
                $INF_DATA{'FullName'} = $1;
        }

        # Get FamilyName from AFM
        if ($AFM_STRING =~ m/FamilyName\s(.*)/) {
                $INF_DATA{'FamilyName'} = $1;
        }

        # Construct AppleName from FamilyName and style specification
        $INF_DATA{'AppleName'} = $INF_DATA{'FamilyName'};
        if (defined ($STYLE_SPEC)) {
                # Ensure that AppleName for a regular font is equal
                # to MSMenuName
                #
                if ($STYLE_SPEC eq 'Regular') {
                        if ($REGULAR_SUFFIX ne '') {
                                $INF_DATA{'AppleName'} .= ' ';
                                $INF_DATA{'AppleName'} .= $REGULAR_SUFFIX;
                        }
                } else {
                        $INF_DATA{'AppleName'} .= ' ';
                        $INF_DATA{'AppleName'} .= $STYLE_SPEC;
                }
        }

        # Construct MSMenuName from FamilyName and style specification
        $INF_DATA{'MSMenuName'} = $INF_DATA{'FamilyName'};
        
        # Adobe Cyrillic fonts usually have the word "Cyrillic"
        # in the FullName, but simply Cyr in the MSMenuName
        #
        $INF_DATA{'MSMenuName'} =~ s/Cyrillic/Cyr/;
        
        # Additional information about Roman font style,
        # which may be either retrieved from the font itself,
        # or constructed on the basis of the given Roman font suffix
        #
        $STYLE_ADD = ' ';
        if (defined ($STYLE_SPEC)) {
                # If an uprignt style name can be retrieved from
                # font's full name, use it
                #
                if ($STYLE_SPEC =~ m/(Roman|Upright)/) {
                        $STYLE_ADD .= $1;
                } elsif ($REGULAR_SUFFIX ne '') {
                        $STYLE_ADD .= $REGULAR_SUFFIX;
                }
                
                if ($STYLE_SPEC =~ m/SC|OsF/) {
                        $STYLE_ADD .= 'SC';
                }
        } elsif ($REGULAR_SUFFIX ne '') {
                $STYLE_ADD .= $REGULAR_SUFFIX;
        }
        
        if ($STYLE_ADD ne ' ') {
                $INF_DATA{'MSMenuName'} .= $STYLE_ADD;
        }
        
        # Construct VPMenuName and WORDMenuName from MSMenuName
        $INF_DATA{'VPMenuName'} = $INF_DATA{'MSMenuName'};
        $INF_DATA{'WORDMenuName'} = $INF_DATA{'MSMenuName'};
        $INF_DATA{'WORDMenuName'} =~ s/\s//g;

        # Get version from AFM
        if ($AFM_STRING =~ m/Version\s(.*)/) {
                $INF_DATA{'version'} = $1;
        } else {
                # and from T1A
                if ($T1A_STRING =~ m/\/version\s\((.*)\)/) {
                        $INF_DATA{'version'} = $1;
                }
        }

        # Get isFixedPitch from AFM
        if ($AFM_STRING =~ m/IsFixedPitch\s(true|false)/) {
                $INF_DATA{'isFixedPitch'} = $1;
        } else {
                # and from T1A
                if ($T1A_STRING =~ m/\/isFixedPitch\s(true|false)/) {
                        $INF_DATA{'isFixedPitch'} = $1;
                }
        }

        # Get Character Set, Encoding and Pi from AFM
        if ($AFM_STRING =~ m/EncodingScheme\s(\S*)/) {
                $AFM_CODING = $1;
        } else {
                $AFM_CODING = '';
        }
        # and from T1A
        if ($T1A_STRING =~ m/\/Encoding\s(\S*)\sdef/) {
                $T1A_CODING = $1;
        } else {
                $T1A_CODING = '';
        }
        
        if ($T1A_CODING eq 'StandardEncoding' ||
                $AFM_CODING eq 'AdobeStandardEncoding') {
                
                $INF_DATA{'CharacterSet'} = 'isoadobe';
                $INF_DATA{'Encoding'} = 'StandardEncoding';
                $INF_DATA{'Pi'} = 'false';
        } else {
                $INF_DATA{'CharacterSet'} = 'custom';
                $INF_DATA{'Encoding'} = 'FontSpecificEncoding';
                $INF_DATA{'Pi'} = 'true';
        }

        # Get ItalicAngle from AFM
        if ($AFM_STRING =~ m/ItalicAngle\s(-?\d*)/) {
                $INF_DATA{'ItalicAngle'} = $1;
        } else {
                $INF_DATA{'ItalicAngle'} = 0;
        }

        # Get CapHeight from AFM
        if ($AFM_STRING =~ m/CapHeight\s(\d*)/) {
                $INF_DATA{'CapHeight'} = $1;
        } else {
                # if CapHeight is not specified explicitly, retrieve it from 
                # bounding box for the letter 'H'
                #
                if ($AFM_STRING =~ m/N\sH\s*;\sB\s\d*\s\d*\s\d*\s(\d*)\s/) {
                        $INF_DATA{'CapHeight'} = $1;
                }
        }

        # Get UnderlinePosition from AFM
        if ($AFM_STRING =~ m/UnderlinePosition\s(-?\d*)/) {
                $INF_DATA{'UnderlinePosition'} = $1;
        } else {
                # and from T1A
                if ($T1A_STRING =~ m/\/UnderlinePosition\s(-?\d*)/) {
                        $INF_DATA{'UnderlinePosition'} = $1;
                }
        }

        # Get UnderlineThickness from AFM
        if ($AFM_STRING =~ m/UnderlineThickness\s(-?\d*)/) {
                $INF_DATA{'UnderlineThickness'} = $1;
        } else {
                # and from T1A
                if ($T1A_STRING =~ m/\/UnderlineThickness\s(-?\d*)/) {
                        $INF_DATA{'UnderlineThickness'} = $1;
                }
        }

        # Get StemWidth from AFM
        if ($AFM_STRING =~ m/StdVW\s(\d*)/) {
                $INF_DATA{'StemWidth'} = $1;
        } else {
                # and from T1A
                if ($T1A_STRING =~ m/\/StdVW\s\[(\d*)\]/) {
                        $INF_DATA{'StemWidth'} = $1;
                }
        }

        # Add Serif to the INF hash
        if ($serif eq True) {
                $INF_DATA{'Serif'} = 'true';
        } else {
                $INF_DATA{'Serif'} = 'false';
        }
        
        # Get SuperiorBaseline from AFM
        if ($AFM_STRING =~ m/twosuperior\s*;\sB\s\d*\s(\d*)\s\d*\s\d*/) {
                $INF_DATA{'SuperiorBaseline'} = $1;
        }
        
        # Obtain VPStyle
        if ($AFM_STRING =~ m/Weight\s(\S*)/) {
                $FONT_WEIGHT = $1;

                # Obtain PCLStrokeWeight from the font weight information
                if ($FONT_WEIGHT =~ m/^(UltraThin)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -7;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(ExtraThin)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -6;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(Thin)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -5;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(ExtraLight)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -4;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(Light)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -3;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(Titling)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = -1;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(SemiBold)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 1;
                        $BOLD = True;
                } elsif ($FONT_WEIGHT =~ m/^(Medium)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 2;
                        $BOLD = False;
                } elsif ($FONT_WEIGHT =~ m/^(Bold)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 3;
                        $BOLD = True;
                } elsif ($FONT_WEIGHT =~ m/^(ExtraBold)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 4;
                        $BOLD = True;
                } elsif ($FONT_WEIGHT =~ m/^(Black)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 5;
                        $BOLD = True;
                } elsif ($FONT_WEIGHT =~ m/^(ExtraBlack)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 6;
                        $BOLD = True;
                } elsif ($FONT_WEIGHT =~ m/^(UltraBlack)$/i) {
                        $INF_DATA{'PCLStrokeWeight'} = 7;
                        $BOLD = True;
                } else {
                        $INF_DATA{'PCLStrokeWeight'} = 0;
                        $BOLD = False;
                }
        } else {
                $INF_DATA{'PCLStrokeWeight'} = 0;
                $BOLD = False;
        }

        
        if ($BOLD eq True) {
                if ($INF_DATA{'ItalicAngle'} != 0) {
                        $INF_DATA{'VPStyle'} = 'T';
                } else {
                        $INF_DATA{'VPStyle'} = 'B';
                }
        } else {
                if ($INF_DATA{'ItalicAngle'} != 0) {
                        $INF_DATA{'VPStyle'} = 'I';
                } else {
                        $INF_DATA{'VPStyle'} = 'N';
                }
        }

        # Obtain WindowsFirstChar according to the font full name
        if ($INF_DATA{'Encoding'} eq 'StandardEncoding') {
                $INF_DATA{'WindowsCharSet'} = 0;
        } else {
                if ($INF_DATA{'FullName'} =~ m/Cyrillic/) {
                        $INF_DATA{'WindowsCharSet'} = 204;
                } elsif ($INF_DATA{'FullName'} =~ m/Greek/) {
                        $INF_DATA{'WindowsCharSet'} = 161;
                } elsif ($INF_DATA{'FullName'} =~ m/Turkish/) {
                        $INF_DATA{'WindowsCharSet'} = 162;
                } elsif ($INF_DATA{'FullName'} =~ m/Baltic/) {
                        $INF_DATA{'WindowsCharSet'} = 186;
                } elsif ($INF_DATA{'FullName'} =~ m/Hebrew/) {
                        $INF_DATA{'WindowsCharSet'} = 177;
                } elsif ($INF_DATA{'FullName'} =~ m/Eastern\s?European/ ||
                        $INF_DATA{'FullName'} =~ m/Central\s?European/) {
                        
                        $INF_DATA{'WindowsCharSet'} = 238;
                } else {
                        $INF_DATA{'WindowsCharSet'} = 0;
                }
        }
        
        # Set PCFileNamePrefix
        $INF_DATA{'PCFileNamePrefix'} = $BASENAME;
        
        # Set WindowsFirstChar and WindowsLastChar
        $INF_DATA{'WindowsFirstChar'} = 32;
        $INF_DATA{'WindowsLastChar'} = 255;
        
	return (\%INF_DATA);
}

sub add_inf_entry {
        my $INF_DATA_REF = shift;
        my $INF_KEY = shift;
        my $IS_STRING = shift || False;
        
        my %INF_DATA = %$INF_DATA_REF;
        
        if (defined $INF_DATA{$INF_KEY}) {
                if ($IS_STRING eq True) {
                        $INF_ENTRY = sprintf 
                                ("%s (%s)\r\n", $INF_KEY, $INF_DATA{$INF_KEY});
                } else {
                        $INF_ENTRY = sprintf 
                                ("%s %s\r\n", $INF_KEY, $INF_DATA{$INF_KEY});
                }
        } else {
                $INF_ENTRY = ''
        }
        
        return $INF_ENTRY;
}

sub get_inf {
        my $INF_DATA_REF = shift;
        # my %INF_DATA = %$INF_DATA_REF;
        
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'FontName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'FullName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'AppleName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'AppleFONDID', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'FamilyName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'version', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'isFixedPitch', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'CharacterSet', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'Encoding', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'ItalicAngle', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'CapHeight', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'UnderlinePosition', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'UnderlineThickness', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'StemWidth', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'Serif', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'Pi', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'SuperiorBaseline', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'MSMenuName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'VPMenuName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'WORDMenuName', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'VPTypefaceID', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'VPStyle', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'PCFileNamePrefix', True);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'PCLTypefaceID', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'PCLStrokeWeight', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'PCLStyle', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'WindowsCharSet', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'WindowsFirstChar', False);
        $INF_STRING .= add_inf_entry ($INF_DATA_REF, 'WindowsLastChar', False);

        # foreach $KEY (sort keys %INF_DATA){
        #    if (test_numeric ($INF_DATA{$KEY}) eq True) {
        #            $INF_STRING .= sprintf ("%s %s\r\n", $KEY, $INF_DATA{$KEY});
        #    } elsif (test_boolean ($INF_DATA{$KEY}) eq True) {
        #            $INF_STRING .= sprintf ("%s %s\r\n", $KEY, $INF_DATA{$KEY});
        #    } else {
        #            $INF_STRING .= sprintf ("%s (%s)\r\n", $KEY, $INF_DATA{$KEY});
        #    }
        #}
        
        return $INF_STRING;
}

sub analyze_args {
	my $i = 0;
	
	for ($i = 0; $i <= $#ARGV; $i++) {
		if (index ($ARGV[$i], '-') == 0) {
			if ($ARGV[$i] eq '-t') {
				if ($i == $#ARGV) {
					show_help ();
				} else {
					$i++;
					$INP_T1A = $ARGV[$i];
				}
			} elsif ($ARGV[$i] eq '-r') {
				if ($i == $#ARGV) {
					show_help ();
				} else {
					$i++;
					$REGULAR_SUFFIX = $ARGV[$i];
				}
			} elsif ($ARGV[$i] eq '-s') {
				if ($i == $#ARGV) {
					show_help ();
				} else {
					$i++;
					if ($ARGV[$i] eq 'false') {
                                                $SERIF = False;
                                        } else {
                                                $SERIF = True;
                                        }
				}
                        }
		} else {
			if ($INP_AFM eq '') {
				$INP_AFM = $ARGV[$i];
			} elsif ($OUT_INF eq '') {
				$OUT_INF = $ARGV[$i];
			} else {
				show_help ();
			}
		}
	}
	
	if ($INP_AFM eq '') {
		show_help ();
	}

	if ($INP_T1A eq '') {
		$INP_T1A = '';
	}
        
	unless (defined ($REGULAR_SUFFIX)) {
		$REGULAR_SUFFIX = '';
	}

	unless (defined ($SERIF)) {
		$SERIF = True;
	}

        # afm file name without extension
        $BASENAME = $INP_AFM;
        $BASENAME =~ s/\..*$//;

        if ($OUT_INF eq '') {
            $OUT_INF = $BASENAME;
        }
        
        return $INP_AFM, $BASENAME, $SERIF, $INP_T1A, $OUT_INF, $REGULAR_SUFFIX;
}

sub show_help {
	die (
                "Usage: fixoortf.pl [options] <afmfile[.afm]> [<inffile[.inf]>]\n\n" .
                "The following options are available:\n" .
                "-t Specifies a disassembled Type 1 font file, which may\n" .
                "   contain some information not found in the afm file;\n" .
                "-r for bold or slanted fonts may be used to specify the\n" .
                "   style name of the corresponding upright font (e. g. Roman,\n" .
                "   Upright or Regular), which may be useful for constructing\n" .
                "   the MSMenuName entry;\n" .
                "-s (valid values are true or false)\n" .
                "   specifies if the font which should be processed has the\n" .
                "   serif design.\n"
                );
}

# Program body begins here

# First, global variables should be initialized
$INP_AFM = '';
$INP_T1A = '';
$OUT_INF = '';

($INP_AFM, $BASENAME, $SERIF, $INP_T1A, $OUT_INF, $REGULAR_SUFFIX) = analyze_args ();

$INP_AFM = make_path ($INP_AFM, ('.afm', '.AFM'));
if ($INP_T1A ne '') {
	$INP_T1A = make_path ($INP_T1A, ('.t1a', '.T1A'));
}

$OUT_INF = make_path ($OUT_INF, ('.inf', '.INF'));

open (AFM, "< $INP_AFM") or 
die ("Can't open $INP_AFM.\n");

while ($AFM_LINE = <AFM>) {
	$AFM_STRING .= $AFM_LINE;
}
close (AFM);

if ($INP_T1A ne "") {
	open (PS, "< $INP_T1A") or 
	die ("Can't open $INP_T1A.\n");

	while ($T1A_LINE = <PS>) {
		$T1A_STRING .= $T1A_LINE;
	}
close (PS);
} else {
	$T1A_STRING = '';
}

$INF_DATA = get_inf_data 
        ($AFM_STRING, $T1A_STRING, $BASENAME, $REGULAR_SUFFIX, $SERIF);
$INF_STRING = get_inf ($INF_DATA);

open (OUTINF, "> $OUT_INF") or
die ("Can't write to $OUT_INF.\n");

print OUTINF $INF_STRING;
close (OUTINF);

exit (0);
