#!/bin/sh

# Copyright (C) 2024 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr/pkg"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/usr/pkgsrc/net/kea/work/kea-2.6.1/src/share/database/scripts/pgsql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/usr/pkgsrc/net/kea/work/kea-2.6.1/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

VERSION=$(pgsql_version "$@")

if [ "$VERSION" != "22.0" ]; then
    printf 'This script upgrades 22.0 to 22.1. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${VERSION}"
    exit 0
fi

psql "$@" >/dev/null <<EOF
START TRANSACTION;

-- This line starts the schema upgrade to version 22.1.

-- First we migrate existing OPT_RECORD_TYPE values.
SELECT set_config('kea.disable_audit', 'true', false);
UPDATE dhcp4_option_def SET type = 254 WHERE record_types IS NOT NULL;
UPDATE dhcp6_option_def SET type = 254 WHERE record_types IS NOT NULL;

-- Create the table that enumerates option definition data types.
CREATE TABLE option_def_data_type (
    id smallint NOT NULL,
    name VARCHAR(32) NOT NULL,
    PRIMARY KEY (id)
);

-- Now insert supported types.
-- We skip (9, 'any-address') as it is not externally supported.
INSERT INTO option_def_data_type VALUES
    (0, 'empty'),
    (1, 'binary'),
    (2, 'boolean'),
    (3, 'int8"'),
    (4, 'int16'),
    (5, 'int32'),
    (6, 'uint8'),
    (7, 'uint16'),
    (8, 'uint32'),
    (10, 'ipv4-address'),
    (11, 'ipv6-address'),
    (12, 'ipv6-prefix'),
    (13, 'psid'),
    (14, 'string'),
    (15, 'tuple'),
    (16, 'fqdn'),
    (17, 'internal'),
    (254, 'record');

--  Add foreign key constraints to enforce only valid types.
ALTER TABLE dhcp4_option_def
    ADD CONSTRAINT fk_option_def_data_type4 FOREIGN KEY (type) REFERENCES option_def_data_type(id);

ALTER TABLE dhcp6_option_def
    ADD CONSTRAINT fk_option_def_data_type6 FOREIGN KEY (type) REFERENCES option_def_data_type(id);

SELECT set_config('kea.disable_audit', 'false', false);
UPDATE schema_version
    SET version = '22', minor = '1';

-- This line concludes the schema upgrade to version 22.1.

-- Commit the script transaction.
COMMIT;

EOF
