# Copyright (C) 2008 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""Contains classes for all messages put out by the NewzbinDownloader module."""

from lottanzb.hellalog import LogRecord, nzbdownloader
from lottanzb.util import _

class DNZBRequestFailedInfo(LogRecord): # NewzbinDownloader.gotHeaders(...)
    PATTERN = r"^DNZB request failed: (?P<error_message>.+)$"
    TEMPLATE = _("DNZB request failed: %(error_message)s")

class DownloadingInfo(nzbdownloader.DownloadingInfo): # NewzbinDownloader.download()
    PATTERN = r"^Downloading newzbin NZB: (?P<id>\d+) $"
    TEMPLATE = _("Downloading Newzbin NZB: %(id)s")

class DownloadError(LogRecord): # NewzbinDownloader.handleEnqueueNZB(...)
    PATTERN = r"^Unable to download newzbin NZB: (?P<id>\d+) \((?P<error_message>\D.*)\)$"
    TEMPLATE = _("Unable to download Newzbin NZB: %(id)s (%(error_message)s)")
    
    def set_data(self, data):
        self.invalid_id = data["error_message"] == "This appears to be an invalid msgid"
        self.incorrect_credentials = data["error_message"] == "Incorrect NEWZBIN_USERNAME/PASSWORD?"
        
        if self.invalid_id:
            data["error_message"] = _("This appears to be an invalid Newzbin ID")
        elif self.incorrect_credentials:
            data["error_message"] = _("Incorrect Newzbin username or password")
        
        LogRecord.set_data(self, data)

class RateLimitDownloadError(DownloadError): # NewzbinDownloader.errBack(...)
    PATTERN = r"^Unable to download newzbin NZB: (?P<id>\d+) due to rate limiting\. Will not retry$"
    TEMPLATE = _("Unable to download Newzbin NZB %(id)s due to rate limiting. Will not retry")

class RetryDownloadError(DownloadError): # NewzbinDownloader.errBack(...)
    PATTERN = r"^Unable to download newzbin NZB: (?P<id>\d+) \(newzbin said: (?P<error_message>.*)\) will retry in (?P<retry_in>\d+) seconds \(attempt: (?P<attempts>\d+)\)$"
    TEMPLATE = _("Unable to download Newzbin NZB: %(id)s (%(error_message)s). Will retry in %(retry_in)s seconds (attempts: %(attempts))")

class OtherDownloadError(DownloadError): # NewzbinDownloader.errBack(...)
    PATTERN = r"^Unable to download newzbin NZB: (?P<id>\d+) \((?P<code>\d+): (?P<text>.+)\)$"
    TEMPLATE = _("Unable to download Newzbin NZB: %(id)s (%(code)s: %(text)s)")
