PK
     JU[p  p    package.xmlUT	 hhux r      <package>
  <name>WebServer</name>
  <namespace>NetClients.WikiWorks</namespace>
  <prereq>NetClients</prereq>

  <filein>WebServer.st</filein>
  <filein>FileServer.st</filein>
  <filein>WikiServer.st</filein>
  <filein>STT.st</filein>
  <filein>Haiku.st</filein>
  <file>edit.jpg</file>
  <file>example1.stt</file>
  <file>example2.stt</file>
  <file>find.jpg</file>
  <file>head.jpg</file>
  <file>help.jpg</file>
  <file>history.jpg</file>
  <file>next.jpg</file>
  <file>prev.jpg</file>
  <file>recent.jpg</file>
  <file>rename.jpg</file>
  <file>test.st</file>
  <file>top.jpg</file>
  <file>ChangeLog</file>
</package>PK
     
wB|E  E    WebServer.stUT	 NQhux r      "======================================================================
|
|   Generic web-server framework
|
|
 ======================================================================"

"======================================================================
|
| Copyright 2000, 2001 Travis Griggs and Ken Treis
| Written by Travis Griggs, Ken Treis and others.
| Port to GNU Smalltalk, enhancements and refactoring by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
|
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
|
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.	If not, write to the Free Software
| Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
|
 ======================================================================"



NetServer subclass: WebServer [
    | virtualHosts defaultVirtualHost log |
    
    <comment: 'A WebServer keeps a socket listening on a port, and dispatches incoming
requests to Servlet objects.  Thus, it is extremely expandable through
`servlets'' which subclass Servlet.  A separate Process is devoted to HTTP
serving.'>
    <category: 'Web-Framework'>

    Version := nil.

    WebServer class >> version [
	<category: 'accessing'>
	| number |
	Version isNil ifFalse: [^Version].
	number := Smalltalk version subStrings 
		    detect: [:each | (each at: 1) isDigit]
		    ifNone: 
			["???"

			'0.0'].
	^Version := 'GNU-WikiWorks/' , number
    ]

    log: action uri: location time: time [
	"self times nextPut: (Array with: action with: location with: time)"

	<category: 'logging'>
	Transcript
	    print: time;
	    space;
	    nextPutAll: action;
	    space;
	    print: location;
	    nl
    ]

    log [
	"self times"

	<category: 'logging'>
	log isNil ifTrue: [log := WriteStream on: Array new].
	^log
    ]

    depth [
	<category: 'accessing'>
	^-1
    ]

    addVirtualHost: aServlet [
	<category: 'accessing'>
	virtualHosts addComponent: aServlet
    ]

    defaultVirtualHost [
	<category: 'accessing'>
	^defaultVirtualHost
    ]

    defaultVirtualHost: anHost [
	<category: 'accessing'>
	virtualHosts rootServlet: (virtualHosts componentNamed: anHost).
	defaultVirtualHost := anHost
    ]

    handler [
	<category: 'accessing'>
	^virtualHosts rootServlet
    ]

    handler: aServlet [
	<category: 'accessing'>
	aServlet name: self defaultVirtualHost.
	virtualHosts
	    addComponent: aServlet;
	    rootServlet: aServlet
    ]

    respondTo: aRequest [
	<category: 'accessing'>
	| host handler |
	host := aRequest at: #HOST ifAbsent: [self defaultVirtualHost].
	(virtualHosts hasComponentNamed: host) 
	    ifFalse: [host := self defaultVirtualHost].
	(virtualHosts componentNamed: host) respondTo: aRequest
    ]

    initialize [
	<category: 'initialize-release'>
	super initialize.
	virtualHosts := CompositeServlet new.
	virtualHosts parent: self.
	self
	    defaultVirtualHost: Sockets.SocketAddress localHostName;
	    handler: CompositeServlet new
    ]

    uriOn: aStream [
	<category: 'private'>
	aStream nextPutAll: 'http:/'
    ]

    newSession [
	<category: 'private'>
	^WebSession new
    ]
]



NetSession subclass: WebSession [
    
    <comment: 'A WebSession is the NetSession object created by a WebServer.'>
    <category: 'Web-Framework'>

    next [
	<category: 'private'>
	^WebRequest for: self socket
    ]

    log: req time: time [
	<category: 'private'>
	self server 
	    log: req action
	    uri: req location
	    time: time
    ]
]



Object subclass: Servlet [
    | name parent |
    
    <category: 'Web-Framework'>
    <comment: 'A Servlet handles WebRequests that are given to it. WebRequests 
come from a WebServer, but often a Servlet will pass them on to
other Servlets.  Thus, sometimes there is a tree of Servlets.'>

    Servlet class >> named: aString [
	<category: 'instance creation'>
	^(self new)
	    name: aString;
	    yourself
    ]

    depth [
	<category: 'accessing'>
	^parent depth + 1
    ]

    name [
	<category: 'accessing'>
	^name
    ]

    name: aString [
	<category: 'accessing'>
	name := aString
    ]

    parent [
	<category: 'accessing'>
	^parent
    ]

    parent: anObject [
	<category: 'accessing'>
	parent := anObject
    ]

    uriOn: aStream [
	<category: 'accessing'>
	self printOn: aStream
    ]

    printOn: aStream [
	<category: 'accessing'>
	parent uriOn: aStream.
	self name isNil ifTrue: [^self].
	aStream nextPut: $/.
	aStream nextPutAll: self name
    ]
]



Servlet subclass: CompositeServlet [
    | components rootServlet errorServlet |
    
    <category: 'Web-Framework'>
    <comment: 'Handles a request by looking at the next name in the path.  If there is
no name, it uses the root handler. If there is no handler for the name,
uses the error handler.  Names are case sensitive.'>

    CompositeServlet class >> new [
	<category: 'instance creation'>
	^self onError: ErrorServlet new
    ]

    CompositeServlet class >> onError: aServlet [
	<category: 'instance creation'>
	^self onError: aServlet onRoot: ServletList new
    ]

    CompositeServlet class >> onError: aServlet onRoot: anotherServlet [
	<category: 'instance creation'>
	^super new onError: aServlet onRoot: anotherServlet
    ]

    errorServlet [
	<category: 'accessing'>
	^errorServlet
    ]

    errorServlet: aServlet [
	<category: 'accessing'>
	errorServlet := aServlet.
	aServlet parent: self
    ]

    rootServlet [
	<category: 'accessing'>
	^rootServlet
    ]

    rootServlet: aServlet [
	<category: 'accessing'>
	rootServlet := aServlet.
	aServlet parent: self
    ]

    addComponent: aServlet [
	<category: 'accessing'>
	components at: aServlet name put: aServlet.
	aServlet parent: self
    ]

    componentNamed: aString [
	<category: 'accessing'>
	^components at: aString ifAbsent: [errorServlet]
    ]

    components [
	<category: 'accessing'>
	^components copy
    ]

    hasComponentNamed: aString [
	<category: 'accessing'>
	^components includesKey: aString
    ]

    onError: aServlet onRoot: anotherServlet [
	<category: 'initialize release'>
	components := Dictionary new.
	self errorServlet: aServlet.
	self rootServlet: anotherServlet.
	anotherServlet parent: self
    ]

    respondTo: aRequest [
	<category: 'interaction'>
	| componentName |
	aRequest location size < self depth 
	    ifTrue: [^rootServlet respondTo: aRequest].
	componentName := aRequest location at: self depth.
	(self hasComponentNamed: componentName) 
	    ifFalse: [^errorServlet respondTo: aRequest].
	^(self componentNamed: componentName) respondTo: aRequest
    ]
]



Servlet subclass: ServletList [
    
    <category: 'Web-Framework'>
    <comment: 'A ServletList output a list of servlets that are children of its parent.
It is typically used as the root handler of a CompositeServlet.'>

    respondTo: aRequest [
	<category: 'interaction'>
	| stream |
	stream := aRequest stream.
	parent components isEmpty 
	    ifTrue: 
		[^(ErrorResponse unavailable)
		    respondTo: aRequest;
		    nl].
	aRequest pageFollows.
	stream
	    nextPutAll: '<HTML><TITLE>Top page</TITLE><BODY>';
	    nl.
	stream
	    nextPutAll: '<H2>Welcome to my server!!</H2>';
	    nl.
	stream
	    nextPutAll: 'This server contains the following sites:';
	    nl.
	stream
	    nextPutAll: '<UL>';
	    nl.
	parent components keys asSortedCollection do: 
		[:each | 
		stream
		    nextPutAll: '  <LI><A HREF="/';
		    nextPutAll: each;
		    nextPutAll: '">';
		    nextPutAll: each;
		    nextPutAll: '</A>';
		    nextPutAll: ', a ';
		    print: (parent componentNamed: each) class;
		    nl].
	stream
	    nextPutAll: '</UL>';
	    nl.
	stream
	    nextPutAll: '</BODY></HTML>';
	    nl;
	    nl
    ]
]



Servlet subclass: ErrorServlet [
    
    <category: 'Web-Framework'>
    <comment: 'An ErrorServlet gives a 404 (not found) or 503 (unavailable) error,
depending on whether its parent has children or not.  It is typically used
as the error handler of a CompositeServlet.'>

    respondTo: aRequest [
	<category: 'interaction'>
	| response |
	response := parent components isEmpty 
		    ifFalse: [ErrorResponse notFound]
		    ifTrue: [ErrorResponse unavailable].
	(#('HEAD' 'GET' 'POST') includes: aRequest action) 
	    ifFalse: [response := ErrorResponse acceptableMethods: #('HEAD' 'GET' 'POST')].
	response respondTo: aRequest
    ]
]



Stream subclass: WebResponse [
    | responseStream request |
    
    <category: 'Web-Framework'>
    <comment: 'WebResponse is an object that can emit an HTTP entity.  There can be
different subclasses of WebResponse for the various ways a page can be
rendered, such as errors, files from the file system, or Wiki pages.
Although you are not forced to use WebResponse to respond to requests
in your Servlet, doing so means that a good deal of code is already
there for you, including support for emitting headers, distinguishing
HEAD requests, HTTP/1.1 multi-request connections, and If-Modified-Since
queries.

All subclasses must implement sendBody.'>

    << anObject [
	<category: 'streaming'>
	responseStream display: anObject
    ]

    nl [
	<category: 'streaming'>
	responseStream nl
    ]

    nextPut: aCharacter [
	<category: 'streaming'>
	responseStream nextPut: aCharacter
    ]

    nextPutUrl: aString [
	<category: 'streaming'>
	responseStream nextPutAll: (URL encode: aString)
    ]

    nextPutAll: aString [
	<category: 'streaming'>
	responseStream nextPutAll: aString
    ]

    do: aBlock [
	<category: 'streaming'>
	self shouldNotImplement
    ]

    next [
	<category: 'streaming'>
	self shouldNotImplement
    ]

    atEnd [
	<category: 'streaming'>
	^true
    ]

    isErrorResponse [
	<category: 'testing'>
	^false
    ]

    modifiedTime [
	<category: 'response'>
	^DateTime now
    ]

    respondTo: aRequest [
	<category: 'response'>
	responseStream := aRequest stream.
	request := aRequest.
	self notModified 
	    ifTrue: [self sendNotModifiedResponse]
	    ifFalse: 
		[self sendHeader.
		aRequest isHead ifFalse: [self sendBody]].
	responseStream := request := nil
    ]

    notModified [
	<category: 'response'>
	| ifModSince modTime |
	ifModSince := request dateTimeAt: #'IF-MODIFIED-SINCE' ifAbsent: [nil].
	modTime := self modifiedTime.
	^ifModSince notNil and: [modTime <= ifModSince]
    ]

    request [
	<category: 'response'>
	^request
    ]

    responseStream [
	<category: 'response'>
	^responseStream
    ]

    sendBody [
	<category: 'response'>
	
    ]

    contentLength [
	<category: 'response'>
	^nil
    ]

    sendHeader [
	<category: 'response'>
	| stream |
	stream := responseStream.
	responseStream := CrLfStream on: stream.
	self sendResponseType.
	self sendServerHeaders.
	self sendStandardHeaders.
	self sendModifiedTime.
	self sendMimeType.
	self sendHeaderSeparator.

	"Send the body as binary"
	responseStream := stream
    ]

    sendHeaderSeparator [
	<category: 'response'>
	self nl
    ]

    sendNotModifiedResponse [
	<category: 'response'>
	^self
	    nextPutAll: 'HTTP/1.1 304 Not modified';
	    sendServerHeaders;
	    sendModifiedTime;
	    sendHeaderSeparator;
	    yourself
    ]

    sendMimeType [
	<category: 'response'>
	self
	    nextPutAll: 'Content-Type: text/html';
	    nl
    ]

    sendResponseType [
	<category: 'response'>
	self
	    nextPutAll: 'HTTP/1.1 200 Page follows';
	    nl
    ]

    sendServerHeaders [
	<category: 'response'>
	self
	    nextPutAll: 'Date: ';
	    sendTimestamp: DateTime now;
	    nl;
	    nextPutAll: 'Server: ';
	    nextPutAll: WebServer version;
	    nl
    ]

    sendStandardHeaders [
	<category: 'response'>
	| length |
	length := self contentLength.
	length isNil 
	    ifTrue: [request moreRequests: false]
	    ifFalse: 
		[self
		    << 'Content-Length: ';
		    << length;
		    nl].
	self
	    << 'Connection: ';
	    << (request at: #Connection);
	    nl
    ]

    sendModifiedTime [
	<category: 'response'>
	self
	    << 'Last-Modified: ';
	    sendTimestamp: self modifiedTime;
	    nl
    ]

    sendTimestamp: aTimestamp [
	<category: 'response'>
	| utc |
	utc := aTimestamp offset = Duration zero 
		    ifTrue: [aTimestamp]
		    ifFalse: [aTimestamp asUTC].
	self
	    nextPutAll: aTimestamp dayOfWeekAbbreviation;
	    nextPutAll: (aTimestamp day < 10 ifTrue: [', 0'] ifFalse: [', ']);
	    print: aTimestamp day;
	    space;
	    nextPutAll: aTimestamp monthAbbreviation;
	    space;
	    print: aTimestamp year;
	    space;
	    print: aTimestamp asTime;
	    nextPutAll: ' GMT'
    ]

    lineBreak [
	<category: 'html'>
	self
	    << '<BR>';
	    nl
    ]

    heading: aBlock [
	<category: 'html'>
	self heading: aBlock level: 1
    ]

    heading: aBlock level: anInteger [
	<category: 'html'>
	self << '<H' << anInteger << '>'.
	aBlock value.
	self
	    << '</H';
	    << anInteger;
	    << '>';
	    nl
    ]

    horizontalLine [
	<category: 'html'>
	self
	    << '<HR>';
	    nl
    ]

    image: fileNameBlock linkTo: urlBlock titled: titleBlock [
	<category: 'html'>
	self << '<A href="'.
	urlBlock value.
	self << '"><IMG src="'.
	fileNameBlock value.
	self << '" alt="'.
	titleBlock value.
	self << '" border=0></A>'
    ]

    image: fileNameBlock titled: titleBlock [
	<category: 'html'>
	self << '<IMG src="'.
	fileNameBlock value.
	self << '" alt="'.
	titleBlock value.
	self << '">'
    ]

    linkTo: urlBlock titled: titleBlock [
	<category: 'html'>
	self << '<A href="'.
	urlBlock value.
	self << '">'.
	titleBlock value.
	self << '</A>'
    ]

    listItem: aBlock [
	<category: 'html'>
	self << '<LI>'.
	aBlock value.
	self
	    << '</LI>';
	    nl
    ]

    monospace: aBlock [
	<category: 'html'>
	self << '<PRE>'.
	aBlock value.
	self
	    << '</PRE>';
	    nl
    ]

    para: aBlock [
	<category: 'html'>
	self << '<P>'.
	aBlock value.
	self
	    << '</P>';
	    nl
    ]

    bold: aBlock [
	<category: 'html'>
	self << '<B>'.
	aBlock value.
	self
	    << '</B>';
	    nl
    ]

    italic: aBlock [
	<category: 'html'>
	self << '<I>'.
	aBlock value.
	self
	    << '</I>';
	    nl
    ]

    tr: aBlock [
	<category: 'html'>
	self << '<TR>'.
	aBlock value.
	self
	    << '</TR>';
	    nl
    ]

    td: aBlock [
	<category: 'html'>
	self << '<TD>'.
	aBlock value.
	self
	    << '</TD>';
	    nl
    ]
]



Object subclass: WebRequest [
    | originator stream action clientData postData location uri |
    
    <category: 'Web-Framework'>
    <comment: 'WebRequests know how to parse HTTP requests, organizing the data
according to the requested header fields and to the form keys
(encoded in the URL for GET requests and in the request for POST
requests).'>

    EndOfLine := nil.
    EndOfRequest := nil.

    WebRequest class >> initialize [
	<category: 'initialization'>
	EndOfLine := String with: Character cr with: Character nl.
	EndOfRequest := EndOfLine , EndOfLine
    ]

    WebRequest class >> for: aClientConnection [
	<category: 'instance creation'>
	^self new initConnection: aClientConnection
    ]

    WebRequest class >> new [
	<category: 'instance creation'>
	^super new initialize
    ]

    action [
	<category: 'accessing'>
	^action
    ]

    action: aString [
	<category: 'accessing'>
	action := aString
    ]

    at: aSymbol [
	<category: 'accessing'>
	^clientData at: aSymbol
    ]

    at: aSymbol ifAbsent: aBlock [
	<category: 'accessing'>
	^clientData at: aSymbol ifAbsent: aBlock
    ]

    at: aSymbol ifPresent: aBlock [
	<category: 'accessing'>
	^clientData at: aSymbol ifPresent: aBlock
    ]

    dateTimeAt: aSymbol [
	<category: 'accessing'>
	^self parseTimestamp: (clientData at: aSymbol)
    ]

    dateTimeAt: aSymbol ifAbsent: aBlock [
	<category: 'accessing'>
	^self parseTimestamp: (clientData at: aSymbol ifAbsent: [^aBlock value])
    ]

    dateTimeAt: aSymbol ifPresent: aBlock [
	<category: 'accessing'>
	^clientData at: aSymbol
	    ifPresent: [:value | aBlock value: (self parseTimestamp: value)]
    ]

    enumeratePostData: aBlock [
	<category: 'accessing'>
	postData keysAndValuesDo: aBlock
    ]

    getRequest [
	<category: 'accessing'>
	| saveStream version |
	saveStream := stream.
	stream := CrLfStream on: saveStream.
	self extractAction.
	self extractLocation.
	version := stream upTo: Character cr.
	stream next.	"Get nl"
	self extractClientData: version.
	(action sameAs: 'POST') 
	    ifTrue: 
		[self extractPostData: version
		    contentLength: (clientData at: #'CONTENT-LENGTH' ifAbsent: [nil])].

	"Get back to binary mode"
	stream := saveStream
    ]

    hasPostData [
	<category: 'accessing'>
	^postData notEmpty
    ]

    postDataAt: aSymbol ifPresent: aBlock [
	<category: 'accessing'>
	^postData at: aSymbol ifPresent: aBlock
    ]

    location [
	<category: 'accessing'>
	^location
    ]

    isHead [
	<category: 'accessing'>
	^action sameAs: 'HEAD'
    ]

    originator [
	<category: 'accessing'>
	^originator
    ]

    pageFollows [
	<category: 'accessing'>
	WebResponse new respondTo: self
    ]

    moreRequests [
	<category: 'accessing'>
	^(self at: #Connection) sameAs: 'keep-alive'
    ]

    moreRequests: aBoolean [
	<category: 'accessing'>
	self at: #Connection
	    put: (aBoolean ifTrue: ['Keep-Alive'] ifFalse: ['close'])
    ]

    postDataAt: aSymbol [
	<category: 'accessing'>
	^postData at: aSymbol
    ]

    postDataAt: aSymbol ifAbsent: aBlock [
	<category: 'accessing'>
	^postData at: aSymbol ifAbsent: aBlock
    ]

    stream [
	<category: 'accessing'>
	^stream
    ]

    stream: aStream [
	<category: 'accessing'>
	stream := aStream.
	originator := stream remoteAddress name
    ]

    uri [
	<category: 'accessing'>
	^uri
    ]

    initConnection: aClientConnection [
	<category: 'initialize-release'>
	| ec |
	self
	    stream: aClientConnection;
	    getRequest
    ]

    initialize [
	<category: 'initialize-release'>
	postData := IdentityDictionary new.
	clientData := IdentityDictionary new.
	location := OrderedCollection new
    ]

    release [
	<category: 'initialize-release'>
	stream flush.
	self moreRequests ifFalse: [stream close].
	^super release
    ]

    parseTimestamp: ts [
	<category: 'private'>
	| tok d m y time |
	tok := ts subStrings.
	(tok at: 1) last = $, 
	    ifFalse: 
		["asctime:  Sun Nov  6 08:49:37 1994"

		ts size = 5 ifFalse: [^nil].
		m := (ts at: 2) asSymbol.
		d := (ts at: 3) asInteger.
		y := (ts at: 5) asInteger.
		time := ts at: 4.
		^self 
		    makeTimestamp: d
		    month: m
		    year: y
		    time: time].
	(tok at: 1) size = 4 
	    ifTrue: 
		["RFC 822:  Sun, 06 Nov 1994 08:49:37 GMT"

		ts size = 6 ifFalse: [^nil].
		d := (ts at: 2) asInteger.
		m := (ts at: 3) asSymbol.
		y := (ts at: 4) asInteger.
		time := ts at: 5.
		^self 
		    makeTimestamp: d
		    month: m
		    year: y
		    time: time].
	"RFC 850 (obsolete):  Sunday, 06-Nov-94 08:49:37 GMT"
	ts size = 4 ifFalse: [^nil].
	d := ts at: 2.
	time := ts at: 3.
	d size = 9 ifFalse: [^nil].
	y := (d at: 8) base10DigitValue * 10 + (d at: 9) base10DigitValue + 1900.
	m := (d copyFrom: 4 to: 6) asSymbol.
	d := (d at: 1) base10DigitValue * 10 + (d at: 2) base10DigitValue.
	^self 
	    makeTimestamp: d
	    month: m
	    year: y
	    time: time
    ]

    makeTimestamp: d month: m year: y time: t [
	<category: 'private'>
	| month sec |
	t size = 8 ifFalse: [^nil].
	month := #(#Jan #Feb #Mar #Apr #May #Jun #Jul #Aug #Sep #Oct #Nov #Dec) 
		    indexOf: m
		    ifAbsent: [^nil].
	sec := ((t at: 1) base10DigitValue * 10 + (t at: 2) base10DigitValue) 
		    * 3600 
			+ (((t at: 4) base10DigitValue * 10 + (t at: 5) base10DigitValue) * 60) 
			+ ((t at: 7) base10DigitValue * 10 + (t at: 8) base10DigitValue).
	^(DateTime 
	    newDay: d
	    monthIndex: month
	    year: y) addSeconds: sec
    ]

    at: aSymbol put: aValue [
	<category: 'private'>
	^clientData at: aSymbol put: aValue
    ]

    endOfLine [
	<category: 'private'>
	^EndOfLine
    ]

    endOfRequest [
	<category: 'private'>
	^EndOfRequest
    ]

    extractAction [
	<category: 'private'>
	action := stream upTo: Character space
    ]

    extractClientData: clientVersion [
	<category: 'private'>
	"Default depends on version"

	| rs |
	self at: #Connection
	    put: (clientVersion = '1.0' ifTrue: ['close'] ifFalse: ['keep-alive']).
	rs := (stream upToAll: self endOfRequest) readStream.
	[rs atEnd] whileFalse: 
		[self at: (rs upTo: $:) trimSeparators asUppercase asSymbol
		    put: (rs upTo: Character cr) trimSeparators]
    ]

    extractLocation [
	<category: 'private'>
	uri := (stream upToAll: 'HTTP/') trimSeparators.
	location := uri subStrings: $?.
	location isEmpty ifTrue: [self error: 'Empty uri: ' , uri , '.'].
	location size = 2 ifTrue: [self extractQueryData: (location at: 2)].
	location := (location at: 1) subStrings: $/.
	location := location collect: [:each | URL decode: each].
	location := location reject: [:each | each isEmpty]
    ]

    extractPostData: clientVersion contentLength: contentLength [
	<category: 'private'>
	| s |
	clientVersion ~= '1.0' 
	    ifTrue: 
		[stream
		    nextPutAll: 'HTTP/1.1 100 Continue';
		    nl;
		    nl].
	(self at: #'CONTENT-TYPE' ifAbsent: [nil]) 
	    ~= 'application/x-www-form-urlencoded' ifTrue: [^self].

	"TODO: Parse the stream directly, rather than loading it all into
	 memory, because it could be large."
	s := contentLength notNil 
		    ifTrue: [stream next: contentLength asInteger]
		    ifFalse: [stream upTo: Character cr].
	^self extractQueryData: s
    ]

    extractQueryData: query [
	<category: 'private'>
	(query subStrings: $&) do: 
		[:each | 
		| pair |
		pair := each subStrings: $=.
		self postDataAt: (URL decode: pair first) asSymbol
		    put: (URL decode: (pair at: 2 ifAbsent: ['']))]
    ]

    postDataAt: aSymbol put: aValue [
	<category: 'private'>
	^postData at: aSymbol put: aValue
    ]
]



WebResponse subclass: ErrorResponse [
    | errorCode additionalHeaders |
    
    <category: 'Web-Framework'>
    <comment: 'An ErrorResponse generates responses with 3xx, 4xx or 5xx status codes,
together with their explaining HTML entities.'>

    ErrorNames := nil.
    ErrorDescriptions := nil.

    ErrorResponse class >> three [
	<category: 'initialize'>
	^#(#(300 'Multiple Choices' '<P>The requested resource corresponds to any one of a set of
representations. You can select a preferred representation.</P>') #(301 'Moved Permanently' '<P>The requested resource has been assigned a new permanent URL
and any future references to this resource should be done using
one of the returned URLs.</P>') #(302 'Moved Temporarily' '<P>The requested resource resides temporarily under a different
URI.  This is likely to be a response to a POST request which
has to retrieve a fixed entity, since many clients do not interpret
303 responses (See Other) correctly.</P>') #(303 'See Other' '<P>The response to the request can be found under a different
URL and should be retrieved using the supplied Location.</P>') #(304 'Not Modified' '') #(305 'Use Proxy' '<P>The requested resource must be accessed through the proxy given by
the Location field. </P>'))
    ]

    ErrorResponse class >> four [
	<category: 'initialize'>
	^#(#(400 'Bad Request' '<P>The request could not be understood by the server due to malformed
syntax.</P>') #(401 'Unauthorized' '<P>The request requires user authentication.</P>') #(402 'Payment Required' '<P>This code is reserved for future use.</P>') #(403 'Forbidden' '<P>The server understood the request, but is refusing to fulfill it.</P>') #(404 'Not Found' '<P>The requested URL was not found on this server.</P>') #(405 'Method Not Allowed' '<P>The specified method is not allowed for the resource identified by
the specified URL.</P>') #(406 'Not Acceptable' '<P>The resource identified by the request is only capable of generating
response entities which have content characteristics not acceptable
according to the accept headers sent in the request.</P>') #(407 'Proxy Authentication Required' '<P>To proceed, the client must first authenticate itself with the proxy.</P>') #(408 'Request Timeout' '<P>The client did not produce a request within the time that the server
was prepared to wait.</P>') #(409 'Conflict' '<P>The request could not be completed due to a conflict with the current
state of the resource. </P>') #(410 'Gone' '<P>The requested resource is no longer available at the server and no
forwarding address is known. This condition should be considered
permanent.</P>') #(411 'Length Required' '<P>The server refuses to accept the request without a defined
Content-Length header field.</P>') #(412 'Precondition Failed' '<P>The precondition given in one or more of the request-header fields
evaluated to false when it was tested on the server.</P>') #(413 'Request Entity Too Large' '<P>The server is refusing to process a request because the request
entity is larger than the server is willing or able to process.</P>') #(414 'Request-URI Too Long' '<P>The server is refusing to service the request because the requested
URL is longer than the server is willing to interpret. This condition
is most likely due to a client''s improper conversion of a POST request
with long query information to a GET request.</P>') #(415 'Unsupported Media Type' '<P>The server is refusing to service the request because the entity of
the request is in a format not supported by the requested resource
for the requested method.</P>'))
    ]

    ErrorResponse class >> five [
	<category: 'initialize'>
	^#(#(500 'Internal Server Error' '<P>The server encountered an unexpected condition which prevented it
from fulfilling the request.</P>') #(501 'Not Implemented' '<P>The server does not support the functionality required to fulfill the
request. The server does not recognize the request method and is not
capable of supporting it for any resource.</P>') #(502 'Bad Gateway' '<P>The server, while acting as a gateway or proxy, received an invalid
response from the upstream server it accessed in attempting to
fulfill the request.</P>') #(503 'Service Unavailable' '<P>The server is currently unable to handle the request due to a
temporary overloading or maintenance of the server. This is a temporary
condition.</P>') #(504 'Gateway Timeout' '<P>The server, while acting as a gateway or proxy, did not receive a
timely response from the upstream server it accessed in attempting to
complete the request.</P>') #(505 'HTTP Version Not Supported' '<P>The server does not support, or refuses to support, the HTTP protocol
version that was used in the request message.</P>'))
    ]

    ErrorResponse class >> initialize [
	<category: 'initialize'>
	ErrorNames := IdentityDictionary new.
	ErrorDescriptions := IdentityDictionary new.
	self initialize: self three.
	self initialize: self four.
	self initialize: self five
    ]

    ErrorResponse class >> initialize: arrayOfArrays [
	<category: 'initialize'>
	arrayOfArrays do: 
		[:array | 
		ErrorNames at: (array at: 1) put: (array at: 2).
		ErrorDescriptions at: (array at: 1) put: (array at: 3)]
    ]

    ErrorResponse class >> nameAt: error [
	<category: 'accessing'>
	^ErrorNames at: error
	    ifAbsent: 
		[(error < 300 or: [error > 599]) 
		    ifTrue: [self nameAt: 500]
		    ifFalse: [self nameAt: error // 100 * 100]]
    ]

    ErrorResponse class >> descriptionAt: error [
	<category: 'accessing'>
	^ErrorDescriptions at: error
	    ifAbsent: 
		[(error < 300 or: [error > 599]) 
		    ifTrue: [self descriptionAt: 500]
		    ifFalse: [self descriptionAt: error // 100 * 100]]
    ]

    ErrorResponse class >> errorCode: code [
	<category: 'instance creation'>
	^self new errorCode: code
    ]

    ErrorResponse class >> notModified [
	<category: 'instance creation'>
	^self errorCode: 304
    ]

    ErrorResponse class >> noContent [
	<category: 'instance creation'>
	^self errorCode: 204
    ]

    ErrorResponse class >> resetContent [
	<category: 'instance creation'>
	^self errorCode: 205
    ]

    ErrorResponse class >> unavailable [
	<category: 'instance creation'>
	^self errorCode: 503
    ]

    ErrorResponse class >> forbidden [
	<category: 'instance creation'>
	^self errorCode: 403
    ]

    ErrorResponse class >> notFound [
	<category: 'instance creation'>
	^self errorCode: 404
    ]

    ErrorResponse class >> gone [
	<category: 'instance creation'>
	^self errorCode: 410
    ]

    ErrorResponse class >> seeOtherURI: anotherURI [
	<category: 'instance creation'>
	^(self errorCode: 303)
	    addHeader: 'Location: ' , anotherURI;
	    yourself
    ]

    ErrorResponse class >> movedTemporarilyTo: anotherURI [
	<category: 'instance creation'>
	^(self errorCode: 302)
	    addHeader: 'Location: ' , anotherURI;
	    yourself
    ]

    ErrorResponse class >> movedPermanentlyTo: anotherURI [
	<category: 'instance creation'>
	^(self errorCode: 301)
	    addHeader: 'Location: ' , anotherURI;
	    yourself
    ]

    ErrorResponse class >> unauthorized: aString [
	<category: 'instance creation'>
	^(self errorCode: 401)
	    addHeader: 'WWW-Authenticate: ' , aString;
	    yourself
    ]

    ErrorResponse class >> acceptableMethods: anArray [
	<category: 'instance creation'>
	| header |
	header := String streamContents: 
			[:s | 
			s nextPutAll: 'Allow: '.
			anArray do: [:each | s nextPutAll: each] separatedBy: [s nextPutAll: ', ']].
	^(self errorCode: 405)
	    addHeader: header;
	    yourself
    ]

    isErrorResponse [
	<category: 'testing'>
	^true
    ]

    errorCode: code [
	<category: 'initialize'>
	errorCode := code.
	^self
    ]

    addHeader: aString [
	<category: 'initialize'>
	additionalHeaders isNil 
	    ifTrue: [additionalHeaders := OrderedCollection new].
	^additionalHeaders add: aString
    ]

    sendResponseType [
	<category: 'emit'>
	self
	    << 'HTTP/1.1 ';
	    << errorCode;
	    space;
	    << (self class nameAt: errorCode);
	    nl
    ]

    sendStandardHeaders [
	<category: 'emit'>
	super sendStandardHeaders.
	additionalHeaders isNil ifTrue: [^self].
	additionalHeaders do: 
		[:each | 
		self
		    << each;
		    nl]
    ]

    noMessageBody [
	<category: 'emit'>
	^#(204 205 304) includes: errorCode
    ]

    sendBody [
	<category: 'emit'>
	| description |
	self noMessageBody ifTrue: [^self].
	description := self class descriptionAt: errorCode.
	description isEmpty ifTrue: [^self].
	self
	    << '<HTML>';
	    nl;
	    << '<HEAD><TITLE>';
	    << errorCode;
	    space;
	    << (self class nameAt: errorCode);
	    << '</TITLE></HEAD>';
	    nl;
	    << '<BODY>';
	    nl;
	    heading: 
		    [self
			<< errorCode;
			space;
			<< (self class nameAt: errorCode)];
	    << description;
	    << 'originator: ';
	    << request originator displayString;
	    lineBreak;
	    << 'action: ';
	    << request action displayString;
	    lineBreak;
	    << 'location: '.
	request location do: [:each | self << $/ << each].
	request enumeratePostData: 
		[:key :val | 
		self
		    lineBreak;
		    << key;
		    << ' = ';
		    nl;
		    << val;
		    nl].
	self
	    lineBreak;
	    horizontalLine;
	    italic: [self << WebServer version];
	    << '</BODY></HTML>'
    ]
]



Object subclass: WebAuthorizer [
    | authorizer |
    
    <category: 'Web-Framework'>
    <comment: 'A WebAuthorizer checks for the correctness login/password couplets in an
HTTP request using the Basic authentication scheme.'>

    WebAuthorizer class >> fromString: aString [
	<category: 'private'>
	^self new authorizer: aString
    ]

    WebAuthorizer class >> loginID: aLoginID password: aPassword [
	<category: 'private'>
	^(self new)
	    loginID: aLoginID password: aPassword;
	    yourself
    ]

    authorize: aRequest [
	<category: 'accessing'>
	| trial |
	trial := aRequest at: #AUTHORIZATION ifAbsent: [nil].
	^trial = self authorizer
    ]

    authorizer [
	<category: 'accessing'>
	^authorizer
    ]

    authorizer: aString [
	<category: 'accessing'>
	authorizer := aString
    ]

    challengeFor: aServlet [
	<category: 'accessing'>
	^'Basic realm="%1"' % {aServlet name}
    ]

    authorize: aRequest in: aServlet ifAuthorized: aBlock [
	<category: 'accessing'>
	^(self authorize: aRequest) 
	    ifTrue: [aBlock value]
	    ifFalse: 
		[(ErrorResponse unauthorized: (self challengeFor: aServlet)) 
		    respondTo: aRequest.
		^nil]
    ]

    loginID: aName password: aPassword [
	"(self loginID: 'aName' password: 'aPassword') authorizer =
	 'Basic YU5hbWU6YVBhc3N3b3Jk'"

	<category: 'private'>
	| plain plainSize i chars stream |
	aName isNil | aPassword isNil ifTrue: [^nil].
	chars := 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'.
	plain := (aName , ':' , aPassword) asByteArray.
	plainSize := plain size.
	plain size \\ 3 = 0 
	    ifFalse: [plain := plain , (ByteArray new: 3 - (plain size \\ 3))].
	i := 1.
	stream := WriteStream on: String new.
	stream nextPutAll: 'Basic '.
	[i < plain size] whileTrue: 
		[stream
		    nextPut: (chars at: (plain at: i) // 4 + 1);
		    nextPut: (chars at: (plain at: i) \\ 4 * 16 + ((plain at: i + 1) // 16) + 1);
		    nextPut: (chars 
				at: (plain at: i + 1) \\ 16 * 4 + ((plain at: i + 2) // 64) + 1);
		    nextPut: (chars at: (plain at: i + 2) \\ 64 + 1).
		i := i + 3].
	authorizer := stream contents.
	i := authorizer size.
	plain size - plainSize timesRepeat: 
		[authorizer at: i put: $=.
		i := i - 1]
    ]
]



Character extend [

    base10DigitValue [
	<category: 'converting'>
	^self isDigit ifTrue: [self asciiValue - 48] ifFalse: [0]
    ]

]



Eval [
    ErrorResponse initialize.
    WebRequest initialize
]

PK
     
wBjk|^  |^  
  FileServer.stUT	 NQhux r      "======================================================================
|
|   File server plug-in
|
|
 ======================================================================"

"======================================================================
|
| Copyright 2000, 2001, 2008 Travis Griggs and Ken Treis
| Written by Travis Griggs, Ken Treis and others.
| Port to GNU Smalltalk, enhancements and refactory by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
|
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
|
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.	If not, write to the Free Software
| Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
|
 ======================================================================"



WebResponse subclass: FileSystemResponse [
    | file |
    
    <comment: '
A FileSystemResponse, being tied to a File object, knows about its
last modification time.'>
    <category: 'Web-File Server'>

    FileSystemResponse class >> file: aFile [
	<category: 'instance creation'>
	^self new file: aFile
    ]

    file: aFile [
	<category: 'initialize-release'>
	file := aFile
    ]

    modifiedTime [
	<category: 'response'>
	^file lastModifyTime
    ]
]



FileSystemResponse subclass: DirectoryResponse [
    
    <comment: '
A DirectoryResponse formats output of the contents of a Directory object.'>
    <category: 'Web-File Server'>

    chopName: aString [
	<category: 'response'>
	^aString size > self maxNameLength 
	    ifTrue: [(aString copyFrom: 1 to: self maxNameLength - 3) , '...']
	    ifFalse: [aString]
    ]

    maxNameLength [
	<category: 'response'>
	^30
    ]

    maxSizeLength [
	<category: 'response'>
	^6
    ]

    sendMetaHeaders [
	"While caching of file responses is generally desirable (even though
	 it can be incorrect if somebody does some uploading), caching
	 directory responses can be extremely confusing and could yield
	 incorrect uploads (where someone thinks he uploaded something and
	 actually didn't)"

	<category: 'response'>
	self
	    << '<meta http-equiv="Pragma" content="no-cache">';
	    nl.
	self
	    << '<meta http-equiv="Cache-control" content="no-cache">';
	    nl
    ]

    sendBody [
	<category: 'response'>
	self
	    << '<html><head><title>Directory Listing for ';
	    << request uri;
	    << '</title>';
	    nl;
	    sendMetaHeaders;
	    << '</head><body><h1>Directory Contents:</h1><pre>';
	    nl;
	    << 'Name';
	    next: self maxNameLength - 1 put: $ ;
	    << 'Modified on	       Size';
	    nl;
	    << '<hr>';
	    nl.
	(File name: file name) entryNames asSortedCollection 
	    do: [:each | self sendFileProperties: each].
	self << '</pre><hr><FORM ACTION="' << request uri.
	self
	    << '" METHOD="post" ENCTYPE="multipart/form-data">';
	    nl.
	self
	    << '<INPUT TYPE="file" NAME="contents">';
	    nl.
	self
	    << '<INPUT TYPE="submit" VALUE="Upload"></FORM>';
	    nl.
	self << '</body></html>'
    ]

    sendFileProperties: each [
	<category: 'response'>
	| isDirectory choppedName name subDirFile parent slash |
	each = '.' ifTrue: [^self].
	subDirFile := file / each.
	subDirFile isReadable ifFalse: [^self].
	isDirectory := subDirFile isDirectory.
	choppedName := isDirectory 
		    ifTrue: [self chopName: (each copyWith: $/)]
		    ifFalse: [self chopName: each].
	each = '..' 
	    ifTrue: 
		[slash := request uri findLast: [:each | each == $/].
		slash = 1 ifTrue: [^self].
		self << '<a href="' << (request uri copyFrom: 1 to: slash)]
	    ifFalse: [self << '<a href="' << request uri << $/ << each].
	self << '">' << choppedName << '</a>'.
	self next: self maxNameLength - choppedName size + 3 put: $ .
	self sendModifyTimeFor: subDirFile.
	isDirectory ifFalse: [self sendFileSizeFor: subDirFile].
	self nl
    ]

    sendModifyTimeFor: aFile [
	<category: 'response'>
	| date |
	date := aFile lastModifyTime at: 1.
	date day < 10 ifTrue: [self nextPut: $0].
	self << date << '	  '
    ]

    sendFileSizeFor: aFile [
	<category: 'response'>
	| size type printString |
	size := [aFile size] on: Error do: [:ex | ex return: nil].
	size isNil ifTrue: [^self].
	printString := String new: self maxSizeLength withAll: $ .
	type := #('Bytes' 'KB' 'MB' 'GB' 'TB') detect: 
			[:each | 
			| found |
			found := size < 10000.
			found ifFalse: [size := (size + 512) // 1024].
			found]
		    ifNone: 
			[^self
			    next: self maxSizeLength put: $*;
			    << ' huge!'].
	printString := printString , size rounded printString.
	printString := printString 
		    copyFrom: printString size + 1 - self maxSizeLength.
	self
	    << printString;
	    space;
	    << type
    ]
]



DirectoryResponse subclass: UploadResponse [
    
    <comment: '
An UploadResponse formats output of the contents of a Directory object,
and interprets multipart/form-data contents sent by a client that wants
to upload a file.'>
    <category: 'Web-File Server'>

    respondTo: aRequest [
	<category: 'response'>
	self doUpload: aRequest.
	super respondTo: aRequest
    ]

    doUpload: aRequest [
	"This is not a general multipart/form-data parser. The only things
	 it lacks is the ability to parse more than one field (with the
	 last boundary identified by two trailing dashes) and to build a
	 dictionary with the contents of each form field."

	<category: 'multipart'>
	| boundary str i remoteName uploadStream subHeaders |
	request := aRequest.
	boundary := self boundaryString.
	boundary isNil ifTrue: [^self].
	(request stream)
	    skipToAll: boundary;
	    nextLine.
	subHeaders := self getSubHeaders.
	subHeaders isEmpty ifTrue: [^self].
	str := subHeaders at: #'CONTENT-DISPOSITION' ifAbsent: [''].
	i := str indexOfSubCollection: 'filename="' ifAbsent: [0].
	i = 0 ifTrue: [^self].
	i := i + 10.
	(str at: i) == $" ifTrue: [^self].
	remoteName := str copyFrom: i to: (str indexOf: $" startingAt: i) - 1.
	remoteName := URL decode: remoteName.	"### not sure about this..."
	uploadStream := (self localFileFor: remoteName) writeStream.

	"Collect at least 128 bytes of content (of course, stop if we see a
	 boundary).	 We need this quantity because M$ Internet Explorer 4.0
	 for Mac appends 128 bytes of Mac file system info which we must
	 remove."
	boundary := boundary precompileSearch.
	str := self nextChunk.
	
	[i := boundary searchIn: str startingAt: 1.
	i notNil and: [str size < 128]] 
		whileTrue: [str := str , self nextChunk].
	((str at: 1) asciiValue = 0 and: 
		[(str at: 2) asciiValue = remoteName size 
		    and: [(str copyFrom: 3 to: remoteName size + 2) = remoteName]]) 
	    ifTrue: 
		[str := str copyFrom: 129 to: str size.
		i := i - 128].

	"Now do the real work"
	[i > 0] whileFalse: 
		[request stream isPeerAlive 
		    ifFalse: 
			[uploadStream close.
			(self localFileFor: remoteName) remove.
			^self].

		"While we don't encounter a chunk which could contain the
		 boundary, copy at maximum speed."
		
		[i := boundary possibleMatchSearchIn: str startingAt: 5.
		i > 0] 
			whileFalse: 
			    [uploadStream nextPutAll: str.
			    str := self nextChunk].

		"The boundary could be here. We have to look more carefully."
		i := boundary searchIn: str startingAt: i - 4.
		i > 0 
		    ifFalse: 
			["Not found, but it might finish in the next chunk..."

			uploadStream nextPutAll: (str copyFrom: 1 to: i - 5).
			str := (str copyFrom: i - 4 to: str size) , self nextChunk.
			i := boundary searchIn: str startingAt: 1]].

	"Save the last chunk in the file (the first if we didn't go through
	 the while loop."
	i > 5 ifTrue: [uploadStream nextPutAll: (str copyFrom: 1 to: i - 5)].

	"Clean things up..."
	uploadStream close
    ]

    nextChunk [
	<category: 'multipart'>
	
	request stream isPeerAlive ifFalse: [^''].
	^request stream nextAvailable: 1024
    ]

    localFileFor: remoteName [
	<category: 'multipart'>
	| idx fileName |
	idx := remoteName findLast: [:each | ':/\' includes: each].
	fileName := remoteName copyFrom: idx + 1.
	^file at: fileName
    ]

    getSubHeaders [
	<category: 'multipart'>
	| hdr subHeaders line colon |
	subHeaders := LookupTable new.
	
	[line := request stream nextLine.
	colon := line indexOf: $:.
	colon = 0] 
		whileFalse: 
		    [subHeaders at: (line copyFrom: 1 to: colon - 1) asUppercase asSymbol
			put: (line copyFrom: colon + 1) trimSeparators].
	^subHeaders
    ]

    boundaryString [
	"Decode multipart form data boundary information from a
	 header line that looks like the following line:
	 Content-Type: multipart/form-data; boundary=-----"

	<category: 'multipart'>
	| str |
	str := (request at: #'CONTENT-TYPE' ifAbsent: ['']) readStream.
	(str upTo: $;) = 'multipart/form-data' ifFalse: [^nil].
	str skipTo: $=.

	"Boundary lines *always* start with two dashes"
	^'--' , str upToEnd
    ]
]



FileSystemResponse subclass: FileResponse [
    | fileStream |
    
    <comment: '
A FileResponse outputs the contents of a whole file onto an HTTP
data stream.'>
    <category: 'Web-File Server'>

    FileResponse class >> file: aFile [
	<category: 'instance creation'>
	^
	[| fileStream |
	fileStream := aFile readStream.
	(super file: aFile)
	    fileStream: fileStream;
	    yourself] 
		on: Error
		do: [:ex | ex return: ErrorResponse forbidden]
    ]

    mimeType [
	<category: 'accessing'>
	^ContentHandler contentTypeFor: file name
    ]

    respondTo: aRequest [
	<category: 'response'>
	[super respondTo: aRequest] ensure: [fileStream close]
    ]

    sendBody [
	<category: 'response'>
	| size data read |
	size := fileStream size.
	[size > 0] whileTrue: 
		[data := fileStream next: (read := size min: 2000).
		size := size - read.
		self nextPutAll: data]
    ]

    contentLength [
	<category: 'response'>
	^fileStream size
    ]

    sendMimeType [
	<category: 'response'>
	self
	    << 'Content-Type: ';
	    << self mimeType;
	    nl
    ]

    sendStandardHeaders [
	<category: 'response'>
	super sendStandardHeaders.
	self
	    << 'Accept-Ranges: bytes';
	    nl
    ]

    fileStream: aStream [
	<category: 'initialize-release'>
	fileStream := aStream
    ]
]



FileResponse subclass: RangeResponse [
    | range |
    
    <comment: '
A RangeResponse outputs the contents of a single interval of a file
onto an HTTP data stream.'>
    <category: 'Web-File Server'>

    RangeResponse class >> file: aFile range: aRangeSpecification [
	<category: 'response'>
	| response |
	response := self file: aFile.
	^response isErrorResponse 
	    ifTrue: [response]
	    ifFalse: [response range: aRangeSpecification]
    ]

    range: aRangeSpecification [
	<category: 'initialize-release'>
	range := aRangeSpecification.
	range fileSize: fileStream size
    ]

    sendBody [
	<category: 'response'>
	self sendBody: range
    ]

    sendBody: range [
	<category: 'response'>
	| size data read |
	size := range last - range first + 1.
	fileStream position: range first.
	[size > 0] whileTrue: 
		[data := fileStream next: (read := size min: 2000).
		size := size - read.
		self nextPutAll: data]
    ]

    sendStandardHeaders [
	<category: 'response'>
	super sendStandardHeaders.
	range sendStandardHeadersOn: self
    ]

    contentLength [
	<category: 'response'>
	^range last - range first + 1
    ]
]



RangeResponse subclass: MultiRangeResponse [
    | mimeType boundary |
    
    <comment: '
A MultiRangeResponse outputs the contents of more than one interval of a
file onto an HTTP data stream, in multipart/byteranges format.'>
    <category: 'Web-File Server'>

    getBoundary [
	<category: 'caching'>
	^'------%1-!-GST-!-%2' % 
		{Time secondClock.
		Time millisecondClock}
    ]

    mimeType [
	"Cache the MIME type as computed by the FileResponse implementation"

	<category: 'caching'>
	mimeType isNil ifTrue: [mimeType := super mimeType].
	^mimeType
    ]

    sendBody [
	<category: 'response'>
	range do: 
		[:each | 
		self
		    << '--';
		    << boundary;
		    nl.
		self
		    << 'Content-type: ';
		    << self mimeType;
		    nl.
		each
		    sendStandardHeadersOn: self;
		    nl.
		self sendBody: each].
	self
	    << '--';
	    << boundary;
	    << '--';
	    nl
    ]

    sendMimeType [
	<category: 'response'>
	boundary := self getBoundary.
	self
	    << 'Content-type: multipart/byteranges; boundary=';
	    << boundary;
	    nl
    ]

    contentLength [
	<category: 'response'>
	^nil
    ]
]



Object subclass: RangeSpecification [
    
    <category: 'Web-File Server'>
    <comment: '
Subclasses of RangeSpecification contain information on the data requested
in a Range HTTP request header.'>

    RangeSpecification class >> on: aString [
	"Parse the `Range' header field, answer an instance of a subclass of
	 RangeSpecification. From RFC 2068 (HTTP 1.1) -- 1# means comma-separated
	 list with at least one element:
	 byte-ranges-specifier = bytes-unit '=' byte-range-set
	 byte-range-set  = 1#( byte-range-spec | suffix-byte-range-spec )
	 byte-range-spec = first-byte-pos '-' [last-byte-pos]
	 first-byte-pos  = 1*DIGIT
	 last-byte-pos   = 1*DIGIT
	 suffix-byte-range-spec = '-' suffix-length
	 suffix-length = 1*DIGIT'
	 "

	<category: 'parsing'>
	| stream partial current n first which ch |
	stream := ReadStream on: aString.
	partial := nil.
	which := #first.

	"Read the unit"
	(stream upToAll: 'bytes=') isEmpty ifFalse: [^nil].
	stream atEnd ifTrue: [^nil].
	
	[n := nil.
	
	[ch := stream atEnd 
		    ifTrue: [$,	"Fake an empty entry at end"]
		    ifFalse: [stream next].
	ch isDigit] 
		whileTrue: 
		    [n := n isNil ifTrue: [ch digitValue] ifFalse: [n * 10 + ch digitValue]].
	ch == $- 
	    ifTrue: 
		["Check for invalid range specifications"

		which == #last ifTrue: [^nil].
		which := #last.
		first := n].
	ch == $, 
	    ifTrue: 
		["Check for invalid range specifications"

		which == #first ifTrue: [^nil].
		first > n ifTrue: [^nil].
		n = -1 & (first = -1) ifTrue: [^nil].
		which := #first.
		current := SingleRangeSpecification new.
		current
		    first: first;
		    last: n.
		partial := partial isNil ifTrue: [current] ifFalse: [partial , current].
		stream atEnd ifTrue: [^partial]]] 
		repeat
    ]

    , anotherRange [
	<category: 'overridden'>
	self subclassResponsibility
    ]

    do: aBlock [
	<category: 'overridden'>
	self subclassResponsibility
    ]

    fileSize: size [
	<category: 'overridden'>
	self subclassResponsibility
    ]

    sendStandardHeadersOn: aStream [
	<category: 'overridden'>
	
    ]

    printOn: aStream [
	<category: 'printing'>
	self do: [:each | each sendStandardHeadersOn: aStream]
    ]
]



RangeSpecification subclass: SingleRangeSpecification [
    | first last size |
    
    <category: 'Web-File Server'>
    <comment: '
A SingleRangeSpecification contains information that will result in a
Content-Range HTTP header or multipart/byteranges subheader.'>

    first [
	<category: 'accessing'>
	^first
    ]

    last [
	<category: 'accessing'>
	^last
    ]

    first: anInteger [
	<category: 'accessing'>
	first := anInteger
    ]

    last: anInteger [
	<category: 'accessing'>
	last := anInteger
    ]

    , anotherRange [
	<category: 'overridden'>
	^(MultiRangeSpecification with: self)
	    , anotherRange;
	    yourself
    ]

    do: aBlock [
	<category: 'overridden'>
	aBlock value: self
    ]

    fileSize: fSize [
	<category: 'overridden'>
	size := fSize.

	"-500: first = nil, last = 500"
	first isNil 
	    ifTrue: 
		[first := last + size - 1.
		last := size - 1].

	"9500-: first = 9500, last = nil"
	last isNil ifTrue: [last := size - 1]
    ]

    sendStandardHeadersOn: aStream [
	<category: 'overridden'>
	aStream << 'Content-range: bytes ' << first << $- << last << $/ << size.
	aStream nl
    ]

    size [
	<category: 'overridden'>
	^1
    ]
]



RangeSpecification subclass: MultiRangeSpecification [
    | subranges |
    
    <category: 'Web-File Server'>
    <comment: '
A MultiRangeSpecification contains information on a complex Range request
header, that will result in a multipart/byteranges (MultiRangeResponse)
response.'>

    MultiRangeSpecification class >> with: aRange [
	<category: 'instance creation'>
	^(self new initialize)
	    , aRange;
	    yourself
    ]

    initialize [
	<category: 'initialize-release'>
	subranges := OrderedCollection new
    ]

    , anotherRange [
	<category: 'overridden'>
	anotherRange do: [:each | subranges add: each].
	^self
    ]

    do: aBlock [
	<category: 'overridden'>
	subranges do: aBlock
    ]

    fileSize: fSize [
	<category: 'overridden'>
	self do: [:each | each fileSize: fSize]
    ]

    sendStandardHeadersOn: aStream [
	<category: 'overridden'>
	
    ]

    size [
	<category: 'overridden'>
	^subranges size
    ]
]



Servlet subclass: FileWebServer [
    | initialDirectory uploadAuthorizer |
    
    <comment: '
A FileWebServer transforms incoming requests into appropriate FileResponses
and DirectoryResponses.'>
    <category: 'Web-File Server'>

    FileWebServer class >> named: aString [
	<category: 'instance creation'>
	^self new name: aString
    ]

    FileWebServer class >> named: aString directory: dirString [
	<category: 'instance creation'>
	^(self new)
	    name: aString;
	    directory: dirString;
	    yourself
    ]

    FileWebServer class >> new [
	<category: 'instance creation'>
	^(super new)
	    initialize;
	    yourself
    ]

    fileResponse: file request: aRequest [
	<category: 'interaction'>
	| range |
	range := aRequest at: #RANGE ifAbsent: [nil].
	range isNil ifTrue: [^FileResponse file: file].
	range := RangeSpecification on: range.
	range size = 1 ifTrue: [^RangeResponse file: file range: range].
	^MultiRangeResponse file: file range: range
    ]

    directoryResponse: aDirectory request: aRequest [
	<category: 'responding'>
	| listable |
	listable := aDirectory isReadable.
	(aRequest action sameAs: 'POST') 
	    ifTrue: 
		[^listable 
		    ifTrue: [self uploadResponse: aDirectory request: aRequest]
		    ifFalse: [ErrorResponse acceptableMethods: #('HEAD' 'GET')]].
	^(self indexResponse: aDirectory request: aRequest) ifNil: 
		[listable 
		    ifTrue: [DirectoryResponse file: aDirectory]
		    ifFalse: [ErrorResponse forbidden]]
    ]

    indexResponse: aDirectory request: aRequest [
	<category: 'interaction'>
	self indexFileNames do: 
		[:each | 
		| indexFile |
		indexFile := aDirectory / each.
		indexFile isReadable 
		    ifTrue: [^self fileResponse: indexFile request: aRequest]].
	^nil
    ]

    respondTo: aRequest [
	<category: 'interaction'>
	| response |
	response := (#('HEAD' 'GET' 'POST') includes: aRequest action asUppercase) 
		    ifTrue: [self responseFor: aRequest]
		    ifFalse: [ErrorResponse acceptableMethods: #('HEAD' 'GET' 'POST')].
	response isNil ifFalse: [response respondTo: aRequest]
    ]

    responseFor: aRequest [
	<category: 'interaction'>
	| file path |
	path := aRequest location.
	file := initialDirectory.
	path 
	    from: self depth
	    to: path size
	    do: 
		[:each | 
		(self isValidName: each) ifFalse: [^ErrorResponse notFound].
		file isDirectory ifFalse: [^ErrorResponse notFound].
		file := file directoryAt: each.
		file isReadable 
		    ifFalse: 
			[^file isDirectory 
			    ifTrue: [ErrorResponse notFound]
			    ifFalse: [ErrorResponse forbidden]]].
	file isDirectory ifTrue: [^self directoryResponse: file request: aRequest].
	^self fileResponse: file request: aRequest
    ]

    directory: aDirectory [
	<category: 'accessing'>
	initialDirectory := File name: aDirectory
    ]

    indexFileNames [
	<category: 'accessing'>
	^#('index.html' 'index.htm' 'default.html' 'default.htm')
    ]

    initialize [
	<category: 'initialize-release'>
	initialDirectory := Directory working.
	uploadAuthorizer := WebAuthorizer new.
	name := 'File'
    ]

    isValidName: aString [
	"Don't allow people to put strange characters or .. in a file directory.
	 If we allowed .., then someone could grab our password file."

	<category: 'testing'>
	^(aString indexOfSubCollection: '..') = 0 and: 
		[aString 
		    conform: [:each | each asInteger >= 32 and: [each asInteger < 127]]]
    ]

    uploadAuthorizer [
	<category: 'accessing'>
	^uploadAuthorizer
    ]

    uploadAuthorizer: aWebAuthorizer [
	<category: 'accessing'>
	uploadAuthorizer := aWebAuthorizer
    ]

    uploadLoginID: aLoginID password: aPassword [
	<category: 'accessing'>
	uploadAuthorizer := WebAuthorizer loginID: aLoginID password: aPassword
    ]

    uploadResponse: aDirectory request: aRequest [
	<category: 'responding'>
	^uploadAuthorizer 
	    authorize: aRequest
	    in: self
	    ifAuthorized: [UploadResponse file: aDirectory]
    ]
]



CharacterArray extend [

    precompileSearch [
	"Compile the receiver into some object that answers
	 #searchIn:startingAt: and #possibleMatchSearchIn:startingAt:"

	<category: 'Boyer-Moore search'>
	| encoding size |
	size := self size.
	encoding := size > 254 
		    ifTrue: [Array new: 513 withAll: size]
		    ifFalse: [ByteArray new: 513 withAll: size].

	"To find the last char of self, moving forwards"
	1 to: size do: [:i | encoding at: 2 + (self valueAt: i) put: size - i].

	"To find the first char of self, moving backwards"
	size to: 1
	    by: -1
	    do: [:i | encoding at: 258 + (self valueAt: i) put: i - 1].
	^Array with: self with: encoding
    ]

    boyerMooreSearch: string encoding: encoding startingAt: minPos [
	<category: 'Boyer-Moore search'>
	| idx searchSize size ofs |
	searchSize := encoding at: 1.
	idx := minPos + searchSize - 1.
	size := self size.
	[idx < size] whileTrue: 
		[ofs := encoding at: 2 + (self valueAt: idx).
		ofs = 0 
		    ifTrue: 
			["Look behind for the full searched string"

			ofs := searchSize.
			
			[(ofs := ofs - 1) == 0 ifTrue: [^idx - searchSize + 1].
			(string at: ofs) == (self at: idx - searchSize + ofs)] 
				whileTrue.

			"Sorry not found... yet"
			ofs := 1].
		idx := idx + ofs].
	^0
    ]

    boyerMoorePossibleMatchSearch: encoding startingAt: minPos [
	<category: 'Boyer-Moore search'>
	| idx searchSize ofs result |
	searchSize := encoding at: 1.
	idx := self size.
	result := 0.
	[idx > minPos] whileTrue: 
		[ofs := encoding at: 258 + (self valueAt: idx).
		ofs = 0 
		    ifTrue: 
			[result := idx.
			ofs := 1].
		idx := idx - ofs].
	^result
    ]

]



ArrayedCollection extend [

    searchIn: aString startingAt: minPos [
	"Same as `aString indexOfSubCollection: ... ifAbsent: [ 0 ]', where
	 the searched string is the string that was precompiled in the
	 receiver.	Optimized for minPos < self size - minPos (otherwise, you're
	 likely to win if you first use #possibleMatchSearchIn:startingAt:)"

	<category: 'Boyer-Moore search'>
	^aString 
	    boyerMooreSearch: (self at: 1)
	    encoding: (self at: 2)
	    startingAt: minPos
    ]

    possibleMatchSearchIn: aString startingAt: minPos [
	"Search for the first possible match starting from the minPos-th
	 item in the string that was precompiled in the receiver.  This
	 is not necessarily the first occurrence of the first character
	 (a later occurrence, or none at all, could be returned if the
	 algorithm discovers that the first cannot be part of a match).
	 Optimized for minPos > self size - minPos (otherwise, you're
	 likely to win if you use #searchIn:startingAt: directly)"

	<category: 'Boyer-Moore search'>
	^aString boyerMoorePossibleMatchSearch: (self at: 2) startingAt: minPos
    ]

]



WebServer class extend [

    publishMyHomeDir [
	"WebServer myHomeDirWiki"

	<category: 'examples'>
	| handler name dir |
	self terminateServer: 8080.
	name := '~' , (File stripPathFrom: Directory home).
	dir := Directory home , '/pub-www'.

	"Add a file server on a particular directory."
	handler := (self initializeServer: 8080) handler.
	handler addComponent: (FileWebServer named: name directory: dir)
    ]

]

PK
     
wBSYa    
  WikiServer.stUT	 NQhux r      "======================================================================
|
|   Wiki-style web server plug-in
|
|
 ======================================================================"

"======================================================================
|
| Copyright 2000, 2001 Travis Griggs and Ken Treis
| Written by Travis Griggs, Ken Treis and others.
| Port to GNU Smalltalk, enhancements and refactoring by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
|
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
|
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.	If not, write to the Free Software
| Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
|
 ======================================================================"



Object subclass: WikiPage [
    | author timestamp |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    WikiPage class >> newVersionOf: aWikiPage by: anAuthor [
	<category: 'instance creation'>
	^(self new)
	    previousVersion: aWikiPage;
	    author: anAuthor;
	    yourself
    ]

    WikiPage class >> new [
	<category: 'instance creation'>
	^super new initialize
    ]

    allTitles [
	<category: 'accessing'>
	| oc |
	oc := OrderedCollection new.
	self allTitlesInto: oc.
	^oc
    ]

    allTitlesInto: aCollection [
	<category: 'accessing'>
	self subclassResponsibility
    ]

    author [
	<category: 'accessing'>
	^author
    ]

    contents [
	<category: 'accessing'>
	^self subclassResponsibility
    ]

    references: aString [
	<category: 'accessing'>
	^(aString match: self contents) or: [aString match: self title]
    ]

    operationSynopsis [
	<category: 'accessing'>
	^self subclassResponsibility
    ]

    timestamp [
	<category: 'accessing'>
	^timestamp
    ]

    title [
	<category: 'accessing'>
	^self subclassResponsibility
    ]

    versionAt: aNumber [
	<category: 'accessing'>
	self versionsDo: [:each | each versionNumber = aNumber ifTrue: [^each]].
	^self subscriptBoundsError: aNumber
    ]

    versionNumber [
	<category: 'accessing'>
	self subclassResponsibility
    ]

    versionsDo: aBlock [
	<category: 'accessing'>
	self subclassResponsibility
    ]

    versionsReverseDo: aBlock [
	<category: 'accessing'>
	self subclassResponsibility
    ]

    printOn: aStream [
	<category: 'displaying'>
	aStream
	    nextPut: $[;
	    nextPutAll: self title;
	    nextPut: $(;
	    print: self versionNumber;
	    nextPut: $);
	    nextPut: $];
	    nl;
	    nextPutAll: self contents;
	    nl.
	aStream
	    nextPut: ${;
	    nextPutAll: author;
	    space;
	    print: timestamp;
	    nextPut: $}
    ]

    changeTitle: aTitle by: anAuthor [
	<category: 'editing'>
	| newGuy |
	aTitle = self title ifTrue: [^self].
	newGuy := RenamedWikiPage newVersionOf: self by: anAuthor.
	newGuy title: aTitle.
	^newGuy
    ]

    newContents: aContents by: anAuthor [
	<category: 'editing'>
	| newGuy |
	aContents = self contents ifTrue: [^self].
	newGuy := EditedWikiPage newVersionOf: self by: anAuthor.
	newGuy contents: aContents.
	^newGuy
    ]

    author: anObject [
	<category: 'initialize'>
	author := anObject
    ]

    initialize [
	<category: 'initialize'>
	timestamp := DateTime now.
	author := ''
    ]

    saveToFile: aFileStream under: aWikiPM [
	<category: 'flat file'>
	aFileStream
	    nextPutAll: author;
	    nl.
	aFileStream
	    print: timestamp asSeconds;
	    nl.
	^self
    ]

    loadFromFile: rs under: aWikiPM [
	<category: 'flat file'>
	| timestamp author seconds |
	author := rs nextLine.
	seconds := rs nextLine asNumber.
	timestamp := (Date 
		    year: 1901
		    day: 1
		    hour: 0
		    minute: 0
		    second: 0) + (Duration seconds: seconds).
	self
	    author: author;
	    timestamp: timestamp
    ]

    timestamp: value [
	<category: 'flat file'>
	timestamp := value
    ]
]



WikiPage subclass: OriginalWikiPage [
    | title |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    allTitlesInto: aCollection [
	<category: 'accessing'>
	aCollection add: title
    ]

    contents [
	<category: 'accessing'>
	^'Describe ' , title , ' here...'
    ]

    operationSynopsis [
	<category: 'accessing'>
	^'Created'
    ]

    title [
	<category: 'accessing'>
	^title
    ]

    title: aString [
	<category: 'accessing'>
	title := aString
    ]

    versionNumber [
	<category: 'accessing'>
	^0
    ]

    versionsDo: aBlock [
	<category: 'accessing'>
	aBlock value: self
    ]

    versionsReverseDo: aBlock [
	<category: 'accessing'>
	aBlock value: self
    ]

    saveToFile: aFileStream under: aWikiPM [
	<category: 'flat file'>
	super saveToFile: aFileStream under: aWikiPM.
	aFileStream nextPutAll: title.
	^self
    ]

    loadFromFile: rs under: aWikiPM [
	<category: 'flat file'>
	super loadFromFile: rs under: aWikiPM.
	self title: rs upToEnd
    ]
]



WikiPage subclass: ChangedWikiPage [
    | previousVersion |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    allTitlesInto: aCollection [
	<category: 'accessing'>
	previousVersion allTitlesInto: aCollection
    ]

    contents [
	<category: 'accessing'>
	^previousVersion contents
    ]

    previousVersion [
	<category: 'accessing'>
	^previousVersion
    ]

    previousVersion: anObject [
	<category: 'accessing'>
	previousVersion := anObject
    ]

    title [
	<category: 'accessing'>
	^previousVersion title
    ]

    versionNumber [
	<category: 'accessing'>
	^previousVersion versionNumber + 1
    ]

    versionsDo: aBlock [
	<category: 'accessing'>
	aBlock value: self.
	previousVersion versionsDo: aBlock
    ]

    versionsReverseDo: aBlock [
	<category: 'accessing'>
	previousVersion versionsReverseDo: aBlock.
	aBlock value: self
    ]

    saveToFile: aFileStream under: aWikiPM [
	<category: 'flat file'>
	super saveToFile: aFileStream under: aWikiPM.
	aFileStream
	    print: (aWikiPM idForPage: self previousVersion);
	    nl.
	^self
    ]

    loadFromFile: rs under: aWikiPM [
	<category: 'flat file'>
	| id |
	super loadFromFile: rs under: aWikiPM.
	id := rs nextLine.
	self previousVersion: (aWikiPM loadPage: id)
    ]
]



ChangedWikiPage subclass: EditedWikiPage [
    | contents |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    contents [
	<category: 'accessing'>
	^contents
    ]

    contents: aString [
	"trim off trailing CRs"

	<category: 'accessing'>
	| index |
	index := aString size.
	[index > 1 and: [(aString at: index) = Character nl]] 
	    whileTrue: [index := index - 1].
	contents := aString copyFrom: 1 to: index
    ]

    operationSynopsis [
	<category: 'accessing'>
	^'Edited'
    ]

    saveToFile: aFileStream under: aWikiPM [
	<category: 'flat file'>
	super saveToFile: aFileStream under: aWikiPM.
	aFileStream nextPutAll: contents.
	^self
    ]

    loadFromFile: rs under: aWikiPM [
	<category: 'flat file'>
	super loadFromFile: rs under: aWikiPM.
	self contents: rs upToEnd
    ]
]



ChangedWikiPage subclass: RenamedWikiPage [
    | title |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    allTitlesInto: aCollection [
	<category: 'accessing'>
	aCollection add: title.
	^super allTitlesInto: aCollection
    ]

    operationSynopsis [
	<category: 'accessing'>
	^'Renamed'
    ]

    title [
	<category: 'accessing'>
	^title
    ]

    title: aString [
	<category: 'accessing'>
	title := aString
    ]

    saveToFile: aFileStream under: aWikiPM [
	<category: 'flat file'>
	super saveToFile: aFileStream under: aWikiPM.
	aFileStream nextPutAll: title.
	^self
    ]

    loadFromFile: rs under: aWikiPM [
	<category: 'flat file'>
	super loadFromFile: rs under: aWikiPM.
	self title: rs upToEnd
    ]
]



Object subclass: WikiSettings [
    | dictionary |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    WikiSettings class >> cookieString: aString [
	<category: 'instance creation'>
	^self new fromCookieString: aString
    ]

    WikiSettings class >> new [
	<category: 'instance creation'>
	^super new initialize
    ]

    loadFromFile: aFileStream [
	<category: 'flat file'>
	| line |
	[(line := aFileStream nextLine) isEmpty] whileFalse: 
		[line := line substrings: $=.
		line size = 2 
		    ifTrue: [self at: (line at: 1) put: (line at: 2)]
		    ifFalse: [self at: (line at: 1) put: true]]
    ]

    saveToFile: ws [
	<category: 'flat file'>
	| line |
	self settingsDo: 
		[:key :value | 
		value == false 
		    ifFalse: 
			[line := key.
			value == true ifFalse: [line := line , '=' , 'value'].
			ws
			    nextPutAll: line;
			    nl]].
	ws nl
    ]

    initialize [
	<category: 'private'>
	dictionary := Dictionary new
    ]

    at: name put: value [
	<category: 'private'>
	^dictionary at: name put: value
    ]

    at: name default: default [
	<category: 'private'>
	^dictionary at: name ifAbsentPut: [default]
    ]

    backgroundColor [
	<category: 'settings'>
	^self at: 'bc' default: '#ffffff'
    ]

    backgroundColor: anObject [
	<category: 'settings'>
	self at: 'bc' put: anObject
    ]

    linkColor [
	<category: 'settings'>
	^self at: 'lc' default: '#0000ff'
    ]

    linkColor: anObject [
	<category: 'settings'>
	self at: 'lc' put: anObject
    ]

    tableBackgroundColor [
	<category: 'settings'>
	^self at: 'tbc' default: '#ffe0ff'
    ]

    tableBackgroundColor: anObject [
	<category: 'settings'>
	self at: 'tbc' put: anObject
    ]

    textColor [
	<category: 'settings'>
	^self at: 'tc' default: '#000000'
    ]

    textColor: anObject [
	<category: 'settings'>
	self at: 'tc' put: anObject
    ]

    visitedLinkColor [
	<category: 'settings'>
	^self at: 'vlc' default: '#551a8b'
    ]

    visitedLinkColor: anObject [
	<category: 'settings'>
	self at: 'vlc' put: anObject
    ]
]



Servlet subclass: Wiki [
    | settings pages rootPageTitle syntaxPageTitle fileServer persistanceManager |
    
    <comment: 'A Wiki is made up of four kinds of classes; Wiki, WikiPersistanceManager,
WikiPage, and WikiHTML.  A Wiki has a collection of WikiPages, which can be read or
edited over the web, and is able to select a WikiHTML to match the command to
be performed.  WikiHTML objects produce HTML for the page, which the WebServer
will send back to the web browser.  A WikiPersistanceManager knows how to save to
disk and then retrieve the pages that make up a Wiki; the reason why it is separated
from the Wiki class is that, this way, you can use any kind of persistance (binary,
flat file,...) with any kind of Wiki (password-protected, normal,
read-only,...).

There are many subclasses of WikiHTML, one for each way that a page can be
converted into HTML.  Each subclass represents a different command, such as editing,
changing the name of a page, or looking at old versions of a page.

There are also many subclasses of WikiPage.  Except for the original page, each
version points to the previous version of the page.  Since the original page is always
of the form "Describe XXX here", it is not very interesting.  Other versions of the page
can have a custom contents or can be renamed.'>
    <category: 'Web-Wiki'>

    Wiki class >> named: aString [
	<category: 'instance creation'>
	^self new name: aString
    ]

    Wiki class >> new [
	<category: 'instance creation'>
	^super new initialize
    ]

    initialize [
	<category: 'initialize'>
	pages := Dictionary new.
	settings := WikiSettings new.
	self name: 'Wiki'.
	self rootPageTitle: 'Duh Tawp'.
	self syntaxPageTitle: 'Duh Rools'
    ]

    redirectToRootPage: aRequest [
	<category: 'interaction'>
	aRequest location addLast: self rootPageTitle , '.html'.
	"self sendPageFor: aRequest."
	^(ErrorResponse 
	    movedTemporarilyTo: self printString , '/' , aRequest location last) 
		respondTo: aRequest
    ]

    removeHTMLFrom: pageTitle [
	<category: 'interaction'>
	pageTitle size > 5 ifFalse: [^pageTitle].
	^(pageTitle copyFrom: pageTitle size - 4 to: pageTitle size = '.html') 
	    ifTrue: [pageTitle copyFrom: 1 to: pageTitle size - 5]
	    ifFalse: [pageTitle]
    ]

    sendPageFor: aRequest [
	<category: 'interaction'>
	| pageTitle |
	pageTitle := self removeHTMLFrom: aRequest location last.
	^(self hasPageTitled: pageTitle) 
	    ifTrue: [WikiPageHTML respondTo: aRequest in: self]
	    ifFalse: [WikiAbsentPageHTML respondTo: aRequest in: self]
    ]

    replyToGetRequest: aRequest [
	<category: 'interaction'>
	| rClass size |
	size := aRequest location size - self depth + 1.
	size < 2 
	    ifTrue: 
		[size = 0 ifTrue: [^self redirectToRootPage: aRequest].
		^(aRequest location last sameAs: 'RECENT CHANGES') 
		    ifTrue: [WikiChangesHTML respondTo: aRequest in: self]
		    ifFalse: [self sendPageFor: aRequest]].
	rClass := size = 2 
		    ifTrue: [self classForCommand: aRequest]
		    ifFalse: [WikiErrorHTML].
	^rClass respondTo: aRequest in: self
    ]

    classForCommand: aRequest [
	<category: 'interaction'>
	| cmd page |
	cmd := aRequest location at: self depth.
	page := aRequest location last.
	(cmd sameAs: 'CREATE') 
	    ifTrue: 
		[self createPageFor: aRequest.
		^WikiEditHTML].
	(self hasPageTitled: page) ifFalse: [^WikiAbsentPageHTML].
	(cmd sameAs: 'EDIT') ifTrue: [^WikiEditHTML].
	(cmd sameAs: 'HISTORY') ifTrue: [^WikiHistoryHTML].
	(cmd sameAs: 'RENAME') ifTrue: [^WikiRenameHTML].
	(cmd sameAs: 'REFS') ifTrue: [^WikiReferencesHTML].
	(cmd sameAs: 'VERSION') ifTrue: [^WikiVersionHTML].
	^WikiErrorHTML
    ]

    replyToPostEditRequest: aRequest [
	<category: 'interaction'>
	| newPage currentPage newContents |
	currentPage := self pageTitled: aRequest location last.
	newContents := aRequest postDataAt: #NEWCONTENTS.
	newPage := currentPage newContents: newContents by: aRequest originator.
	self addPage: newPage.
	self sendPageFor: aRequest
    ]

    replyToPostRenameRequest: aRequest [
	<category: 'interaction'>
	| currentPage newTitle newPage |
	currentPage := self pageTitled: aRequest location last.
	newTitle := aRequest postDataAt: #NEWTITLE.
	((self hasPageTitled: newTitle) 
	    and: [(self pageTitled: newTitle) ~= currentPage]) 
		ifTrue: [^WikiRenameConflictHTML respondTo: aRequest in: self].
	newPage := currentPage changeTitle: newTitle by: aRequest originator.
	self addPage: newPage.
	self sendPageFor: aRequest
    ]

    replyToPostRequest: aRequest [
	<category: 'interaction'>
	| cmd |
	cmd := aRequest postDataAt: #COMMAND.
	(cmd sameAs: 'EDIT') ifTrue: [^self replyToPostEditRequest: aRequest].
	(cmd sameAs: 'RENAME') ifTrue: [^self replyToPostRenameRequest: aRequest].
	(cmd sameAs: 'SEARCH') ifTrue: [^self replyToPostSearchRequest: aRequest].
	self replyToUnknownRequest: aRequest
    ]

    replyToPostSearchRequest: aRequest [
	<category: 'interaction'>
	^WikiReferencesHTML respondTo: aRequest in: self
    ]

    replyToUnknownRequest: aRequest [
	<category: 'interaction'>
	^WikiErrorHTML respondTo: aRequest in: self
    ]

    respondTo: aRequest [
	<category: 'interaction'>
	(aRequest action sameAs: 'HEAD') 
	    ifTrue: [^self replyToGetRequest: aRequest].
	(aRequest action sameAs: 'GET') 
	    ifTrue: [^self replyToGetRequest: aRequest].
	(aRequest action sameAs: 'POST') 
	    ifTrue: [^self replyToPostRequest: aRequest].
	^(ErrorResponse acceptableMethods: #('HEAD' 'GET' 'POST')) 
	    respondTo: aRequest
    ]

    syntaxPageTitle [
	<category: 'accessing'>
	^(self pageTitled: syntaxPageTitle) title
    ]

    syntaxPageTitle: aString [
	<category: 'accessing'>
	syntaxPageTitle notNil 
	    ifTrue: [pages removeKey: syntaxPageTitle asUppercase].
	syntaxPageTitle := aString.
	self addPage: self newSyntaxPage
    ]

    filesPath [
	<category: 'accessing'>
	^fileServer isNil ifTrue: [nil] ifFalse: [fileServer printString]
    ]

    filesPath: aString [
	<category: 'accessing'>
	| path |
	aString isNil ifTrue: [^self fileServer: nil].
	path := (aString at: 1) == $/ 
		    ifTrue: [WebServer current handler]
		    ifFalse: [self parent].
	(aString substrings: $/) 
	    do: [:each | each isEmpty ifFalse: [path := path componentNamed: each]].
	self fileServer: path
    ]

    fileServer [
	<category: 'accessing'>
	^fileServer
    ]

    fileServer: aString [
	<category: 'accessing'>
	fileServer := aString
    ]

    name [
	<category: 'accessing'>
	^name
    ]

    name: aString [
	<category: 'accessing'>
	name := aString
    ]

    persistanceManager: aWikiPersistanceManager [
	<category: 'accessing'>
	persistanceManager := aWikiPersistanceManager.
	aWikiPersistanceManager wiki: self
    ]

    rootPageTitle [
	<category: 'accessing'>
	^(self pageTitled: rootPageTitle) title
    ]

    rootPageTitle: aString [
	<category: 'accessing'>
	rootPageTitle notNil ifTrue: [pages removeKey: rootPageTitle asUppercase].
	rootPageTitle := aString.
	self addPage: (OriginalWikiPage new title: rootPageTitle)
    ]

    save [
	<category: 'accessing'>
	persistanceManager save
    ]

    settings [
	<category: 'accessing'>
	^settings
    ]

    startDate [
	<category: 'accessing'>
	^((self pageTitled: self rootPageTitle) versionAt: 0) timestamp
    ]

    loadFromFile: aFileStream [
	<category: 'flat file'>
	| path |
	settings loadFromFile: aFileStream.
	self name: aFileStream nextLine.
	self rootPageTitle: aFileStream nextLine.
	self syntaxPageTitle: aFileStream nextLine.
	path := aFileStream nextLine.
	path = '<none>' ifTrue: [path := nil].
	self filesPath: path.
	^self
    ]

    saveToFile: ws [
	<category: 'flat file'>
	settings saveToFile: ws.
	ws
	    nextPutAll: self name;
	    nl.
	ws
	    nextPutAll: self rootPageTitle;
	    nl.
	ws
	    nextPutAll: self syntaxPageTitle;
	    nl.
	self filesPath isNil 
	    ifTrue: 
		[ws
		    nextPutAll: '<none>';
		    nl]
	    ifFalse: 
		[ws
		    nextPutAll: self filesPath;
		    nl].
	^self
    ]

    addPage: aPage [
	<category: 'pages'>
	aPage allTitles do: [:each | pages at: each asUppercase put: aPage].
	persistanceManager isNil ifFalse: [persistanceManager addPage: aPage]
    ]

    currentPageTitleFor: aString [
	<category: 'pages'>
	^(aString sameAs: 'Changes') 
	    ifTrue: ['Recent Changes']
	    ifFalse: [(pages at: aString asUppercase) title]
    ]

    currentTitleOf: aString [
	<category: 'pages'>
	^(aString sameAs: 'RECENT CHANGES') 
	    ifTrue: [aString]
	    ifFalse: [(self pageTitled: aString) title]
    ]

    syntaxPage [
	<category: 'pages'>
	^self pageTitled: syntaxPageTitle
    ]

    hasPageTitled: aString [
	<category: 'pages'>
	^(pages includesKey: aString asUppercase) 
	    or: [aString sameAs: 'RECENT CHANGES']
    ]

    allPagesDo: aBlock [
	<category: 'pages'>
	pages do: aBlock
    ]

    pagesDo: aBlock [
	"when enumerating the pages dictionary, we want to filter to only those entries whose titles are current, this avoids double enumerating a page that might have two or more titles in it's history"

	<category: 'pages'>
	pages 
	    keysAndValuesDo: [:title :page | (page title sameAs: title) ifTrue: [aBlock value: page]]
    ]

    pageTitled: aString [
	<category: 'pages'>
	^pages at: aString asUppercase
    ]

    createPageFor: aRequest [
	<category: 'private'>
	(self hasPageTitled: aRequest location last) 
	    ifFalse: 
		[self addPage: ((OriginalWikiPage new)
			    author: aRequest originator;
			    title: aRequest location last;
			    yourself)]
    ]

    newSyntaxPage [
	<category: 'private'>
	^(OriginalWikiPage new title: syntaxPageTitle) 
	    newContents: self newSyntaxPageContents
	    by: ''
    ]

    newSyntaxPageContents [
	<category: 'private'>
	^'The Wiki''s a place where anybody can edit anything. To do so just follow the <I>Edit this page</I> link at the top or bottom of a page. The formatting rules are pretty simple:
. Links are created by placing square brackets around the link name (e.g. [[aPageName]). If you need to create a [[ character, use two of them (e.g. "[[[["). You don''t need to double up the ] character unless you actually want to use it as part of the link name.
. If you want to create a link to an "outside" source, just include the full internet protocol name (e.g. [[http://www.somesite.com] or [[mailto:someone@somewhere.com] or [[ftp://somesite.ftp]).
. If you want a link (either internal or outside) by another name, then place both the desired name and the actual link target as a pair separated by > character (e.g. [[The Top > Home Page] or [[me > mailto:myname@myplace.com]).
. Carriage returns create a new paragraph
. Use any HTML you want. The Wiki formatting rules will not be applied between a PRE tag.
. To create a horizontal line, start a line with ''----''.
. To create a bullet list item, start a line with a . character.
. To create a numbered list item, start a line with a # character.
. To create a heading, start a line with a * character.  More consecutive asterisks yield lower level headings.
. To create a table, start the line with two | (vertical bar) characters. For each cell in the row, separate again by two | characters. Successive lines that start with the two | characters are made into the same table.
. To publish your edits, press the save button. If you don''t want to publish, just press your browser''s Back button.
'
    ]
]



Wiki subclass: ProtectedWiki [
    | authorizer |
    
    <comment: nil>
    <category: 'Web-Wiki'>

    replyToRequest: aRequest [
	<category: 'authentication'>
	self authorizer 
	    authorize: aRequest
	    in: self
	    ifAuthorized: [super replyToRequest: aRequest]
    ]
]



ProtectedWiki subclass: ReadOnlyWiki [
    
    <comment: nil>
    <category: 'Web-Wiki'>

    replyToPostEditRequest: aRequest [
	<category: 'authentication'>
	self authorizer 
	    authorize: aRequest
	    in: self
	    ifAuthorized: [super replyToPostEditRequest: aRequest]
    ]

    replyToPostRenameRequest: aRequest [
	<category: 'authentication'>
	self authorizer 
	    authorize: aRequest
	    in: self
	    ifAuthorized: [super replyToPostRenameRequest: aRequest]
    ]
]



ProtectedWiki subclass: PasswordWiki [
    
    <comment: nil>
    <category: 'Web-Wiki'>

    authorizer [
	<category: 'authentication'>
	^authorizer
    ]

    authorizer: aWebAuthorizer [
	<category: 'authentication'>
	authorizer := aWebAuthorizer.
	self fileServer isNil 
	    ifFalse: [self fileServer uploadAuthorizer: aWebAuthorizer]
    ]

    loginID: aLoginID password: aPassword [
	<category: 'authentication'>
	self authorizer: (WebAuthorizer loginID: aLoginID password: aPassword)
    ]

    loadFromFile: aFileStream [
	<category: 'flat file'>
	super loadFromFile: aFileStream.
	self authorizer: (WebAuthorizer fromString: aFileStream nextLine).
	^self
    ]

    saveToFile: ws [
	<category: 'flat file'>
	super saveToFile: ws.
	ws
	    nextPutAll: self authorizer authorizer;
	    nl.
	^self
    ]
]



WebResponse subclass: WikiHTML [
    | wiki page |
    
    <comment: 'WikiHTML is an object that can convert a WikiPage to HTML.  There are
different subclasses of WikiHTML for the various ways a page can be rendered,
such as when it is edited or renamed.

All subclasses must implement sendBody.'>
    <category: 'Web-WikiRendering'>

    WikiHTML class >> new [
	<category: 'instance creation'>
	^super new initialize
    ]

    WikiHTML class >> respondTo: aRequest in: aWiki [
	<category: 'instance creation'>
	^(self new)
	    wiki: aWiki;
	    respondTo: aRequest
    ]

    initialize [
	<category: 'initialize'>
	
    ]

    browserTitle [
	<category: 'accessing'>
	^self wikiName , ': ' , self pageTitle
    ]

    encodedPageTitle [
	<category: 'accessing'>
	^(URL encode: self page title) , '.html'
    ]

    settings [
	<category: 'accessing'>
	^wiki settings
    ]

    page [
	<category: 'accessing'>
	page isNil ifTrue: [page := wiki pageTitled: request location last].
	^page
    ]

    pageTitle [
	<category: 'accessing'>
	^self page title
    ]

    emitIcon: imageBlock linkTo: nameBlock titled: titleBlock [
	<category: 'accessing'>
	self wiki filesPath isNil 
	    ifFalse: 
		[^self
		    image: imageBlock
			linkTo: nameBlock
			titled: titleBlock;
		    nl].
	self td: [self linkTo: nameBlock titled: titleBlock]
    ]

    emitCommonIcons [
	<category: 'accessing'>
	self
	    emitIcon: [self << self wiki filesPath << '/help.jpg']
		linkTo: 
		    [self
			<< self wiki;
			<< $/;
			nextPutUrl: self wiki syntaxPageTitle]
		titled: [self << self wiki syntaxPageTitle];
	    emitIcon: [self << self wiki filesPath << '/recent.jpg']
		linkTo: [self << self wiki << '/RECENT+CHANGES']
		titled: [self << 'Recent changes'];
	    emitIcon: [self << self wiki filesPath << '/top.jpg']
		linkTo: [self << self wiki << $/]
		titled: [self << 'Back to Top']
    ]

    sendBody [
	"subclasses will usually want to do more here"

	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self emitFinish
    ]

    emitFinish [
	<category: 'accessing'>
	self
	    nl;
	    << '</FONT>';
	    nl;
	    << '</BODY></HTML>'
    ]

    emitSearch: aString [
	<category: 'accessing'>
	self horizontalLine.
	(self << '<FORM ACTION="' << wiki name)
	    << '" METHOD=POST>';
	    nl.
	self
	    << '<INPUT TYPE="HIDDEN" NAME="COMMAND" VALUE="SEARCH">';
	    nl.
	(self << '<INPUT TYPE= "TEXT" NAME="SEARCHPATTERN" VALUE="' << aString)
	    << '" SIZE=40>';
	    nl.
	self wiki filesPath isNil 
	    ifFalse: 
		[self << '<INPUT TYPE="image" ALIGN="absmiddle" BORDER="0" SRC="' 
		    << self wiki filesPath << '/find.jpg" ALT=']
	    ifTrue: [self << '<INPUT TYPE="submit" VALUE='].
	self
	    << '"Find..."></FORM>';
	    nl
    ]

    emitStart [
	<category: 'accessing'>
	(self << '<HTML><HEAD><TITLE>' << self browserTitle 
	    << '</TITLE></HEAD><BODY bgcolor=' << self settings backgroundColor 
	    << ' link=' << self settings linkColor 
	    << ' vlink=' << self settings visitedLinkColor)
	    << $>;
	    nl.
	(self << '<FONT color=' << self settings textColor)
	    << $>;
	    nl
    ]

    emitIcons [
	<category: 'accessing'>
	self
	    emitIconsStart;
	    emitCommonIcons;
	    emitIconsEnd
    ]

    emitIconsEnd [
	<category: 'accessing'>
	self wiki filesPath isNil 
	    ifFalse: 
		[self
		    << '<BR>';
		    nl]
	    ifTrue: 
		[self
		    nl;
		    << '</TR>';
		    nl;
		    << '</TABLE>';
		    nl]
    ]

    emitIconsStart [
	<category: 'accessing'>
	self wiki filesPath isNil 
	    ifFalse: 
		[^self image: [self << self wiki filesPath << '/head.jpg']
		    titled: [self << self wiki]].
	self << '<TABLE width=100% bgcolor=' << self settings tableBackgroundColor.
	self
	    << '><TR>';
	    nl
    ]

    emitUrlForCommand: commandName [
	<category: 'accessing'>
	self << self wiki << $/ << commandName << $/ << self encodedPageTitle
    ]

    emitUrlOfPage [
	<category: 'accessing'>
	self << self wiki << $/ << self encodedPageTitle
    ]

    linkToPage: aPage [
	<category: 'accessing'>
	self linkTo: 
		[self
		    << self wiki;
		    << $/;
		    nextPutUrl: aPage title]
	    titled: [self << aPage title]
    ]

    wiki [
	<category: 'accessing'>
	^wiki
    ]

    wiki: anObject [
	<category: 'accessing'>
	wiki := anObject
    ]

    wikiName [
	<category: 'accessing'>
	^wiki name
    ]
]



WikiHTML subclass: WikiPageHTML [
    | contentStream currentChar lastChar inBullets inNumbers heading inTable |
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    ParseTable := nil.

    WikiPageHTML class >> initialize [
	<category: 'initialize'>
	ParseTable := Array new: 256.
	ParseTable at: 1 + Character cr asciiValue put: #processCr.
	ParseTable at: 1 + Character nl asciiValue put: #processNl.
	ParseTable at: 1 + $[ asciiValue put: #processLeftBracket.
	ParseTable at: 1 + $. asciiValue put: #processDot.
	ParseTable at: 1 + $# asciiValue put: #processPound.
	ParseTable at: 1 + $- asciiValue put: #processDash.
	ParseTable at: 1 + $* asciiValue put: #processStar.
	ParseTable at: 1 + $| asciiValue put: #processPipe.
	ParseTable at: 1 + $< asciiValue put: #processLeftAngle
    ]

    isExternalAddress: linkAddress [
	"Faster than #match:"

	<category: 'private-HTML'>
	^#('http:' 'https:' 'mailto:' 'file:' 'ftp:' 'news:' 'gopher:' 'telnet:') 
	    anySatisfy: 
		[:each | 
		each size < linkAddress size and: 
			[(1 to: each size) 
			    allSatisfy: [:index | (each at: index) == (linkAddress at: index)]]]
    ]

    isImage: linkAddress [
	"Faster than #match:"

	<category: 'private-HTML'>
	^#('.gif' '.jpeg' '.jpg' '.jpe') anySatisfy: 
		[:each | 
		each size < linkAddress size and: 
			[(1 to: each size) allSatisfy: 
				[:index | 
				(each at: index) == (linkAddress at: linkAddress size - each size + index)]]]
    ]

    linkAddressIn: aString [
	<category: 'private-HTML'>
	| rs |
	rs := aString readStream.
	rs skipTo: $>.
	^(rs atEnd ifTrue: [aString] ifFalse: [rs upToEnd]) trimSeparators
    ]

    linkNameIn: aString [
	<category: 'private-HTML'>
	| rs |
	rs := aString readStream.
	^(rs upTo: $>) trimSeparators
    ]

    addCurrentChar [
	<category: 'parsing'>
	self responseStream nextPut: currentChar
    ]

    atLineStart [
	<category: 'parsing'>
	^lastChar == Character nl or: [lastChar == nil]
    ]

    closeBulletItem [
	<category: 'parsing'>
	self
	    << '</LI>';
	    nl.
	contentStream peek == $. 
	    ifFalse: 
		[inBullets := false.
		self
		    << '</UL>';
		    nl]
    ]

    closeHeading [
	<category: 'parsing'>
	(self << '</H' << heading)
	    << '>';
	    nl.
	heading := nil
    ]

    closeNumberItem [
	<category: 'parsing'>
	self
	    << '</LI>';
	    nl.
	contentStream peek == $# 
	    ifFalse: 
		[inNumbers := false.
		self
		    << '</OL>';
		    nl]
    ]

    closeTableRow [
	<category: 'parsing'>
	| pos |
	self
	    << '</TD></TR>';
	    nl.
	pos := contentStream position.
	(contentStream peekFor: $|) 
	    ifTrue: 
		[(contentStream peekFor: $|) 
		    ifTrue: 
			[inTable := false.
			self
			    << '</TABLE>';
			    nl]].
	contentStream position: pos
    ]

    processNextChar [
	<category: 'parsing'>
	| selector |
	lastChar := currentChar.
	currentChar := contentStream next.
	selector := ParseTable at: currentChar value + 1.
	^selector isNil 
	    ifTrue: [self addCurrentChar]
	    ifFalse: [self perform: selector]
    ]

    processDot [
	<category: 'parsing'>
	self atLineStart ifFalse: [^self addCurrentChar].
	inBullets 
	    ifFalse: 
		[self
		    << '<UL>';
		    nl.
		inBullets := true].
	self << ' <LI>'
    ]

    processStar [
	<category: 'parsing'>
	self atLineStart ifFalse: [^self addCurrentChar].
	heading := 2.
	[contentStream peekFor: $*] whileTrue: [heading := heading + 1].
	self << '<H' << heading << '>'
    ]

    processCr [
	<category: 'parsing'>
	contentStream peekFor: Character nl.
	currentChar := Character nl.
	self processNl
    ]

    processNl [
	<category: 'parsing'>
	inBullets ifTrue: [^self closeBulletItem].
	inNumbers ifTrue: [^self closeNumberItem].
	inTable ifTrue: [^self closeTableRow].
	heading isNil ifFalse: [^self closeHeading].
	self lineBreak
    ]

    processDash [
	<category: 'parsing'>
	self atLineStart ifFalse: [^self addCurrentChar].
	contentStream skipTo: Character nl.
	self horizontalLine.
	lastChar := Character nl
    ]

    processLeftAngle [
	<category: 'parsing'>
	| s |
	s := String new writeStream.
	self addCurrentChar.
	
	[currentChar := contentStream next.
	currentChar == $> or: [currentChar == $ ]] 
		whileFalse: [s nextPut: currentChar].
	self << (s := s contents) << currentChar.
	(s sameAs: 'PRE') ifFalse: [^self].
	
	[contentStream atEnd ifTrue: [^self].
	self << (contentStream upTo: $<) << $<.
	self << (s := contentStream upTo: $>) << $>.
	s sameAs: '/PRE'] 
		whileFalse
    ]

    processLeftBracket [
	<category: 'parsing'>
	| linkAddress linkName link |
	(contentStream peekFor: $[) ifTrue: [^self addCurrentChar].
	link := contentStream upTo: $].
	[contentStream peekFor: $]] 
	    whileTrue: [link := link , ']' , (contentStream upTo: $])].
	linkName := self linkNameIn: link.
	linkAddress := self linkAddressIn: link.
	(self isExternalAddress: linkAddress) 
	    ifTrue: 
		["external outside link"

		^self << '<A HREF="' << linkAddress << '">' << linkName << '</A>'].
	linkAddress = linkName 
	    ifTrue: [self emitLink: linkName]
	    ifFalse: [self emitLink: linkName to: linkAddress]
    ]

    processPipe [
	<category: 'parsing'>
	(contentStream peekFor: $|) 
	    ifTrue: 
		[self atLineStart 
		    ifTrue: 
			[inTable 
			    ifFalse: 
				[self
				    << '<TABLE BORDER=2 CELLPADDING=4 CELLSPACING=0 >';
				    nl.
				inTable := true].
			self << '<TR><TD>']
		    ifFalse: [self << '</TD><TD>']]
	    ifFalse: [self addCurrentChar]
    ]

    processPound [
	<category: 'parsing'>
	self atLineStart ifFalse: [^self addCurrentChar].
	inNumbers 
	    ifFalse: 
		[self
		    << '<OL>';
		    nl.
		inNumbers := true].
	self << ' <LI>'
    ]

    emitLink: linkAddress [
	<category: 'parsing'>
	| currentTitle |
	(self isImage: linkAddress) 
	    ifTrue: 
		["graphic image link"

		(self isExternalAddress: linkAddress) 
		    ifTrue: [^self << '<img src="' << linkAddress << '">']
		    ifFalse: 
			[^self << '<img src="' << '/' << self wiki filesPath << '/' << linkAddress 
			    << '">']].
	(wiki hasPageTitled: linkAddress) 
	    ifTrue: 
		["simple one piece existing link"

		currentTitle := self wiki currentTitleOf: linkAddress.
		self linkTo: 
			[self
			    << self wiki;
			    << $/;
			    nextPutUrl: currentTitle]
		    titled: [self << currentTitle]]
	    ifFalse: 
		["simple one piece non existant link"

		self << '<U>' << linkAddress << '</U>'.
		self linkTo: 
			[self
			    << self wiki;
			    << '/CREATE/';
			    nextPutUrl: linkAddress]
		    titled: [self << $?]]
    ]

    emitLink: linkName to: linkAddress [
	<category: 'parsing'>
	| currentTitle |
	(wiki hasPageTitled: linkAddress) 
	    ifTrue: 
		["two piece existing link"

		currentTitle := self wiki currentTitleOf: linkAddress.
		self linkTo: 
			[self
			    << self wiki;
			    << $/;
			    nextPutUrl: currentTitle]
		    titled: [self << linkName]]
	    ifFalse: 
		["two piece non existant link"

		self << '<U>' << linkName << '</U>'.
		self linkTo: 
			[self
			    << self wiki;
			    << '/CREATE/';
			    nextPutUrl: linkAddress]
		    titled: [self << $?]]
    ]

    sendBody [
	<category: 'HTML'>
	self emitStart.
	self emitIcons.
	self emitTitle.
	self emitContents.
	self emitSearch: ''.
	self emitFinish
    ]

    emitCommand: commandName text: textString [
	<category: 'HTML'>
	^self 
	    emitIcon: 
		[self << self wiki filesPath << $/ << commandName asLowercase << '.jpg']
	    linkTo: [self emitUrlForCommand: commandName]
	    titled: [self << textString]
    ]

    emitIcons [
	<category: 'HTML'>
	self emitIconsStart.
	self emitCommonIcons.
	self emitCommand: 'EDIT' text: 'Edit this page'.
	self emitCommand: 'RENAME' text: 'Rename this page'.
	self emitCommand: 'HISTORY' text: 'History of this page'.
	self emitIconsEnd
    ]

    emitContents [
	<category: 'HTML'>
	contentStream := self page contents readStream.
	[contentStream atEnd] whileFalse: [self processNextChar].
	lastChar == Character nl ifFalse: [self processNl].
	contentStream := nil
    ]

    emitTitle [
	<category: 'HTML'>
	self heading: 
		[self linkTo: [self emitUrlForCommand: 'REFS']
		    titled: [self << self page title]]
    ]

    initialize [
	<category: 'initialization'>
	super initialize.
	heading := nil.
	inBullets := inNumbers := inTable := false
    ]
]



WikiHTML subclass: WikiAbsentPageHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    browserTitle [
	<category: 'accessing'>
	^self wikiName , ': `' , self pageTitle , ''' not found'
    ]

    pageTitle [
	<category: 'accessing'>
	^request location last
    ]

    sendResponseType [
	<category: 'accessing'>
	self
	    << 'HTTP/1.1 404 Not Found';
	    nl
    ]

    sendBody [
	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self heading: 
		[self << self wikiName << ' contains no page titled: "' 
		    << request location last]
	    level: 2.
	self emitSearch: request location last.
	self emitFinish
    ]
]



WikiHTML subclass: WikiReferencesHTML [
    | referringPages |
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    actualSearchString [
	<category: 'private'>
	^self searchString isEmpty 
	    ifTrue: [self searchString]
	    ifFalse: ['*' , self searchString , '*']
    ]

    findMatches [
	<category: 'private'>
	| match |
	referringPages := SortedCollection sortBlock: [:a :b | a title < b title].
	match := self actualSearchString.
	Processor activeProcess lowerPriority.
	wiki 
	    pagesDo: [:each | (each references: match) ifTrue: [referringPages add: each]].
	Processor activeProcess raisePriority
    ]

    browserTitle [
	<category: 'accessing'>
	| ws |
	ws := String new writeStream.
	ws
	    nextPutAll: 'SEARCH ';
	    nextPutAll: self wikiName;
	    nextPutAll: ':"';
	    nextPutAll: self searchString;
	    nextPut: $".
	^ws contents
    ]

    sendBody [
	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self emitMatchList.
	self emitSearch: self searchString.
	self emitFinish
    ]

    emitMatchList [
	<category: 'accessing'>
	self findMatches.
	referringPages isEmpty ifTrue: [^self emitNoMatches].
	self heading: 
		[self 
		    << ('There %<is|are>2 %1 reference%<|s>2 to the phrase:' % 
				{referringPages size.
				referringPages size = 1})].
	self
	    << '<I>  ...';
	    << self searchString;
	    << '...</I>';
	    lineBreak.
	self
	    << '<UL>';
	    nl.
	referringPages do: [:each | self listItem: [self linkToPage: each]].
	self
	    << '</UL>';
	    nl
    ]

    emitNoMatches [
	<category: 'accessing'>
	self
	    << '<H1>No references to the phrase</H1>';
	    nl.
	self
	    << '<I>    ...';
	    << self searchString;
	    << '...</I>';
	    lineBreak
    ]

    searchString [
	<category: 'accessing'>
	^request postDataAt: #SEARCHPATTERN ifAbsent: [request location last]
    ]
]



WikiPageHTML subclass: WikiVersionHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    page [
	<category: 'accessing'>
	^super page versionAt: self versionNumber
    ]

    emitIcons [
	<category: 'accessing'>
	self emitIconsStart.
	self emitCommonIcons.
	self emitCommand: 'HISTORY' text: 'History of this page'.
	self emitPreviousVersion.
	self emitNextVersion.
	self emitIconsEnd
    ]

    emitNextVersion [
	<category: 'accessing'>
	self versionNumber < (wiki pageTitled: self page title) versionNumber 
	    ifFalse: [^self].
	self 
	    emitIcon: [self << self wiki filesPath << '/next.jpg']
	    linkTo: [self emitUrlForVersionNumber: self versionNumber + 1]
	    titled: [self << 'Previous']
    ]

    emitPreviousVersion [
	<category: 'accessing'>
	self versionNumber <= 0 ifTrue: [^self].
	self 
	    emitIcon: [self << self wiki filesPath << '/prev.jpg']
	    linkTo: [self emitUrlForVersionNumber: self versionNumber - 1]
	    titled: [self << 'Previous']
    ]

    emitTitle [
	<category: 'accessing'>
	self heading: 
		[self linkTo: [self emitUrlForCommand: 'REFS']
		    titled: [self << self page title].
		self << ' (Version ' << self versionNumber << ')']
    ]

    versionNumber [
	<category: 'accessing'>
	^((request postDataAt: #n) asNumber max: 0) min: super page versionNumber
    ]

    emitUrlForVersionNumber: aNumber [
	<category: 'html'>
	self << self wiki << '/VERSION/' << self encodedPageTitle << '?n=' 
	    << aNumber
    ]
]



WikiHTML subclass: WikiChangesHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    numberOfChanges [
	<category: 'accessing'>
	^20
    ]

    numberOfDays [
	<category: 'accessing'>
	^7
    ]

    pageTitle [
	<category: 'accessing'>
	^'Recent Changes'
    ]

    sendBody [
	<category: 'accessing'>
	| day genesis minDate changesShown |
	self emitStart.
	self emitIcons.
	self emitChanges.
	self emitSearch: ''.
	self emitFinish
    ]

    emitChangedPage: aPage [
	<category: 'accessing'>
	self listItem: 
		[self
		    linkToPage: aPage;
		    space.
		self << aPage timestamp asTime << ' (' << aPage author << ')']
    ]

    emitChanges [
	<category: 'accessing'>
	| day genesis minDate changesShown |
	self heading: [self << 'Recent Changes'].
	genesis := wiki startDate printNl.
	day := Date today.
	minDate := (day subtractDays: self numberOfDays) printNl.
	changesShown := 0.
	
	[day < genesis ifTrue: [^self].
	day >= minDate or: [changesShown < self numberOfChanges]] 
		whileTrue: 
		    [changesShown := changesShown + (self emitChangesFor: day).
		    day := day subtractDays: 1]
    ]

    emitChangesFor: aDate [
	<category: 'accessing'>
	| sc |
	sc := SortedCollection new 
		    sortBlock: [:a :b | a timestamp > b timestamp] wiki
		    pagesDo: [:each | each timestamp asDate = aDate ifTrue: [sc add: each]].
	sc isEmpty 
	    ifFalse: 
		[self heading: 
			[(self responseStream)
			    nextPutAll: aDate monthName;
			    space;
			    print: aDate day;
			    space;
			    print: aDate year]
		    level: 3.
		self
		    << '<UL>';
		    nl.
		sc do: [:each | self emitChangedPage: each].
		self
		    << '</UL>';
		    nl].
	^sc size
    ]
]



WikiHTML subclass: WikiErrorHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    browserTitle [
	<category: 'accessing'>
	^self pageTitle
    ]

    emitDescription [
	<category: 'accessing'>
	self
	    << 'The ';
	    << self wiki;
	    << ' wiki is not able to process this request. '.
	self 
	    << 'This can be due to a malformed URL, or (less likely) to an internal server error'.
	self
	    lineBreak;
	    lineBreak.
	self
	    << 'originator: ';
	    << request originator displayString;
	    lineBreak.
	self
	    << 'action: ';
	    << request action displayString;
	    lineBreak.
	self << 'location: '.
	request location do: [:each | self << $/ << each].
	self lineBreak.
	request enumeratePostData: 
		[:key :val | 
		self
		    lineBreak;
		    << key;
		    << ' = ';
		    nl;
		    << val;
		    nl].
	self
	    lineBreak;
	    horizontalLine;
	    italic: [self << WebServer version]
    ]

    pageTitle [
	<category: 'accessing'>
	^'Bad request'
    ]

    sendBody [
	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self emitDescription.
	self emitFinish
    ]
]



WikiHTML subclass: WikiRenameConflictHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    newTitle [
	<category: 'accessing'>
	^request postDataAt: #NEWTITLE
    ]

    emitDescription [
	<category: 'accessing'>
	self heading: 
		[self << 'This name ('.
		self linkTo: [self << self wiki << $/ << self newTitle]
		    titled: [self << self newTitle].
		self << ') is in use already. Sorry, cannot complete this rename.']
	    level: 2
    ]

    sendBody [
	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self emitDescription.
	self emitSearch: self newTitle.
	self emitFinish
    ]
]



WikiHTML subclass: WikiCommandHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    browserTitle [
	<category: 'accessing'>
	^super browserTitle , self titleSuffix
    ]

    titleSuffix [
	<category: 'accessing'>
	^self subclassResponsibility
    ]
]



WikiCommandHTML subclass: WikiEditHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    titleSuffix [
	<category: 'accessing'>
	^' (edit)'
    ]

    emitForm [
	<category: 'HTML'>
	self heading: 
		[self << 'Edit '.
		self linkTo: [self emitUrlForCommand: 'REFS']
		    titled: [self << self pageTitle]].
	self
	    << 'Don''t know how to edit a page? Visit ';
	    linkToPage: wiki syntaxPage;
	    << '.';
	    nl.
	self
	    << '<FORM ACTION="';
	    emitUrlOfPage;
	    << '" METHOD=POST>';
	    nl.
	self
	    << '<INPUT TYPE="HIDDEN" NAME="COMMAND" VALUE="EDIT">';
	    nl.
	self
	    << '<TEXTAREA NAME="NEWCONTENTS"  WRAP=VIRTUAL COLS=80 ROWS=20>';
	    nl.
	self
	    << self page contents;
	    nl.
	self
	    << '</TEXTAREA>';
	    lineBreak.
	self
	    << '<INPUT TYPE="submit" VALUE="Save">';
	    nl.
	self
	    << '</FORM>';
	    nl
    ]

    sendBody [
	<category: 'HTML'>
	self emitStart.
	self emitIcons.
	self emitForm.
	self emitFinish
    ]
]



WikiCommandHTML subclass: WikiHistoryHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    sendBody [
	<category: 'HTML'>
	self emitStart.
	self emitIcons.
	self emitTitle.
	self emitTable.
	self emitSearch: ''.
	self emitFinish
    ]

    emitTitle [
	<category: 'HTML'>
	self heading: 
		[self << 'History of '.
		self linkTo: [self emitUrlForCommand: 'REFS']
		    titled: [self << self page title]]
    ]

    emitTable [
	<category: 'HTML'>
	self
	    << '<TABLE WIDTH="95%" BORDER="1">';
	    nl.
	self
	    << '<TR>';
	    nl.
	self
	    td: [self << '<B>Version</B>'];
	    td: [self << '<B>Operation</B>'];
	    td: [self << '<B>Author</B>'];
	    td: [self << '<B>Creation Time</B>'].
	self
	    << '</TR>';
	    nl.
	self page versionsDo: [:each | self emitPageVersion: each].
	self
	    << '</TABLE>';
	    nl
    ]

    emitPageVersion: each [
	<category: 'HTML'>
	self
	    << '<TR>';
	    nl.
	self td: 
		[self linkTo: 
			[self
			    << self wiki;
			    << '/VERSION/';
			    nextPutUrl: each title;
			    << '?n=';
			    << each versionNumber]
		    titled: [self << each versionNumber]].
	self td: [self << each operationSynopsis].
	self td: [self << each author].
	self td: [self sendTimestamp: each timestamp].
	self
	    << '</TR>';
	    nl
    ]

    titleSuffix [
	<category: 'accessing'>
	^' (history)'
    ]
]



WikiCommandHTML subclass: WikiRenameHTML [
    
    <comment: nil>
    <category: 'Web-WikiRendering'>

    titleSuffix [
	<category: 'accessing'>
	^' (rename)'
    ]

    emitForm [
	<category: 'accessing'>
	self heading: 
		[self << 'Rename'.
		self linkTo: [self emitUrlForCommand: 'REFS']
		    titled: [self << self pageTitle]].
	self
	    << '<FORM ACTION="';
	    emitUrlOfPage;
	    << '" METHOD=POST>';
	    nl.
	self
	    << '<INPUT TYPE="HIDDEN" NAME="COMMAND" VALUE="RENAME">';
	    nl.
	self
	    << '<INPUT TYPE= "TEXT" NAME="NEWTITLE" SIZE=80 VALUE="';
	    << self pageTitle;
	    << '">';
	    lineBreak.
	self
	    << '<INPUT TYPE="submit" VALUE="Save">';
	    nl.
	self
	    << '</FORM>';
	    nl
    ]

    sendBody [
	<category: 'accessing'>
	self emitStart.
	self emitIcons.
	self emitForm.
	self emitFinish
    ]
]



Object subclass: WikiPersistanceManager [
    | wiki |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    wiki [
	<category: 'accessing'>
	^wiki
    ]

    wiki: aWiki [
	<category: 'accessing'>
	wiki := aWiki.
	self reset
    ]

    allPagesDo: aBlock [
	<category: 'accessing'>
	wiki allPagesDo: aBlock
    ]

    addPage: aPage [
	<category: 'persistance'>
	
    ]

    load [
	<category: 'persistance'>
	self subclassResponsibility
    ]

    save [
	<category: 'persistance'>
	self subclassResponsibility
    ]
]



WikiPersistanceManager subclass: FlatFileWiki [
    | directory fileCounter idMap |
    
    <category: 'Web-Wiki'>
    <comment: nil>

    FlatFileWiki class >> directory: aDirectory [
	<category: 'instance creation'>
	^self new directory: aDirectory
    ]

    reset [
	<category: 'initialize'>
	directory exists ifFalse: [Directory create: directory name].
	idMap := IdentityDictionary new.
	fileCounter := -1
    ]

    idForPage: aPage [
	<category: 'private-persistance'>
	^idMap at: aPage ifAbsentPut: [self savePage: aPage]
    ]

    indexIn: aFilename [
	<category: 'private-persistance'>
	| tail |
	tail := aFilename stripPath.
	^(tail copyFrom: 1 to: tail size - 4) asNumber
    ]

    nextFileCounter [
	<category: 'private-persistance'>
	^fileCounter := fileCounter + 1
    ]

    loadPage: id [
	<category: 'private-persistance'>
	^self loadPageInFile: (directory at: id , '.pag')
    ]

    loadPageInFile: aFilename [
	<category: 'private-persistance'>
	| index rs page |
	index := self indexIn: aFilename.
	^idMap at: index
	    ifAbsentPut: 
		[| type |
		Transcript show: '.'.
		rs := aFilename readStream.
		type := rs nextLine asSymbol.
		
		[page := (Smalltalk at: type) new.
		page loadFromFile: rs under: self] 
			ensure: [rs close].
		page]
    ]

    loadPages [
	<category: 'private-persistance'>
	| latestVersions pageMap |
	idMap := pageMap := IdentityDictionary new.
	directory filesMatching: '*.pag' do: [:fn | self loadPageInFile: fn].
	idMap := IdentityDictionary new.
	pageMap keysAndValuesDo: [:i :page | idMap at: page put: i].
	latestVersions := pageMap asSet.
	pageMap do: 
		[:page | 
		"Remove all versions older than `each' from latest"

		page 
		    versionsDo: [:each | each == page ifFalse: [latestVersions remove: each ifAbsent: []]]].
	latestVersions do: [:page | self wiki addPage: page]
    ]

    load [
	<category: 'private-persistance'>
	| rs fn |
	self reset.
	(fn := directory at: 'wiki.conf') exists 
	    ifFalse: [self error: 'wiki directory doesn''t exist'].
	rs := fn readStream.
	
	[| type |
	type := rs nextLine asSymbol.
	self wiki: (Smalltalk at: type) new.
	self wiki loadFromFile: rs] 
		ensure: [rs close].
	self loadPages.
	self wiki persistanceManager: self.
	^self wiki
    ]

    savePage: aPage [
	<category: 'private-persistance'>
	| id ws |
	id := self nextFileCounter.
	idMap at: aPage put: id.
	ws := (self directory at: id printString , '.pag') writeStream.
	
	[ws
	    nextPutAll: aPage class name;
	    nl.
	aPage saveToFile: ws under: self] 
		ensure: [ws close].
	^id
    ]

    savePages [
	<category: 'private-persistance'>
	self allPagesDo: [:aPage | self savePage: aPage]
    ]

    save [
	<category: 'private-persistance'>
	| ws |
	self reset.
	directory exists ifFalse: [Directory create: directory name].
	ws := (directory at: 'wiki.conf') writeStream.
	
	[ws
	    nextPutAll: wiki class name;
	    nl.
	wiki saveToFile: ws] 
		ensure: [ws close].
	self savePages
    ]

    directory [
	<category: 'accessing'>
	^directory
    ]

    directory: aFilename [
	<category: 'accessing'>
	directory := File name: aFilename
    ]

    addPage: aPage [
	<category: 'pages'>
	self idForPage: aPage.
	^self
    ]
]



WebServer class extend [

    wikiDirectories [
	<category: 'examples'>
	^#('GnuSmalltalkWiki')
    ]

    initializeImages [
	<category: 'examples'>
	(self at: 8080) handler addComponent:
	    (FileWebServer
		named: 'images'
		directory: (Directory kernel / '../WebServer.star') zip)
    ]

    initializeWiki [
	"Only run this method the first time."

	"WikiServer initializeNormalWiki"

	<category: 'examples'>
	self initializeImages.
	self wikiDirectories do: 
		[:eachName | 
		"Only run this method the first time."

		| wiki |
		wiki := Wiki new.
		wiki persistanceManager: (FlatFileWiki directory: eachName).
		wiki name: eachName.
		wiki rootPageTitle: 'Home Page'.
		wiki syntaxPageTitle: 'Wiki Syntax'.
		wiki filesPath: '/images'.
		wiki save.
		(self at: 8080) handler addComponent: wiki].
	(self at: 8080) start
    ]

    initializeWikiNoImages [
	"Only run this method the first time."

	"WikiServer initializeWikiNoImages"

	<category: 'examples'>
	self wikiDirectories do: 
		[:eachName | 
		"Only run this method the first time."

		| wiki |
		wiki := Wiki new.
		wiki persistanceManager: (FlatFileWiki directory: eachName).
		wiki name: eachName.
		wiki rootPageTitle: 'Home Page'.
		wiki syntaxPageTitle: 'Wiki Syntax'.
		wiki save.
		(self at: 8080) handler addComponent: wiki].
	(self at: 8080) start
    ]

    restartWiki [
	"WikiServer restartWiki"

	<category: 'examples'>
	self initializeImages.
	self wikiDirectories do: 
		[:eachName | 
		(self at: 8080) handler 
		    addComponent: (FlatFileWiki directory: eachName) load].
	(self at: 8080) start
    ]

    restartWikiNoImages [
	"WikiServer restartWikiNoImages"

	<category: 'examples'>
	self wikiDirectories do: 
		[:eachName | 
		(self at: 8080) handler 
		    addComponent: (((FlatFileWiki directory: eachName) load)
			    filesPath: nil;
			    yourself)].
	(self at: 8080) start
    ]

]



Eval [
    WikiPageHTML initialize
]

PK
     
wB(9#&  #&    STT.stUT	 NQhux r      "=====================================================================
|
|   Smalltalk templates
|
|
 ======================================================================"

"======================================================================
|
| Copyright 2002 Federico G. Stilman
| Porting by Markus Fritsche and Paolo Bonzini
| Integration with the web server framework by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LIB.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02110-1301, USA.  
|
 ======================================================================"



Object subclass: STTTemplate [
    | sttCode cache asStringSelector |
    
    <category: 'Web-STT'>
    <comment: 'This class implements template  la JSP, PHP, ASP (ugh!), and so on.
Smalltalk code is included between {% and %} tags.  The only caution
is not to include comments between a period or an open parentheses
of any kind, and the closing %}.

For example

    %{ "Comment" 1 to: 5 do: [ %} yes<br> %{ ] %}    is valid 
    %{ 1 to: 5 do: [ "Comment" %} yes<br> %{ ] %}    is not valid

This restriction might be removed in the future.

The template is evaluated by sending #evaluateOn: or #evaluateOn:stream:
and returns the output stream (available to the code as the variable `out'').
The first (or only) argument of these two methods is available to the
code as `self'').'>

    STTTemplate class >> test [
	<category: 'unit testing'>
	| sttTest |
	sttTest := '
        <html>
	<head><title>{%= self class %}</title></head>
	<body>
		<table>
			{% self to: 10 do: [ :each | %}
		        <tr>
				   <td>{%= each printString %}</td>
				   <td>{%= (each * 2) printString %}</td>
		        </tr>
		   	{% ] %}
		</table>
	</body>
	</html>'.
	^(STTTemplate on: sttTest) evaluateOn: 1
    ]

    STTTemplate class >> test2 [
	<category: 'unit testing'>
	| sttTest |
	sttTest := '
	<html>
	<head><title>{%= self class %}</title></head>

	{% 
		out nextPutAll: ''This is another test''; nl.

		1 to: 15 do: [:x |
                    out nextPutAll: ''<p>This paragraph was manually sent out '',
                                    (self * x) printString, ''</p>''; nl ].

		out nextPutAll: ''After all this ST code goes the final HTML closing tag''.
	%}

	</html>'.
	^(STTTemplate on: sttTest) evaluateOn: 3
    ]

    STTTemplate class >> on: aString [
	"Creates an instance of the receiver on aString"

	<category: 'instance creation'>
	^self on: aString asStringSelector: self defaultAsStringSelector
    ]

    STTTemplate class >> on: aString asStringSelector: aSymbol [
	"Creates an instance of the receiver on aString"

	<category: 'instance creation'>
	^self new initializeOn: aString asStringSelector: aSymbol
    ]

    STTTemplate class >> defaultAsStringSelector [
	<category: 'defaults'>
	^#displayString
    ]

    cache [
	"Returns the receiver's cached object"

	<category: 'caching'>
	^cache
    ]

    cache: anObject [
	"Save anObject in the receiver's cache"

	<category: 'caching'>
	cache := anObject
    ]

    initializeCache [
	"Initialize the receiver's cache"

	<category: 'caching'>
	cache := nil
    ]

    isCached [
	"Tell if the receiver is cached or not. In the future
	 this will consider the fact that a cached object may
	 become old after some time, and that means that the
	 object is NOT cached anymore."

	<category: 'caching'>
	^self cache notNil
    ]

    asSmalltalkCodeOn: anObject [
	"Returns the equivalent version of the receiver as a Smalltalk
	 CompiledMethod"

	<category: 'private'>
	| method stream |
	self isCached ifTrue: [^self cache].
	stream := String new writeStream.
	self writeSmalltalkCodeOn: stream.
	method := anObject class compile: stream.
	self cache: method.
	anObject class removeSelector: method selector.
	^method
    ]

    writeSmalltalkCodeOn: stream [
	"Write the equivalent version of the receiver as Smalltalk code
	 on the given stream"

	<category: 'private'>
	| sttOpenIndex sttCloseIndex lastIndex sttCodeIndex smalltalkExpression |
	stream
	    nextPutAll: 'STT_Cache';
	    print: self asOop;
	    nextPutAll: ': out [';
	    nl.
	lastIndex := 1.
	
	[(sttOpenIndex := self sttCode indexOfSubCollection: '{%'
		    startingAt: lastIndex) > 0] 
		whileTrue: 
		    [self 
			writeOutputCodeFor: (self sttCode copyFrom: lastIndex to: sttOpenIndex - 1)
			on: stream.
		    sttCloseIndex := self sttCode 
				indexOfSubCollection: '%}'
				startingAt: sttOpenIndex
				ifAbsent: [^self error: 'Missing closing tag'].
		    sttCodeIndex := sttOpenIndex + 2.
		    (sttCode at: sttOpenIndex + 2) = $= 
			ifTrue: 
			    [stream nextPutAll: 'out nextPutAll: ('.
			    sttCodeIndex := sttCodeIndex + 1].
		    smalltalkExpression := sttCode copyFrom: sttCodeIndex to: sttCloseIndex - 1.
		    smalltalkExpression := smalltalkExpression trimSeparators.
		    stream nextPutAll: smalltalkExpression.
		    (sttCode at: sttOpenIndex + 2) = $= 
			ifTrue: 
			    [stream nextPutAll: ') ' , self asStringSelector asString.
			    sttCodeIndex := sttCodeIndex + 1].
		    ('|[({.' includes: smalltalkExpression last) ifFalse: [stream nextPut: $.].
		    stream nl.
		    lastIndex := sttCloseIndex + 2].
	self 
	    writeOutputCodeFor: (self sttCode copyFrom: lastIndex to: sttCode size)
	    on: stream.
	stream nextPutAll: '^out ]'
    ]

    writeOutputCodeFor: aString on: aStream [
	"Writes on aStream the required Smalltalk code for outputing aString on 'out'"

	<category: 'private'>
	aStream
	    nextPutAll: 'out nextPutAll: ''';
	    nextPutAll: aString;
	    nextPutAll: '''.';
	    nl
    ]

    evaluateOn: anObject [
	"Evaluates the receiver to anObject"

	<category: 'evaluating'>
	^(self evaluateOn: anObject stream: String new writeStream) contents
    ]

    evaluateOn: anObject stream: out [
	"Evaluates the receiver to anObject"

	<category: 'evaluating'>
	^anObject perform: (self asSmalltalkCodeOn: anObject) with: out
    ]

    sttCode [
	"Returns the receiver's Smalltalk Template code"

	<category: 'accessing'>
	^sttCode
    ]

    asStringSelector [
	"Returns the selector used to show objects as Strings on the receiver"

	<category: 'accessing'>
	^asStringSelector
    ]

    asStringSelector: aSymbol [
	"Sets the selector used to show objects as Strings on the receiver"

	<category: 'accessing'>
	asStringSelector := aSymbol
    ]

    initializeOn: aString asStringSelector: aSymbol [
	<category: 'initializing'>
	sttCode := aString.
	asStringSelector := aSymbol.
	self initializeCache
    ]
]



WebResponse subclass: STTResponse [
    | stt |
    
    <comment: 'A WebResponse that uses STTTemplate to implement #sendBody.'>
    <category: 'Web-STT'>

    STTResponse class >> respondTo: aRequest with: aSTTTemplate [
	<category: 'responding'>
	(self new)
	    stt: aSTTTemplate;
	    respondTo: aRequest
    ]

    sendBody [
	<category: 'sending'>
	[self stt evaluateOn: self stream: responseStream] on: Error
	    do: 
		[:ex | 
		responseStream
		    << ex messageText;
		    nl;
		    << '<pre>'.
		Smalltalk backtraceOn: responseStream.
		responseStream
		    nl;
		    << '</pre>'.
		ex return]
    ]

    stt [
	<category: 'accessing'>
	^stt
    ]

    stt: aSTTTemplate [
	<category: 'accessing'>
	stt := aSTTTemplate
    ]
]



Servlet subclass: STTServlet [
    | stt |
    
    <comment: 'A Servlet that uses a STTResponse to implement #respondTo:.  Pass
a File, Stream, String or STTTemplate to its #stt: instance-side
method to complete the initialization of the servlet.'>
    <category: 'Web-STT'>

    respondTo: aRequest [
	<category: 'accessing'>
	STTResponse respondTo: aRequest with: self stt
    ]

    stt [
	<category: 'accessing'>
	^stt
    ]

    stt: aSTTTemplate [
	<category: 'accessing'>
	(aSTTTemplate isKindOf: File) 
	    ifTrue: 
		[self stt: aSTTTemplate readStream contents.
		^self].
	(aSTTTemplate isKindOf: Stream) 
	    ifTrue: 
		[self stt: aSTTTemplate contents.
		^self].
	(aSTTTemplate isKindOf: STTTemplate) 
	    ifFalse: 
		[self stt: (STTTemplate on: aSTTTemplate).
		^self].
	stt := aSTTTemplate
    ]
]



FileWebServer subclass: STTFileWebServer [
    | knownSTTs |
    
    <comment: 'A FileWebServer that uses STT to process .stt files.  Templates are
cached.'>
    <category: 'Web-STT'>

    initialize [
	<category: 'accessing'>
	super initialize.
	knownSTTs := LookupTable new
    ]

    fileResponse: file request: aRequest [
	<category: 'accessing'>
	| stt |
	('*.stt' match: file name) 
	    ifFalse: [^super fileResponse: file request: aRequest].
	stt := knownSTTs at: file name
		    ifAbsentPut: [STTTemplate on: file readStream contents].
	^STTResponse new stt: stt
    ]
]



WebServer class extend [

    publishMyFileSystem [
	"Watch out!! Security hole, they could steal /etc/passwd!!"

	"WebServer publishMyFileSystem"

	<category: 'testing'>
	| handler |
	self terminateServer: 8080.

	"Add a file server on a particular directory."
	handler := (self initializeServer: 8080) handler.
	handler addComponent: (STTFileWebServer named: 'disk' directory: '/')
    ]

]

PK
     
wB^fr      Haiku.stUT	 NQhux r      ErrorResponse class extend [

    haikuErrorMessages: aBoolean [
	<category: 'haiku'>
	aBoolean ifFalse: [self initialize] ifTrue: [self initialize: self haiku]
    ]

    haiku [
	<category: 'haiku'>
	^#(#(404 'Not found' '<P><BLOCKQUOTE><I>Rather than a beep<BR>
Or a rude error message,<BR>
These words: "File not found."</I></BLOCKQUOTE></P>

<P>The requested URL was not found on this server.</P>') #(410 'Gone' '<P><BLOCKQUOTE><I>You step in the stream,<BR>
but the water has moved on.<BR>
This page is not here.</I></BLOCKQUOTE></P>

<P>The requested resource is no longer available at the server and no
forwarding address is known. This condition should be considered
permanent.</P>') #(414 'Request-URI Too Long' '<P><BLOCKQUOTE><I>Out of memory.<BR>
We wish to hold the whole sky,<BR>
But we never will.</I></BLOCKQUOTE></P>

<P>The server is refusing to service the request because the requested
URL is longer than the server is willing to interpret. This condition
is most likely due to a client''s improper conversion of a POST request
with long query information to a GET request.</P>') #(503 'Service unavailable' '<P><BLOCKQUOTE><I>Stay the patient course<BR>
Of little worth is your ire<BR>
The network is down.</I></BLOCKQUOTE></P>

<P>The server is currently unable to handle the request due to a
temporary overloading or maintenance of the server. This is a temporary
condition.</P>'))
    ]

]



Eval [
    ErrorResponse haikuErrorMessages: true
]

PK    
wB4      edit.jpgUT	 NQhux r      wTWL `*(ӈ
EdaB XF(",1*+BZ! AV@V@bз9o}=>{   D PcSgН& D
@Qw *9S(.$,"*&K`JPÄpA5VP!o̅e}EB2EOֶvck(&.oC##&,ml\x=牻@ >,<"2szRrJjZzVvNy䗔?xH~DyZG5񢳫e#)ٹEoܵ
pskzʅpAp\d1 B	˘;ʪ'ʝ,mS7pa>O\^pjg7
|<VSjc/5hNt?Q(Ʀ!#QQ?|t.fX`ݤqzy)gvW(r'JfsGa{qa/8emB$(VFsB+qE"lHkῨ`tG}ݙ>!e_i/mJD8ׁ\i}mS-$>l(mlkW4W!{nbY#_7q|l\" ~$pߙQ))LZ%&:V$ɫAhCkc+z3Ry[+˘4-KmuBVy{4I׳/[MT9TTqy{^3ڽov:rrpzVF
W{m&lα~N&PX&]iۚmȕ+bU{K-Nu2>d	bh֥UyC}^^go9
F5`;lwLiV )sd؉j>R:|k1T%;OXr{cc$?o3@"^gct[~XsL 6Eva\7Sil}Ȫk.pd]ߗmԷFTX:}KWT0]4aAI6-6+jIc^XUVbK1J'|T!R,bEp95Ѩ~oQP4;;+Petvy<ѯЬf){@RJZ:o?Hrm6й珛BKJm?*7_ALK]:h-wkur}qkWkmDὝp3rYeƧF@RPX!goW/q_-Y]9=xNZw d8C8eb@3/`"1a
t16pL3%-muxj8TdRMvIkbcW*pӫSr`M%|ӵZ0tRr{OsQҷF6}r<cR5٣?X('ho-(cyp!dg.ܨ>?b&Dwē)&Z9g۲ܯp{QL` :|zthe>զLS{1iL-Oi3̀xx x]NJnٴUchwmq*x6~AKhV3?^ϋeb%
E	'ȱT6=9C\rtwbԡndHV}e	7U4e|r$ c `W(SYPK    
wB        example1.sttUT	 NQhux r      m!gx.$xqK3Gsq02@SiL4Pn6{(1Ͼ0/`1<VHVj6 -4[OisVRuK_l%v`E$v՘hkn^C~PK    
wBeR   U    example2.sttUT	 NQhux r      e?k0Sx	1 [V:IXg.mstxz3JZydg͵xAQJ(n*SZ@v;j0&H<	lnj5Ohz|)&"MO\(L~&>şTpׂ8,glM]\$wcSh-`Xt'}*$P0F7$7PK    
wB\      find.jpgUT	 NQhux r      y4kٻDrDVA(ג,s2Ҩ	ٙ[Ӥ-[SȒ2_fTsn~wy9yߗjg}@  "h [PG돶	d md7 EB`H(Ή'-U(jP8\,p$BJQ\HWxw+E"J}Ǌoߣwa+k#v.]=N?p:0

@7o%SQ<zT]S[W?G^e?LLNMa/-ZY]?orA ط+RapM.¦	G(I;	HED"UT>1.s{ρ/o`߹F	Dpy0$`lÞ3c!-YY;JoD7O*Uo|Mc<n+DC5<>$T]]ޠn=S: nm	'7E'QZ@0О$0%q1d\i϶bIT/L'Ttkn\r޻ܮ`Óo\
"\y08kUk
){,Gb.֧HК8y~dl	#GYK@2\ʼ{Qa$jݹEVgEǟ5Ѽ^~XƵĬr`߹]ERAݝهE͌^M*/F>8"zNcr9+q{hz[$yφ>/''V*Mt]sk,In cE	)t2XVѶ1WFݧ
xMZi5'g2HWݳ><ORcӦRҋOѠ&ÜY|*x:4F}Q fF8b7UQz-Y9¼f͔t	;GB:ݡwѬ-	A;v)3HU%FG\t-)4Ӽ!}QqRB)Ǒ],4qiĮϗ'Ģ<
=r/s&<o3J`rAW8#Je- _;Qڃ3Q`~\D
㌘lV}+ߡ6eo=FpZrW(fvdB4sԸGAz4K
He4⓰U\Bsf{JYf%O4W߹vvy7&Ft_ZՒp|5
¡F@?אT;_N!V7
f[g5(zE	]YwuwH=grs[ɳ
^g)
d{<޴pEWPUY+ZTZ5cUv9,&W`#'RC7n-|*.oJi劕0PK    
wBD]]#  %    head.jpgUT	 NQhux r      xwTo
G"-* -  p B	D#UAPC ADZN4QB{oΚ53k?{w3k	obhl;r;>`Y>eqwGO	8"
;*pC&ŶVu(Ǳ'8OrqZaGpp=qcc\z󄠥kOkdU<xIu
M-m[FMX;8x
	Ņ=OHLy2XQYm{cԩ34կ?A%u1cuhpęnמ<KF
0-,F
?{쿀7)#q0-6Ôe#P2ۛs 4vqԔM6-ԃ&|<~]@+5&ta	je'>@@>RfnS$]tܞ
QBB'Qip0-~L2]̊-vCiKd,;lZb$qh4+8C
3B6vcO-|uBBv
:>ӭ齺ymjntբ*Klww(q9Y7t%cV9:˱>id[}N(O-P42*
Q!ڡb?_ B\HG3ؚo,X2BU[f)c7NLLݧv1)
%u0ű{v^mT↚Lg!$;g>Ѧq7huSxo
:߼<N25&eo,xbwVj]ߊ(n^(!1Mf#ҍP̓tLIJI\k]GqP$S%OGɱOO3j#Qbv.*m{[&|]KSOuoჩ|(A
,XBѭFY8[횾?3Cqî̸<9fZdou,Gn;^|=+=}%jXskjv@br甌750;JiBr7;رD;](0҇>]g(×+.T7"Gp!dE,r
XAS}	MƇ>gy)^r9>a ! }se,Uԏ?A}dWۿN-Bvgv׌շH>68-x'(J`;~4?fRa5.% CZqyMT=B؋Wsѻp#kޚᯞ3kUDK嬵vOدJ2ڈѶ#aҒ-jQW˂Mz+
9LXZN8Sx9LnK`cIHIGQ#|j#/9	@s	
SH)1^[!ʛkڦ/50EԎBZ[7^):׻u;*|W*PP~ȋ=ߎAvcM)o\xaנ>ȢO䲂0/s	kZ^;P<Dɱ:ד*mӣkVd3.DZTrɁJG5F)	ѱ$@&}ɼ{G3Q%-Eoݏ

D&N7ƅ^O,&tA$'&DB9td::qh*RNh(W{r?IW8:5[vt4$tnUgZ|2r&{2;٢7:z'꺦mG2-DL}ÂrU(\x_ud%	#Tݿ]g\-u';-3f2_M<cX0OB\)
Mސ	B8}r-_f2/Rioҫ?'}s%4w}J&Boܸ*f,5GZN~
o.ydY}*P✙UAz)"|PU͔enzqb߆"<y=@7&Sp}1;> <ص XoRsUV|0	f>)@0.{I'"ie$7@`"m{={+@ZtWbYI[أ|l~1 3O:O:>?OQ1㘩ĳsO.>A+4R~S"6I
O_MmHEEIԶgܪ¦#B9|=c*[_2˴_Wh_)/`+0{ST'WÈdJل鍖Ȫ&LR.edqIznlPGR:o^cxTa/6M(!*]RWNV8=eqR]R"K'Шǰ;#_r(3"2`8Y[.7%uf0smu-fdR5z 86065%bWGSߞ?]pMoK'2^Z|аPYbanl+;6nuK-Zg5Xj y>u2`ė)*VUl۷uy{+cK#pу8ҘqgxO?.SCHA$	V(JX"aM{VoK3Q
#E93_p8ƯL+/j
ieo*<:6SHi'gߦteA 0@(=L_
;w&@/>VPldޗ@hߪ779X$ռqӋ6+%})0Hi(2_~kz#/θu%^58, )F fgB:fTzv86	slw~k)cIG\Ӡm]XJ6 @ݫAXho
0=^[#lz!{~cyFLa脁r?@V eHƀ2֪&t3A[n])v/O-z3:qqkYn @ݗ@V>CC/bxu`Pg6k䂋&==XGutL]MgXkSr$`>2C!kY^6.>d2?F^95n2IʟuN:e4oɜB]$b`%,8ɥ!$;ח*R@M~p?@]ԸR48ruЭYzʂ&tkn87-o?uU/kg]|gٻPN%(gֺ#x^.,̭=2;soy$*2O:4~풂{u/LwgRLZ˾n̀eRUiZ7NfZTب^
DQ<gW[ s@'Ue»%V]Xer:sz:~'3i$tz(N^*iɤzt?#IDySE3l<
Ȩ0uRoD«FTuݧֈKi|(nmOYM,$+\D"tk<%Zvн+՜x*#rz6x+KJ2ǂ-`O~=ېJ`|>mY(`	@
x%>[a41q|W(^yD}ɨֱnDo%~ï-d6Ri2*tbs!1JQ惢mL;3
ގ	O֏
~g4TjXp6iONvgNc~c[,Ewqh$yƵyuK9gF$G>oG%cL8 7*Q)=췟߄o@2>	@U	̚
]蝰;r`̺D[R<{VPVsCRh*UEMI,虚wZwG/ϟwh]hn6ta0?E* q'
HdHBO: <~DxT-]RZL}Cd/%,K/IٺEH[</6
*@I⏀D"hDw򷙏Ps+yNS+zB)sSok?_84LH<JZ'qKuZg-%PoRwIԇZ٦; !
rdg^voF|~]FPLde3)qE՘ԇɢ).<-!qZ7pF)~]ih=1qNŮ?T+s/W:{T?JHGw=$xf5V)ثkU,#7)?`
Kl`N
nz7]	4GD8Y^}#"nB$`1^yH'zE Y'{(BH+E
L2|l:oV;6E?|5e-e䮷
GB$(i
w}yt-lI>{O\3T5]} Oӯ9=CiV:! ܰoэ %[~eͅ
y3p܍%KX6BnW>#~&cmx)~zNmXT:k᭝_]2wp1o{J拗ԁV0P F鋼 فIec"xS)f6fefL׮3VsGB$"M'$pO/I	 &dJZa@[0nPyзK,2yRm,Fvk'-w^$ER^wCy<Ŋ^`r6vمs|(.ؾltTgc3υj+;=-̂	oN}hwm;KwFc4=wO:ec_+_'^RPӪ/:9<hrے8Xe`\>Vfવ,re䯾~s}{K_Xj}n2e(1Cu>_P:ϺT(1Q[ -QtQ2'[W֎ěZDٿQNIA-Ϥ(iB`v#wd}q~S8K1'56B u^n7Sϣ ۅ͓Ɯ֛_$,}eӃ/@	6>ر澱j	ճ
CMoY7a75|[RkĈ__z\e-jUTde-*Ү7/}4XAo&:ςdN+NT{@nޑ^si@N¥Իcj6J3;g^}/$	5	lH5qnEqrIَ~\W/|MV
бTRE@:pszl(5qF۠#BY˒pȁR5mN
1'z-  r\cIj3d槵3R J+\_
YFn'm\,+uPo{$\̽hq1$VTF?\&?k|nQ=)ȇ&DI<nQ4Ğ;
X:䙔7|옔%BY;4{PZNwǬ?؏<[#qtm4O2V.+;%~!.̂!O(=3Z0	:BG޸1\^[m ^r楀qi0/aTIa%pjtenmAC]#B.FOm%32>ٛ7Fid'%GFoͨ_,/ zVeGڟq'hUQ&+s]̵¯Cǻ?Bt;>P:Q'~GkW2'ϖc쭓+{:Dg-M$vXcC,A8ː9:@Nzh}Dܜ & ѳ|lhi,ZRe&_m#l5NxT'K
g@=pGԣ;hS2!R*6Rh/ŊjP$ A+w̹\#]/Mg(m1[vu<U۵͘
9U5 b%١3W~|q1UQ+J>X4}@QӋ%bTDg(d\
&
ɇcD=!9sd[lҺm8QctdڭA>y
d&1"4DY7mmIa2Z*j_N:|~72$Zn(iA%
> V_^bɧZs
==ѪCTr-ŦoU̰*+V4iPyf[чf|ypsmn[lMMRy`2AT11*7]W$9Pa>(L5Iq>h3-n'ױ;}=2?ayPWe%wT6ux[cJ.h|F}L{6st(ح
_ꘚᣐcXq.rwoY&_Z3E0jլww<_K(QhtQu2
X/|K%a#g[>mnf4!(®i0!fwml骃eʯgoYOz/}e0%%: ^/	K:OTjf\:IսM?~
~D=J	
<"gSҾA\oDdy?덝rmGvia{gīS:k_5
HaU'I$Jv
f(sSُ#+MLr'l|3)nH6Z,139e"Ur𪩇hlfVZc+b|9/BQ1g/[8+'_hZ&/R#A^>aTmkbKu?./KF\Oql9AĢ"BP2}_"f6L+tHRnȆ$Î=F.r;fi'0#{#wKEvJ`<<>q1~qߥ;9sT߽\C{tNr?{$.Ӆ3lz52u'.EG
OP:y
#)ĂB3ŸȑrL_CSkXGFw6,_T. Z96b̀H"˂qq&z2'$(E)0+O9q.\HL|ߕNxЎwHK͆Ç:]SDkKbv%޸j6QA
YoK_ղl֞;ׅn8/(Ny~6d#N
F;rJx4w?[2|^ L.SO$vUb#QH`NR+usV9"yZpUlsot|.:.|yco[WK84:M4jRj+s<mMdC4zNԮN'~71ז2s6fR|NHr<51uq%X¸ڭ)d#xXtvǡw_! aN:qNS|nKZGCuJWgÔ'?z
>7?w݊|͉{݆W&$'0Jyc(V9{,LDEݽƣ1=J6en-Is3x{zqi$QX.=u>VOy߶Dmwsr&AfҦʂy
B7{)@)RFZhNKjaHn4m	e/Fa1gGe4|Vke
ՆѲk7$=3Ghd2W8w]WOI`/ڤ:Ii572M4)1O2^wP 2-/Y :[C'ܴgg]_ z^n7)wAdDg-:Fb8	@d!xzN[v(`BpҠ;
M70<+7-k*FBobrgD<JG1EC9nHĢ'KWE-"jcdG3=4	T!#iTRu0('
:lVp]r\S38q]vJ$ޅ΃˽NwUȬڴ
$8>I԰ ZWc_BӰZ}Az>={ety)vZni.3eB?B%gζϗR˒?zVfbp:J3YHS9meq9Q:SܚgQ<PK    
wB)	ſm  1    help.jpgUT	 NQhux r      yPƿl+@W"$H E"E&e]BYBX-,	
D)E%(ɇARrciǝsg3g}S7@  "<`V[P0~ː[n<d 4`PYL"P=
#DD%$6 
p> EȩA0)S;u 9m/.!p谢)qgrEwo.W#GRnD%$&%ݺMUkj[Z]}GC9195=3Yxnqpͭm/\ W.Y!EpA_pQ9Qq؁S9{L܃~%ѦXE'_\3>Dy0Y iQű'd}bSFHְP ^&(c@iH4)a_2rFL0лR9as-Ff%>XLaZԧ-!vHIǜcVziAP]&]N[Ėkz?$m{̷ukTܓ@ҠRL,҅kEQM5#
]kq=6HHtis+dOgs1Z3-'xMRފ\T
fh})cC_+B\f_;{<Z3kuFCDtk͒fpUW(gK5XiuNܮۤ@IǱ2,<h@KXRuL{owZxռ,@ҪKWJ~CԄq	 +AbB޳RV(vt	+GR6~[ADI|V;X}y^[Z{#١,shwkUS{g\O0%_?"jq5p=OWͪ,r;Y\Y7 4)m${/^1Wb_ )}T4LOvn޶5ޘ$XCUS	RB'fE-y-> zprvO]|'urG&Zz+L0m +@T~>Ųp:ؔ(R\DC0EFiG3K<k2{I$F;i1{4}:X}S]ĭdGZ#'[^@3kUz
I!Ug'Kv2qt'KO8VJ5WoC"xr&l6B"ceRIb?£.3>K>( V~
{ssW F?t6kydM$EI =z8H H3ɕm]Kk~@	/=|lm53Th0&|	X*OM$q#dL9n68?"롯y^A(JqQZNɇVfx>7]v7;:h~~9|9u`&}N)K.t\]d:xstxޮnZ]^Y幥[
FSZFa2Ӂ\}>n<Ti갂XuǳuD!(fs&ɵN9w³Yo˿!\ҀBODjpsZ7]&g=+^:%ok8t<mEfє	7HK\'򮘇~򌝝][RXjmyF5zdƯmwϏ{f1'0Ý]],*㮑3׵,>Ff;J&p%tl_|B:%PK    
wB  >    history.jpgUT	 NQhux r      yTW!aUHٔF(Be52$E%TE@AvHJ( kdEY" F $93wy}I9  hov/ۿϛ~Zjd% - DdA`YP{$KHJ3" 0X@< S?~F+"{y6
{8ZFH8glbjv򔹭Y9W77^/]x+619y,?Q
OKJ5/j^3tt?
g>:;7?_dD 0D|$o;"8񃄼ͽfIMoH}XE؟\A0h5 1)Ƅm*bs&PҕeɳQN+|۴Ċx9qjdEKJxE;$B-',|zZr}Nj~Dv>Fr:5L55/2GpsZuٻc]KQS!SИ$5Z-J8^u>g%)NR)iA\w;\F7/O}tlQ|LT_,2w.ήEBXby	TL?m֦ļ(;wXIq骑aiC~AH^7J$[&=v(|BCg?7eژNX4]̾\ 7iH3VԶ_Q̌Rbp-7t{75vK
f*oGSfJː
'/*ZQ{רY	9F_}c0KrX<b:㶕b$P?Pgx,=P>es~HW+ߒ@EGVv&jSr$mNiwf{墨=8;zvr
<&nj#839pw)Ne#W D=!P8N}[4-ruS\M7NmJ{E%t-~Mlҫt&Rhr(bL6YVx1S>Y3~ע|UY8ɺOC1]Y&N^?A[zZ-qCFL!	_&.W ?WUk|3i#GXmL>4${F(Ȃެzqi(J~pnU8w2,/h̦5EɺNXP
q26R8eS*j\->!z٨Yz2;Ův
C$_=d7B::_ߤ[n-Pq.dnʜR*3f]5C{DDXuc*0[I'(6aPUSe`M?qMAgu#!fˬOsn<2UrmֲU\MэMłXn hTܢWZ>8zqq`oC;)]C{ʶ'ϱ$</=t4}$i^H7b׫/Vj}@
ȷf~e7]M$bYx{˚:w"3(LRj;rFExUByWUOhm,z	߂D-o}3KwP#SO.":1RTӜBo6W}*QǻfT튦$9{PhOӪUǓ,],u7F"/=#

PK    
wBf>      next.jpgUT	 NQhux r      {4ǿssBFeJku2
sI֤!VpPfU%Ke0R1aZKrl(f-ߞwo9y>y9
Dv   $?lpW[`<xk?kp%0 0dm|O
@a
J*rw; A `(B+> v[Sh별B%S@#1<[aA#6v1.Xiחg)/豗Mz-7sn%?pju
[mO?084|D<5ofo--_r\`(Ui[8o㘇b@Nv9eVD湨-C+t7?mbrن>2q%|
[8uÛB9C+KjأCb:X7#4xJ_ɚ:{q]]gkg'֮?ҌqMhG^\Y66iIݯ((qSPd.	/zO΅Դ
~*zI;|i'>6h:Ja;RPbJ+4([JK'qd ^Ě܅rLDW
DƺgyaTOw_h֘N]$mk1	wC3r@ xuzZ3'BNnbl(5t޸I$,:uesqG~xt͆"QƈCg(%O1}-'Vy6g9v:g_
BqOQv=K"m,f0)ƈ*qk6j:JOT}|1O*;=l4NU:C=*vaL`ʃM9xI tMdBa	v;hOѢRS+ɤ	嫣 S*kK,Ѩ
3z>oׄ}fSoGWNK&_IbZ GS2G1ϬÝI-fa(p_Ӥneu><rgjx?e<9e?[EC3vf+%܁AG-z8&Ad9c&L--C=sa<NfuNIw50b*@aHVu+X$FTZSkggwȀ(B^'𼾴=/3Mjr}>m=)<}f<V_Ynk[셺͞|Lz.98^a̭o"1{Ah{Xz\pǨ|HթY,M#	g܌d{nOߡeq	n_] k^l>q|-ls车,o+7%I4ߞ|Ln
44a.u7?\mGU\Pl+i[,!h1P]}]n]
0="+b;2O(/zx[E}QHIer줇䮈d߼I~<MT12P
eLktAUU$<cX3R1)-JS;nVlPK    
wBAiw  \    prev.jpgUT	 NQhux r      {4ǿsC$M$rJCLL.jY
!ɥ2&afg'
KX#e]]u)J25*(a}2Xgw{ssy|V6:80 :_#o!z@+  a4l[HE5Cf C H
Du ڲ^N3X~gyrNC^C.Q)շihjaw[⭬ٷAWo|}A=ultL,\\ꅋi2s~g\-,bW.^w/cτ"7Sogf?gu.+zD"\0xD0b)<YA!}/pɨ!E]":d,"o.![< +++2MBy!#lkҤ;ҁ2=P
1O:X>ٔp(,F4	o󮒰_/U'>vzBP J\I&p`{ 
:C!"xo5$6tt\BO,|/k֊D=.
g:Hj8A8/QWaRB3o0.kh|j>Y,PjT,qFvҠsMfc\9QsɋZ5K@;Aeg>˰}\M(($N|Yy5(>RD6Stt\Fh5YmuJoU=R8%@Wc/q:>_9o<Y,Jx埸&I*R[9N;oroŞ،cb[gikM'SƋ:6~-"Mj917zwOZ	
[vzN5E.tz;}RjM8;"Y15%ʽ:GSۊI#6{ s(a]C}׶'\SvrMo,E+x!d()Ua22ODp5ɻڐGWWnL$jd9@i_炡Dk:"R7lEDf,5|Xpô9K rÕKS"Brd@jz8>E1	xMqbo񞉴A5rZ`θ,ST
R2I`RDZ
ۥ(j:L jheɑiqV\_Wv	nCRÊBeb2LLfl~n>'2S-=T$q10y~AsX붢L p Z鶌HFƁ!y>ǂҺtw[P^|L#CDa{C-y/Ӣ'՘IôU@9>CԘaYPFY{*ϸr%;TR]d3廬L⪳eS#\KΑc#N|WU)T^1=}fF}Ί|pi"PYTdK $)g\˸
Yw9!(~QgВSʃA|{xH
ulrypp{{ځ4=$&(;g^GBJ|n1:Fmmؔ@?ZыٽngN=#Ӵ)<wڢi:6HkϪT	aԏ_c5cQe>/@]<ϰu-_i~,7-XŴ{'U]hVԔoeκuZCqb65M)&iL_$

-o瞪yV`z#;/n%-q>Rzч+[>yc%!~PK    
wB:R%&	  	  
  recent.jpgUT	 NQhux r      {Pgƿp"ʒRRQ@B`)
D@nЂ Ԡ#(&PAC m[IK@R4!Yv33=s9s~﫚P |<= `kPMkZ7OCl/K@j
90@oZ:k&@
!Z7{ zf{hlEi~ow^caXJ㭖ۿr=wAOC>C?~:y|aQWJqHŸvn>?d)?2:6>!Ē;yN:/[zO\0 W.Gh~₩e|2 f{447nvcms XʰR	WX}	JՍR`Z5DaKtNo}v"[l-%8KCj.OkߴXܺO	
ǞwRfzuF#KܨjeɠδU8^Z+8vYJi<en[J=0_Ewa9!EI:
+my֒^<n9:xX2VH:]uΕ6
*!4rQ
/(Ei*u}Dol8uЈKqegvh43a/6Zb0c\K37UjqEt;>E= V2G!^nj{ߞO=*wV:~M* LC9C* c6*uQyS#a(~@O%ϙ9lsbDHGY$D @ogʹxƃS"xa?m+B9s9<\a|2

BR3Rm9J?[t輏QZAeU]xѬL}h+=v*;+ej9+gfB)\v>*CYJA:
|DsZw[&+n]!c]}n©.oU4CE4,ŷ}ӌEZ&s3)`
,E2$c_pKTkG&^
_PL_T"z:WJ)/7
9c[JYM8a{rFIqJzVv
ڛELjQn>7s2ZcWT@
_Ȳ6(v=q+,c7bR_u)~qDruM,/JGqn4kZ	p{`'[ӱ#?@KM-83!܊+GgdT),Ϋ
$Ί~gC_CT%#1>bh[.ñ\im[Q6Ӄ6d}!Lw]\/L h:SMv>5^X;*t{9Zυڞdr0G#r("ļ9auR4-5ǜ:!`I^-pZKˉOexTmE8u5 U9Q[U )Su{AA|{- 0iLIӗS&_(`rwr/M,26c7PM7p<r8d<.24t^xpϘ15JJP nPPQM|oՇG+l~gun0|yM>m>zےK	7ӆdZ0znc*n_d[koo,RzF{pawVzR9ҝO&!q!b]DuQw3W\tg{LJߞ:O@L;nF"jG7&49k"W)-$ի/"Ť`SCJpIz#JjڝXU5FY/>ؑ^X~ץd#{
.VsMQpQ{A}\$wLntZc
Q0	z+AqH`MTJO a)$6PLRwz=yo46Y=]Edx7C=ڤp P
5D=aSB	(3DBő[dI/uC׶zA/FN?.{KP{3%8j5.t꟤_YoRfIwۃs~72qa8g$JoPK    
wBm#63    
  rename.jpgUT	 NQhux r      wP!TT]]@"	nhH]X%j`BKxQ
F6H};Ǜy;{9gwΜϽҷ)@	   q (Qol36ֿ &@8d 0$`{g,@2`[6d%@@F7e@zɩxjvbӞ?yr/\I߼E]Cs^}Fp#Vpt:<y!aWc032srn-.S)_U]CoyB"SZRi]/߼g~V0'YeWo\  ǃW.$|3@!z䔏yȟQmv}79)TpE]ߜg K l6>}]bQO']ou=fRQE#Kl+`viS~_>hE}5hM}a/sܞ
5+"zKIrRE49}0#6mg+`DtIA9,є;|JlOc3l',)36=zo$_ lZBB=o_ĊU3CGX%UpnHrM'5\o"!oSn+0L(?=}\j ewQ|>qahVk0jlnm^Y
.O/UP.w `BP
NwEL܄ }Gڴ6;8{GA\"MIxB]_\֠|>pgk[WI	j8]_r}ć=IPWn
u5&ΉRL?jcnwSqgDܹ#.]Yhh$$tNTy5NzJ O,I7*/)\n/sXPξ}!2af	7ёEL;R?ߓ[d%Zț1/
kI$'EF(m|'}s5`:yd䴂]DkN*5j.D?=*Jɞ+HSBV*\.,@(cPB},zÁHWvOGU(:m,{:Ir++َ>_z(ɚNIJW}yDdneZ5ՄAT7qP(i(MIUb
lx\0DoZuįE:Xց8V'z~f~L!	x?glт)wLZүo8+_VTNV64x1Lw6/Xr팬S*Q`"0e)B.CAWչCnA泏>R] yye!$ 0y)pv ^a m~PWXf+!E-2ĴPz/l1˛|WNk1LtOY9Z
CL;@EqZ^v릻Gi^/}zF6CHͺn9޺sF:مÁ3-#~4-K[XdTh[W=.mU	r 	Les8kaJL>Mi$WF+âKߏ?0YA*zjBW$m43c^́rRs,ٰp9\E-N〤v8@nW~L^,"^\*Hb$ԋg:kۑ"/WTU[-H@ѳ'/زz1>\1 hy[5:OVh!<|=dgO	pߘ3H
.lJv&r
Vt;)/k'|E߫	Fظ,:_
]v>br1LESNLwz8n>
ZV5
oX6x C	īґPK
     
wBvj      test.stUT	 NQhux r      Eval [
    PackageLoader fileInPackage: 'WebServer'
]



Namespace current: NetClients.WikiWorks [
    Smalltalk arguments do: [:each | FileStream fileIn: each]
]



Eval [
    ObjectMemory snapshot
]



Namespace current: NetClients.WikiWorks [
    WebServer publishMyFileSystem.
    "WebServer initializeWiki."
    "WebServer restartWikiNoImages."
    "WebServer restartWiki."

    Processor activeProcess suspend
]

PK    
wB\A      top.jpgUT	 NQhux r      }4{{"YI9f.qWN
Ja6%yh,-Is=l*Oa#k7Tsn?{y|_߯CX4  H]ت>`ѷ<7  A8HU,UX@`3O`B`0(TP _C/X2TJgI"|.$"^[d+ml;9ou
َ@صoCEFĞN`9{|bj߯rrynۥx|AraMm]}C'/[D:^%Ҿ7o'>L*>*4 __j.0
jqs8lfp)-Cϔjmux٢#d|3ُ/`߸Ā.<pfgW$bd/6pztTlL;ȀfL4\J&unm{3ߣ-I#N	ZaH
tӦUfWffsBVMTCJN(&qxwkBڅfT\_jPDːtJc\1sm'd}WH~K6=L{KuѤsɡD#XSr$w9׎i_8$xUG݊OOhw
>cܤCb{	~.-gsEAD96BOf
7#Ov}dNcd"-wXI®[F%דƐձOg1.Sgx"VI^!JncHJ+2yY%lR/hht3$Q<žGWjiU]ZJuO-/
5kR{c{xLlA{8-Mun)rK] ߼1iEP(g 5%|EqLB|&dQ$ɐO>WziB{t ={v|/c]֦e7Ӻp~&\WBڐZC\q݂e/iFr`ڦy8oU-7 !cJd:VB|UF5L%3iiauwY	0g6p1)A*X
Ƭ<‰plz`qa:Kj;(`L8MU#/-'0?0b3tSaEaS_f/21I3w7Cb@E6Pè
o)%z#SdK̘ވDN!*PstHG$8Q	gO.oms+=^F"ٲnP;o8?C~m.">f22a >&fkOlV~/m9'=%kOjY>Wo +̻W؃Ĩ6GpϝKQv90`dbW!b?x%/3L"RqMz7PK    
wB¦`    	  ChangeLogUT	 NQhux r      Qk0_qAKڹMEĹ1Y>m$Vtﻻ$b!g=`J[*	0Z$UU@:{\d<`[CĂ6j0UzWTdc<5EE~8xs1
!XTr%_r	V`i%V>QQcK&MZ*fZvג
={[[^}Ç.Ϩ_vP@^O؞#_mUX rn6zɳ{2Z_W^lƭu!$RJ% (.s7F!FXoȋvl\WэhÉ"U3u2N+~ PK
     JU[p  p                package.xmlUT hux r      PK
     
wB|E  E              WebServer.stUT NQux r      PK
     
wBjk|^  |^  
          @  FileServer.stUT NQux r      PK
     
wBSYa    
            WikiServer.stUT NQux r      PK
     
wB(9#&  #&            X STT.stUT NQux r      PK
     
wB^fr               Haiku.stUT NQux r      PK    
wB4               edit.jpgUT NQux r      PK    
wB               3 example1.sttUT NQux r      PK    
wBeR   U            example2.sttUT NQux r      PK    
wB\              2 find.jpgUT NQux r      PK    
wBD]]#  %            m head.jpgUT NQux r      PK    
wB)	ſm  1             help.jpgUT NQux r      PK    
wB  >            L history.jpgUT NQux r      PK    
wBf>              " next.jpgUT NQux r      PK    
wBAiw  \            ") prev.jpgUT NQux r      PK    
wB:R%&	  	  
          0 recent.jpgUT NQux r      PK    
wBm#63    
          i: rename.jpgUT NQux r      PK
     
wBvj              B test.stUT NQux r      PK    
wB\A              D top.jpgUT NQux r      PK    
wB¦`    	         IK ChangeLogUT NQux r      PK      5  L   