#!/usr/pkg/bin/perl -wT
#
# ==========================================================================
#
# ZoneMinder WatchDog Script, $Date$, $Revision$
# Copyright (C) 2001-2008 Philip Coombes
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# ==========================================================================

=head1 NAME

zmwatch.pl - ZoneMinder WatchDog Script

=head1 SYNOPSIS

zmwatch.pl

=head1 DESCRIPTION

This does some basic setup for ZoneMinder to run and then periodically
checks the fps output of the active daemons to check they haven't 
locked up. If they have then they are killed and restarted

=cut
use strict;
use bytes;

# ==========================================================================
#
# These are the elements you can edit to suit your installation
#
# ==========================================================================

use constant START_DELAY => 30; # To give everything else time to start

# ==========================================================================
#
# Don't change anything below here
#
# ==========================================================================

# Include from system perl paths only
use ZoneMinder;
use POSIX;
use DBI;
use autouse 'Data::Dumper'=>qw(Dumper);

$| = 1;

$ENV{PATH}  = '/usr/pkg/bin:/bin:/usr/bin';
$ENV{SHELL} = '/bin/sh' if exists $ENV{SHELL};
delete @ENV{qw(IFS CDPATH ENV BASH_ENV)};

logInit();
logSetSignal();

Info( "Watchdog starting\n" );
Info( "Watchdog pausing for ".START_DELAY." seconds\n" );
sleep( START_DELAY );

my $dbh = zmDbConnect();

my $sql = $Config{ZM_SERVER_ID} ? 'SELECT * FROM Monitors WHERE ServerId=?' : 'SELECT * FROM Monitors';
my $sth = $dbh->prepare_cached( $sql )
    or Fatal( "Can't prepare '$sql': ".$dbh->errstr() );

while( 1 )
{
    my $now = time();
    my $res = $sth->execute( $Config{ZM_SERVER_ID} ? $Config{ZM_SERVER_ID} : () )
        or Fatal( "Can't execute: ".$sth->errstr() );
    while( my $monitor = $sth->fetchrow_hashref() )
    {
        next if $monitor->{Function} eq 'None';
        my $restart = 0;
        if ( zmMemVerify( $monitor ) ) {
            # Check we have got an image recently
            my $image_time = zmGetLastWriteTime( $monitor );
            if ( !defined($image_time) ) {
				 # Can't read from shared data
				Debug( "LastWriteTime is not defined." );
        zmMemInvalidate( $monitor );
				next;
			}
            if ( !$image_time ) {
				# We can't get the last capture time so can't be sure it's died.
				Debug( "LastWriteTime is = $image_time." );
        zmMemInvalidate( $monitor );
				next;
			}

            my $max_image_delay = ( $monitor->{MaxFPS}
                                    &&($monitor->{MaxFPS}>0)
                                    &&($monitor->{MaxFPS}<1)
                                  ) ? (3/$monitor->{MaxFPS})
                                    : $Config{ZM_WATCH_MAX_DELAY}
            ;
            my $image_delay = $now-$image_time;
            Debug( "Monitor $monitor->{Id} last captured $image_delay seconds ago, max is $max_image_delay\n" );
            if ( $image_delay > $max_image_delay )
            {
                Info( "Restarting capture daemon for "
                      .$monitor->{Name}.", time since last capture $image_delay seconds ($now-$image_time)\n"
                );
                $restart = 1;
            }
        }
        else
        {
            Info( "Restarting capture daemon for ".$monitor->{Name}.", shared data not valid\n" );
            $restart = 1;
        }

        if ( $restart )
        {
            my $command;
            if ( $monitor->{Type} eq 'Local' )
            {
                $command = "zmdc.pl restart zmc -d $monitor->{Device}";
            }
            else
            {
                $command = "zmdc.pl restart zmc -m $monitor->{Id}";
            }
            runCommand( $command );
        }
        elsif ( $monitor->{Function} ne 'Monitor' )
        {
            # Now check analysis daemon
            $restart = 0;
            # Check we have got an image recently
            my $image_time = zmGetLastReadTime( $monitor );
            if ( !defined($image_time) ) {
                # Can't read from shared data
                $restart = 1;
                Error( "Error reading shared data for $$monitor{id} $$monitor{Name}\n");
            } elsif ( !$image_time ) {
                # We can't get the last capture time so can't be sure it's died.
                $restart = 1;
                Error( "Error getting last capture time for $$monitor{id} $$monitor{Name}\n");
            } else {

                my $max_image_delay = ( $monitor->{MaxFPS}
                                        &&($monitor->{MaxFPS}>0)
                                        &&($monitor->{MaxFPS}<1)
                                      ) ? (3/$monitor->{MaxFPS})
                                        : $Config{ZM_WATCH_MAX_DELAY}
                ;
                my $image_delay = $now-$image_time;
                Debug( "Monitor $monitor->{Id} last analysed $image_delay seconds ago, max is $max_image_delay\n" );
                if ( $image_delay > $max_image_delay )
                {
                    Info( "Analysis daemon for $$monitor{id} $$monitor{Name} needs restarting,"
                          ." time since last analysis $image_delay seconds ($now-$image_time)\n"
                    );
                    $restart = 1;
                }
            }

            if ( $restart )
            {
                Info( "Restarting analysis daemon for $$monitor{id} $$monitor{Name}\n");
                my $command = "zmdc.pl restart zma -m ".$monitor->{Id};
                runCommand( $command );
            } # end if restart
        } # end if check analysis daemon
        # Prevent open handles building up if we have connect to shared memory
        zmMemInvalidate( $monitor );
    } # end foreach monitor
    sleep( $Config{ZM_WATCH_CHECK_INTERVAL} );
} # end while (1)
Info( "Watchdog exiting\n" );
exit();
