//
// $Id: BzlibInStream.h,v 1.12 2007/04/16 01:48:15 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(BZLIBINSTREAM_OL_GUARD)
#define BZLIBINSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredInStream.h>

/**
 * @class OLBzlibInStream BzlibInStream.h ObjectiveLib/BzlibInStream.h
 *
 * A stream for bzlib decompression. This is a concrete layered stream that decompresses all
 * data read using the library bzlib. It is suitable for reading data compressed with
 * OLBzlibOutStream.
 *
 * @note OLBzlibInStream is suitable for reading data written using the command-line tool
 * @c bzip2.
 *
 * @sa OLZlibOutStream
 *
 * @ingroup Streams
 */
@interface OLBzlibInStream : OLLayeredInStream
{
@protected
    /**
     * The opaque data type used by bzlib to represent the stream
     */
    void*               bzstream;

    /**
     * The buffer required by bzlib
     */
    uint8_t*            buffer;

    /**
     * The size of the bzlib buffer
     */
    unsigned            bufferSize;
}

/**
 * Create and return a newly created stream. The stream is created with a buffer size
 * of 1,024. The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @return a reference to a newly created stream
 */
+ (id) streamWithInStream: (OLInStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The stream is created with a buffer size
 * of 1,024. The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream;

/**
 * Initialize the stream. The stream is created with the given buffer size.
 * The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @param size the buffer size
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size;

/**
 * Initialize the stream. The stream is created with the given buffer size.
 * The given stream @a underStream will become
 * the next layer down in the stack of streams.
 *
 * The parameter @a small should be set to YES if the bzlib library should try to
 * conserve memory. All other initializers set this value to NO.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @param size the buffer size
 * @param small should be YES if the library should try to conserve memory, NO otherwise
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size smallMode: (BOOL)sm;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

- (void) close;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Read a sequence of bytes from the stream. Bytes are decompressed up to the value
 * of @a max and the number decompressed is returned. If no more bytes are available
 * from the stream, then UINT_MAX is returned.
 *
 * @exception OLInputOutputException if there is an error reading from the stream
 * @param dest the address to which the bytes should be read
 * @param max the maximum number bytes that should be read
 * @return the number of bytes read or UINT_MAX if no bytes could be read
 */
- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max;

@end

#endif
