/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.cram.ref;

import htsjdk.samtools.Defaults;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.cram.io.InputStreamUtils;
import htsjdk.samtools.cram.ref.CRAMLazyReferenceSource;
import htsjdk.samtools.cram.ref.CRAMReferenceSource;
import htsjdk.samtools.cram.ref.GaveUpException;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFile;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

public class ReferenceSource
implements CRAMReferenceSource {
    private static final Log log = Log.getInstance(ReferenceSource.class);
    private final ReferenceSequenceFile rsFile;
    private int downloadTriesBeforeFailing = 2;
    private final Map<String, WeakReference<byte[]>> cacheW = new HashMap<String, WeakReference<byte[]>>();
    private static final Pattern chrPattern = Pattern.compile("chr.*", 2);

    public ReferenceSource(File file) {
        this(IOUtil.toPath(file));
    }

    public ReferenceSource(Path path) {
        this(path == null ? null : ReferenceSequenceFileFactory.getReferenceSequenceFile(path));
    }

    public ReferenceSource(ReferenceSequenceFile rsFile) {
        this.rsFile = rsFile;
    }

    public static CRAMReferenceSource getDefaultCRAMReferenceSource() {
        if (null != Defaults.REFERENCE_FASTA) {
            if (Defaults.REFERENCE_FASTA.exists()) {
                log.info(String.format("Default reference file %s exists, so going to use that.", Defaults.REFERENCE_FASTA.getAbsolutePath()));
                return new ReferenceSource(Defaults.REFERENCE_FASTA);
            }
            throw new IllegalArgumentException("The file specified by the reference_fasta property does not exist: " + Defaults.REFERENCE_FASTA.getName());
        }
        if (Defaults.USE_CRAM_REF_DOWNLOAD) {
            log.info("USE_CRAM_REF_DOWNLOAD=true, so attempting to download reference file as needed.");
            return new ReferenceSource((ReferenceSequenceFile)null);
        }
        return new CRAMLazyReferenceSource();
    }

    private byte[] findInCache(String name) {
        byte[] bytes;
        WeakReference<byte[]> weakReference = this.cacheW.get(name);
        if (weakReference != null && (bytes = (byte[])weakReference.get()) != null) {
            return bytes;
        }
        return null;
    }

    private byte[] addToCache(String sequenceName, byte[] bases) {
        for (int i = 0; i < bases.length; ++i) {
            bases[i] = StringUtil.toUpperCase(bases[i]);
        }
        this.cacheW.put(sequenceName, new WeakReference<byte[]>(bases));
        return bases;
    }

    @Override
    public synchronized byte[] getReferenceBases(SAMSequenceRecord record, boolean tryNameVariants) {
        String name = record.getSequenceName();
        byte[] bases = this.findInCache(name);
        if (bases != null) {
            return bases;
        }
        String md5 = record.getAttribute("M5");
        if (md5 != null) {
            bases = this.findInCache(md5);
            if (bases != null) {
                return bases;
            }
            bases = this.findInCache(md5.toLowerCase());
            if (bases != null) {
                return bases;
            }
            bases = this.findInCache(md5.toUpperCase());
            if (bases != null) {
                return bases;
            }
        }
        if ((bases = this.findBasesByName(record.getSequenceName(), tryNameVariants)) != null) {
            return this.addToCache(record.getSequenceName(), bases);
        }
        if (Defaults.USE_CRAM_REF_DOWNLOAD) {
            if (md5 != null) {
                bases = this.findBasesByMD5(md5.toLowerCase());
            }
            if (bases != null) {
                return this.addToCache(md5, bases);
            }
        }
        return null;
    }

    private byte[] findBasesByName(String name, boolean tryVariants) {
        if (this.rsFile == null || !this.rsFile.isIndexed()) {
            return null;
        }
        ReferenceSequence sequence = null;
        try {
            sequence = this.rsFile.getSequence(name);
        }
        catch (SAMException sAMException) {
            // empty catch block
        }
        if (sequence != null) {
            return sequence.getBases();
        }
        if (tryVariants) {
            for (String variant : this.getVariants(name)) {
                try {
                    sequence = this.rsFile.getSequence(variant);
                }
                catch (SAMException e) {
                    log.warn("Sequence not found: " + variant);
                }
                if (sequence == null) continue;
                return sequence.getBases();
            }
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private byte[] findBasesByMD5(String md5) {
        String url = String.format(Defaults.EBI_REFERENCE_SERVICE_URL_MASK, md5);
        int i = 0;
        while (i < this.downloadTriesBeforeFailing) {
            try (InputStream is = new URL(url).openStream();){
                if (is == null) {
                    byte[] byArray = null;
                    return byArray;
                }
                log.info("Downloading reference sequence: " + url);
                byte[] data = InputStreamUtils.readFully(is);
                log.info("Downloaded " + data.length + " bytes for md5 " + md5);
                String downloadedMD5 = SequenceUtil.calculateMD5String(data);
                if (md5.equals(downloadedMD5)) {
                    byte[] byArray = data;
                    return byArray;
                }
                String message = String.format("Downloaded sequence is corrupt: requested md5=%s, received md5=%s", md5, downloadedMD5);
                log.error(message);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            ++i;
        }
        throw new GaveUpException("Giving up on downloading sequence for md5 " + md5);
    }

    private List<String> getVariants(String name) {
        boolean chrPatternMatch;
        ArrayList<String> variants = new ArrayList<String>();
        if (name.equals("M")) {
            variants.add("MT");
        }
        if (name.equals("MT")) {
            variants.add("M");
        }
        if (chrPatternMatch = chrPattern.matcher(name).matches()) {
            variants.add(name.substring(3));
        } else {
            variants.add("chr" + name);
        }
        if ("chrM".equals(name)) {
            variants.add("MT");
        }
        return variants;
    }

    public int getDownloadTriesBeforeFailing() {
        return this.downloadTriesBeforeFailing;
    }

    public void setDownloadTriesBeforeFailing(int downloadTriesBeforeFailing) {
        this.downloadTriesBeforeFailing = downloadTriesBeforeFailing;
    }
}

