/*
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2006 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2015)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/

package GoGrinder;

import java.io.*;
import java.util.*;
import javax.swing.JOptionPane;
import java.nio.charset.*;

/**
 *
 * @author  tkington
 * @author  Ruediger Klehn
 */
 // seemingly has always the data for the current problem: NAME.sgf, NAME.dat, stats-data and tags (but not the sgf-data)
public class ProbData implements Comparable {

    private File file;
    private int num;
    
    private ProbCollection parent;
    private int timesTried;
    private int timesRight;
    private long totalTime;
    private HashSet tags;
    private long lastTried;
    private String msg = "";
    private static final String NL = Main.NEW_LINE;
    private boolean loaded;
    private boolean exporting = false;
    public CharsetWorks charsetW;
    
    public ProbData() { /* Just for serialization */ }
    
    /** Creates a new instance of ProbData */
    public ProbData(ProbCollection parent, File f) {
        
        file = f;
        this.parent = parent;
        tags = new HashSet();
        lastTried = 0;
        loaded = false;
        charsetW = new CharsetWorks(file);
    }
    
    // #### TAKE CARE #### we read possibly another encoding as when writing!
    public void export(ObjectOutputStream out) throws IOException { // called by an iterator, 
      exporting = true; // this way getSGF should go around using found or set charsets
      // better of course would be to write the encoding (utf-8) into the file ( CA[UTF-8] )
      // export sgf + tags to ~zip file, for e.g. another user  
      File fileNameRelativeF = null;
      int pathToSettingsLength = Main.pathToSettings.length() + 1; 
                // c:\Users\Axel\MyGoGrinderSettings\problems\veryeasy\easy1.sgf
                //                       cut before "problems"
    	load();
      
      fileNameRelativeF = new File(file.getPath().substring(pathToSettingsLength)); // -> "problems\veryeasy\easy1.sgf"

      out.writeObject(fileNameRelativeF);
      out.writeLong(file.lastModified());
      out.writeObject(getSGF()); // ########### TAKE CARE ############### // maybe getSGFIgnoreEncoding()?
      out.writeObject(tags);     // we read possibly another encoding then write!
      exporting = false; 
    }
    
    public double getAvgTime() {
    	load();
    	
        if(timesRight == 0)
            return 60000;
        return (double)totalTime / timesRight;
    }
    
    private String getDatName() { // it seems, here is the place to handle stats for shared problems
        String fileNameBody = file.getName().substring(0, file.getName().length() - 4);
        String statsFilePath = getStatsParent() + Main.SLASH + fileNameBody + ".dat";
        return statsFilePath;
    }

    private String getStatsParent(){
      String path = file.getPath();
      String subPath = path.substring(Main.pathToProblems.length(), path.length() - (file.getName().length() + 1) ); // clean
      return Main.pathToStats + subPath;
    }

    private void load() {
    	if(!loaded)
    		loadStats();
    }
    
    private static final int STATREVISION = 1;
    public void loadStats() { // try to read corresponding .dat file in stats folder
        try {                 // can this be used to rebuild the tags pool?
            File f = new File(getDatName());
            
            File parentFile = f.getParentFile();
            if(!parentFile.exists()) {
                if(!parentFile.mkdirs()){
                  String msg = Messages.getString("couldnt_create_dir")+ " " + parentFile.getPath(); 
                  JOptionPane.showMessageDialog(null, msg);
                  ExceptionHandler.logSilent(msg);
                }
            }
            
            ObjectInputStream in = new ObjectInputStream(new FileInputStream(f));
            
            int rev = in.readInt();
            if(rev > STATREVISION) { // so we got newer stats files with changed composition
                msg = Messages.getString("err_data_file_newer"); 
                JOptionPane.showMessageDialog(null, msg); 
                Main.logSilent(new Exception(msg));
                System.exit(-1); // I don't like this - why not try to go on without 
                                 //   this content and don't allow to write?
                                 //   (writing would destroy the content: the given 
                                 //   tags and the results of the tries)
            }                    //   ... and we get a message: "Seemingly there are stats files, which have been
                                 //                                written by a new program version -> get it!"
            
            timesTried = in.readInt();
            timesRight = in.readInt();
            totalTime = in.readLong();
            tags = (HashSet)in.readObject();
            lastTried = in.readLong();
            in.close();
            
            loaded = true;
        }
        catch(FileNotFoundException e) {
            //  No stats for this problem yet
        	
        	//	Don't try to load file again
        	loaded = true;
        }
        catch(Exception e) {
            JOptionPane.showMessageDialog(null, Messages.getString("err_reading") + " " + getDatName());  
            Main.logSilent(e);
        }
    }
    
    public void probDone(boolean right, long time) {
    	load();
    	
        timesTried++;
        if(right) {
            timesRight++;
            totalTime += time;
        }
        
        lastTried = System.currentTimeMillis();
        parent.probDone(right, time);
        saveStats();
    }
    
    public void resetStats() { // called by an iterator
    	load();
    	
        totalTime = timesTried = timesRight = 0;
        saveStats();
    }
    
    public void saveStats() { // save status to corresponding .dat file in stats folder
    	load();                
    	
        String filename = getDatName();
        try {
            ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(filename));
            out.writeInt(STATREVISION);
            out.writeInt(timesTried);
            out.writeInt(timesRight);
            out.writeLong(totalTime);
            out.writeObject(tags);
            out.writeLong(lastTried);
            out.close();
        }
        catch(IOException e) {
            JOptionPane.showMessageDialog(null, Messages.getString("err_writing") + " " + filename);  
            Main.logSilent(e);
        }
    }
    
    public void getSelectionStats(SelectionStats s) {
    	load();
        s.addStats(timesTried, timesRight, totalTime);
    }
    
    public String getSGF() { // used only by ProbData.export() + SGFController (handed through to SGFParser.parse())
      if (!file.exists()){
        String stackTop = NL + "Happened where (~): " + new Exception().getStackTrace()[0];
        msg = "File not found (moved?):" + NL
           + file.getPath() 
           + NL + "We use the default code.";
        if (Main.TEST) msg = msg + stackTop; 
        JOptionPane.showMessageDialog(null, msg);
        return Main.SGF_DEFAULT_CODE; // we need to return sgf code here
      }
      long fileSize = FileWorks.getFileSize(file.toString());
      if (fileSize > GS.getParserSGFMaxFileSize() ){
        String stackTop = NL + "Happened where (~): " + new Exception().getStackTrace()[0];
        msg = "File: \"" + file.toString() + NL 
            + "File is very big: " + fileSize + " bytes. We bypass this file and use " + NL 
            + "the default problem data. You can change the maximum size for sgf files " + NL
            + "manually in the settings file or by the system variable MYGG_SGF_MAX_SZ.";
        if (Main.DEBUG4) msg = msg + stackTop; 
        GoGrinder.sgf.SGFLog.SGFLog(msg);
        JOptionPane.showMessageDialog(null, msg);
        return Main.SGF_DEFAULT_CODE;  // we need to return sgf code here
      }
    	load(); // loads the stats file (stats/5k/prob004.dat for problems/5k/prob004.sgf)
      String useCharset = "";
      if (exporting)useCharset = Charset.defaultCharset().toString(); // ####### LOOK OUT !! ########
      else {
        useCharset = charsetW.getCharset();
 if(Main.DEBUG7)d.b.g(useCharset);
        charsetW.setUseCharset(useCharset);
      }
// if(Main.DEBUG7)d.b.g(useCharset + "\n" + charsetSGF.file.toString() + "\n" + file.toString());
      String sgfFromFile = FileWReadWrite.readFileStrCharset(file, useCharset);
      if (sgfFromFile.equals("")){
        String stackTop = NL + "Happened where (~): "+ new Exception().getStackTrace()[0];
        sgfFromFile = Main.SGF_DEFAULT_CODE;
        msg = "Error while reading the file - we use the default data.";
        if (Main.DEBUG4) msg = msg + stackTop;
        JOptionPane.showMessageDialog(null, msg);
        // we don't do any file moving here
        // but we could delete the file from the Selection (ArrayList<ProbData>)
      }
      return sgfFromFile;
    }
    
    public void addTag(String t) { // this could be used for merge tags?
        load();
        if(tags.add(t))
            saveStats();
    }
    
    public HashSet getTags() { // use this to rebuild tags pool?
    	load();
    	return tags;
    }
    
    public boolean matchesTags(ArrayList t, int matchType) {
    	load();
    	
        switch(matchType) {
            case Selection.MATCH_ALL:
                for(int i = 0; i < t.size(); i++) {
                    if(!tags.contains(t.get(i)))
                        return false;
                }
                return true;
            case Selection.MATCH_ANY:
                for(int i = 0; i < t.size(); i++) {
                    if(tags.contains(t.get(i)))
                        return true;
                }
                return false;
            case Selection.MATCH_NONE:
                for(int i = 0; i < t.size(); i++) {
                    if(tags.contains(t.get(i)))
                        return false;
                }
                return true;
        }
        
        throw new RuntimeException();
    }
    
    public void removeTag(String t) {
    	load();
    	
        if(tags.remove(t))
            saveStats();
    }
    
    //  This could only save the parent dir, the problem number, and filename so that 
    //  we can find this problem again
    private static final int REVISION = 1;
    public void readData(ObjectInput in) throws IOException, ClassNotFoundException { // -> Externizable!
    	//	No load here
        int rev = in.readInt();
        if(rev > REVISION) {
            msg = Messages.getString("err_data_file_newer"); 
            JOptionPane.showMessageDialog(null, msg);
            Main.logSilent(new Exception(msg));
            System.exit(-1);
        }
        
        file = (File)in.readObject(); // here we read from grind.dat
        if(!file.getPath().equals(file.getAbsolutePath()) || !file.exists()){
        // if the grind.dat is of a previous version of Grinder (2.0...2.1), we 
        // want to make sure, that we don't add path parts where they are not needed
        // maybe we need to check, if Main.pathToSettings contains the string "problems" (lowercase) 
        // and then we need to work around that
          int posProblemsInFilePath = file.getPath().indexOf("problems");
          if (posProblemsInFilePath < 0) d.b.g("UHHH - " + new Exception().getStackTrace()[0]); 
            // shouldn't happen: currently "problems" is always part of the path of the sgf
          String fileStr = Main.pathToSettings + Main.SLASH + file.getPath().substring(posProblemsInFilePath);
          // if the path was c:\Users\Peter\GrinderSettings\problems\3k\probl-3224.sgf
          // it gives back -> problems\3k\probl-3224.sgf and adds the current path to 
          // settings, which can, but needn't be the same pathToSettings
          file = new File(fileStr);
        }
        // we could (for the next some subversions) look, if (file.getPath == file.getCanonicalPath) - or equals()?
        // and change this - if needed - to the full path
        // uses system-dependent default name-separator character
        num = in.readInt();
    }
    
    public void writeData(ObjectOutput out) throws IOException { // -> Externizable!
    	//	No load here
        out.writeInt(REVISION);
        int pathToSettingsLength = Main.pathToSettings.length() + 1; // no slash/backslash at the end
        String fileStr = file.getPath().substring(pathToSettingsLength);
          // if the path was c:\Users\Peter\GrinderSettings\problems\3k\probl-3224.sgf
          // it gives back -> problems\3k\probl-3224.sgf
        File fileF = new File(fileStr);
        out.writeObject(fileF); // HERE BACK TO RELATIVE PATH
        out.writeInt(num);
    }
    
    public int compareTo(Object o) {
    	//	No load here - only compares file
        ProbData p = (ProbData)o;
        return file.compareTo(p.file);
    }
    
    public File getFile() {
    	//	No load here - file not in stats
    	return file;
    }
    
    public int getNum() {
    	//	No load here - num not in stats
    	return num;
    }
    
    public void setNum(int n) {
    	//	No load here - num not in stats
    	num = n;
    }
    
    public long getLastTried() {
    	load();
    	return lastTried;
    }
    
    public void setTags(HashSet tags) {
    	load();
    	this.tags.clear();
    	this.tags.addAll(tags);
    }
}
