package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````(`\P`!````\!P!``````!``````````+B6````
M````!0```$``.``(`$``&P`:``8````$````0`````````!```$``````$``
M`0``````P`$```````#``0````````@``````````P````0``````@``````
M```"`0````````(!```````7`````````!<``````````0`````````#``!P
M!````'R5`````````````````````````````%,````````````````````!
M``````````$````%``````````````````$``````````0``````#)``````
M```,D``````````0`````````0````8````0D````````!"@`0``````$*`!
M```````H!````````#@'`````````!`````````"````!@```""0````````
M(*`!```````@H`$``````+`!````````L`$````````(``````````0````$
M````&`(````````8`@$``````!@"`0``````+``````````L``````````0`
M````````4.5T9`0```#0B0```````-")`0``````T(D!``````#L````````
M`.P`````````!``````````O=7-R+VQI8F5X96,O;&0N96QF7W-O```'````
M!`````$```!.971"4T0```"KD$$$````!`````,```!085@```````````!#
M````80`````````=````6@````,````7````1P```&````!*`````````$8`
M```/````+@````````!2`````````%\```!7````6P```"D````L````"```
M```````?````6````$P`````````(@```$$```!<````-@```%4`````````
M``````````!=````!0``````````````*P```%0`````````4P```!8````X
M`````````%D````2````)@````````!#`````````#(`````````!P```$(`
M````````&0````````!6````,0````````!1````7@```$L```!/````/```
M`$X```````````````````````````````(````````````````````$````
M````````````````````````````````````````````````````````````
M``````````8````*````&```````````````````````````````````````
M```:````$P`````````0````%``````````,```````````````-````````
M`!P`````````"0````$````.````+P`````````+````+0```"$`````````
M%0```````````````````"``````````.P```#<````;`````````",```!`
M`````````!X````^`````````#H````U````)0```#,````G````1````$D`
M``!%````/P```$@````J````.0```#`````D`````````$T````T````$0``
M````````````*````%`````]````````````````````````````````````
M````````````````<@```!(````P&0$``````````````````0```!(```!`
M&0$`````````````````"0```!(```!0&0$`````````````````%`,``!(`
M"@`$(@$```````0`````````$0```!(```!@&0$`````````````````%P``
M`!(```!P&0$`````````````````)0,``!(`"@"<*0$``````!@`````````
M0`,``!``%@!(IP$`````````````````'@```!(```"`&0$`````````````
M````K`(``!(```"0&0$`````````````````)@```!(```"@&0$`````````
M````````3`,``!(`"@"",`$``````)P`````````8P```!(```"P&0$`````
M````````````+@```!(```#`&0$`````````````````6`,``!``%``XI`$`
M````````````````7P,``!``%``HI`$`````````````````;P,``!``$P`0
MI`$`````````````````/@```!(```#0&0$`````````````````30```!(`
M``#@&0$`````````````````5````!(```#P&0$`````````````````7@``
M`!(`````&@$`````````````````:````!(````0&@$`````````````````
M<0```!(````@&@$`````````````````XP$``!(````P&@$`````````````
M````=P```!(```!`&@$`````````````````?@```!(```!0&@$`````````
M````````C````!(```!@&@$`````````````````G0```!$`%@"`I`$`````
M``$`````````6`(``"(`````````````````````````J@```!(```!P&@$`
M````````````````?@,``!(`"@#,(P$``````"(`````````BP,``!(`"@#N
M(P$``````.P`````````EP,``!(`"@!.*@$``````)H$````````N````!8`
M````````````````````````RP```!(```"`&@$`````````````````O@(`
M`!(```"0&@$`````````````````U@```!(```"@&@$`````````````````
MWP```!$`%@!`IP$```````@`````````YP```!(```"P&@$`````````````
M````H@(``!(```#`&@$`````````````````\P```!(```#0&@$`````````
M````````^@```!(```#@&@$`````````````````I`,``!(`"@#B0P$`````
M`-8`````````#0$``!$`%@"@I`$``````$``````````'P$``!(```#P&@$`
M````````````````+P$``!(`````&P$`````````````````/0$``!(````0
M&P$`````````````````0@$``!(````@&P$`````````````````KP,``!$`
M%``PI`$```````@`````````40$``!(````P&P$`````````````````70$`
M`!(```!`&P$`````````````````<@$``!(```!0&P$`````````````````
M?0$``!(```!@&P$`````````````````#P,``!``%@!(IP$`````````````
M````;@(``!(```!P&P$`````````````````D`$``!(```"`&P$`````````
M````````_`,``!(`"@#P'`$```````P`````````G@$``!(```"0&P$`````
M````````````N0,``!``\?\0K`$`````````````````I@$``!(```"@&P$`
M````````````````L@$``!(```"P&P$`````````````````RP,``!(`"@#:
M)`$``````"(#````````N`$``!(```#`&P$`````````````````PP$``!(`
M``#0&P$`````````````````V`,``!(`"@"X1`$``````*X!````````U`$`
M`!(```#@&P$`````````````````X@,``!``%0`XI`$`````````````````
M[@,``!(`"@#>(0$``````"8`````````]@,``!(`"@"V1@$``````(P!````
M````VP$``!(```#P&P$`````````````````.0(``"(`````````````````
M````````^P,``!(`"@#P'`$```````P``````````P0``!(`"@`((@$`````
M`,0!````````$`0``!(`"@"T*0$``````)H`````````X@$``!(`````'`$`
M````````````````(@0``!(`"@#\)P$``````$H`````````ZP$``!(````0
M'`$`````````````````_`$``!(````@'`$`````````````````40(``!(`
M```P'`$`````````````````,`0``!(`"@!&*`$``````#8``````````P(`
M`!(```!`'`$`````````````````#`(``!(```!0'`$`````````````````
M$@(``!(```!@'`$`````````````````E0(``!(```!P'`$`````````````
M````%P(``!(```"`'`$`````````````````B`(``!$`%@"PI@$```````@`
M````````'P(``!(```"0'`$`````````````````LP(``!(```"@'`$`````
M````````````)@(``!(```"P'`$`````````````````.`0``!(`"@".0P$`
M`````%0`````````+0(``!$`%@#@I`$``````,@!````````?0(``!$`$P`0
MI`$```````@`````````000``!(`"@#H+@$``````)H!````````!@$``!(`
M``#`'`$`````````````````,@(``!(```#0'`$`````````````````Q0(`
M`!(```#@'`$``````````````````')E86QL;V,`;65M;6]V90!R;61I<@!S
M=')C:'(`<W1R;F-M<`!S=')R8VAR`&)O;W1?1'EN84QO861E<@!P97)L7V-O
M;G-T<G5C=`!U;FQI;FL`7U]L<W1A=#4P`'!E<FQ?9G)E90!?7W-T870U,`!?
M97AI=`!M96UC<'D`<&5R;%]D97-T<G5C=`!?7W-T86-K7V-H:U]F86EL`%!,
M7V1O7W5N9'5M<`!097)L7W-V7W-E='!V`%!,7V-U<G)E;G1?8V]N=&5X=`!P
M97)L7V%L;&]C`&-L;W-E9&ER`&5N=FER;VX`7U]O<&5N9&ER,S``;65M<V5T
M`%!E<FQ?<V%F97-Y<VUA;&QO8P!?7W-T86-K7V-H:U]G=6%R9`!097)L7V=V
M7V9E=&-H<'8`7U]S<')I;G1F7V-H:P!R96%D`%!E<FQ?<WES7VEN:70S`%!E
M<FQ?9V5T7W-V`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E<FQ?;F5W6%,`4&5R
M;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`7U]E<G)N;P!097)L7VUG
M7W-E=`!M:V1I<@!P97)L7W!A<G-E`%!E<FQ?;F5W6%-?9FQA9W,`86-C97-S
M`&=E='!I9`!V9G!R:6YT9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W-&`'-T
M<F-M<`!?7V1E<F5G:7-T97)?9G)A;65?:6YF;P!S=')D=7``7U]R96=I<W1E
M<E]F<F%M95]I;F9O`'!T:')E861?871F;W)K`%]?<')O9VYA;64`7U]P<U]S
M=')I;F=S`%]?9V5T<'=U:60U,`!?7W-Y<V-A;&P`871E>&ET`%]L:6)C7VEN
M:70`<W1R=&]K`%]?<F5A9&1I<C,P`&QI8G!E<FPN<V\`;&EB;2YS;RXP`&QI
M8F-R>7!T+G-O+C$`;&EB<'1H<F5A9"YS;RXQ`&QI8F,N<V\N,3(`7V5N9`!P
M87)?8W5R<F5N=%]E>&5C`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`%]?
M0E-37T5.1%]?`'!A<E]C;&5A;G5P`%]E9&%T80!?7U-$051!7T)%1TE.7U\`
M7U]$051!7T)%1TE.7U\`<&%R7V)A<V5N86UE`'!A<E]D:7)N86UE`'!A<E]M
M:W1M<&1I<@!S:&%?=7!D871E`&UY7W!A<E]P;`!?7V=L;V)A;%]P;VEN=&5R
M)`!P87)?:6YI=%]E;G8`<VAA7V9I;F%L`%]?8G-S7W-T87)T`'AS7VEN:70`
M;6%I;@!?7W-T87)T`'!A<E]F:6YD<')O9P!P87)?<V5T=7!?;&EB<&%T:`!P
M87)?96YV7V-L96%N`'!A<E]D:64`<VAA7VEN:70`6%-?26YT97)N86QS7U!!
M4E]"3T]4`"]U<W(O<&MG+VQI8B]P97)L-2\U+C0R+C`O<FES8W8V-"UN971B
M<V0M=&AR96%D+6UU;'1I+T-/4D4Z+W5S<B]P:V<O;&EB``#HH0$```````(`
M```=``````````````#PH0$```````L````B``````````````#XH0$`````
M``(````F```````````````(H@$```````(```!'```````````````0H@$`
M``````(```!6```````````````8H@$```````(```!<``````````````"`
MI`$```````0````<``````````````"@I`$```````0````L````````````
M``#@I`$```````0```!;```````````````PH@$```````4````!````````
M```````XH@$```````4````"``````````````!`H@$```````4````#````
M``````````!(H@$```````4````%``````````````!0H@$```````4````&
M``````````````!8H@$```````4````)``````````````!@H@$```````4`
M```*``````````````!HH@$```````4````+``````````````!PH@$`````
M``4````-``````````````!XH@$```````4````.``````````````"`H@$`
M``````4````2``````````````"(H@$```````4````3``````````````"0
MH@$```````4````4``````````````"8H@$```````4````5````````````
M``"@H@$```````4````6``````````````"HH@$```````4````7````````
M``````"PH@$```````4````8``````````````"XH@$```````4````9````
M``````````#`H@$```````4````:``````````````#(H@$```````4````;
M``````````````#0H@$```````4````>``````````````#8H@$```````4`
M```C``````````````#@H@$```````4````D``````````````#HH@$`````
M``4````E``````````````#PH@$```````4````G``````````````#XH@$`
M``````4````H````````````````HP$```````4````I```````````````(
MHP$```````4````J```````````````0HP$```````4````M````````````
M```8HP$```````4````N```````````````@HP$```````4````O````````
M```````HHP$```````4````P```````````````PHP$```````4````R````
M```````````XHP$```````4````S``````````````!`HP$```````4````T
M``````````````!(HP$```````4````U``````````````!0HP$```````4`
M```W``````````````!8HP$```````4````X``````````````!@HP$`````
M``4````Z``````````````!HHP$```````4````\``````````````!PHP$`
M``````4````]``````````````!XHP$```````4````_``````````````"`
MHP$```````4```!```````````````"(HP$```````4```!"````````````
M``"0HP$```````4```!&``````````````"8HP$```````4```!+````````
M``````"@HP$```````4```!-``````````````"HHP$```````4```!.````
M``````````"PHP$```````4```!/``````````````"XHP$```````4```!1
M``````````````#`HP$```````4```!2``````````````#(HP$```````4`
M``!3``````````````#0HP$```````4```!4``````````````#8HP$`````
M``4```!5``````````````#@HP$```````4```!7``````````````#HHP$`
M``````4```!8``````````````#PHP$```````4```!9``````````````#X
MHP$```````4```!>````````````````I`$```````4```!?````````````
M```(I`$```````4```!@``````````````"7DP``,P/#00.^`Y$3`T/]DX(#
MD1-3$P"#LH(`9P`.`!>>```#/@Z09P,.`!,````7G@```SZ.CV<##@`3````
M%YX```,^#H]G`PX`$P```!>>```#/HZ.9P,.`!,````7G@```SX.CF<##@`3
M````%YX```,^CHUG`PX`$P```!>>```#/@Z-9P,.`!,````7G@```SZ.C&<#
M#@`3````%YX```,^#HQG`PX`$P```!>>```#/HZ+9P,.`!,````7G@```SX.
MBV<##@`3````%YX```,^CHIG`PX`$P```!>>```#/@Z*9P,.`!,````7G@``
M`SZ.B6<##@`3````%YX```,^#HEG`PX`$P```!>>```#/HZ(9P,.`!,````7
MG@```SX.B&<##@`3````%YX```,^CH=G`PX`$P```!>>```#/@Z'9P,.`!,`
M```7G@```SZ.AF<##@`3````%YX```,^#H9G`PX`$P```!>>```#/HZ%9P,.
M`!,````7G@```SX.A6<##@`3````%YX```,^CH1G`PX`$P```!>>```#/@Z$
M9P,.`!,````7G@```SZ.@V<##@`3````%YX```,^#H-G`PX`$P```!>>```#
M/HZ"9P,.`!,````7G@```SX.@F<##@`3````%YX```,^CH%G`PX`$P```!>>
M```#/@Z!9P,.`!,````7G@```SZ.@&<##@`3````%YX```,^#H!G`PX`$P``
M`!>.```#/HY_9P,.`!,````7C@```SX.?V<##@`3````%XX```,^CGYG`PX`
M$P```!>.```#/@Y^9P,.`!,````7C@```SZ.?6<##@`3````%XX```,^#GUG
M`PX`$P```!>.```#/HY\9P,.`!,````7C@```SX.?&<##@`3````%XX```,^
MCGMG`PX`$P```!>.```#/@Y[9P,.`!,````7C@```SZ.>F<##@`3````%XX`
M``,^#GIG`PX`$P```!>.```#/HYY9P,.`!,````7C@```SX.>6<##@`3````
M%XX```,^CGAG`PX`$P```!>.```#/@YX9P,.`!,````7C@```SZ.=V<##@`3
M````%XX```,^#G=G`PX`$P```!>.```#/HYV9P,.`!,````7C@```SX.=F<#
M#@`3````%XX```,^CG5G`PX`$P```!>.```#/@YU9P,.`!,````7C@```SZ.
M=&<##@`3````%XX```,^#G1G`PX`$P```!>.```#/HYS9P,.`!,````7C@``
M`SX.<V<##@`3````%XX```,^CG)G`PX`$P```)>1``"3@0'R;P!``P$1(N@F
MY`;L%X0``!,$1#&7A```@[3$3&-WE``<8"$$@I?C;93^XF!"9*)D!6&"@'EQ
M!O0B\";L2NA.Y%+@X<&<89AI%XH```,ZBDN<8Y>&``"#MF9,C.(C,.H`+HFJ
MB=G'%X8```,VIDL<X@/'!P"3!O`"`<N%!V,3UP`<X@/'!P!U^Y>$``"#M`1&
M%X0``!,$1"CO\!_Q8W>4`!Q@(02"E^-ME/YCA0D`3H7O\)^^EX0``(.T1$47
MA```$P3$)1<%```3!07T[_#?O&-WE``<8"$$@I?C;93^`S8*`(,U"0`#)8D`
M[R!0#>_PO[27AP``@[?G0A.'@:F8XV&WS487-@``$P;&I(E%$47O\)_+!47O
M\#_!.7$&_"+X)O1*\$[L4NA6Y%K@+<T#NP&S8PH+!H-'!0"JB2Z**HD3!]`#
MO<-CAN<`@T<9``4)_?L["3E!`S0+`!W`VH23"M`#(H5*ALZ%[_`?LK,')`$)
MY8/'!P!C@%<#@&2A!'7PXG`BA4)THG0">>)I0FJB:@)K(6&"@+.$9$&-A`4)
M(R":`$J4\;\!1.&_`4EMMS%Q(ODF]0;]DX0!B8J%G&`^[8%'*H3O\%^S#>&B
M1[UF$6?UCV.%YP`I9V.:YP")12*%[_"?T!,U%0`1H`%%:F><8+F/`4>1Y^IP
M2G2J="EA@H#O\)^V57%.]8;EHN$F_4KY4O&3B0&)@[<)`#[M@4<)Q8-'!0`J
MA)GO:F>#MPD`N8\!1^'CKF`.9.IT2GFJ>0IZ:6&"@._P7[>JA'G=)H7O\+_9
M)<V#1]4`$PG5`)N')_V!YX-'&0#]TX-'U0";AR?]I<<BA>_P7]0JBDJ%[_#?
MTSL%J@`))>_P/]3*AB*&ES4``).%I8TJB>_P'\^*A4J%[_"?I_U78PCU`*)'
MO6819_6/8XSG`DJ%[_`?HTJ%[_"_GR:%[_!?TE'U)H7O\-^M(H7O\'^9B;^#
M1QD`FX<G_<G[@T<I`+W3:;=*A>_PG_'QM^_P?Z<=<:+HIN3*X%+XANQ._%;T
M6O!>[&+H@\<%`!.)`8FJA).'-_R3MQ<`,X3U`"*%@S<)`#[D@4?O\-_(*HI,
M`":%[_!?WGG)JHGO\+_'8W9%"Y)*$XL!LXX*`\<$`*:'DP;0`V,$!Q)C!M<`
M`\<7`(4'??N%GX,Y"P`3!2H`/I76F>_P/\4CL*D`,<4#-PL`@\<$`!.%%`!6
MEQ1C$P;0`R.`]@"!ZXVH@T<%``4%HP#W`+W#-H>%!N.8Q_Z3!M`#DP<7`",`
MUP`#1P0`A0<%!*./Y_YU^R)G@S<)`+F/`4?=Z^9@1F2F9`9IXGE">J)Z`GOB
M:T)L)6&"@(-'!`"%"04$HX_Y_N';@T<$`(4)!02CC_G^_?/!MS:'3;<3BP&S
M`SP+`(,W#`"EQ^*'@4F89\Z&H0>%*67_C2:3E38`EX<``).'YS"80P7K!4<N
MA9C#[_!?N*J+0=&3FCD`5H;BA>_P7X`C,'L!FX<9`(X'OILCL`L`3L+5M6*%
M[_"O_2,PJP"JBRG9DYHY`/F_@4?EM<%%@4EMM^_P_XU!$2+@%X0```,T9`8&
MY)=F``"3AD9_LP=$`(AC%P8``!,&YH&7)0``DX7E:^_P_YD2E`A@`F2B8!<W
M```3!R>B@4>ZAA<6```3!N;1ER4``).%96M!`=6ZEX<``(.W)P&79@``DX8F
M>A?V__\3!B9]DI>(8Y<E``"3A>5FN;(!18*`$P$!KR,P@5`C."%/(S0Q3R,T
M<4TC-!%0(SR13B,P04\C/%%-(SAA32,P@4TC/)%+(SBA2Y.+`8FJB2Z$%R4`
M`!,%169L`(.W"P`C-/%*@4?O\+^[*HF3!?`"3H7O\,_P(<T7)0``$P7E9,Z%
M[_"_UTZ%`S>!2H.W"P"YCP%'8Y$'%(,P@5`#-`%0@S2!3P,Y`4^#.8%.`SH!
M3H,Z@4T#.P%-@SN!3`,\`4R#/(%+`ST!2Q,!`5&"@"*%[_#_EI<E``"3A25@
M[_`O_"J$6=F#1P0`$PKP/Q<K```3"RM?ERH``).**E^];"%LDP3P`L'+8P<)
M`**%2H7O\!^=)<4BA>_PGYJ3!_7_HI<JC6-H]``QJ".`!P#]%V,&]``#QP<`
MXPF7_B*%[_`_F"J-3H7O\+^7:I4%!>-FJO*!14Z(VH<BA]:&$P8`0"@1[_#O
M^@P(*!'O\&_K&>7B1[/WEP&!)V.`AP67)0``DX7%50%%[_"O\2J$XP<%[H-'
M!`"U^V,'"0"#1PD`FX<G_9G'!4T7)```$P2D4%F_@T<9`.'7_;>%12@1[_!?
M@U7]+!$7)0``$P5%4._P/\,H$>_P_X95O>_PC^E!$9,%\`(BX`;D*H3O\(_<
M&<$3!!4`(H4"9*)@00%O\'^$$P$!O",\,4$C/!%"(SB!0B,TD4(C,"%#DXD!
MB8.W"0`C-/%`@4=9P8-'!0`JB;W/[_`_B1,&%0"3!P!`*H1CX\<*)`#*A2:%
M[_"OX),']/^FEY,&\`)CZ/0`$:@C@`<`_1=CAI<``\<'`.,)U_Z3!?`")H7O
M\._3%<ECXJ0&)H7O\"_\`S>!0(.W"0"YCP%'H>^#,(%#`S0!0X,T@4(#.0%"
M@SF!01,!`42"@`,W@4"#MPD`N8\!1X7K`S0!0X,P@4.#-(%"`SD!0H,Y@4$7
M)0``$P5%/A,!`41O\(_V(P`%`'&_`45YO^_PK]AY<2+P)NQ*Z`;T$X0!B1Q@
M/N2!1Y<D``"3A"0_$XD!LTP`)H7O\%^1%<$21X,W"0`.![J7F&>8XV77F&NA
M!YCC;?],`":%[_`_CV7QER0``).$I#P3B0&S3``FA>_PWXT5P1)'@S<)``X'
MNI>89YCC9=>8:Z$'F.-M_TP`)H7O\+^+9?&7)```DX2D.1.)`;-,`":%[_!?
MBA7!$D>#-PD`#@>ZEYAGF.-EUYAKH0>8XVW_3``FA>_P/XAE\9<D``"3A"0W
M$XD!LTP`)H7O\-^&%<$21X,W"0`.![J7F&>8XV77F&NA!YCC;?],`":%[_"_
MA&7QER0``).$)#03B0&S3``FA>_P7X,5P1)'@S<)``X'NI>89YCC9=>8:Z$'
MF.-M_TP`)H7O\#^!9?&7)```DX2D,1.)`;-,`":%[_#/_Q7!$D>#-PD`#@>Z
MEYAGF.-EUYAKH0>8XVW_3``FA>_PK_UE\9<D``"3A*0F$XD!LTP`)H7O\$_\
M%<$21X,W"0`.![J7F&>8XV77F&NA!YCC;?],`":%[_`O^F7QER0``).$I"L3
MB0&S3``FA>_PS_@5P1)'@S<)``X'NI>89YCC9=>8:Z$'F.-M_TP`)H7O\*_V
M9?&7)```DX0D*1.)`;-,`":%[_!/]17!$D>#-PD`#@>ZEYAGF.-EUYAKH0>8
MXVW_3``FA>_P+_-E\9<D``"3A*0F$XD!LTP`)H7O\,_Q%<$21X,W"0`.![J7
MF&>8XV77F&NA!YCC;?],`":%[_"O[V7QER0``).$I!D3B0&S3``FA>_P3^X5
MP1)'@S<)``X'NI>89YCC9=>8:Z$'F.-M_TP`)H7O\"_L9?%,`!<E```3!84@
M[_`OZZJ%&<47)0``$P6%'>_P_X=,`!<E```3!24@[_!/Z:J%&<47)0``$P6E
M(._P'X9,`!<E```3!<4@[_!OYZJ%'<D7)0``$P5%$._P/X0B9QQ@N8\!1Z'C
M`G2B<.)D0FF7)0``DX5E(!<E```3!64416%O\+^!3``7)0``$P5E'>_P#^.J
MA6'5%R4``!,%913O\,__;;_O\(^F`1$BZ`;L$X0!B4P`%R4``!,%A1(<8#[D
M@4?O\,_?%<&#1P4``46)QY.'!_TS-?``(F<<8+F/`4>9Y^)@0F0%88*``47U
MM^_P[Z$=<1,#@0(N]#+XON0&[#;\NN#"Z,;LDX<!B:J%&H83A0&@F&,ZY`%'
M&N#O\`^Y$P7P#^_PCXM9<:+PP7)!9,[DTN!F[(;TINS*Z%;\6OA>]&+P$P>$
M`!:1"I>3C`&)@[<,`!SC@4<)1H%%*HKO\.^YDPGU_Q/7^4-!D[.'Z0#"%\&3
MF8^SB?E`8\<)"D%[DP0+_Y,'!`&FE\%K_5JSA"<`H0L=3!>)```3">FR`4;.
MA5*%[_"/M6,(50E>AJ:%4H7O\*^@`25C"E4'8W.L!F$5,X2D`&-VE`"AJ'T4
M8VJ4!"%&RH4BA>_P3Z]E^<%W06?!%T$'/I>S!R<`'8PS!30!P6>A!XJ7F&.#
MMPP`N8\!1ZGCP6(6D:9P!G3F9$9IIFD&:N)Z0GNB>P)\XFQE88*`VIGCG6GW
M?57AMQ<E```3!04&[_#_ZQ<E```3!44$[_`_Z^_PCXQ!$0;D[_`/KP$E[_"O
MK!'!"&&B8$$!@H`Y<2;T2O`&_"+X3NP3B0&)JH1,`!<E```3!84"@S<)`#[D
M@4?O\*_#"<6#1P4`*H2%YR)G@S<)`+F/`4>FA:'O0G3B<*)T`GGB:1<E```3
M!27_(6%O\&_>)H7O\"^JJHDBA>_PKZE.E0D%[_`OJB*'ER4``).%1>N7)@``
MDX;&Z2:&*H3O\(^D(F>#-PD`N8\!1Z*%U=?O\&^!EV<``).'I_-E<8.X!P$#
MN(<!D'.4=YA[G'_*^L[V!N<BXZ;^TO+6[MKJWN;BXI.)`8D#LPD`&OX!0Q<C
M```3`P/XOO"7)P``DX?G]RJ)&N0^\!<C```3`P/@ER<``).'!_=,`!<E```3
M!67?&N@"[$;\PN"RY+;HNNP^]`+X[_!OM!'%@T<%`"J$8Y,',._P3YL!)>_P
M[YAC!@46`&%C`P06(H7O\.^;!@4%!>_P;YP#1@0`JH0JBA'.ERH``).*"O(F
MA=:%!03O\*^6`T8$`(D$9?HD&!<E```3!<7F(:"(9*$$#<%,`._PSZTJA&W9
M@T<%`/77[_`OMGW1(H7O\$^.*H09[20`@&`!S(-'!`!CB`<.(H7O\"^T5>6`
M9*$$=?0BA>_P[Y.JA%*%[_!ODZF<FX1$0":%[_"OD]*'%R<``!,'1^J7)@``
MDX;&TR*&ER4``).%I>DJBN_PCXV3!0`<4H7OX/___5=C!?4&K!!2A>_@'^,J
MA!GEQE>]9A%G]8]C@^<,`S8)`-*&ER4``).%Y>H3A0&@[^"_Y(%$<G>#MPD`
MN8\!1V.1!RZZ8!ID5GFV>19Z]FI6:[9K%FPFA?9T66&"@"*%[_"O@2J$H00%
MW8&_[^#_]AA!Q4?C"??X`S0)`._@__480=*&(H:7)0``DX7%X!.%`:#OX)_>
M>;\!1`6W`G4$$!G)3`#O\(^;`<6#1P4`A>.(9*$$??4U1>_P+X6JA"J*%R0`
M`!,$I-@3!C`%6;7OX-_Z*H1QW>_P3X(&!04%G;6:2N_@G_^;!P4`XYKZ\L97
M$P<`')/W]Q_CD^?R)H7O\,^`3`"JA!<E```3!47B[_#OE*J%8P$%%`,U"0#O
M\&_3*HMC"@426H7OX#_]FPH%`(U'8]A7`Q.%RO^7)0``DX6%WUJ5[^!__0GM
MD4=CA?H:FX>Z_]J7`\<'`),'\`)C#/<83``7)0``$P5EP>_P#X]C!@4.@T<%
M`&."!PX3!P`#8X[G#)<E``"3A>7(%R4``!,%Y;[O\$^J[^`_ZBJ'ER<``).'
MQ^B7)@``DX9&MU*&ER4``).%)=LFA>_@'_%]6<5*%RP``!,,C.:7*P``DXL+
MM1<K```3"PO:%:#OX/_?'$%CE5<#[^!?Y04D*H=BB**'WH92AMJ%)H7OX!_M
MDP4`'":%[^!_W^,*)?U2A>_@W[ZFA1<E```3!>6M[_#/H2:%[_#?O0&][_!_
MO7)W@[<)`+F/`4=CE@<.(H4:9+I@]G16>;9Y%GKV:E9KMFL6;%EA;^"?X@,[
M"0#QM8%%6H7OX+_D*HGC#P7P[_#?I:J*XTH%\),%9?\!1DJ%[^#_X1E&#`)*
MA>_@7\V91V,=]0@#)P$0MT=!0Y.'!S#C%/?N`U=!$)%GDX>'5.,=]^P!1I.%
MZOQ*A>_@7]X3!H`"+`)*A>_@G\F3!X`"8Q[U!)<G``"3A\?5.`*7)@``DX8F
MI%*&ER4``).%!<<FA2,(`1+OX+_=(;\#.8D`XP0)YDJ%[^"_WXU'X_ZGY'$5
MER4``).%Y<)*E>_@7^#C%`7D2HN)M>_@G[@7)0``$P7%P>_P/Y9Y<2+P%W0`
M``,T1#!*Z`;T)NRS!T0`A&,3B0&)3`"#QP0+%R4``!,%9:`#-PD`.N0!1Z.(
M]`KOX%_O&<4#1P4`DP<0`V,(]Q+O\*_:LP=$`(1C)H6-1@5&ER4``).%!<#O
MX)^ZJH0=P1QI,P=$``ACC&,7)@``$P;FON_@_["<:(QCW$79@X6+8Y$'$+,'
M1`"(8XU&!4:7)0``DX4EO>_@O[:JA`W!3``7)0``$P6%DN_@O^>S!T0`*H:(
M8YQH8<*,8^_@?ZRS!T0`A&,%1I<E``"3A26Z)H7OX!^WJH47)@``$P:FNB:%
M[^`?JA*4'&!,`!<E```3!>6,HX@'"N_@W^(7=```$P1D10C@,<63!0`<[^!_
MNQL'%0"%1V/RYP+%1V,/]0"7=P``@[>G0Q!@C&.JAA<E```3!46X[^#?L2)G
M@S<)`+F/`4>QYZ)P`G3B9$)I16&"@!=U```#-85`[_!_H`C@1?47)0``$P6%
MLN_@GZZ#1Q4`XXT'[/&U%W<```,W)SX08R6_LP=$`(AC[^"?LN6][^`_GGEQ
M(O`F[`;T2N@3A`&)JH1,`!<E```3!<6)'&`^Y(%'[^`?UP'-@T<%`(G+$P<`
M`V.&YP"!Q(/'!`"![R)G'&"YCP%'L>NB<`)TXF1":45A@H`FA>_PC[&3!?`"
M*HGOX!^,&<$3"14`2H7OX%^T$4:7)0``DX7ED^_@?XAA\2)G'&"YCP%'B>L"
M=*)P0FDFA>)D16%OX'_>[^!_E!,!`=,C,($L(S01+",\D2HC."$K(S0Q*R,P
M02LC/%$I$X0!B30`?5_!;AQ@(S3Q*(%'DPB%`Y,"A0<VB!-?#P*W#_\`DXX.
M\`.V"`!!"*$(DU4&`A,7A@&3$X8`DU>&`A->A@"35(8#$Y.%`3-WYP&S\_,!
ML_?7`3-G=P`S?MX!Q8\S<^,!H@4S9\<!&U:&`;/G9P"S]?4!48_-CR,XZ/XC
M//C^XY(2^_U5QG]F?PIGXFX"?B)SPGAB>(9CDX0&((&1$:"R@[!BG&*Z@J$&
ML8^SQ_<!L\?7`1.7%P#]@UV/;8^X_OJ/\HX6CQJ.1H/"B!Z(XYF6_(,^!0"B
M;P,X!0$#,X4`@SB%`0,^!0(WAX):$P>7F9/7O@$3EEX`78ZZGQ-?(P"3%^,!
M;8[RG[-&&`&SY^<!L_9F`'Z6PF\S_[<`L\86`3:6LT;H`6V.L_;6`;J?DQ)6
M`)-4M@'&G[/&!@'^EK/OD@`3V2X`DY/N`?Z6XF^SXR,!L\=W`/&/NI_MCK/'
MYP'"GY.45@`3V;8!DQ+F`?Z7"8*"?[/BP@`SYB0!LI<SQE,`NI^S\[,`=8[M
MCWZ?,T9V`).45P`3V;<!DY_F`7J6B8(B?[/OWP"SYB0!-I:SQO(!;8ZS\K(`
M_8XZGQZ?DQ16`!-9M@&SQE8`DY/G`8F#L^/S`/J6L^<D`;Z6,_^V`,)VL\=_
M`/&/L_^_`+J6EI:3%%\`$UF_`;/']P&VE[/F)`&VEY,2Y@$)@K/BP@`S]K<`
MXG>SQE,`L_;F`;/SLP"ZE_Z7DQ16`!-9M@&SQG8`OI:SYR0!OI:3'^\!$U\O
M`+/O[P$S_[8`AF:SQ_(!\8^S\K(`NI:>EI,47P`36;\!L\=7`+:7L^8D`;:7
MDQ/F`0F"L^/#`#/VMP"F9[/&?P"S]N8!L_^_`+J7EI>3%%8`$UFV`;/&]@&^
MEK/G)`&^EI,2[P$37R\`L^+B`3/_M@#&9K/'4P#QC[/SLP"ZEOZ6DQ1?`!-9
MOP&SQW<`MI>SYB0!MI>3'^8!"8*S[\\`,_:W`.9GL\;R`;/VY@&S\K(`NI>>
MEY,45@`36;8!L\96`+Z6L^<D`;Z6DQ/O`1-?+P"SX^,!,_^V`(9VL\=_`/&/
ML_^_`+J6EI:3%%\`$UF_`;/']P&VE[/F)`&VEY,2Y@$)@K/BP@`S]K<`IG>S
MQE,`L_;F`;/SLP"ZE_Z7DQ16`!-9M@&SQG8`OI:SYR0!OI:3'^\!$U\O`+/O
M[P$S_[8`QG:SQ_(!\8^S\K(`NI:>EI,47P`36;\!L\=7`+:7L^8D`;:7DQ/F
M`0F"L^/#`#/VMP#F=[/&?P"S]N8!L_^_`+J7EI>3%%8`$UFV`;/&]@&^EK/G
M)`&^EI,2[P$37R\`L^+B`3/_M@"*9K/'4P#QC[/SLP"ZEOZ6DQ1?`!-9OP&S
MQW<`MI>SYB0!MI>3'^8!"8*S[\\`,_:W`*IGL\;R`;/VY@&S\K(`NI>>EY-4
MM@&3$U8`L\96`!,9[P&^EA-?+P"SYY,`RF,S;^D!OI:SQ^\![8ZS_[\`NI/Q
MCYZ2D]2V`;/']P&3DU8`L^.3`):7ZF23$N8!"8*SXL(`GI?MCS-&7P"ZE#-_
MOP`3V;<!=8Z3DU<`II\S1N8!L^,C`;/SLP!^EI.?Y@&)@AZ6L^_?``IYLW2V
M`+/&\@']CK/RL@"3DU0`D]FT`3J9,\=6`+/F,P&3D^<!B8.SX_,`2I^SQW\`
M[8YZE[&/*G8S#]<`-_?9;A,'%[JS>;\`$]FY`3J6DY99`!:6[8^SYB8!LI?M
MCKZ6DY+D`<IWB8"SXI(`,_FV`#/&4P"S_[\`DU2Y`;J7,T_F`1,660`S?[\`
M_I=%COJ7;8X^EI.?Z0'J=Y/9*0"S[S\!LW2V`#//\@&S\[,`D]FT`;J7,T_?
M`).65``S?[\`GI>SYC8!^I?MCKZ6$Q_I`8YG$UDI`#-O+P&S\[8`,\GO`;/Y
ML@"ZE[-"R0`3V;,!$Y93`+/RL@#.ES-F)@&6EVV./I:3DN0!KF>)@+/BD@`S
M25\`LW2V`+/YOP"ZE[-/V0`3V;0!DY94`+/_OP#.E[/F)@'^E^V.OI:3G^,!
MSF>3TR,`L^]_`#/)\@&S\[8`LWF_`+J7,T_)`!/9LP$3EE,`,W^_`,Z7,V8F
M`?J7;8X^EA.?Y`'N9XF`,V^?`#/)[P&S=+8`L_FR`+J7LT+9`!/9M`&3EE0`
ML_*R`,Z7L^8F`9:7[8Z^EI.2XP&.=Y/3(P"SXG(`,TE?`+/SM@"S^;\`NI>S
M3\D`$]FS`1.64P"S_[\`SI<S9B8!_I=MCCZ6DY_D`:YWB8"S[Y\`,\GR`;-T
MM@"S>;\`NI<S3]D`$]FT`9.65``S?[\`SI>SYB8!^I?MCKZ6$Y_C`<YWD],C
M`#-O?P`SR>\!L_.V`+/YL@"ZE[-"R0`3V;,!$Y93`+/RL@#.ES-F)@&6EVV.
M/I:3DN0![G>)@+/BD@`S25\`LW2V`+/YOP"ZE[-/V0`3V;0!DY94`+/_OP#.
ME[/F)@'^E^V.OI:3G^,!DF>3TR,`L^]_`#/)\@&S\[8`LWF_`+J7,T_)`!/9
MLP$3EE,`,W^_`,Z7,V8F`?J7;8X^EA.?Y`&R9XF`,V^?`#/)[P&S=+8`L_FR
M`+J7LT+9`!/9M`&3EE0`L_*R`,Z7L^8F`9:7[8Z^EI.2XP'29Y/3(P"SXG(`
M,TE?`+/SM@"S^;\`NI>S3\D`$]FS`1.64P"S_[\`SI<S9B8!_I=MCCZ6DY_D
M`?)GB8"S[Y\`,\GR`;-TM@"S>;\`NI<S3]D`$]FT`9.65``S?[\`SI>SYB8!
M^I?MCKZ6$Y_C`9)WD],C`#-O?P`SR>\!L_.V`+/YL@"ZE[-"R0`3V;,!$Y93
M`+/RL@#.ES-F)@&6EVV./I:3DN0!LG>)@+/BD@`S25\`LW2V`+/YOP"ZE[-/
MV0`3V;0!DY94`+/_OP#.E[/F)@'^E^V.OI:3G^,!TG>3TR,`L^]_`#/YM@"S
MP_(!LWF_`+J7,\_#`)-3N0$3%ED`,W^_`,Z7,V9V`&V.^I<SC\<`DY/D`?)W
MB8"SXY,`,\9_`+-TOP"S\K(`NI>QCNV.D]FT`9:7DY)4`+:7L^8R`;:7$Q;I
M`99F$UDI`#-F)@$SR<,`L_*W`+/_OP"ZEC-'Z0%MC_Z6$YE2`)/?L@$S:?D!
MNI;*EA.?Y`&)@#-OGP#]5[-TM@`W]\8C]8VV9H&3"@<3!\?-LW__`#-FY@$S
M^?0!NI8S=E8`L_/S`+:3D]FU`3-F)@$3F54`'I:S8SD!'I:S<_8`DY;B`59F
MD](B`+/F5@"S\O8`,V_?`#/Y7P`ZEC-_OP"RE)/9LP$S;R\!$YE3`":?LV0Y
M`1.6Y0$FGXF!]F1-CK-U]@#1CC-__P`S^;(`L_9V`+J4_I236;\!L^8F`1,9
M7P"FEK-D.0$3FN,!D]\C`*:6EG2S;_H!L_/_`#-F]@']CC/Y=0"ZE#-VY@&F
MDC-F)@&3E%8`$]FV`;/D)`$6EI,9[P$FEA-?+P"V=#/OZ0&S<O\`L^_O`7V.
M,_E3`+/_WP"ZE+/O+P&NE!,95@"36;8!DY7F`8F"U8VSAI\`LV\Y`?Z6UG^S
M]/4`,V^_`/V.,_F2`+J?,W_/`#-O+P'^DQ.95@"3V;8!DQ_F`0F"L^_/`#,&
M?P`S;SD!>I9V?[/S_P"SY?4!?8XS^70`.I_UC;/E)0'ZDA,95@"36;8!$Y_F
M`8F",V_?`+.&50"S93D!KI::9;-R_P"S[^\!_8XS^5,`NI6S_\\`L^\O`:Z4
M$YE6`)/9M@&3%>8!"8+1C3.&GP"S;SD!?I:Z;[/T]0`S;[\`?8XS^9(`NI\S
M?]\`,V\O`?Z3$QE6`)-9M@&3G^8!B8*S[]\`LP9_`#-O.0'ZEEIOL_/_`+/E
M]0']CC/Y=``ZG_&-L^4E`?J2$YE6`)/9M@$3'^8!"8(S;\\`,X95`+-E.0$N
MEOIELW+_`+/O[P%]CC/Y4P"ZE;/_WP"S[R\!KI03&58`DUFV`9.5Y@&)@M6-
MLX:?`+-O.0'^EII_L_3U`#-OOP#]CC/YD@"ZGS-_SP`S;R\!_I,3F58`D]FV
M`9,?Y@$)@K/OSP`S!G\`,V\Y`7J6.G^S\_\`L^7U`7V.,_ET`#J?]8VSY24!
M^I(3&58`DUFV`1.?Y@&)@C-OWP"SAE4`LV4Y`:Z6VG6S<O\`L^_O`?V.,_E3
M`+J5L__/`+/O+P&NE!.95@"3V;8!DQ7F`0F"T8TSAI\`LV0Y`2:6^G2S__4`
M,V^_`'V.,_GR`3-_WP"ZE#-O+P&>E!,95@"36;8!DY/F`8F"L^/3`+,&GP"S
M9#D!'FDS__,`L^5U`*:6\8VS].\!.IG]CA:9Q8V3F58`$]JV`9-2)@"3%.8!
MRI4^:;/D5``SYDD!L_+T`+*5,^:3`'6.L_/U`#J9LW5?`'Z9DYE3`!/:LP%-
MCI/?)@"3E>8!2I:SYDD!L^7U`3:6L^:T`+-T]@!>9K/_]0`S^?(!L_9V`#J6
M>I:SYB8!$Y]4`!/9M`$S;R\!LI;ZEA.6XP%^;Y/3(P`S9G8`LW/V`-&-_8XS
M^G\`Y8TZGY/9M@$6GQ.95@"SY44!,VDY`?J5GGD3WR0`DY+D`;/BX@$S__(`
M,WGY`#-F5@#*E76.L_KC`;J9L_3U`#-G5@$SAOD!,I<3F50`$]8F`!/:M`&3
MG^8!L^_/`#-I20&SQO(!,WGY`#J9K8XWMY@ROG4*!Q,'9QVS>?D`D]*Y`;J5
M$Y99`)Z5_8XS9E8`?8ZNEI.2Y`'><XF`LX7&`+/BD@`SQE\`L_3U`+J3,T8F
M`9.65``3V;0!'I]]CK/F)@']CGJ6-I:3D^D!_G:3V2D`L^,S`3-Y]@`SSW(`
ML___`)-9N0&ZEC-/OP"3%5D`,W__`/Z6L^4U`?J6_8VVE1.?Y`&#-@$@B8`S
M;Y\`L\_C`;/T]0"S\O(`D]FT`;J6L\_/`!.65`"S__\`EI8S9C8!_I9]CC:6
MDQ+I`8,V@2`362D`L^(B`;-/7P`S>?8`L_/S`)-9N0&ZEK//OP"3%5D`L___
M`)Z6L^4U`?Z6_8VVE9.?Y`&#-@$AB8"S[Y\`L\/R`;/T]0"S>?\`NI8SS\,`
MD].T`1.65``S?_\`SI8S9G8`^I9]CC:6$Q_I`8,V@2$362D`,V\O`;-S]@`S
MR>\!L_GR`+J6LT*Y`!/9LP&3E5,`L_+R`,Z6L^4E`9:6_8VVE9.2Y`&#-@$B
MB8"SXI(`,TE?`+/T]0"S^?\`NI:S3\D`$]FT`1.65`"S__\`SI8S9B8!_I9]
MCC:6DY_C`8,V@2*3TR,`L^]_`#/)\@&S<_8`LWG_`+J6,T^Y`!/9LP&3E5,`
M,W__`,Z6L^4E`?J6_8VVE1.?Y`&#-@$CB8`S;Y\`,\GO`;/T]0"S^?(`NI:S
M0LD`$]FT`1.65`"S\O(`SI8S9B8!EI9]CC:6DY+C`8,V@2.3TR,`L^)R`#-)
M7P"S<_8`L_G_`+J6LT^Y`!/9LP&3E5,`L___`,Z6L^4E`?Z6_8VVE9.?Y`&#
M-@$DB8"S[Y\`,\GR`;/T]0"S>?\`NI8S3\D`$]FT`1.65``S?_\`SI8S9B8!
M^I9]CC:6$Y_C`8,V@223TR,`,V]_`#/)[P&S<_8`L_GR`+J6LT*Y`!/9LP&3
ME5,`L_+R`,Z6L^4E`9:6_8VVE9.2Y`&#-@$EB8"SXI(`,TE?`+/T]0"S^?\`
MNI:S3\D`$]FT`1.65`"S__\`SI8S9B8!_I9]CC:6DY_C`8,V@263TR,`L^]_
M`#/)\@&S<_8`LWG_`+J6,T^Y`!/9LP&3E5,`,W__`,Z6L^4E`?J6_8VVE1.?
MY`&#-@$FB8`S;Y\`,_GU`+/$[P&S^?(`NI:SPL0`DU2Y`1,660"S\O(`SI9%
MCI:6?8XVEI.4XP&#-H$FD],C`+/D=`"S0I\`LW/V`+/Y_P"ZEK//L@"3TK,!
MDY53`+/__P#.EK/E50#]C?Z6LX^V`),2Z0&#-@$G$UDI`+/B(@&SQ50`,_G_
M`#-__P"36;D!NI:QC1,660#]C?J6,V8V`7V.KI:SA<8`$Y_C`0,V@2>3TR,`
M,V]_`+/&X@&S\_4`_8PZEK/&]@']CB:6D]FS`9.44P`VEK/F-`$VEH,V`2B3
M'^D!$UDI`+/O+P&ZEK-$_P$S=_8`]I:S\O(`,\:T`'V.DQ57`)->MP&6EK*6
M,^;5`9.5XP&3TR,`LI:SY74`_8XZDS,&SP$SAP4!LX4?`7V/%.']C;-V\P#Q
MCQCI'/$4Y0SM`S>!*!Q@N8\!1Y7C@S"!+`,T`2R#-($K`SD!*X,Y@2H#.@$J
M@SJ!*1,!`2V"@._0;^U!$1,%``@&Y._0OY*W=EU,A@:3AN;/-W?I83<F16<4
MZ1,&%C"W5C(0!@<0X9.&9D<79@```S:F!A,'!Q\0Y13M&/$C-`4"(S@%`B,L
M!0:B8$$!@H`Y<2+X)O1*\%+H!OQ.[%;D&'63%S8`%'FZEX(7@9,SM^<`JH2Z
MEBA=$U?6`3:7G/28^#*)+H03BH0##<V3"0`$NX>I0+Z)8T3V")N*"0!6AJ*%
M4I7OT*_@O%P3!P`$NX;Y`+3<8YSF!":%.PDY05:4[^#?S),'\`-CW2<%&PD)
M_)M::0";B1H`F@FBF:*%$P8`!%*%[]"OW!,$!`0FA>_@_\GC%33_FYIJ`#L)
M64%*ALZ%4H7OT*_:(ZPD!^)P0G2B=`)YXFE":J)J(6&"@+*)K;^BB>F_>7$B
M\";L4N`&]$KH3N2#N84"$XJ%`Y,&`/@3UCD`$W;V`S,'R@"3!Q8``[D%`R,`
MUP`3!X`#JH0NA`4F,P7Z`&-5]Q*3!P`$D9\11V/SYQ0)1V/WYQ)CD@<2(H7O
MX+_`$P:``X%%4H7OT`_;DU8)`1-7B0"3]O8/H@:;5XD!$W?W#T('U8\3>?D/
M8@G9CQO6B0&3U@D!`A:SYR<!$]>)`)/V]@^B%M&/$W?W#T(7U8_9C^(9L^<W
M`2*%//COX+^Z'&`BA>&#(X#T`!Q@P8.C@/0`'&"A@R.!]``<8*.!]``<9.&#
M(X+T`!QDP8.C@O0`'&2A@R.#]``<9*.#]``<:.&#(X3T`!QHP8.CA/0`'&BA
M@R.%]``<:*.%]``<;.&#(X;T`!QLP8.CAO0`'&RA@R.']``<;*.']``<<.&#
M(XCT`!QPP8.CB/0`''"A@R.)]``<<*.)]``"=*)PXF1"::)I`FI%86_0+[F3
M!X`#.X;'0(%%[]!/RM6](P`%`/&](P`%`*,`!0#Y%PD%XX<'[.VW(P`%`*,`
M!0`C`04`HP$%`/$7"4<1!>/GY^K1OY.'@:H#QX<$$>\%1R.$YP279P``@[<G
MN9''%T4``!,%!42"AX*`DX>!J@/'!P`%XP5'(X#G`)=G``"#M\>T@<N3A0&K
M%T4``!,%94&"AX*`'7&&[*;DJH>BZ,K@3OR3A`&)+N@RY`P(,`!H"#[.G&`^
M](%'[]!OQ!?6__\3!B9(LH47U?__$P6%4N_0#\B#QP&'^<^79P``@[=GKQ=D
M```3!.34DI>88W))@T<G31L%20`.!9/G-P`C"?=,[]#ON\)E\D<79@``(S^F
MT)AA#2FJAACA%P<``!,'YTH8Y1=G```#-\?.&.D7!P``$P<G2ACM!4=C7?<,
M&X@W`$*&H0411XAADQ<W`+:7!2>(XZ$%XQD'_YL'!@".!PA@MI<!1TJ&E^7_
M_Y.%Y9\CL`<`[]#/PBW%"&#OT$^K*HE,$!<%```3!<47[]!OYBG-"&#OT,^D
M[]"/O")WG&"YCP%'K>_F8$9DIF3B>4J%!FDE88*`[]"OJA=D```3!&3'".`Q
MR>_0KYZ79P``@[=GH(5F,X='`!AC-I<C"P>4*;<(8._0S\-)OY=I``"3B6G#
M`[4)``'%[^"_AEF_3!`7!0``$P6%"._0K]XCL*D`2='=MY%'H;<%1>_0;X_O
MT"^B````````<'-?<W1R:6YG<R!M:7-S:6YG"@``````)7,O)7,```!$>6YA
M3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(``````$EN=&5R;F%L<SHZ4$%2.CI"
M3T]4`````"X`````````4$%27U1%35```````````%!!4E]04D]'3D%-10``
M```Z`````````"\`````````)7,E<R5S``!015),-4Q)0@``````````4$52
M3$Q)0@!015),-4]05```````````4$523$E/``!005)?24Y)5$E!3$E:140`
M4$%27U-005=.140``````%!!4E]#3$5!3@````````!005)?1$5"54<`````
M````4$%27T-!0TA%`````````%!!4E]'3$]"04Q?1$5"54<``````````%!!
M4E]'3$]"04Q?5$U01$E2`````````%!!4E]435!$25(```````!005)?1TQ/
M0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``````````,0````````!L<V5E
M:R!F86EL960`````<F5A9"!F86EL960``````$Q$7TQ)0E)!4EE?4$%42`!3
M65-414T``"]T;7`O````55-%4@````!54T523D%-10``````````)3`R>```
M``!P87(M`````"5S)7,E<R5S```````````E<SH@8W)E871I;VX@;V8@<')I
M=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````
M)7,Z('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E
M(')E;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@```````%!!
M5$@`````<&%R;``````N<&%R`````'-H;W)T(')E860````````E<R5S8V%C
M:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````
M`````!@`````````<&5R;``````P`````````%!!4CHZ4&%C:V5R.CI615)3
M24].`````#$N,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O
M<GD``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``````````"UE````````
M+2T```````!435!$25(``%1%35!$25(`5$5-4`````!435```````&5V86P@
M)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C
M(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P
M;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?
M;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB
M.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE
M7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"([
M"GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H
M=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@
M)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK
M7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H
M=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@
M,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?
M34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I
M*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?
M<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE
M+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%2
M7T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I
M;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M
M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN
M9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?
M=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D
M7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*
M5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE
M.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@
M/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT
M("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*
M<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@
M=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR
M;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I
M;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@
M)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@
M<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@
M?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F
M26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D[
M"FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A
M=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?
M97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO
M(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7
M:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U
M=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E
M;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-
M4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N
M*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N
M;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q
M6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T
M('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'
M24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII
M9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC
M('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL
M87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO
M;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?
M;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L
M("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(
M12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R
M:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K
M97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S
M971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S
M*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E
M96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F
M<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY
M("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@
M("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE
M("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?
M1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D
M8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D[
M"FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@
M*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@
M/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0[
M"G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN
M960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2![
M"FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W
M-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL
M96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F
M=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2![
M"F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L
M"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D
M8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D
M=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4[
M"FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL
M97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.
M0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.
M5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM
M>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C
M<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R
M:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F
M:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[
M8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL
M96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*
M?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L
M92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U
M:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@
M17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H
M<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD
M9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I
M;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*
M96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E
M8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I
M;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ
M<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P
M<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL
M(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY
M($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE
M7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ
M92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A
M<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI
M9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@
M(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV
M("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M
M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]
M('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U
M<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^
M("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E
M<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q
M("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*
M=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*
M)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S
M:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*
M?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q
M(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A
M;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G
M5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM
M>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%
M35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ
M1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C
M;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`
M861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL
M93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$[
M"GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN
M960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@
M<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H
M+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z
M:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z
M.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A
M<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II
M<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@
M>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O
M;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H
M9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R
M(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D
M8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@
M82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S
M92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE
M+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ
M3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I
M92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM
M;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE
M9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R
M+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]
M('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"
M051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R
M:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R
M*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D
M(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(
M96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E
M7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P
M('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N
M9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I
M=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`
M:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z
M.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@
M/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB
M7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@
M*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD
M969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI
M*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O
M<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<
M12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S
M=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII
M9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M
M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L*
M*"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]
M"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E
M;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@
M+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&
M24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I
M;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT
M(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO
M<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@[
M"E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M
M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N
M86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H
M0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D[
M"GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I
M("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U
M=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D
M8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@
M96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N
M(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&
M:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@
M<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D
M;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE
M;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$
M:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D
M9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X
M+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA
M;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@
M>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F
M:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E
M="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP
M86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R
M(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T[
M"F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I
M;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005([
M"E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I
M;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/
M.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R
M8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O
M<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL
M960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*
M;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:
M7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D
M<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q
M,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C
M:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN
M(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E
M('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP
M+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM
M>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE
M<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ
M0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T
M7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I
M;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E
M<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H
M*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET
M(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO
M=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA
M;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?
M;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R
M;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H
M;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]
M"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\
M/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;
M("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP
M;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM0
M05)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD[
M"GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]
M($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*
M<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I
M('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]
M"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$
M>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR
M97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*
M<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%
M>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N
M=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR
M97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ
M2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:
M;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(
M03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!0
M05(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U
M:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB
M=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I
M;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@
M<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I
M<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@
M4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U
M:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D
M("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I
M('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H
M("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$
M25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P
M871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@
M/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H
M)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO
M9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE
M(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%
M?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@
M(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M
M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H
M)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*
M96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G
M:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F
M:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<
M,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H
M(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H
M+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP
M0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F
M+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D
M:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C
M:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,
M14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D
M:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D
M<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]4
M14U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V
M95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@
M)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U
M;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D
M9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@
M)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP
M;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L
M;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA
M;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!
M4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I
M('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%
M3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%
M35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I
M;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%
M3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E
M;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE
M*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES
M>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]
M"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%
M+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0
M?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y
M<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y
M<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA
M;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII
M9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O
M9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]
M("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD
M969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P
M=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1
M)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]0
M4D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII
M9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*
M?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W
M*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I
M('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!4
M14U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)0
M05)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B
M?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,
M14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L
M;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B
M0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S
M:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]
M+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L
M:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E
M(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD
M7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@
M0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F
M:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D
M<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?
M7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`
M(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?
M5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%
M4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS:2YC`````````)!)`0``````^$L!
M````````3`$```````A,`0``````$$P!``````````````````!#04-(10``
M`1L#.^P````<````+)/__P0!``!<D___*`$``$*4__]X`0``UI3__[0!```L
ME?__X`$``!Z6__\4`@``MI?__U@"```.F/__?`(``#28__^0`@``.)C__Z0"
M``#\F?__]`(``!Z:__\4`P``"IO__U@#```LGO__B`,``':>__^L`P``K)[_
M_\0#``#,G___%`0``.2?__\P!```?J#__V`$```8I?__Q`0``+*F___T!```
M3J?__V@%``"^N?__I`4``!*Z___$!0``Z+K___P%``"6O/__9`$``+R\__]0
M`0``YKS__S`%````````$``````````#>E(``7P!`1L-`@`@````&````""2
M__\P`````$(.($:(!(D&@0)@P4+(0LE"#@`D````/````"R2___F`````$(.
M,$R!`H@$B0:2"),*E`P`````````$````&0```!DN___*@`````````0````
M>````"J[__\F`````````#@```",````PI+__Y0`````0@Y`4($"B`2)!I((
MDPJ4#)4.EA`"5@K!1,A"R4+20M-"U$+50M9"#@!""R@```#(````&I/__U8`
M````0@[``4:(!(D&@0("0@K!0LA"R4(.`$(+````,````/0```!$D___\@``
M``!"#M`!3),*@0*(!(D&D@B4#&0*P4+(0LE"TD+30M1"#@!""T`````H`0``
M`I3__Y@!````0@Y@5(@$B0:2")0,@0*3"I4.EA"7$I@4`LP*P4+(0LE"TD+3
M0M1"U4+60M="V$(.`$(+(````&P!``!6E?__6`````!"#A!"B`1*@0)HR$+!
M7@X`````$````)`!``"*E?__)@`````````0````I`$``)R5__\$````````
M`$P```"X`0``C)7__\0!````1`Z0"G"(!)((DPJ7$H$"B0:4#)4.EA"8%)D6
MFA@"4@K!1,A$R4321--$U$351-9$UT381-E$VD0.`$(+````'`````@"````
ME___(@````!"#A!(B`2!`E#(0L%"#@!`````*`(```*7___L`````$0.P`A4
MDPJ!`H@$B0:2"`)^"L%$R$3)1-)$TT0.`$(+4@K(1,%$R4321--,#@!$"P``
M`"P```!L`@``JI?__R(#````0@XP2(@$B0:2"($"`]@""LA"P4+)0M)2#@!$
M"P```"````"<`@``G)K__TH`````0@X@1(@$@0)V"L%"R$(.`$(+`!0```#`
M`@``PIK__S8`````0@Y@3($2`$P```#8`@``X)K__R`!````0@YP1(@$7`[P
M@`23"I0,F1:!`HD&D@B5#I80EQ*8%`*^"@YP0L%"R$+)0M)"TT+40M5"UD+7
M0MA"V4(.`$(+&````"@#``"PF___&`````!"#A!"@0)0P4(.`"P```!$`P``
MK)O__YH`````0@Y`2HD&D@B!`H@$DPIV"LA"P4+)0M)"TTH.`$0+`&````!T
M`P``%IS__YH$````2@Z0`V22"),*@0*(!(D&E`R5#I80EQ*8%`.``0K!0LA"
MTD+30M1"U4+60M="V$3)0@X`0@L#X`$*R$+!0LE"TD+30M1"U4+60M="V$(.
M`$0+```L````V`,``$R@__^:`0```$(.,$*(!$Z2"($"B08#/`$*P4+(0LE"
MTD(.`$(+```X````"`0``+:A__^<`````$(.,$B(!(D&@0*2"`(^"L%"R$+)
M0M)"#@!""P(X"LA"P4+21,E"#@!$"P`T````1`0``*ZW__^,`0```$(.8$R!
M`HD&B`22"),*`P0!"L%"R$+)0M-$TD(.`$(+`````````#@```!\!```WJ'_
M_W`2````1`[0!5R(!($"B0:2"),*E`R5#@,N$@K!1,A$R4321--$U$351`X`
M0@L``!P```"X!```$K3__U0`````0@X01H$"`DC!0@X`````-````-@$``!&
MM/__U@````!"#D!.B`2)!I((E`R!`I,*E0X"K@K!0LA"R4+20M-"U$+50@X`
M0@LT````$`4``.2T__^N`0```$(.,$R(!(D&E`R!`I((DPH#3`$*R$+!0LE"
MTD+30M1"#@!$"P```````````````````(Q&`0`````````````````!````
M`````-$"`````````0````````#<`@````````$`````````Y@(````````!
M`````````/0"`````````0`````````$`P````````\`````````5P0`````
M```9`````````!"@`0``````&P`````````(``````````0`````````2`(!
M```````%`````````/@-`0``````!@````````#@!`$```````H`````````
MGP0````````+`````````!@`````````%0````````````````````,`````
M````(*(!```````"`````````*`%````````%``````````'`````````!<`
M````````<!,!```````'`````````)@2`0``````"`````````!X!@``````
M``D`````````&```````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````@H`$``````!B@`0``````#*`!````
M```````````````````````````````````8H`$`````````````````````
M`````````````````/__________```````````0&0$``````!`9`0``````
M$!D!```````0&0$``````!`9`0``````$!D!```````0&0$``````!`9`0``
M````$!D!```````0&0$``````!`9`0``````$!D!```````0&0$``````!`9
M`0``````$!D!```````0&0$``````!`9`0``````$!D!```````0&0$`````
M`!`9`0``````$!D!```````0&0$``````!`9`0``````$!D!```````0&0$`
M`````!`9`0``````$!D!```````0&0$``````!`9`0``````$!D!```````0
M&0$``````!`9`0``````$!D!```````0&0$``````!`9`0``````$!D!````
M```0&0$``````!`9`0``````$!D!```````0&0$``````!`9`0``````$!D!
M```````0&0$``````!`9`0``````$!D!```````0&0$``````!`9`0``````
M$!D!```````0&0$``````!`9`0``````$!D!```````0&0$``````!`9`0``
M````$!D!```````0&0$``````!`9`0``````$!D!```````0&0$``````!`9
M`0``````$!D!``````"HI@$```````I005(N<&T*``````````")J\WO````
M`!A,`0``````1T-#.B`H3F5T0E-$(&YB,2`R,#(U,#<R,2D@,3(N-2XP`"1.
M971"4T0Z(&-R=#`N4RQV(#$N,R`R,#(S+S`U+S`W(#$R.C0Q.C0V('-K<FQL
M($5X<"`D`"1.971"4T0Z(&-R=#`M8V]M;6]N+F,L=B`Q+C,P(#(P,C4O,#4O
M,#(@,C,Z,#0Z,#8@<FEA<W1R861H($5X<"`D`"1.971"4T0Z(&-R=&DN4RQV
M(#$N,B`R,#$Y+S`R+S$P(#`W.C(X.C`V(&UA>6$@17AP("0`)$YE=$)31#H@
M8W)T8F5G:6XN8RQV(#$N,3<@,C`Q."\Q,B\R."`Q.#HQ-SHQ,2!C:')I<W1O
M<R!%>'`@)``D3F5T0E-$.B!C<G1E;F0N4RQV(#$N,2`R,#$T+S`Y+S$Y(#$W
M.C,V.C(U(&UA='0@17AP("0`05(```!R:7-C=@`!2`````00!7)V-C1I,G`Q
M7VTR<#!?83)P,5]F,G`R7V0R<#)?8S)P,%]Z:6-S<C)P,%]Z:69E;F-E:3)P
M,%]Z;6UU;#%P,```+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N;F5T8G-D+FED
M96YT`"YN;W1E+FYE=&)S9"YP87@`+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YR
M96QA+F1Y;@`N<F5L82YP;'0`+G1E>'0`+G)O9&%T80`N96A?9G)A;65?:&1R
M`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N:F-R`"YD>6YA;6EC`"YG;W0`+F=O
M="YP;'0`+F1A=&$`+G-D871A`"YS8G-S`"YB<W,`+F-O;6UE;G0`+G)I<V-V
M+F%T=')I8G5T97,`````````````````````````````````````````````
M```````````````````````````````````````````````````+`````0``
M``(```````````(!`````````@```````!<````````````````````!````
M````````````````$P````<````"`````````!@"`0``````&`(````````8
M````````````````````!````````````````````"8````'`````@``````
M```P`@$``````#`"````````%`````````````````````0`````````````
M```````W````!0````(`````````2`(!``````!(`@```````)@"````````
M!0`````````(``````````0`````````/0````L````"`````````.`$`0``
M````X`0````````8"0````````8````!````"``````````8`````````$4`
M```#`````@````````#X#0$``````/@-````````GP0`````````````````
M``$```````````````````!-````!`````(`````````F!(!``````"8$@``
M`````-@`````````!0`````````(`````````!@`````````5P````0```!"
M`````````'`3`0``````<!,```````"@!0````````4````2````"```````
M```8`````````%P````!````!@`````````0&0$``````!`9````````X`,`
M`````````````````!``````````$`````````!A`````0````8`````````
M\!P!``````#P'````````%(K```````````````````"````````````````
M````9P````$````"`````````$A(`0``````2$@```````"'00``````````
M````````"````````````````````&\````!`````@````````#0B0$`````
M`-")````````[`````````````````````0```````````````````!]````
M`0````(`````````P(H!``````#`B@```````$P%```````````````````(
M````````````````````AP````X````#`````````!"@`0``````$)``````
M```(````````````````````"``````````(`````````),````!`````P``
M```````8H`$``````!B0````````"`````````````````````@`````````
M``````````"8````!@````,`````````(*`!```````@D````````+`!````
M````!@`````````(`````````!``````````H0````$````#`````````-"A
M`0``````T)$```````!0````````````````````"``````````(````````
M`*8````!`````P`````````@H@$``````""2````````\`$`````````````
M``````@`````````"`````````"O`````0````,`````````$*0!```````0
ME````````!$````````````````````(````````````````````M0````$`
M```#`````````"BD`0``````*)0````````0````````````````````"```
M`````````````````+P````(`````P`````````XI`$``````#B4````````
M(`````````````````````@```````````````````#"````"`````,`````
M````@*0!```````XE````````,@"``````````````````!`````````````
M````````QP````$````P````````````````````.)0````````B````````
M`````````````0`````````!`````````!\````!````,```````````````
M`````%J4````````(@$```````````````````$``````````0````````#0
M`````P``<`````````````````````!\E0```````%,`````````````````
M```!`````````````````````0````,`````````````````````````SY4`
E``````#B`````````````````````0``````````````````````
