;;; format-spec.el --- functions for formatting arbitrary formatting strings

;; Copyright (C) 1999, 2000, 2001, 2002, 2003, 2004,
;;   2005, 2006, 2007, 2008 Free Software Foundation, Inc.

;; Author: Lars Magne Ingebrigtsen <larsi@gnus.org>
;; Keywords: tools

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Commentary:

;;; Code:

(eval-when-compile (require 'cl) (require 'wid-edit))

(eval-when-compile
 (unless (fboundp 'widget-parsing-escapes)
   (defmacro widget-parsing-escapes (control-string escape-symbol &rest body)
     "Insert text at point formatted according to CONTROL-STRING.

If there are any two-character %-sequences in CONTROL-STRING, for each of them
encountered, bind ESCAPE-SYMBOL to the second, non-% character, and execute
BODY.

BODY is usually a (case ...) expression responsible for the specific
interpretation of the %-sequence, inserting glyphs, widget values and so
forth.

Return nil."
     (declare (indent 1))
     `(let* ((w-p-e-string ,control-string)
             (w-p-e-max (1- (length w-p-e-string)))
             (w-p-e-last 0)
             (w-p-e-position (string-match-p "%" w-p-e-string w-p-e-last))
             ,escape-symbol)
       ;; Parse escapes in format. XEmacs; these format strings are short, so
       ;; this code isn't as much of a loop hotspot as you might worry.
       ;;
       ;; We used to insert the format string into the current buffer and
       ;; parse it there, using #'re-search-forward and then (replace-match ""
       ;; t t).  This turns out to be more expensive than parsing the string
       ;; and only inserting what you need to insert.
       ;;
       ;; SJT suggested that #'skip-chars-forward would be less expensive than
       ;; #'re-search-forward; against all intuition, this is not true, it
       ;; seems the compiled regexp caching does a better job than the
       ;; nonexistent #'skip-chars-forward caching, and the fact that
       ;; #'skip-chars-forward has a byte code doesn't outweigh that.
       ;;
       ;; On checking, #'position is faster than #'string-match-p, as
       ;; intuition would suggest. However, #'position is in Lisp pre
       ;; XEmacs-21.5, and is not available in GNU Emacs (they have renamed it
       ;; to cl-position and it remains in Lisp).
       (while w-p-e-position
         (if (eql w-p-e-position w-p-e-max) ;; Percent as the last character
                                            ;; in the string?
             (progn
               (insert (substring w-p-e-string w-p-e-last)) ;; Insert it.
               (setq w-p-e-position nil
                     w-p-e-last (length w-p-e-string)))
           ;; (insert (substring STR S E)) is equivalent to (write-sequence STR
           ;; (current-buffer) :start S :end E), except that the former is
           ;; faster since both function calls have byte-codes. That said, the
           ;; latter generates no garbage.
           (insert (substring w-p-e-string w-p-e-last w-p-e-position))
           (setq ,escape-symbol (aref w-p-e-string (1+ w-p-e-position))
                 w-p-e-last (+ w-p-e-position 2)
                 w-p-e-position (string-match-p "%" w-p-e-string w-p-e-last))
           ,@body))
       ;; Insert the trailing part of STRING.
       (when (<= w-p-e-last w-p-e-max)
         (insert (substring w-p-e-string w-p-e-last)))))))

(defun format-spec (format specification)
  "Return a string based on FORMAT and SPECIFICATION.
FORMAT is a string containing `format'-like specs like \"bash %u %k\",
while SPECIFICATION is an alist mapping from format spec characters
to values.  Any text properties on a %-spec itself are propagated to
the text that it generates."
  (with-temp-buffer
    (widget-parsing-escapes format escape
      (cond 
	;; Quoted percent sign.
	((eql escape ?%)
	 (insert (substring format (1- w-p-e-last) w-p-e-last)))
        ;; Valid format spec.
        ((string-match "\\([-0-9.]*\\)\\([a-zA-Z]\\)"
		       format (1- w-p-e-last))
         (let ((val (cdr (assq
			  (string-to-char (match-string 2 format))
			  specification))))
           (unless val
             (error "Invalid format character" (match-string 2 format)))
	   ;; Pad result to desired length, insert 
	   (insert (format (concat "%" (match-string 1 format) "s")
                           val))))
        (t
         (error "Invalid format string" format))))
    (buffer-string)))

(defun format-spec-make (&rest pairs)
  "Return an alist suitable for use in `format-spec' based on PAIRS.
PAIRS is a list where every other element is a character and a value,
starting with a character."
  (let (alist)
    (while pairs
      (unless (cdr pairs)
	(error "Invalid list of pairs"))
      (push (cons (car pairs) (cadr pairs)) alist)
      (setq pairs (cddr pairs)))
    (nreverse alist)))

(provide 'format-spec)

;;; arch-tag: c22d49cf-d167-445d-b7f1-2504d4173f53
;;; format-spec.el ends here
