/*
 * Table.java
 *
 * Created on 31 de julio de 2002, 09:58
 */

package pgclibrary.objects;
import pgclibrary.dataaccess.*;
import pgclibrary.utilities.*;
import java.sql.*;
import java.util.*;

/**
 *
 * @author  nestor
 */
public class Table extends DBObject {
    private long pages_number = 0;
    private Vector columns = null;
    private Vector index = null;
    private String description = null;
    
    /** Creates a new instance of Table */
    public Table(DBObject owner, long oid) {
        super(owner);
        search(oid);
    }
    public Table(DBObject owner) {
        super(owner);
    }
    
    public void insertRecord(Vector values) throws SQLException {
        PreparedStatement prep = prepareInsertStatement();
        for(int i=0; i< getColumns().size(); i++){
            setStatementValue(prep, (String) values.get(i),  (TableColumn) getColumns().get(i), i+1);
        }
        prep.executeUpdate();
        prep.close();
    }
    public void updateRecord(Vector values) throws SQLException {
        PreparedStatement prep = prepareUpdateStatement();
        
        int values_idx = 0;
        for(int i=0; i< getColumns().size(); i++){
            setStatementValue(prep, (String) values.get(values_idx),  (TableColumn) getColumns().get(i), values_idx+1);
            values_idx++;
        }
        Vector cols = ((TableIndex) getPrimaryKeys().get(0)).getColumns();
        for (int i=0; i< cols.size(); i++) {
            setStatementValue(prep, (String) values.get(values_idx),  (TableColumn) cols.get(i), values_idx+1);
            values_idx++;
        }
        
        prep.executeUpdate();
        prep.close();
    }
    public void deleteRecord(Vector values) throws SQLException {
        PreparedStatement prep = prepareDeleteStatement();
        Vector cols = ((TableIndex) getPrimaryKeys().get(0)).getColumns();
        for (int i=0; i< cols.size(); i++) {
            setStatementValue(prep, (String) values.get(i),  (TableColumn) cols.get(i), i+1);
        }
        prep.executeUpdate();
        prep.close();
    }
    
    /** Retrieve the SQL string that eliminate this object from
     * the database.
     * @return SQL DML string.
     */
    public String getDeleteSQL() {
        StringBuffer retorno = new StringBuffer();
        retorno.append("DROP TABLE ");
        retorno.append(getName());
        retorno.append("; \n\t");
        return retorno.toString();
    }
    
    /** Returns the SQL string that
     * generate this object.
     * @return SQL DML instruction for generation.
     */
    public String getInsertSQL() {
        StringBuffer retorno = new StringBuffer();
        retorno.append("CREATE TABLE ");
        retorno.append(getName());
        retorno.append(" ( \n\t");
        for(int i=0; i<getColumns().size(); i++){
            TableColumn tc = (TableColumn) getColumns().get(i);
            retorno.append(tc.getInsertSQL());
            retorno.append(", \n\t");
        }
        
        Vector tmp = getPrimaryKeys();
        for(int i=0; i<tmp.size(); i++){
            retorno.append(((TableIndex) tmp.get(i)).getInsertSQL());
        }
        
        retorno.append(" );\n");
        
        tmp = getNotPrimaryKeyIndex();
        for(int i=0; i<tmp.size(); i++){
            retorno.append(((TableIndex) tmp.get(i)).getInsertSQL());
            retorno.append("\n");
        }
        return retorno.toString();
    }
    
    /** Retrieve the SQL String that implements this
     * object modifications.
     * @return SQL DML String.
     */
    public String getUpdateSQL() {
        return new String();
    }
    
    /** Find the object asociated with the OID.
     * @param _oid OID to search.
     * @return true if the object was found
     */
    public boolean search(long _oid) {
        boolean retorno = false;
        try{
            initialize();
            PreparedStatement prep = getConnection().prepareStatement("SELECT OID, * FROM PG_CLASS WHERE OID=?");
            prep.setLong(1, _oid);
            ResultSet rs = prep.executeQuery();
            if(rs.next()){
                setName(rs.getString("RELNAME"));
                setOID(rs.getLong("OID"));
                setNumberOfPages(rs.getLong("RELPAGES"));
                setCreator(new User(this, rs.getLong("RELOWNER")));
                retorno = true;
            }
            rs.close();
            prep.close();
        } catch (SQLException e){
        }
        return retorno;
    }
    
    public void initialize(){
        super.initialize();
        pages_number = 0;
        columns = null;
        index = null;
        description = null;
    }
    
    public dataSource getColumnsTable(){
        dataSourceVector dsve = new dataSourceVector();
        dsve.addColumnName("Col #");
        dsve.addColumnName("Name");
        dsve.addColumnName("Type");
        dsve.addColumnName("Size");
        dsve.addColumnName("Nulls");
        dsve.addColumnName("Default");
        dsve.addColumnName("Description");
        dsve.addColumnSize(10);
        dsve.addColumnSize(20);
        dsve.addColumnSize(15);
        dsve.addColumnSize(5);
        dsve.addColumnSize(5);
        dsve.addColumnSize(30);
        dsve.addColumnSize(30);
        
        Vector temp = getColumns();
        for(int i=0; i<temp.size(); i++){
            TableColumn tc = (TableColumn) temp.get(i);
            
            Vector elem = new Vector();
            elem.add(String.valueOf(tc.getColumnNumber()));
            elem.add(tc.getName());
            elem.add(tc.getColumnType().getName());
            elem.add(String.valueOf(tc.getSize()));
            elem.add(Utilities.formatBoolean(tc.allowNulls()));
            elem.add(tc.getDefaultValue());
            elem.add(tc.getDescription());
            
            dsve.addElement(elem);
        }
        return dsve;
    }
    public dataSource getIndexTable(){
        dataSourceVector dsve = new dataSourceVector();
        dsve.addColumnName("Name");
        dsve.addColumnName("Primary");
        dsve.addColumnName("Unique");
        dsve.addColumnName("Cluster");
        dsve.addColumnName("Columns");
        dsve.addColumnSize(20);
        dsve.addColumnSize(6);
        dsve.addColumnSize(6);
        dsve.addColumnSize(6);
        dsve.addColumnSize(30);
        
        Vector temp = getIndexes();
        for(int i=0; i<temp.size(); i++){
            TableIndex ti = (TableIndex) index.get(i);
            
            Vector elem = new Vector();
            elem.add(ti.getName());
            elem.add(Utilities.formatBoolean(ti.isPrimary()));
            elem.add(Utilities.formatBoolean(ti.isUnique()));
            elem.add(Utilities.formatBoolean(ti.isClustered()));
            elem.add(ti.getColumnsText());
            dsve.addElement(elem);
        }
        return dsve;
    }
    
    // -------SET AND GET ----------
    private void setNumberOfPages(long _pages_number){
        pages_number = _pages_number;
    }
    public long getNumberOfPages(){
        return pages_number;
    }
    
    public Vector getColumns(){
        if(columns == null){
            columns = new Vector();
            try{
                PreparedStatement prep = getConnection().prepareStatement("select "
                + " ATTNUM from pg_attribute "
                + "  where attnum > 0 and  attrelid = ? ORDER BY attnum");
                prep.setLong(1, getOID());
                ResultSet rs = prep.executeQuery();
                while(rs.next()){
                    TableColumn tc = new TableColumn(this, rs.getLong("ATTNUM"));
                    columns.add(tc);
                }
                rs.close();
                prep.close();
            } catch (SQLException e){}
        }
        return columns;
    }
    public int getColumnIndex(String _column){
        _column = _column.toUpperCase();
        int retorno = -1;
        Vector cols = getColumns();
        for(int i =0 ; (i<cols.size()) && (retorno < 0) ; i++){
            if (((TableColumn) cols.get(i)).getName().equals(_column)){
                retorno = i;
            }
        }
        return retorno;
    }
    
    private Vector getIndexes(){
        if(index == null){
            index = new Vector();
            try{
                PreparedStatement prep = getConnection().prepareStatement("select "
                + "indexrelid as oid from pg_index   where indrelid = ?");
                prep.setLong(1, getOID());
                ResultSet rs = prep.executeQuery();
                while(rs.next()){
                    TableIndex ti = new TableIndex(this, rs.getLong("OID"));
                    index.add(ti);
                }
                rs.close();
                prep.close();
            } catch (SQLException e){
            }
        }
        return index;
    }
    public Vector getPrimaryKeys(){
        Vector retorno = new Vector();
        Vector tmp = getIndexes();
        for(int i=0;i < tmp.size(); i++){
            if(((TableIndex) tmp.get(i)).isPrimary()){
                retorno.add(tmp.get(i));
            }
        }
        return retorno;
    }
    public Vector getNotPrimaryKeyIndex(){
        Vector retorno = new Vector();
        Vector tmp = getIndexes();
        for(int i=0;i < tmp.size(); i++){
            if(!((TableIndex) tmp.get(i)).isPrimary()){
                retorno.add(tmp.get(i));
            }
        }
        return retorno;
    }

    private PreparedStatement prepareInsertStatement() throws SQLException{
        StringBuffer sqlComm = new StringBuffer();
        sqlComm.append("INSERT INTO ");
        sqlComm.append(getName());
        sqlComm.append(" ( ");
        for(int i=0; i<getColumns().size(); i++){
            TableColumn tc = (TableColumn) getColumns().get(i);
            sqlComm.append(tc.getName());
            if( i<getColumns().size()-1){
                sqlComm.append(", ");
            }
        }
        sqlComm.append(" ) VALUES ( ");
        for(int i=0; i<getColumns().size(); i++){
            sqlComm.append(" ?");
            if( i<getColumns().size()-1){
                sqlComm.append(", ");
            }
        }
        sqlComm.append(" )");
        return getConnection().prepareStatement(sqlComm.toString());
    }
    
    private PreparedStatement prepareUpdateStatement() throws SQLException{
        StringBuffer sqlComm = new StringBuffer();
        sqlComm.append("UPDATE ");
        sqlComm.append(getName());
        sqlComm.append(" SET ");
        for(int i=0; i<getColumns().size(); i++){
            TableColumn tc = (TableColumn) getColumns().get(i);
            sqlComm.append(tc.getName());
            sqlComm.append("= ? ");
            if( i<getColumns().size()-1){
                sqlComm.append(", ");
            }
        }
        sqlComm.append(" WHERE ");
        Vector cols = ((TableIndex) getPrimaryKeys().get(0)).getColumns();
        for(int i=0; i<cols.size(); i++){
            TableColumn tc = (TableColumn) cols.get(i);
            sqlComm.append(tc.getName());
            sqlComm.append("=? ");
            if(i < cols.size()-1){
                sqlComm.append(" and ");
            }
        }
        return getConnection().prepareStatement(sqlComm.toString());
    }
    
    private PreparedStatement prepareDeleteStatement() throws SQLException{
        StringBuffer sqlComm = new StringBuffer();
        sqlComm.append("DELETE FROM ");
        sqlComm.append(getName());
        sqlComm.append(" WHERE ");
        Vector cols = ((TableIndex) getPrimaryKeys().get(0)).getColumns();
        for(int i=0; i<cols.size(); i++){
            TableColumn tc = (TableColumn) cols.get(i);
            sqlComm.append(tc.getName());
            sqlComm.append("=? ");
            if(i < cols.size()-1){
                sqlComm.append(" and ");
            }
        }
        return getConnection().prepareStatement(sqlComm.toString());
    }
    
    private void setStatementValue(PreparedStatement prep, String value, TableColumn tc, int _prepcolumn){
        int type = (int) tc.getColumnType().getSQLType();
        if(type < 0){
            type = Types.CHAR;
        }
        
        try{
            if ((value != null) && (value.length() > 0)){
                switch (type){
                    case Types.BIT:
                        prep.setBoolean(_prepcolumn, Utilities.parseBoolean(value));
                        break;
                    case Types.CHAR:
                        prep.setString(_prepcolumn, value);
                        break;
                    case Types.INTEGER:
                        prep.setInt(_prepcolumn, Integer.parseInt(value));
                        break;
                    case Types.DATE:
                        prep.setDate(_prepcolumn, new java.sql.Date(Utilities.parseDate(value, Utilities.FORMAT_DATE).getTime()));
                        break;
                    case Types.TIME:
                        prep.setTime(_prepcolumn, new java.sql.Time(Utilities.parseDate(value, Utilities.FORMAT_TIME).getTime()));
                        break;
                    case Types.TIMESTAMP:
                        prep.setTimestamp(_prepcolumn, new java.sql.Timestamp(Utilities.parseDate(value, Utilities.FORMAT_DATETIME).getTime()));
                        break;
                    case Types.DOUBLE:
                        prep.setDouble(_prepcolumn, Double.parseDouble(value));
                        break;
                }
            } else {
                prep.setNull(_prepcolumn, type);
            }
        } catch (Exception e){
            e.printStackTrace();
        }
    }
    
    public String getDescription(){
        if(description == null){
            try{
                PreparedStatement prep;
                
                if(getConnection().getDatabaseVersion() < 7.2){
                    prep = getConnection().prepareStatement("select obj_description(oid) as DESCRIPTION from pg_class where oid=? ");
                    prep.setLong(1, getOID());
                } else {
                    prep = getConnection().prepareStatement("select obj_description(oid, relname) as DESCRIPTION from pg_class where oid=? ");
                    prep.setLong(1, getOID());
                    prep.setString(2, getName());
                }
                
                ResultSet rs = prep.executeQuery();
                if(rs.next()){
                    setDescription(rs.getString("DESCRIPTION"));
                } else {
                    setDescription(new String());
                }
                rs.close();
                prep.close();
            } catch (SQLException e){}
        }
        return description;
    }
    public void setDescription(String _description){
        if(_description != null){
            description = _description;
        } else {
            description = new String();
        }
    }
    
    /** refresh the data from database.
     * if oid>0. Elsewere no action.
     */
    public void refresh() {
        search(getOID());
    }
}
