#include <arpa/inet.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <netinet/ether.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>


#include <iostream>
#include <map>
#include <list>
#include <string>

#include <pcap.h>

using namespace std;

#include "sqlwrite.h"
#include "ip.h"
#include "sll.h"
#include "dns.h"
#include "prio.h"

DnsRecord::DnsRecord(struct in_addr ip, string& hostName, int serial, time_t last_access) :
	ip_m(ip), hostName_m(hostName), serial_m(serial), last_access_m(last_access), byDate_m(0) {

}

DnsRecord::DnsRecord(struct in_addr ip, string hostName) :
	ip_m(ip), hostName_m(hostName), serial_m(0), last_access_m(0), byDate_m(0) {
}

void DnsCache::FlushOld()
{
	time_t	now = time((time_t *)0);

	for (DnsRecordsByDateReverseIter iter = recordsByDate_m.rbegin(); iter != recordsByDate_m.rend(); iter++) {
		DnsRecord *dnsRec = *iter;
		if (dnsRec->last_access_m < now - expiryTime_m) {
			// delete from map
			ipToRecord_m.erase(dnsRec);
			// delete from list
			recordsByDate_m.erase(dnsRec->byDate_m);
			delete dnsRec;
		} else 
			break;
	}
}

int	DnsCache::FindHostFromIP(const struct in_addr& ip) {
	char			buf[8192];  // make non local or malloc
	uint32_t addr;
	struct hostent  hpb;
	struct hostent *hp = &hpb;
	int	res;
	int	err;
	string	resolvedName;

	memset(buf, 0, sizeof (buf));
	memset(&hpb, 0, sizeof (hpb));
	// gethost by name using thread safe version
	res = gethostbyaddr_r((char *)&ip, sizeof (struct in_addr),
								 AF_INET, 
								 &hpb,
								 buf, sizeof(buf),
								 &hp, &err);
	if (errno == 0) {
		// convert to lower case
		if (hp == 0) {
			resolvedName = inet_ntoa(ip);
			//cerr << "resolved name '" << resolvedName << "' returned null host entry, res was " << res << endl;
		} else {
			for (char *cp = hp->h_name; *cp; cp++)
				*cp = tolower(*cp);
			resolvedName = hp->h_name;
		}
	} else if (errno == ENOENT) {
		resolvedName = inet_ntoa(ip);
	} else {
		perror("gethostbyaddr_r: ");
		return -1;
	}
	// create temporary Dns record for lookup
	DnsRecord *inDns = new DnsRecord(ip, resolvedName);
	DnsRecord *newDns;

	DnsRecordMapIter		dnsRec = ipToRecord_m.find(inDns);

	if (dnsRec == ipToRecord_m.end()) {
		newDns = new DnsRecord(*inDns);
		newDns->serial_m = SQLGetDNS(resolvedName, ip);
		ipToRecord_m[inDns] = newDns;
	} else { // found
		newDns = (*dnsRec).second;
		delete inDns;
		recordsByDate_m.erase(newDns->byDate_m);
	}
	newDns->last_access_m = time((time_t *)0);
	recordsByDate_m.push_front(newDns);
	newDns->byDate_m = recordsByDate_m.begin();
	return newDns->serial_m;
}

