/*--------------------------------------------------------------------
 * FILE:
 *     replicate_com.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at backend for the replication.
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */

/*--------------------------------------
 * INTERFACE ROUTINES
 *
 * setup/teardown:
 *      PGR_Close_Sock
 *      PGR_Free_Conf_Data
 * I/O call:
 *      PGR_Create_Send_Socket
 *      PGR_Create_Recv_Socket
 *      PGR_Create_Acception
 * table handling:
 *      PGR_Get_Conf_Data
 *-------------------------------------
 */
#ifdef USE_REPLICATION

#include "postgres.h"

#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/in.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#include <arpa/inet.h>
#include <sys/file.h>
#include <netdb.h>

#include "libpq/libpq.h"
#include "miscadmin.h"
#include "nodes/print.h"
#include "utils/guc.h"
#include "parser/parser.h"
#include "access/xact.h"
#include "replicate_com.h"

int PGR_Create_Send_Socket(int * fdP, char * hostName , unsigned short portNumber);
int PGR_Create_Recv_Socket(char * hostName , unsigned short portNumber);
int PGR_Create_Unix_Domain_Socket(char * sock_dir, unsigned short port);
int PGR_Create_Acception(int fd, char * hostName , unsigned short portNumber);
void PGR_Close_Sock(int * sock);
int PGR_Free_Conf_Data(void);
int PGR_Get_Conf_Data(char * dir , char * fname);
void PGRset_recovery_packet_no(RecoveryPacket * packet, int packet_no);
unsigned int PGRget_ip_by_name(char * host);
int PGRget_time_value(char *str);
int PGRget_bw_value(char *str);

static char * get_string(char * buf);
static bool is_start_tag(char * ptr);
static bool is_end_tag(char * ptr);
static void init_conf_data(ConfDataType *conf);
static int get_key(char * key, char * str);
static int get_conf_key_value(char * key, char * value , char * str);
static int add_conf_data(char * parent_table,int parent_rec_no,char *table,int rec_no, char *key,char * value);
static int get_table_data(FILE * fp,char * table, int rec_no, char * parent_table, int parent_rec_no);
static int get_single_data(char * str);
static int get_conf_file(char * fname);

/*--------------------------------------------------------------------
 * SYMBOL
 *     PGR_Create_Send_Socket()
 * NOTES
 *     create new socket
 * ARGS
 *    int * fdP:
 *    char * hostName:
 *    unsigned short portNumber:
 * RETURN
 *    OK: STATUS_OK
 *    NG: STATUS_ERROR
 *--------------------------------------------------------------------
 */
int
PGR_Create_Send_Socket(int * fdP, char * hostName , unsigned short portNumber)
{
	int sock;
	size_t	len = 0;
	struct sockaddr_in addr;
	int fd;
	int one = 1;

	memset((char *)&addr,0,sizeof(addr));

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		* fdP = -1;
		return STATUS_ERROR;
	}
	if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))) == -1)
	{
		PGR_Close_Sock(&fd);
		* fdP = -1;
		return STATUS_ERROR;
	}
	if ((setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, (char *) &one, sizeof(one))) == -1)
	{
		PGR_Close_Sock(&fd);
		* fdP = -1;
		return STATUS_ERROR;
	}
	if (setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, (char *) &one, sizeof(one)) < 0)
	{
		PGR_Close_Sock(&fd);
		* fdP = -1;
		return STATUS_ERROR;
	}
	
	addr.sin_family = AF_INET;
	if ((hostName == NULL) || (hostName[0] == '\0'))
   		addr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		struct hostent *hp;

		hp = gethostbyname(hostName);
		if ((hp == NULL) || (hp->h_addrtype != AF_INET))
		{
			PGR_Close_Sock(&fd);
			* fdP = -1;
			return STATUS_ERROR;
		}
		memmove((char *) &(addr.sin_addr), (char *) hp->h_addr, hp->h_length);
	}

	addr.sin_port = htons(portNumber);
	len = sizeof(struct sockaddr_in);
	
	if ((sock = connect(fd,(struct sockaddr*)&addr,len)) < 0)
	{
		PGR_Close_Sock(&fd);
		* fdP = -1;
		return STATUS_ERROR;
	}
	
	* fdP = fd;
	return	STATUS_OK;
}

int
PGR_Create_Recv_Socket(char * hostName , unsigned short portNumber)
{
	int fd,err;
	size_t	len = 0;
	struct sockaddr_in addr;
	int one = 1;

	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		return -1;
	}
	if ((setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *) &one, sizeof(one))) == -1)
	{
		PGR_Close_Sock(&fd);
		return -1;
	}
	addr.sin_family = AF_INET;
	if ((hostName == NULL) || (hostName[0] == '\0'))
		addr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		struct hostent *hp;

		hp = gethostbyname(hostName);
		if ((hp == NULL) || (hp->h_addrtype != AF_INET))
		{
			PGR_Close_Sock(&fd);
			return -1;
		}
		memmove((char *) &(addr.sin_addr), (char *) hp->h_addr, hp->h_length);
	}

	addr.sin_port = htons(portNumber);
	len = sizeof(struct sockaddr_in);
	
	err = bind(fd, (struct sockaddr *) & addr, len);
	if (err < 0)
	{
		PGR_Close_Sock(&fd);
		return -1;
	}
	err = listen(fd, MAX_SOCKET_QUEUE);
	if (err < 0)
	{
		PGR_Close_Sock(&fd);
		return -1;
	}
	return	fd;
}

int 
PGR_Create_Unix_Domain_Socket(char * sock_dir, unsigned short port)
{
	char * func = "PGR_Create_Unix_Domain_Socket()";
	struct sockaddr_un addr;
	int fd;
	int status;
	int len;

	/* set unix domain socket path */
	fd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (fd == -1)
	{
		fprintf(stderr,"%s:Failed to create UNIX domain socket. reason: %s\n",func,  strerror(errno));
		return -1;
	}
	memset((char *) &addr, 0, sizeof(addr));
	((struct sockaddr *)&addr)->sa_family = AF_UNIX;
	snprintf(addr.sun_path, sizeof(addr.sun_path), "%s/.s.PGSQL.%d",sock_dir,port);
	len = sizeof(struct sockaddr_un);
	status = bind(fd, (struct sockaddr *)&addr, len);
	if (status == -1)
	{
		fprintf(stderr,"%s: bind() failed. reason: %s\n", func, strerror(errno));
		return -1;
	}

	if (chmod(addr.sun_path, 0777) == -1)
	{
		fprintf(stderr,"%s: chmod() failed. reason: %s\n", func, strerror(errno));
		return -1;
	}

	status = listen(fd, MAX_SOCKET_QUEUE);
	if (status < 0)
	{
		fprintf(stderr,"%s: listen() failed. reason: %s\n", func, strerror(errno));
		return -1;
	}
	return fd;
}

int
PGR_Create_Acception(int fd, char * hostName , unsigned short portNumber)
{
	char * func = "PGR_Create_Acception()";
	int sock;
	struct sockaddr  addr;
	socklen_t	len = 0;
	int one = 1;
	int count;

	len = sizeof(struct sockaddr);
	count = 0;
	while ((sock = accept(fd,&addr,&len)) < 0)
	{
		fprintf(stderr,"%s:accept error\n",func);
		PGR_Close_Sock(&fd);
		if ( count > MAX_RETRY_TIMES )
		{
			return -1;
		}
		fd = PGR_Create_Recv_Socket(hostName , portNumber);
		count ++;
	}
	
	count = 0;
	while (setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *) &one, sizeof(one)) < 0)
	{
		fprintf(stderr,"%s: setsockopt TCP_NODELAY error (%s)\n",func, strerror(errno));
		if ( count > MAX_RETRY_TIMES )
		{
			return -1;
		}
		count ++;
	}
	count = 0;
	while (setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *) &one, sizeof(one)) < 0)
	{
		fprintf(stderr,"%s:setsockopt SO_KEEPALIVE error (%s)\n",func,strerror(errno));
		if ( count > MAX_RETRY_TIMES )
		{
			return -1;
		}
		count ++;
	}

	return	sock;
}

void
PGR_Close_Sock(int * sock)
{
	close( (int)*sock);
	*sock = -1;
}

int
PGR_Read_Packet(int sock, char * packet, int size)
{
	int r;
	char * read_ptr;
	int read_size = 0;

	if ((packet == NULL) || (size <= 0))
	{
		return -1;
	}
	read_ptr = (char*)packet;

	for (;;){
		r = recv(sock,read_ptr + read_size ,size, MSG_WAITALL);
		if (r < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				continue;
			} else {
				fprintf(stderr, "read_packet():recv failed\n");
				return -1;
			}
		} else if (r == 0) {
			fprintf(stderr, "read_packet():unexpected EOF\n");
			return -1;
		} else /*if (r > 0)*/ {
			read_size += r;
			if (read_size == size) {
				return read_size;
			}
		}
	}
	return -1;
}

int
PGR_Send_Packet(int  sock, char * packet, int size)
{
	int rtn = 0;
	char * send_ptr = NULL;
	int send_size= 0;
	int buf_size = 0;
	int s;
	fd_set	  wmask;
	struct timeval timeout;

	timeout.tv_sec = 60;
	timeout.tv_usec = 0;

	for (;;)
	{
		FD_ZERO(&wmask);
		FD_SET(sock,&wmask);
		rtn = select(sock+1, (fd_set *)NULL, &wmask, (fd_set *)NULL, &timeout);
		if (rtn < 0)
		{
			if (errno == EINTR || errno == EAGAIN)
			{
				continue;
			}
			break;
		}
		else if (rtn && FD_ISSET(sock, &wmask))
		{
			send_ptr = (char *)packet;
			buf_size = size;
			s = send(sock,send_ptr + send_size,buf_size - send_size ,0);
			if (s < 0)
			{
				if (errno == EINTR || errno == EAGAIN)
				{
					continue;
				}
				fprintf(stderr, "send_recovery_packet():send error\n");
				/* EPIPE || ENCONNREFUSED || ENSOCK || EHOSTUNREACH */
				return STATUS_ERROR;
			}
			else if (s == 0)
			{
				fprintf(stderr, "send_recovery_packet():unexpected EOF\n");
				return STATUS_ERROR;
			}
			else /*if (s > 0)*/
			{
				send_size += s;
				if (send_size == buf_size)
				{
					return STATUS_OK;
				}
			}
		}
	}
	return STATUS_ERROR;
}

static char *
get_string(char * buf)
{
	int i,len1,len2,start_flag;
	char *readp, *writep; 

	writep = readp = buf;
	i = len1 = 0;
	while (*(readp +i) != '\0')
	{
		if (!isspace(*(readp+ i)))
		{
			len1 ++;
		}
		i++;
	}
	start_flag = len2 = 0;
	while (*readp != '\0')
	{
		if (*readp == '#') 
		{
			*writep = '\0';
			break;
		}
		if (isspace(*readp))
		{
			if ((len2 >= len1) || (!start_flag))
			{
				readp++;
				continue;
			}
			*writep = *readp;
		}
		else
		{
			start_flag = 1;
			*writep = *readp;
			len2 ++;
		}
		readp ++;
		writep ++;
	}
	*writep = '\0';
	return buf;
}

static bool
is_start_tag(char * ptr)
{
	if ((*ptr == '<') && (*(ptr+1) != '/'))
	{
		return true;
	}
	return false;
}

static bool
is_end_tag(char * ptr)
{
	if ((*ptr == '<') && (*(ptr+1) == '/'))
	{
		return true;
	}
	return false;
}

static void
init_conf_data(ConfDataType *conf)
{
	memset(conf->parent_table,0,sizeof(conf->parent_table));
	memset(conf->table,0,sizeof(conf->table));
	memset(conf->key,0,sizeof(conf->key));
	memset(conf->value,0,sizeof(conf->value));
	conf->parent_rec_no = 0;
	conf->rec_no = 0;
	conf->last = NULL;
	conf->next = NULL;
}

static int
get_key(char * key, char * str)
{
	int offset = 1;
	char * ptr_s,*ptr_e;

	ptr_s = strchr(str,'<');
	if (ptr_s == NULL)
	{
		return STATUS_ERROR;
	}
	if (*(ptr_s+1) == '/')
	{
		offset = 2;
	}
	ptr_e = strchr(str,'>');
	if (ptr_e == NULL)
	{
		return STATUS_ERROR;
	}
	*ptr_e = '\0';
	strcpy(key,ptr_s + offset);
	*ptr_e = '>';
	return STATUS_OK;
}

static int
get_conf_key_value(char * key, char * value , char * str)
{
	int i;
	int len1,len2,start_flag;
	char * ptr_s,*ptr_e;

	if(get_key(key,str) == STATUS_ERROR)
	{
		return STATUS_ERROR;
	}
	ptr_e = strchr(str,'>');
	if (ptr_e == NULL)
	{
		return STATUS_ERROR;
	}
	ptr_s = ptr_e + 1;

	len1 = 0;
	while ((*ptr_s != '<') && (*ptr_s != '\0'))
	{
			if (! isspace(*ptr_s))
			{
				len1 ++;
			}
			ptr_s ++;
	}
	ptr_s = ptr_e + 1;
	i = len2 = start_flag = 0;
	while ((*ptr_s != '<') && (*ptr_s != '\0'))
	{
		if (isspace(*ptr_s))
		{
			if ((len2 >= len1) || (!start_flag))
			{
				ptr_s ++;
				continue;
			}
			*(value + i) = *ptr_s;
		}
		else
		{
			start_flag = 1;
			*(value + i) = *ptr_s;
			len2 ++;
		}
		i++;
		ptr_s ++;
	}
	*(value + i) = '\0';
	return STATUS_OK;
}

static int
add_conf_data(char * parent_table, int parent_rec_no, char *table,int rec_no, char *key,char * value)
{
	ConfDataType * conf_data;
	ConfDataType * previous_data;

	conf_data = (ConfDataType *)malloc(sizeof(ConfDataType));
	if (conf_data == NULL)
	{
		return STATUS_ERROR;
	}
	init_conf_data(conf_data);
	if (table != NULL)
	{
		memcpy(conf_data->table,table,sizeof(conf_data->table));
		conf_data->rec_no = rec_no;
	}
	if (parent_table != NULL)
	{
		memcpy(conf_data->parent_table,parent_table,sizeof(conf_data->parent_table));
		conf_data->parent_rec_no = parent_rec_no;
	}
	memcpy(conf_data->key,key,sizeof(conf_data->key));
	memcpy(conf_data->value,value,sizeof(conf_data->value));
	if (ConfData_Top == (ConfDataType *)NULL)
	{
		ConfData_Top = conf_data;
		conf_data->last = (char *)NULL;
	}
	if (ConfData_End == (ConfDataType *)NULL)
	{
		conf_data->last = (char *)NULL;
	}
	else
	{
		conf_data->last = (char *)ConfData_End;
		ConfData_End->next = (char *)conf_data;
	}
	ConfData_End = conf_data;
	conf_data->next = (char *)NULL;
	return STATUS_OK;
}

static int
get_table_data(FILE * fp,char * table, int rec_no, char * parent_table, int parent_rec_no)
{
	char buf[1024];
	char key_buf[1024];
	char value_buf[1024];
	char last_key_buf[1024];
	int len = 0;
	char * ptr;
	int sub_rec_no = 0;

	memset(last_key_buf,0,sizeof(last_key_buf));
	memset(key_buf,0,sizeof(key_buf));
	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		/*
		 * pic up a data string
		 */
		ptr = get_string(buf);
		len = strlen(ptr);
		if (len == 0)
		{
			continue;
		}
		if (is_end_tag(ptr))
		{
			if(get_key(key_buf,ptr) == STATUS_ERROR)
			{
				return STATUS_ERROR;
			}
			if (!strcmp(key_buf,table))
			{
				return STATUS_OK;
			}
		}
		if (is_start_tag(ptr))
		{
			if (strstr(ptr,"</") == NULL)
			{
				if(get_key(key_buf,ptr) == STATUS_ERROR)
				{
					return STATUS_ERROR;
				}
				if (strcmp(last_key_buf,key_buf))
				{
					sub_rec_no = 0;
					strcpy(last_key_buf,key_buf);
				}
				get_table_data(fp,key_buf,sub_rec_no,table,rec_no);
				sub_rec_no ++;
			}
			else
			{
				if(get_conf_key_value(key_buf,value_buf,ptr) == STATUS_ERROR)
				{
					return STATUS_ERROR;
				}
				add_conf_data(parent_table,parent_rec_no,table,rec_no,key_buf,value_buf);
			}
		}
	}
	return STATUS_ERROR;
}

static int
get_single_data(char * str)
{
	char key_buf[1024];
	char value_buf[1024];
	if(get_conf_key_value(key_buf,value_buf,str) == STATUS_ERROR)
	{
		return STATUS_ERROR;
	}
	add_conf_data(NULL,0,NULL,0,key_buf,value_buf);
	return STATUS_OK;
}


static int
get_conf_file(char * fname)
{
	FILE * fp = NULL;
	int len;
	char buf[1024];
	char key_buf[1024];
	char last_key_buf[1024];
	char *ptr;
	int rec_no = 0;

	/*
	 * configuration file open
	 */
	if ((fp = fopen(fname,"r")) == NULL)
	{
		return STATUS_ERROR;
	}
	/*
	 * configuration file read
	 */
	memset(last_key_buf,0,sizeof(last_key_buf));
	memset(key_buf,0,sizeof(key_buf));
	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
		/*
		 * pic up a data string
		 */
		ptr = get_string(buf);
		len = strlen(ptr);
		if (len == 0)
		{
			continue;
		}
		if (is_start_tag(ptr))
		{
			if(get_key(key_buf,ptr) == STATUS_ERROR)
			{
				fclose(fp);
				return STATUS_ERROR;
			}
			if (strstr(ptr,"</") == NULL)
			{
				if (strcmp(last_key_buf,key_buf))
				{
					rec_no = 0;
					strcpy(last_key_buf,key_buf);
				}
				get_table_data(fp,key_buf,rec_no,NULL,0);
				rec_no ++;
			}
			else
			{
				get_single_data(ptr);
			}
		}
	}
	fclose(fp);
	return STATUS_OK;
}

int
PGR_Free_Conf_Data(void)
{
	ConfDataType * conf, *nextp;

	if (ConfData_Top == (ConfDataType *)NULL)
	{
		return STATUS_ERROR;
	}
	conf = ConfData_Top;

	while (conf != (ConfDataType *)NULL)
	{
		nextp = (ConfDataType*)conf->next;
		free (conf);
		conf = nextp;
	}
	ConfData_Top = ConfData_End = (ConfDataType *)NULL;
	return STATUS_OK;
}

int
PGR_Get_Conf_Data(char * dir , char * fname)
{

	int status;

	char * conf_file;
	if ((dir == NULL) || ( fname == NULL))
	{
		return STATUS_ERROR;
	}
	conf_file = malloc(strlen(dir) + strlen(fname) + 2);
	if (conf_file == NULL)
	{
		return STATUS_ERROR;
	}
	sprintf(conf_file,"%s/%s",dir,fname);

	ConfData_Top = ConfData_End = (ConfDataType * )NULL;
	status = get_conf_file(conf_file);
	free (conf_file);
	conf_file = NULL;

	return status;
}

void
PGRset_recovery_packet_no(RecoveryPacket * packet, int packet_no)
{
	if (packet == NULL)
	{
		return;
	}
	packet->packet_no = htons(packet_no) ;

}

unsigned int
PGRget_ip_by_name(char * host)
{
	struct hostent *hp = NULL;
	unsigned int ip = 0;
	unsigned char uc = 0;
	int i;

	if ((host == NULL) || (*host == '\0'))
	{
		return 0;
	}
	hp = gethostbyname( host );
	if (hp == NULL)
	{
		return 0;
	}
	for (i = 3 ; i>= 0 ; i --)
	{
		uc = (unsigned char)hp->h_addr_list[0][i];
		ip = ip | uc;
		if (i > 0)
		ip = ip << 8;
	}
	return ip;
}

int
PGRget_time_value(char *str)
{
	int i,len;
	char * ptr;
	int unit = 1;
	
	if (str == NULL)
		return -1;

	len = strlen(str);
	ptr = str;
	for (i = 0; i < len ; i ++,ptr++)
	{
		if ((! isdigit(*ptr)) && (! isspace(*ptr)))
		{
			switch (*ptr)
			{
				case 'm':
				case 'M':
					unit = 60;
					break;
				case 'h':
				case 'H':
					unit = 60*60;
					break;
			}
			*ptr = '\0';
			break;
		}
	}
	return (atoi(str) * unit);
}

int
PGRget_bw_value(char *str)
{
	int i,len;
	char * ptr;
	int unit = 1;
	
	if (str == NULL)
		return -1;

	len = strlen(str);
	ptr = str;
	for (i = 0; i < len ; i ++,ptr++)
	{
		if ((! isdigit(*ptr)) && (! isspace(*ptr)))
		{
			switch (*ptr)
			{
				/* MByte */
				case 'm':
				case 'M':
					unit = 1024;
					break;
				/* GByte */
				case 'g':
				case 'G':
					unit = 1024*1024;
					break;
			}
			*ptr = '\0';
			break;
		}
	}
	return (atoi(str) * unit);
}

int
PGR_Set_IPC_File(char * dir, char * fname, IPCInfo * list)
{
	int fname_size = 0;
	char * ipc_fname = NULL;
	FILE * fd;
	char buf[PGR_MESSAGE_BUFSIZE];

	if ((dir == NULL) || ( fname == NULL))
	{
		fprintf(stderr,"PGR_Set_IPC_File path[%s] fname[%s] need specified\n",dir,fname);
		return STATUS_ERROR;
	}
	fname_size = strlen(dir) + strlen(fname)+4;
	ipc_fname = malloc(fname_size);
	errno = 0;
	if (ipc_fname == NULL)
	{
		fprintf(stderr,"malloc failed:(%s)\n",strerror(errno));
		return STATUS_ERROR;
	}
	memset(ipc_fname,0,fname_size);
	sprintf(ipc_fname,"%s/%s",dir,fname);
fprintf(stderr,"ipc_fname[%s]\n",ipc_fname);

	fd = fopen(ipc_fname, "w");
	if (fd == NULL)
	{
		fprintf(stderr,"%s file could not open: (%s)\n",ipc_fname,strerror(errno));
		free(ipc_fname);
		return STATUS_ERROR;
	}
	while (list->shmid != 0)
	{
		memset(buf,0,sizeof(buf));
		sprintf(buf,"%d,%d,%d\n",list->shmid,list->semid,list->semnum);
		if (fputs(buf,fd) < 0)
		{
			fprintf(stderr,"could not write %s: (%s)\n",ipc_fname,strerror(errno));
			fclose(fd);
			unlink(ipc_fname);
			free(ipc_fname);
			return STATUS_ERROR;
		}
		list ++;
	}
	fclose(fd);
	free(ipc_fname);
	return STATUS_OK;
}

int
PGR_Get_IPC_File(char * dir, char * fname,IPCInfo *list)
{
	char * func="PGR_Get_IPC_File()";
	int fname_size = 0;
	char * ipc_fname = NULL;
	FILE * fd;
	char buf[PGR_MESSAGE_BUFSIZE];
	char * s_ptr = NULL;
	char * e_ptr = NULL;
	int num = 0;
fprintf(stderr,"dir[%s] fname[%s]\n",dir,fname);
	if ((dir == NULL) || ( fname == NULL) || (list == NULL))
	{
		fprintf(stderr,"%s: args error\n",func);
		return -1;
	}
	fname_size = strlen(dir) + strlen(fname)+4;
	ipc_fname = malloc(fname_size);
	if (ipc_fname == NULL)
	{
		fprintf(stderr,"%s: malloc failed(%s)\n",func,strerror(errno));
		return -1;
	}
	memset(ipc_fname,0,fname_size);
	sprintf(ipc_fname,"%s/%s",dir,fname);

	errno = 0;
	fd = fopen(ipc_fname, "r"); 
	if (fd == NULL)
	{
		fprintf(stderr,"%s file could not open: (%s)\n",ipc_fname,strerror(errno));
		free(ipc_fname);
		return -1;
	}
	memset(buf,0,sizeof(buf));
	num = 0;
	while (fgets(buf,sizeof(buf),fd) != NULL)
	{
		int i;
		s_ptr = buf;
		for (i = 0 ; i < 3 ; i ++)
		{
			if (((e_ptr = strchr(s_ptr,',')) != NULL) ||
				((e_ptr = strchr(s_ptr,'\n')) != NULL))
			{
				*e_ptr = '\0';
				switch (i)
				{
					case 0:
						list->shmid = atoi(s_ptr);
						break;
					case 1:
						list->semid = atoi(s_ptr);
						break;
					case 2:
						list->semnum = atoi(s_ptr);
						break;
				}
				s_ptr = e_ptr+1;
			}
		}
		list ++;
		num ++;
	}
	fclose(fd);
	free(ipc_fname);
	return num;
}

void 
PGR_Delete_IPC_File(char * dir, char * fname)
{
	int fname_size = 0;
	char * ipc_fname = NULL;

	if ((dir == NULL) || ( fname == NULL))
	{
		return;
	}
	fname_size = strlen(dir) + strlen(fname)+4;
	ipc_fname = malloc(fname_size);
	if (ipc_fname == NULL)
	{
		return;
	}
	memset(ipc_fname,0,fname_size);
	sprintf(ipc_fname,"%s/%s",dir,fname);
	unlink(ipc_fname);
	free(ipc_fname);
}

char *
PGR_Set_Host_Name(char * dest, char * src)
{
	unsigned int ip;
	if ((dest == NULL) || (src == NULL))
	{
		return NULL;
	}
	ip = PGRget_ip_by_name(src);
	sprintf(dest,
		 "%d.%d.%d.%d",
		 (ip      ) & 0xff ,
		 (ip >>  8) & 0xff ,
		 (ip >> 16) & 0xff ,
		 (ip >> 24) & 0xff );
	return dest;
}
#endif /* USE_REPLICATION */
