/*--------------------------------------------------------------------
 * FILE:
 *    stop.c
 *
 * NOTE:
 *
 * Portions Copyright (c) 2003-2008, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <sys/param.h>
#include <arpa/inet.h>
#include <sys/file.h>
#include "pgc_admin.h"

void PGC_Stop_Process(char * path, char * pid_fname);
void PGC_Status_Process(char * path, char * pid_fname);
bool PGC_Is_Exist_Pid_File(char * path, char * pid_fname);
void PGC_Child_Wait(int signo);
int PGC_Write_Pid_File(char * path, char * pid_fname);
void PGC_Daemonize(void);
int PGC_Admin_Shutdown(char * path);

static int shutdown_pglb(void);
static int shutdown_cluster(void);
static int shutdown_pgrp(void);
static int shutdown_probe(void);

void 
PGC_Stop_Process(char * path, char * pid_fname)
{
	char * func = "PGC_Stop_Process()";
	FILE *fd;
	char fname[256];
	char pidbuf[128];
	pid_t pid;

	snprintf(fname, sizeof(fname), "%s/%s", path, pid_fname);
	fd = fopen(fname, "r");
	if (!fd)
	{
		exit(1);
	}
	memset(pidbuf,0,sizeof(pidbuf));
	fread(pidbuf, sizeof(pidbuf), 1, fd);
	fclose(fd);
	pid = atoi(pidbuf);

	if (kill (pid,SIGTERM) == -1)
	{
		fprintf(stderr,"%s:could not stop pid: %d, reason: %s",func,pid,strerror(errno));
		exit(1);
	}
}

void 
PGC_Status_Process(char * path, char * pid_fname)
{
	char * func = "PGC_Status_Process()";
	FILE *fd;
	char fname[256];
	char pidbuf[128];
	pid_t pid;

	snprintf(fname, sizeof(fname), "%s/%s", path, pid_fname);
	fd = fopen(fname, "r");
	if (!fd)
	{
		fprintf(stderr,"%s:could not open pid file as %s. reason: %s", func,fname, strerror(errno));
		exit(1);
	}
	memset(pidbuf,0,sizeof(pidbuf));
	fread(pidbuf, sizeof(pidbuf), 1, fd);
	fclose(fd);
	pid = atoi(pidbuf);

	if (kill (pid,0) == 0)
	{
		printf(_("server is running (PID: %d)\n"), pid);
	}
	else
	{
		printf(_("no server running\n") );
	}
}

bool
PGC_Is_Exist_Pid_File(char * path, char * pid_fname)
{
	char fname[256];
	struct stat buf;

	snprintf(fname, sizeof(fname), "%s/%s", path, pid_fname);
	if (stat(fname,&buf) == 0)
	{
		/* pid file is exist */
		return true;
	}
	else
	{
		/* pid file is not exist */
		return false;
	}
}

void
PGC_Child_Wait(int signo)
{
	pid_t pid = 0;

	do {
		int ret;
		pid = waitpid(-1,&ret,WNOHANG);
	} while(pid > 0);
}

int
PGC_Write_Pid_File(char * path, char * pid_fname)
{
	char * func = "PGC_Write_Pid_File()";
	FILE *fd;
	char fname[256];
	char pidbuf[128];

	snprintf(fname, sizeof(fname), "%s/%s", path, pid_fname);
	fd = fopen(fname, "w");
	if (!fd)
	{
		fprintf(stderr,"%s:could not open pid file as %s. reason: %s",
				   func, fname, strerror(errno));
		return STATUS_ERROR;
	}
	snprintf(pidbuf, sizeof(pidbuf), "%d", getpid());
	fwrite(pidbuf, strlen(pidbuf), 1, fd);
	if (fclose(fd))
	{
		fprintf(stderr,"%s:could not write pid file as %s. reason: %s",
				   func,fname, strerror(errno));
		return STATUS_ERROR;
	}
	return STATUS_OK;
}

void 
PGC_Daemonize(void)
{
	char * func = "daemonize()";
	int		i;
	pid_t		pid;

	pid = fork();
	if (pid == (pid_t) -1)
	{
		fprintf(stderr,"%s:fork() failed. reason: %s",func, strerror(errno));
		exit(1);
		return;					/* not reached */
	}
	else if (pid > 0)
	{			/* parent */
		exit(0);
	}
	sleep(Fork_Delay_Time);
#ifdef HAVE_SETSID
	if (setsid() < 0)
	{
		fprintf(stderr,"%s:setsid() failed. reason:%s", func,strerror(errno));
		exit(1);
	}
#endif

	i = open("/dev/null", O_RDWR);
	dup2(i, 0);
	dup2(i, 1);
	dup2(i, 2);
	close(i);
}

int
PGC_Admin_Shutdown(char * path)
{
	char * func ="PGC_Admin_Shutdown()";

	if (PGRinit_Admin(path) != STATUS_OK)
	{
		show_error("%s:PGRinit_Admin error",func);
		return STATUS_ERROR;
	}
	if (PGRget_Admin_Conf_Data(path, PGC_ADMIN_CONF_FILE) != STATUS_OK)
	{
		show_error("%s:PGRget_Admin_Conf_Data error",func);
		return STATUS_ERROR;
	}
	shutdown_pglb();
	shutdown_pgrp();
	shutdown_cluster();
	shutdown_probe();

	return STATUS_OK;
}

static int
shutdown_pglb(void)
{
	char * func = "shutdown_pglb()";
	Probe_Header r_header;
	Probe_Header h_data;
	Pglb_Info body;
	int status = STATUS_OK;
	SSL_Info ssl_tbl;
	Pglb_Info * pglb = PglbTbl;

	if (pglb == NULL)
	{
		show_error("%s: pglb info is null",func);
		return STATUS_ERROR;
	}
	memset(&h_data, 0, sizeof(Probe_Header));
	memset(&r_header, 0, sizeof(Probe_Header));
	memset(&ssl_tbl, 0, sizeof(SSL_Info));

	h_data.packet_no = STOP_REQ_PKT;
	h_data.serverType = SERVER_TYPE_PGLB;
	h_data.body_length = sizeof(Pglb_Info);
	h_data.rec_num = 1;
	PGC_Set_Packet_Header(&r_header, &h_data);
	while (pglb->receivePortNumber != 0)
	{
		memset(&body, 0, sizeof(Pglb_Info));
		PGC_Set_Pglb_Info_2_packet(&body, pglb);
		status = PGC_Admin_Send_Packet(&ssl_tbl, &r_header, (char *)&body, pglb->physicalServerId);
		PGC_Close_SSL(&ssl_tbl);
		pglb ++;
	}
	PGC_Close_SSL(&ssl_tbl);
	return STATUS_OK;
}

static int
shutdown_cluster(void)
{
	char * func = "shutdown_cluster()";
	Probe_Header r_header;
	Probe_Header h_data;
	Cluster_Info body;
	int status = STATUS_OK;
	SSL_Info ssl_tbl;
	Cluster_Info * cluster = ClusterDbTbl;

	if (cluster == NULL)
	{
		show_error("%s: cluster info is null",func);
		return STATUS_ERROR;
	}
	memset(&h_data, 0, sizeof(Probe_Header));
	memset(&r_header, 0, sizeof(Probe_Header));
	memset(&ssl_tbl, 0, sizeof(SSL_Info));

	h_data.packet_no = STOP_REQ_PKT;
	h_data.serverType = SERVER_TYPE_CLUSTER;
	h_data.body_length = sizeof(Cluster_Info);
	h_data.rec_num = 1;
	PGC_Set_Packet_Header(&r_header, &h_data);
	while (cluster->portNumber != 0)
	{
		memset(&body, 0, sizeof(Cluster_Info));
		PGC_Set_Cluster_Info_2_packet(&body, cluster);
		status = PGC_Admin_Send_Packet(&ssl_tbl, &r_header,(char *)&body, cluster->physicalServerId);
		PGC_Close_SSL(&ssl_tbl);
		cluster ++;
	}
	PGC_Close_SSL(&ssl_tbl);
	return STATUS_OK;
}

static int
shutdown_pgrp(void)
{
	char * func ="shutdown_pgrp()";
	Probe_Header r_header;
	Probe_Header h_data;
	Pgrp_Info body;
	int status = STATUS_OK;
	SSL_Info ssl_tbl;
	Pgrp_Info * pgrp = PgrpTbl;


	if (pgrp == NULL)
	{
		show_error("%s: pgrp info is null",func);
		return STATUS_ERROR;
	}
	memset(&h_data, 0, sizeof(Probe_Header));
	memset(&r_header, 0, sizeof(Probe_Header));
	memset(&ssl_tbl, 0, sizeof(SSL_Info));

	h_data.packet_no = STOP_REQ_PKT;
	h_data.serverType = SERVER_TYPE_PGRP;
	h_data.body_length = sizeof(Pgrp_Info);
	h_data.rec_num = 1;
	PGC_Set_Packet_Header(&r_header, &h_data);
	while (pgrp->replicationPortNumber != 0)
	{
		memset(&body, 0, sizeof(Pgrp_Info));
		PGC_Set_Pgrp_Info_2_packet(&body, pgrp);
		status = PGC_Admin_Send_Packet(&ssl_tbl, &r_header, (char *)&body, pgrp->physicalServerId);
		PGC_Close_SSL(&ssl_tbl);
		pgrp ++;
	}
	PGC_Close_SSL(&ssl_tbl);
	return STATUS_OK;
}

static int
shutdown_probe(void)
{
	char * func ="shutdown_probe()";
	Probe_Header r_header;
	Probe_Header h_data;
	int status = STATUS_OK;
	SSL_Info ssl_tbl;
	SSL_Server_Info body;
	SSL_Server_Info * probe = ProbeTbl;


	if (probe == NULL)
	{
		show_error("%s: probe info is null",func);
		return STATUS_ERROR;
	}
	memset(&h_data, 0, sizeof(Probe_Header));
	memset(&r_header, 0, sizeof(Probe_Header));
	memset(&ssl_tbl, 0, sizeof(SSL_Info));

	h_data.packet_no = STOP_REQ_PKT;
	h_data.serverType = SERVER_TYPE_PROBE;
	h_data.body_length = sizeof(SSL_Server_Info);
	h_data.rec_num = 1;
	PGC_Set_Packet_Header(&r_header, &h_data);
	while (probe->portNumber != 0)
	{
		memset(&body, 0, sizeof(SSL_Server_Info));
		PGC_Set_SSL_Server_Info_2_packet(&body, probe);
		status = PGC_Admin_Send_Packet(&ssl_tbl, &r_header, (char *)&body, probe->physicalServerId);
		PGC_Close_SSL(&ssl_tbl);
		probe ++;
	}
	PGC_Close_SSL(&ssl_tbl);
	return STATUS_OK;
}
