/*--------------------------------------------------------------------
 * FILE:
 *     probe_info.c
 *
 * NOTE:
 *     This file is composed of the probe process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/file.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "pgc_admin.h"

int PGC_SSL_Server_Info(SSL_Info * ssl_tbl, Probe_Header * header, char * packet);
int PGC_Set_SSL_Admin(Probe_Header * header, SSL_Server_Info * packet);
int PGC_Set_SSL_Probe(Probe_Header * header, SSL_Server_Info * packet);
int PGC_Set_Pglb_Info(Probe_Header * header, Pglb_Info * packet);
int PGC_Set_Cluster_Info(Probe_Header * header, Cluster_Info * packet);
int PGC_Set_Pgrp_Info(Probe_Header * header, Pgrp_Info * packet);
int PGC_Set_Physical_Server_Info(Probe_Header * header, Physical_Server_Info * packet);
SSL_Server_Info * PGC_Get_SSL_Server_Rec(SSL_Server_Info * ssl_server_tbl, int server_no);

static Physical_Server_Info * get_pserver_rec(int server_no);
static Pglb_Info * get_pglb_rec(int server_no);
static Cluster_Info * get_cluster_rec(int server_no);
static Pgrp_Info * get_pgrp_rec(int server_no);

int
PGC_SSL_Server_Info(SSL_Info * ssl_tbl, Probe_Header * header, char * packet)
{
	/* char * func = "PGC_SSL_Server_Info()";*/
	int status = STATUS_OK;
	Probe_Header r_header;
	Probe_Header h_data;

	switch (ntohs(header->serverType))
	{
		case SERVER_TYPE_ADMIN:
			PGC_Set_SSL_Admin(header, (SSL_Server_Info *)packet);
			break;
		case SERVER_TYPE_PROBE:
			PGC_Set_SSL_Probe(header, (SSL_Server_Info *)packet);
			break;
		case SERVER_TYPE_PGLB:
			PGC_Set_Pglb_Info(header, (Pglb_Info *)packet);
			break;
		case SERVER_TYPE_CLUSTER:
			PGC_Set_Cluster_Info(header, (Cluster_Info *)packet);
			break;
		case SERVER_TYPE_PGRP:
			PGC_Set_Pgrp_Info(header, (Pgrp_Info *)packet);
			break;
		case SERVER_TYPE_SYSTEM:
			PGC_Set_Physical_Server_Info(header, (Physical_Server_Info *)packet);
			break;
		default :
			break;
	}
	/* return response packet */
	h_data.packet_no = INFO_ANS_PKT;
	h_data.serverType = SERVER_TYPE_PROBE;
	h_data.body_length = 0;
	h_data.rec_num = 0;
	h_data.status = status;
	PGC_Set_Packet_Header(&r_header, &h_data);
	status = PGC_Send_Status_Packet(ssl_tbl, &r_header,NULL);

	return status;
}

int
PGC_Set_SSL_Admin(Probe_Header * header, SSL_Server_Info * packet)
{
	char * func = "PGC_Set_SSL_Admin()";
	int i = 0;
	int rec_num = 0;
	int server_no = 0;
	SSL_Server_Info * admin = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (AdminTbl == NULL)
	{
		show_error("%s: AdminTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_ADMIN_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		server_no = ntohs(packet->serverNo);
		admin = PGC_Get_SSL_Server_Rec(AdminTbl, server_no);
		if (admin == NULL)
		{
			admin = AdminTbl;
		}
		PGC_Set_Packet_2_SSL_Server_Info(admin , packet);
		packet ++;
	}
	return STATUS_OK;
}

int
PGC_Set_SSL_Probe(Probe_Header * header, SSL_Server_Info * packet)
{
	char * func = "PGC_Set_SSL_Probe()";
	int i = 0;
	int rec_num= 0;
	int server_no = 0;
	SSL_Server_Info * probe = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (ProbeTbl == NULL)
	{
		show_error("%s: ProbeTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_PROBE_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		server_no = ntohs(packet->serverNo);
		probe = PGC_Get_SSL_Server_Rec(ProbeTbl, server_no);
		if (probe == NULL)
		{
			probe = ProbeTbl;
		}
		PGC_Set_Packet_2_SSL_Server_Info(probe , packet);
		packet ++;
	}
	return STATUS_OK;
}

int
PGC_Set_Pglb_Info(Probe_Header * header, Pglb_Info * packet)
{
	char * func = "PGC_Set_Pglb_Info()";
	int i = 0;
	int rec_num = 0;
	int server_no = 0;
	Pglb_Info * pglb = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (PglbTbl == NULL)
	{
		show_error("%s: PglbTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_PGLB_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		server_no = ntohs(packet->serverNo);
		pglb = get_pglb_rec(server_no);
		if (pglb == NULL)
		{
			pglb = PglbTbl;
		}
		PGC_Set_Packet_2_Pglb_Info(pglb , packet);
		packet ++;
	}
	return STATUS_OK;
}

int
PGC_Set_Cluster_Info(Probe_Header * header, Cluster_Info * packet)
{
	char * func = "PGC_Set_Cluster_Info()";
	int i = 0;
	int rec_num = 0;
	int server_no = 0;
	Cluster_Info * cluster = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (ClusterDbTbl == NULL)
	{
		show_error("%s: ClusterDBTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_DB_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		server_no = ntohs(packet->serverNo);
		cluster = get_cluster_rec(server_no);
		if (cluster == NULL)
		{
			cluster = ClusterDbTbl;
		}
		PGC_Set_Packet_2_Cluster_Info(cluster , packet);
		packet ++;
	}
	return STATUS_OK;
}

int
PGC_Set_Pgrp_Info(Probe_Header * header, Pgrp_Info * packet)
{
	char * func = "PGC_Set_Pgrp_Info()";
	int i = 0;
	int rec_num = 0;
	int server_no = 0;
	Pgrp_Info * pgrp = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (PgrpTbl == NULL)
	{
		show_error("%s: PgrpTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_PGRP_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		server_no = ntohs(packet->serverNo);
		pgrp = get_pgrp_rec(server_no);
		if (pgrp == NULL)
		{
			pgrp = PgrpTbl;
		}
		PGC_Set_Packet_2_Pgrp_Info(pgrp , packet);
		packet ++;
	}
	return STATUS_OK;
}

int
PGC_Set_Physical_Server_Info(Probe_Header * header, Physical_Server_Info * packet)
{
	char * func = "PGC_Set_Physical_Server_Info()";
	int i = 0;
	int rec_num = 0;
	Physical_Server_Info * pserver = NULL;

	if ((header == NULL) || (packet == NULL))
	{
		show_error("%s: header or packet is null",func);
		return STATUS_ERROR;
	}
	if (PhysicalServerTbl == NULL)
	{
		show_error("%s: PhysicalServerTbl is not allocated",func);
		return STATUS_ERROR;
	}
	rec_num = ntohl(header->rec_num);
	if (rec_num >= MAX_ADMIN_SERVER)
	{
		show_error("%s: number of received record is out of range",func);
		return STATUS_ERROR;
	}
	for (i = 0 ; i < rec_num ; i ++)
	{
		pserver = get_pserver_rec(ntohs(packet->serverNo));
		if (pserver == NULL)
		{
			pserver = PhysicalServerTbl;
		}
		PGC_Set_Packet_2_Physical_Server_Info(pserver , packet);
		packet ++;
	}
	return STATUS_OK;
}

SSL_Server_Info *
PGC_Get_SSL_Server_Rec(SSL_Server_Info * ssl_server_tbl, int server_no)
{
	SSL_Server_Info * server = ssl_server_tbl;
	while (server->portNumber != 0)
	{
		if (server->serverNo == server_no)
		{
			return server;
		}
		server ++;
	}
	return NULL;
}

static Physical_Server_Info * 
get_pserver_rec(int server_no)
{
	Physical_Server_Info * pserver = PhysicalServerTbl;

	if (pserver == NULL)
		return NULL;
	while (pserver->hostName[0] != 0)
	{
		if (pserver->serverNo == server_no)
		{
			return pserver;
		}
		pserver ++;
	}
	return NULL;
}

static Pglb_Info * 
get_pglb_rec(int server_no)
{
	Pglb_Info * pglb = PglbTbl;

	if (pglb == NULL)
		return NULL;
	while (pglb->receivePortNumber != 0)
	{
		if (pglb->serverNo == server_no)
		{
			return pglb;
		}
		pglb ++;
	}
	return NULL;
}

static Cluster_Info * 
get_cluster_rec(int server_no)
{
	Cluster_Info * cluster = ClusterDbTbl;

	if (cluster == NULL)
		return NULL;
	while (cluster->portNumber != 0)
	{
		if (cluster->serverNo == server_no)
		{
			return cluster;
		}
		cluster ++;
	}
	return NULL;
}

static Pgrp_Info * 
get_pgrp_rec(int server_no)
{
	Pgrp_Info * pgrp = PgrpTbl;

	if (pgrp == NULL)
		return NULL;
	while (pgrp->replicationPortNumber != 0)
	{
		if (pgrp->serverNo == server_no)
		{
			return pgrp;
		}
		pgrp ++;
	}
	return NULL;
}
