/*--------------------------------------------------------------------
 * FILE:
 *    set_conf.c
 *
 * NOTE:
 *
 * Portions Copyright (c) 2003-2009, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#include "postgres.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dirent.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <netdb.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/wait.h>

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"

#include "replicate.h"
#include "libpgc/libpgc.h"
#include "pglb/pglb.h"
#include "pgrp/pgreplicate.h"
#include "admin/pgc_admin.h"

#define BEGIN_COL_OF_CONF	(20)
#define	END_COL_OF_CONF		(44)

int PGRcreate_pgc_admin_conf (char * path);
void PGRcreate_pgcluster_conf (char * path);
char * PGRget_dir_name_with_hostname(char * buf, char * path, Physical_Server_Info *server);
FILE * PGRupdate_conf_contents(FILE * rfp, FILE * wfp, char * tag, char * data);
FILE * PGRcopy_file(FILE * src_fp, FILE * dest_fp);
int PGR_execv(char * command, char *args[]);
FILE * PGRopen_write_file(char * base_dir, char * work_path, char * fname, char * mode);

static int create_pgc_admin_conf (char * path);
static int create_each_admin_conf(char * conf_dir, Physical_Server_Info * server, int cnt);
static int create_pgc_probe_conf (char * path);
static int create_each_probe_conf(char * conf_dir, Physical_Server_Info * server, int cnt);
static FILE * create_physical_server_part(FILE * fp);
static FILE * create_admin_server_part(FILE * fp);
static FILE * put_admin_server_rec(int indent, FILE * fp, Physical_Server_Info * server, int cnt );
static FILE * create_probe_server_part(FILE * fp);
static FILE * put_probe_server_rec(int indent, FILE * fp, Physical_Server_Info * server, int cnt );
static FILE * create_pglb_server_part(FILE * fp);
static FILE * put_pglb_server_rec (int indent, FILE * fp, Pglb_Info * pglb, Physical_Server_Info * server, int cnt );
static FILE * create_cluster_server_part(FILE * fp);
static FILE * put_cluster_server_rec (int indent, FILE * fp, Cluster_Info * cluster, Physical_Server_Info * server, int cnt );
static FILE * create_partial_replication_part(int indent, FILE * fp);
static FILE * create_pgrp_server_part(FILE * fp);
static FILE * put_pgrp_server_rec (int indent, FILE * fp, Pgrp_Info * pgrp, Physical_Server_Info * server, int cnt );
static char * begin_tag(int indent, char * buf,char * tag);
static char * end_tag(int indent, char * buf,char * tag);
static FILE * put_conf_header(int indent, FILE * fp, char * header, int cnt);
static FILE * put_conf_data_with_tag(int indent, FILE * fp, char * tag, char * data);
static char * create_dir(char * base_dir, char * type_dir);
static int create_pglb_conf (char * path);
static int create_each_pglb_conf(char * conf_dir, Physical_Server_Info * server, Pglb_Info * pglb);
static FILE * put_cluster_part_of_pglb_conf (int indent, FILE * fp);
static FILE * put_pglb_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Pglb_Info * pglb );
static FILE * put_pglb_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Pglb_Info * pglb ); 
static int create_pgrp_conf (char * path);
static int create_each_pgrp_conf(char * conf_dir, Physical_Server_Info * server, Pgrp_Info * pgrp);
static FILE * put_cluster_part_of_pgrp_conf (int indent, FILE * fp);
static FILE * put_pglb_part_of_pgrp_conf (int indent, FILE * fp);
static FILE * put_pgrp_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Pgrp_Info * pgrp );
static int create_cluster_conf (char * path);
static int create_each_postgresql_conf(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster );
static int create_each_pg_hba_conf(char * conf_dir, Cluster_Info * cluster );
static int create_each_cluster_conf(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster );
static FILE * put_replicate_part_of_cluster_conf (int indent, FILE * fp);
static FILE * put_cluster_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Cluster_Info * cluster );
static FILE * put_not_replication_part_of_cluster_conf (int indent, FILE * fp);
static int create_db_cluster_dir (char * path);
static int exec_initdb(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster );

int
PGRcreate_pgc_admin_conf (char * path)
{
	int rtn = STATUS_OK;

	rtn = create_pgc_admin_conf (path);
	if (rtn != STATUS_OK)
	{
		show_error("create_pgc_admin_conf failed");
		return rtn;
	}
	rtn = create_pgc_probe_conf (path);
	if (rtn != STATUS_OK)
	{
		show_error("create_pgc_probe_conf failed");
	}

	return rtn;
}

static int
create_pgc_admin_conf (char * path)
{
	char * func = "create_pgc_admin_conf()";
	char conf_dir[512];
	Physical_Server_Info * server = NULL;
	SSL_Server_Info * admin = AdminTbl;
	int i;

	if (path == NULL)
	{
		return STATUS_ERROR;
	}
	for (i = 0 ; i < RecordNumTbl->adminNum ; i ++, admin++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(admin->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			show_error("%s: PGRget_dir_name_with_hostname failed",func);
			continue;
		}
		create_each_admin_conf(conf_dir, server, i);
	}
	return STATUS_OK;
}

static int
create_each_admin_conf(char * conf_dir, Physical_Server_Info * server, int cnt)
{
	char * func = "create_each_admin_conf()";
	FILE * fp = NULL;
	char * work_path = NULL;
	char * fname = PGC_ADMIN_CONF_FILE;

	if ((work_path = strdup(DEFAULT_WORK_DIR_OF_ADMIN)) == NULL)
	{
		show_error("%s:strdup failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	if ((fp = PGRopen_write_file(conf_dir, work_path, fname, "w")) == NULL)
	{
		free(work_path);
		show_error("%s:PGRopen_write_file faied",func);
		return STATUS_ERROR;
	}
	free(work_path);

	/* create physical server part */
	fp = create_physical_server_part(fp);
	/* create admin server part */
	fp = create_admin_server_part(fp);
	/* create probe server part */
	fp = create_probe_server_part(fp);
	/* create pglb server part */
	fp = create_pglb_server_part(fp);
	/* create cluster db server part */
	fp = create_cluster_server_part(fp);
	/* create pgrp server part */
	fp = create_pgrp_server_part(fp);

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static int
create_pgc_probe_conf (char * path)
{
	char * func = "create_pgc_probe_conf()";
	char conf_dir[512];
	int i;
	Physical_Server_Info * server = NULL;
	SSL_Server_Info * probe = ProbeTbl;

	if (path == NULL) 
	{
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->probeNum ; i ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec((probe+i)->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			show_error("%s: PGRget_dir_name_with_hostname failed",func);
			continue;
		}
		create_each_probe_conf(conf_dir, server, i);
	}
	return STATUS_OK;
}


static int
create_each_probe_conf(char * conf_dir, Physical_Server_Info * server, int cnt)
{
	char * func = "create_each_probe_conf()";
	FILE * fp = NULL;
	char * work_path = NULL;
	char * fname = PGC_PROBE_CONF_FILE;

	if ((work_path = strdup(DEFAULT_WORK_DIR_OF_PROBE)) == NULL)
	{
		show_error("%s:strdup failed(%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	if ((fp = PGRopen_write_file(conf_dir, work_path, fname, "w")) == NULL)
	{
		free(work_path);
		return STATUS_ERROR;
	}
	free(work_path);

	/* create admin server part */
	fp = create_admin_server_part(fp);
	/* create probe conf  */
	fp = put_probe_server_rec(0, fp, server, cnt );

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static FILE *
create_physical_server_part(FILE * fp)
{
	char * func = "create_physical_server_part()";
	char buf[1024];
	int i;
	int indent = 0;
	Physical_Server_Info * server = PhysicalServerTbl;

	if (fp ==NULL)
	{
		return NULL;
	}
	if (RecordNumTbl == NULL)
	{
		show_error("%s:RecordNumTbl is NULL",func);
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++, server ++)
	{
		/* set record header */
		put_conf_header(indent, fp, "Physical server", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, PHYSICAL_SERVER_INFO_TAG) , fp);
		fputs("\n",fp);
		indent ++;

		/* Server_ID */
		sprintf(buf,"%d",server->physicalServerId);
		put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

		/* Host_Name */
		put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

		/* User_Name */
		put_conf_data_with_tag(indent, fp, USER_NAME_TAG, server->userName);

		indent --;
		/* end tag*/
		fputs( end_tag(indent, buf, PHYSICAL_SERVER_INFO_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static FILE *
create_admin_server_part(FILE * fp)
{
	char * func = "create_admin_server_part()";
	int i;
	int indent = 0;
	SSL_Server_Info * admin = AdminTbl;
	Physical_Server_Info * server = NULL;

	if (fp ==NULL)
	{
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->adminNum ; i ++)
	{
		server = PGRget_Physical_Server_Rec((admin+i)->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		fp = put_admin_server_rec(indent, fp, server, i );
	}
	return fp;
}

static FILE *
put_admin_server_rec(int indent, FILE * fp, Physical_Server_Info * server, int cnt )
{
	char * func = "put_admin_server_rec()";
	char buf[512];

	if ((fp == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Admin server", cnt+1);

	/* start tag */
	fputs( begin_tag(indent, buf, PGC_ADMIN_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	indent ++;
	/* Server_ID */
	sprintf(buf,"%d",server->physicalServerId);
	put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* Port */
	sprintf(buf,"%d", DEFAULT_PORT_OF_ADMIN + cnt);
	put_conf_data_with_tag(indent, fp, PORT_TAG, buf);

	/* work directory */
	put_conf_data_with_tag(indent, fp, CONF_DIR_TAG, DEFAULT_WORK_DIR_OF_ADMIN);
	
	/* binary directory */
	put_conf_data_with_tag(indent, fp, BIN_DIR_TAG, DEFAULT_BIN_DIR);
	
	/* Cert_File */
	/*
	put_conf_data_with_tag(indent, fp, CERT_FILE_TAG, ADMIN_CERTFILE);
	*/

	/* SSL_Path */
	put_conf_data_with_tag(indent, fp, SSL_PATH_TAG, DEFAULT_SSL_BIN_DIR);

	indent --;
	/* end tag */
	fputs( end_tag(indent, buf, PGC_ADMIN_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	return fp;
}

static FILE *
create_probe_server_part(FILE * fp)
{
	char * func = "create_probe_server_part()";
	int i;
	int indent = 0;
	Physical_Server_Info * server = NULL;
	SSL_Server_Info * probe = ProbeTbl;

	if (fp == NULL)
	{
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->probeNum ; i ++)
	{
		server = PGRget_Physical_Server_Rec((probe+i)->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		fp = put_probe_server_rec(indent, fp, server, i );
	}
	return fp;
}

static FILE *
put_probe_server_rec(int indent, FILE * fp, Physical_Server_Info * server, int cnt )
{
	char * func = "put_probe_server_rec()";
	char buf[512];

	if ((fp == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Probe server", cnt+1);

	/* start tag */
	fputs( begin_tag(indent, buf, PROBE_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	indent ++;
	/* Server_ID */
	sprintf(buf,"%d",server->physicalServerId);
	put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* Port */
	sprintf(buf,"%d", DEFAULT_PORT_OF_PROBE + cnt );
	put_conf_data_with_tag(indent, fp, PORT_TAG, buf);

	/* work directory */
	put_conf_data_with_tag(indent, fp, CONF_DIR_TAG, DEFAULT_WORK_DIR_OF_PROBE);
	
	/* binary directory */
	put_conf_data_with_tag(indent, fp, BIN_DIR_TAG, DEFAULT_BIN_DIR);
	
	/* Cert_File */
	/*
	put_conf_data_with_tag(indent,fp, CERT_FILE_TAG, PROBE_CERTFILE);
	*/

	/* SSL_Path */
	put_conf_data_with_tag(indent, fp, SSL_PATH_TAG, DEFAULT_SSL_BIN_DIR);

	indent --;
	/* end tag */
	fputs( end_tag(indent, buf, PROBE_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	return fp;
}

static FILE *
create_pglb_server_part(FILE * fp)
{
	char * func = "create_pglb_server_part()";
	int i;
	int indent = 0;
	Pglb_Info * pglb = PglbTbl;
	Physical_Server_Info * server = NULL;

	if (fp == NULL) 
	{
		return NULL;
	}
	else if (pglb == NULL)
	{
		show_debug("there is no pglb");
		return fp;
	}
	for (i = 0 ; i < RecordNumTbl->pglbNum ; i ++)
	{
		server = PGRget_Physical_Server_Rec((pglb+i)->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}
		fp = put_pglb_server_rec(indent, fp, pglb, server, i );
	}
	return fp;
}

static FILE *
put_pglb_server_rec (int indent, FILE * fp, Pglb_Info * pglb, Physical_Server_Info * server, int cnt )
{
	char * func = "put_probe_server_rec()";
	char buf[512];

	if ((fp == NULL) || (pglb == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Load Balancer", cnt+1);

	/* start tag */
	fputs( begin_tag(indent, buf, LOAD_BALANCE_SERVER_TAG) , fp);
	fputs("\n",fp);

	indent ++;
	/* Server_ID */
	sprintf(buf,"%d",server->physicalServerId);
	put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* work directory */
	sprintf(buf,"%s_%d",pglb->workPath, cnt+1 );
	put_conf_data_with_tag(indent, fp, CONF_DIR_TAG, buf);
	
	/* binary directory */
	put_conf_data_with_tag(indent, fp, BIN_DIR_TAG, pglb->binPath);

	/* Backend_Socket_Dir */
	put_conf_data_with_tag(indent, fp, BACKEND_SOCKET_DIR_TAG, pglb->backendSocketDir);

	/* Receive_Port */
	sprintf(buf,"%d",pglb->receivePortNumber + cnt );
	put_conf_data_with_tag(indent, fp, RECV_PORT_TAG, buf);

	/* Recovery_Port */
	sprintf(buf,"%d",pglb->recoveryPortNumber + cnt );
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* Max_Cluster_Num */
	sprintf(buf,"%d",pglb->maxClusterNum);
	put_conf_data_with_tag(indent, fp, MAX_CLUSTER_TAG, buf);

	/* Use_Connection_Pooling */
	put_conf_data_with_tag(indent, fp, USE_CONNECTION_POOL_TAG, pglb->usePool);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, pglb->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, pglb->lifecheckInterval);

	/* Connection_Life_Time */
	put_conf_data_with_tag(indent, fp, CONNECTION_LIFE_TIME_TAG, pglb->connectionLifetime);

	/* Log_File_Name */
	put_conf_data_with_tag(indent, fp, LOG_FILE_NAME_TAG, pglb->logFileName);

	/* Log_File_Size */
	put_conf_data_with_tag(indent, fp, LOG_FILE_SIZE_TAG, pglb->logFileSize);

	/* Log_Rotate */
	sprintf(buf,"%d",pglb->logRotate);
	put_conf_data_with_tag(indent, fp, LOG_ROTATION_TAG, buf);

	indent --;
	/* end tag */
	fputs( end_tag(indent, buf, LOAD_BALANCE_SERVER_TAG) , fp);
	fputs("\n",fp);

	return fp;
}

static FILE *
create_cluster_server_part(FILE * fp)
{
	char * func = "create_cluster_server_part()";
	int i;
	int indent = 0;
	Cluster_Info * cluster = ClusterDbTbl;
	Physical_Server_Info * server = NULL;

	if (fp == NULL) 
	{
		return NULL;
	}
	else if (cluster == NULL)
	{
		return fp;
	}
	for (i = 0 ; i < RecordNumTbl->clusterNum ; i ++)
	{
		server = PGRget_Physical_Server_Rec((cluster+i)->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		fp = put_cluster_server_rec(indent, fp, cluster, server, i );
	}
	return fp;
}

static FILE *
put_cluster_server_rec (int indent, FILE * fp, Cluster_Info * cluster, Physical_Server_Info * server, int cnt )
{
	char * func = "put_cluster_server_rec()";
	char buf[512];

	if ((fp == NULL) || (cluster == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
}
	/* set record header */
	put_conf_header(indent, fp, "Cluster DB", cnt+1);

	/* start tag */
	fputs( begin_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
	fputs("\n",fp);

	indent ++;
	/* Server_ID */
	sprintf(buf,"%d",server->physicalServerId);
	put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* work directory */
	sprintf(buf,"%s_%d",cluster->workPath, cnt+1);
	put_conf_data_with_tag(indent, fp, CONF_DIR_TAG, buf);
	
	/* binary directory */
	put_conf_data_with_tag(indent, fp, BIN_DIR_TAG, cluster->binPath);

	/* Max_Connect */
	sprintf(buf,"%d",cluster->maxConnections);
	put_conf_data_with_tag(indent, fp, MAX_CONNECT_TAG, buf);

	/* Port */
	sprintf(buf,"%d",cluster->portNumber + cnt);
	put_conf_data_with_tag(indent, fp, PORT_TAG, buf);

	/* Recovery_Port */
	sprintf(buf,"%d",cluster->recoveryPortNumber + cnt);
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* Rsync_Path */
	put_conf_data_with_tag(indent, fp, RSYNC_PATH_TAG, cluster->rsyncPath);

	/* Rsync_Option */
	put_conf_data_with_tag(indent, fp, RSYNC_OPTION_TAG, cluster->rsyncOption);

	/* Rsync_Compress */
	put_conf_data_with_tag(indent, fp, RSYNC_COMPRESS_TAG, cluster->rsyncCompress);

	/* Rsync_Timeout */
	put_conf_data_with_tag(indent, fp, RSYNC_TIMEOUT_TAG, cluster->rsyncTimeout);

	/* Rsync_Bwlimit */
	put_conf_data_with_tag(indent, fp, RSYNC_BWLIMIT_TAG, cluster->rsyncBwlimit);

	/* Pg_Dump_Path */
	put_conf_data_with_tag(indent, fp, PG_DUMP_PATH_TAG, cluster->pgdumpPath);

	/* Ping_Path */
	put_conf_data_with_tag(indent, fp, PING_PATH_TAG, cluster->pingPath);

	/* When_Stand_Alone */
	put_conf_data_with_tag(indent, fp, STAND_ALONE_TAG, cluster->whenStandAlone);

	/* Replication_Timeout */
	put_conf_data_with_tag(indent, fp, TIMEOUT_TAG, cluster->replicationTimeout);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, cluster->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, cluster->lifecheckInterval);
	
	/* partial replication table part */
	fp = create_partial_replication_part(indent, fp);

	indent --;
	/* end tag */
	fputs( end_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
	fputs("\n",fp);
	return fp;
}

static FILE *
create_partial_replication_part(int indent, FILE * fp)
{
	char * func = "create_partial_replication_part()";
	char buf[512];
	int i;
	Partial_Info * partial = PartialTbl;

	if (fp == NULL) 
	{
		return NULL;
	}
	else if (partial == NULL)
	{
		show_debug("%s: no partial replication part",func);
		return fp;
	}
	for (i = 0 ; i < RecordNumTbl->partialNum ; i ++)
	{
		/* set record header */
		put_conf_header(indent, fp, "Non Replicate", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, NOT_REPLICATE_INFO_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* DB_Name */
		put_conf_data_with_tag(indent, fp, DB_NAME_TAG, partial->dbName);

		/* Table_Name */
		put_conf_data_with_tag(indent, fp, TABLE_NAME_TAG, partial->tableName);

		/* Query */
		put_conf_data_with_tag(indent, fp, QUERY_TAG, partial->query);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, NOT_REPLICATE_INFO_TAG) , fp);
		fputs("\n",fp);
		partial ++;
	}
	return fp;
}

static FILE *
create_pgrp_server_part(FILE * fp)
{
	char * func = "create_pgrp_server_part()";
	int i;
	int indent = 0;
	Pgrp_Info * pgrp = PgrpTbl;
	Physical_Server_Info * server = NULL;

	if (fp ==NULL)
	{
		return NULL;
	}
	else if (pgrp == NULL)
	{
		return fp;
	}
	for (i = 0 ; i < RecordNumTbl->pgrpNum ; i ++ )
	{
		server = PGRget_Physical_Server_Rec((pgrp+i)->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}
		fp = put_pgrp_server_rec(indent, fp, pgrp, server, i );
	}
	return fp;
}

static FILE *
put_pgrp_server_rec (int indent, FILE * fp, Pgrp_Info * pgrp, Physical_Server_Info * server, int cnt )
{
	char * func = "put_pgrp_server_rec()";
	char buf[512];

	if ((fp == NULL) || (pgrp == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Replicator", cnt+1);

	/* start tag */
	fputs( begin_tag(indent, buf, REPLICATION_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	indent ++;
	/* Server_ID */
	sprintf(buf,"%d",server->physicalServerId);
	put_conf_data_with_tag(indent, fp, SERVER_ID_TAG, buf);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* work directory */
	sprintf(buf,"%s_%d",pgrp->workPath, cnt+1);
	put_conf_data_with_tag(indent, fp, CONF_DIR_TAG, buf);
	
	/* binary directory */
	put_conf_data_with_tag(indent, fp, BIN_DIR_TAG, pgrp->binPath);

	/* Replication_Port */
	sprintf(buf,"%d",pgrp->replicationPortNumber + cnt);
	put_conf_data_with_tag(indent, fp, REPLICATE_PORT_TAG, buf);

	/* Recovery_Port */
	sprintf(buf,"%d",pgrp->recoveryPortNumber + cnt);
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* RLOG_Port */
	sprintf(buf,"%d",pgrp->RLogPortNumber + cnt);
	put_conf_data_with_tag(indent, fp, RLOG_PORT_TAG, buf);

	/* Use_Replication_Log */
	put_conf_data_with_tag(indent, fp, USE_REPLICATION_LOG_TAG, pgrp->useRlog);

	/* Replication_Timeout */
	put_conf_data_with_tag(indent, fp, TIMEOUT_TAG, pgrp->replicationTimeout);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, pgrp->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, pgrp->lifecheckInterval);

	/* Log_File_Name */
	put_conf_data_with_tag(indent, fp, LOG_FILE_NAME_TAG, pgrp->logFileName);

	/* Log_File_Size */
	put_conf_data_with_tag(indent, fp, LOG_FILE_SIZE_TAG, pgrp->logFileSize);

	/* Log_Rotate */
	sprintf(buf,"%d",pgrp->logRotate);
	put_conf_data_with_tag(indent, fp, LOG_ROTATION_TAG, buf);

	indent --;
	/* end tag */
	fputs( end_tag(indent, buf, REPLICATION_SERVER_INFO_TAG) , fp);
	fputs("\n",fp);

	return fp;
}

static char *
begin_tag(int indent, char * buf,char * tag)
{
	char tab[24];

	if ((tag == NULL) || (buf == NULL) || (indent >= sizeof(tab)))
		return NULL;
	memset(tab,0,sizeof(tab));
	if (indent > 0)
		memset(tab, '	', indent);
	sprintf(buf,"%s<%s>",tab,tag);
	return buf;
}

static char *
end_tag(int indent, char * buf,char * tag)
{
	char tab[24];

	if ((tag == NULL) || (buf == NULL) || (indent >= sizeof(tab)))
		return NULL;
	memset(tab,0,sizeof(tab));
	if (indent > 0)
		memset(tab, '	', indent);
	sprintf(buf,"%s</%s>",tab,tag);
	return buf;
}

static FILE *
put_conf_header(int indent, FILE * fp, char * header, int cnt)
{
	char * func = "put_conf_header()";
	char buf[512];
	char tab[16];
	char * sp = "#---------------------------------------------------";
	int i;

	if ((fp == NULL) || (header == NULL))
		return NULL;
	if (indent >= sizeof(tab))
	{
		show_error("%s:indent is too deep",func);
		return NULL;
	}
	memset(tab, 0, sizeof(tab));
	for (i = 0; i < indent; i ++)
	{
		strcat(tab,"	");
	}

	sprintf(buf,"%s%s\n",tab,sp);
	fputs( buf, fp);
	sprintf(buf,"%s# %s %d \n", tab, header, cnt);
	fputs( buf, fp);
	sprintf(buf,"%s%s\n",tab,sp);
	fputs( buf, fp);
	return fp;
}

static FILE *
put_conf_data_with_tag(int indent, FILE * fp, char * tag, char * data)
{
	char * func = "put_conf_data_with_tag()";
	char buf [512];
	int data_len,len,begin,end,sp;

	if ((fp == NULL) || (tag == NULL) || (data == NULL))
	{
		show_error("%s:arg is NULL",func);
		return NULL;
	}
	if ((strlen(tag)+indent >= sizeof(buf)) || (strlen(data)+8 >= sizeof(buf)))
	{
		show_error("%s:tag or data is too long",func);
		return fp;
	}
	begin = BEGIN_COL_OF_CONF;
	end = END_COL_OF_CONF;
	memset(buf, 0, sizeof(buf));
	/* set begin tag */
	fputs( begin_tag(indent, buf, tag) , fp);
	/* fill up buffer with space */
	memset(buf, ' ', sizeof(buf)-1);
	/* get begin position */
	len = strlen(tag);
	sp = (len >= begin)?4:(begin - len);
	/* set data strings in the begin point of buffer */
	data_len = strlen(data);
	strncpy(&buf[sp], data, data_len);
	/* get end position */
	data_len += sp;
	len += data_len;
	sp = (len >= end)?4:(end - len);
	/* set terminator in the end point of buffer */
	buf[data_len + sp] = '\0';
	fputs( buf, fp);
	/* set end tag */
	fputs( end_tag(0, buf, tag) , fp);
	fputs("\n",fp);

	return fp;
}

FILE *
PGRopen_write_file(char * base_dir, char * work_path, char * fname, char * mode)
{
	char * func = "PGRopen_write_file()";
	FILE * fp = NULL;
	char * conf_file = NULL;
	char * path = NULL;

	if ((path = create_dir(base_dir, work_path)) == NULL)
	{
		show_error("%s: [%s]create dir failed",func, work_path);
		return NULL;
	}
	if ((conf_file = malloc(strlen(path) + strlen(fname) + 4)) == NULL)
	{
		free(path);
		show_error("%s: malloc failed(%s)",func, strerror(errno));
		return NULL;
	}
	sprintf(conf_file,"%s/%s",path,fname);
	free(path);
	
	if ((fp = fopen(conf_file, mode)) == NULL)
	{
		show_error("%s: %s can not open (%s) mode (%s)",func,conf_file, strerror(errno),mode);
		free(conf_file);
		return NULL;
	}
	free(conf_file);
	
	return fp;
}

static char *
create_dir(char * base_dir, char * type_path)
{
	char * func = "create_dir()";
	DIR * dp = NULL;
	char * p = NULL;
	char * dir_name = NULL;

	/* create directory if it's not yet created */
	if (base_dir == NULL)
	{
		show_error("%s:dir is NULL",func);
		return NULL;
	}
	if ((dp = opendir(base_dir)) == NULL)
	{
		if ((errno == ENOTDIR) || (errno = ENOENT))
		{
			if (mkdir(base_dir, 0700) != 0)
			{
				show_error("%s:mkdir failed (%s)",func,strerror(errno));
				return NULL;
			}
		}
		else
		{
			show_error("%s:opendir failed %d(%s)",func,errno,strerror(errno));
			return NULL;
		}
	}
	else
	{
		closedir(dp);
	}

	if (type_path != NULL)
	{
		if ((p = strrchr(type_path, '/')) == NULL)
		{
			return NULL;
		}
		p ++;
		dir_name = malloc(strlen(base_dir) + strlen(p) + 4);
		sprintf(dir_name,"%s/%s",base_dir,p);
		create_dir(dir_name, NULL);
	}

	return dir_name;
}

char *
PGRget_dir_name_with_hostname(char * buf, char * path, Physical_Server_Info *server)
{
	char *p = NULL;
	if (server == NULL)
	{
		return NULL;
	}
	if ((p = strchr(server->hostName,'.')) != NULL)
	{
		*p = '\0';
	}
	sprintf(buf,"%s/%s",path,server->hostName);
	if (p != NULL)
		*p = '.';
	return buf;
}

void
PGRcreate_pgcluster_conf (char * path)
{
	int rtn = STATUS_OK;

	/* create pglb conf file */
	create_pglb_conf(PGR_Write_Path);

	/* create pgreplicate conf file */
	create_pgrp_conf(PGR_Write_Path);

	/* create db cluster directory */
	rtn = create_db_cluster_dir (PGR_Write_Path);
	if (rtn == STATUS_OK)
	{
		/* create cluster conf file */
		create_cluster_conf(PGR_Write_Path);
	}

}

static int
create_pglb_conf (char * path)
{
	char conf_dir[512];
	int i;
	Physical_Server_Info * server = NULL;
	Pglb_Info * pglb = PglbTbl;

	if (path == NULL) 
	{
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->pglbNum ; i ++, pglb ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(pglb->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		create_each_pglb_conf(conf_dir, server, pglb);
	}
	return STATUS_OK;
}

static int
create_each_pglb_conf(char * conf_dir, Physical_Server_Info * server, Pglb_Info * pglb )
{
	char * func = "create_each_pglb_conf()";
	FILE * fp = NULL;
	char * fname = PGLB_CONF_FILE;

	if ((fp = PGRopen_write_file(conf_dir, pglb->workPath, fname, "w")) == NULL)
	{
		show_error("%s: args is NULL",func);
		return STATUS_ERROR;
	}

	/* create cluster db part of pglb.conf */
	fp = put_cluster_part_of_pglb_conf(0, fp );

	/* create body part of pglb.conf */
	fp = put_pglb_conf_rec(0, fp, server, pglb );

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static FILE *
put_cluster_part_of_pglb_conf (int indent, FILE * fp)
{
	char * func = "put_cluster_part_of_pglb_conf()";
	Physical_Server_Info * server = NULL;
	Cluster_Info * cluster = ClusterDbTbl;
	int i;
	char buf[512];

	if (fp == NULL)
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	show_debug("clusterNum[%d]", RecordNumTbl->clusterNum );
	for (i = 0 ; i < RecordNumTbl->clusterNum ; i ++, cluster ++)
	{
		server = PGRget_Physical_Server_Rec(cluster->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		/* set record header */
		put_conf_header(indent, fp, "Cluster DB", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* Host_Name */
		put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

		/* Port */
		sprintf(buf,"%d",cluster->portNumber );
		put_conf_data_with_tag(indent, fp, PORT_TAG, buf);
	
		/* Max_Connect */
		sprintf(buf,"%d",cluster->maxConnections );
		put_conf_data_with_tag(indent, fp, MAX_CONNECT_TAG, buf);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static FILE *
put_pglb_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Pglb_Info * pglb ) 
{
	char * func = "put_pglb_conf_rec()";
	char buf[512];

	if ((fp == NULL) || (pglb == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Load Balancer", server->physicalServerId);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* Backend_Socket_Dir */
	put_conf_data_with_tag(indent, fp, BACKEND_SOCKET_DIR_TAG, pglb->backendSocketDir);

	/* Receive_Port */
	sprintf(buf,"%d",pglb->receivePortNumber );
	put_conf_data_with_tag(indent, fp, RECV_PORT_TAG, buf);

	/* Recovery_Port */
	sprintf(buf,"%d",pglb->recoveryPortNumber );
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* Max_Cluster_Num */
	sprintf(buf,"%d",pglb->maxClusterNum);
	put_conf_data_with_tag(indent, fp, MAX_CLUSTER_TAG, buf);

	/* Use_Connection_Pooling */
	put_conf_data_with_tag(indent, fp, USE_CONNECTION_POOL_TAG, pglb->usePool);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, pglb->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, pglb->lifecheckInterval);

	/* Connection_Life_Time */
	put_conf_data_with_tag(indent, fp, CONNECTION_LIFE_TIME_TAG, pglb->connectionLifetime);

	/* Log_File_Name */
	put_conf_data_with_tag(indent, fp, LOG_FILE_NAME_TAG, pglb->logFileName);

	/* Log_File_Size */
	put_conf_data_with_tag(indent, fp, LOG_FILE_SIZE_TAG, pglb->logFileSize);

	/* Log_Rotate */
	sprintf(buf,"%d",pglb->logRotate);
	put_conf_data_with_tag(indent, fp, LOG_ROTATION_TAG, buf);

	return fp;
}

static int
create_pgrp_conf (char * path)
{
	char * func = "create_pgrp_conf()";
	char conf_dir[512];
	int i;
	Physical_Server_Info * server = NULL;
	Pgrp_Info * pgrp = PgrpTbl;

	if (path == NULL) 
	{
		show_error("%s: args is NULL",func);
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->pgrpNum ; i ++, pgrp++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(pgrp->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		create_each_pgrp_conf(conf_dir, server, pgrp);
	}
	return STATUS_OK;
}

static int
create_each_pgrp_conf(char * conf_dir, Physical_Server_Info * server, Pgrp_Info * pgrp )
{
	char * func = "create_each_pgrp_conf()";
	FILE * fp = NULL;
	char * fname = PGREPLICATE_CONF_FILE;

	if ((fp = PGRopen_write_file(conf_dir, pgrp->workPath, fname, "w")) == NULL)
	{
		show_error("%s: args is NULL",func);
		return STATUS_ERROR;
	}

	/* create cluster db part of pgrp.conf */
	fp = put_cluster_part_of_pgrp_conf(0, fp );

	/* create pglb part of pgrp.conf */
	fp = put_pglb_part_of_pgrp_conf (0, fp );

	/* create body part of pgrp.conf */
	fp = put_pgrp_conf_rec(0, fp, server, pgrp );

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static FILE *
put_cluster_part_of_pgrp_conf (int indent, FILE * fp)
{
	char * func = "put_cluster_part_of_pgrp_conf()";
	Physical_Server_Info * server = NULL;
	Cluster_Info * cluster = ClusterDbTbl;
	int i;
	char buf[512];

	if (fp == NULL)
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->clusterNum ; i ++, cluster ++)
	{
		server = PGRget_Physical_Server_Rec(cluster->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		/* set record header */
		put_conf_header(indent, fp, "Cluster DB", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* Host_Name */
		put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

		/* Port */
		sprintf(buf,"%d",cluster->portNumber );
		put_conf_data_with_tag(indent, fp, PORT_TAG, buf);
	
		/* Recovery_Port */
		sprintf(buf,"%d",cluster->recoveryPortNumber );
		put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, CLUSTER_SERVER_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static FILE *
put_pglb_part_of_pgrp_conf (int indent, FILE * fp)
{
	char * func = "put_pglb_part_of_pgrp_conf()";
	Physical_Server_Info * server = NULL;
	Pglb_Info * pglb = PglbTbl;
	int i;
	char buf[512];

	if (fp == NULL)
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->pglbNum ; i ++, pglb ++)
	{
		server = PGRget_Physical_Server_Rec(pglb->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		/* set record header */
		put_conf_header(indent, fp, "Load Balance Server ", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, LOAD_BALANCE_SERVER_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* Host_Name */
		put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

		/* Recovery_Port */
		sprintf(buf,"%d",pglb->recoveryPortNumber );
		put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, LOAD_BALANCE_SERVER_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static FILE *
put_pgrp_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Pgrp_Info * pgrp ) 
{
	char * func = "put_pgrp_conf_rec()";
	char buf[512];

	if ((fp == NULL) || (pgrp == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Replicator", server->physicalServerId);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* Replication_Port */
	sprintf(buf,"%d",pgrp->replicationPortNumber );
	put_conf_data_with_tag(indent, fp, REPLICATE_PORT_TAG, buf);

	/* Recovery_Port */
	sprintf(buf,"%d",pgrp->recoveryPortNumber );
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* RLOG_Port */
	sprintf(buf,"%d",pgrp->RLogPortNumber);
	put_conf_data_with_tag(indent, fp, RLOG_PORT_TAG, buf);

	/* Use_Replication_Log */
	put_conf_data_with_tag(indent, fp, USE_REPLICATION_LOG_TAG, pgrp->useRlog);

	/* Replication_Timeout */
	put_conf_data_with_tag(indent, fp, TIMEOUT_TAG, pgrp->replicationTimeout);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, pgrp->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, pgrp->lifecheckInterval);

	/* Log_File_Name */
	put_conf_data_with_tag(indent, fp, LOG_FILE_NAME_TAG, pgrp->logFileName);

	/* Log_File_Size */
	put_conf_data_with_tag(indent, fp, LOG_FILE_SIZE_TAG, pgrp->logFileSize);

	/* Log_Rotate */
	sprintf(buf,"%d",pgrp->logRotate);
	put_conf_data_with_tag(indent, fp, LOG_ROTATION_TAG, buf);

	return fp;
}

static int
create_cluster_conf (char * path)
{
	char * func = "create_cluster_conf()";
	char conf_dir[512];
	int i;
	Physical_Server_Info * server = NULL;
	Cluster_Info * cluster = ClusterDbTbl;

	if (path == NULL) 
	{
		show_error("%s: args is NULL",func);
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->clusterNum ; i ++, cluster ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(cluster->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		/* create postgresql.conf file */
		create_each_postgresql_conf(conf_dir, server, cluster);
show_debug("%s:create_each_postgresql_conf OK",func);
		/* create pg_hba.conf file */
		create_each_pg_hba_conf(conf_dir, cluster);
show_debug("%s:create_each_pg_hba_conf OK",func);
		/* create cluste.conf file */
		create_each_cluster_conf(conf_dir, server, cluster);
show_debug("%s:create_each_cluster_conf OK",func);
	}
	return STATUS_OK;
}

static int
create_each_postgresql_conf(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster )
{
	char * func = "create_each_postgresql_conf()";
	FILE * fp = NULL;
	FILE * tmpfp = NULL;
	char * fname = POSTGRESQL_CONF_FILE;
	char * tmp_fname = "/tmp/create_each_postgresql_conf.tmp";
	char buf[24];

	if ((fp = PGRopen_write_file(conf_dir, cluster->workPath, fname, "r+")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,fname);
		return STATUS_ERROR;
	}
	if ((tmpfp = fopen(tmp_fname, "w")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,tmp_fname);
		return STATUS_ERROR;
	}

	/* change listen address */
	fp = PGRupdate_conf_contents(fp,tmpfp,"listen_addresses = ", "'*'");
	/* change port number */
	sprintf(buf,"%d",cluster->portNumber);
	fp = PGRupdate_conf_contents(fp,tmpfp,"port = ", buf);
	/* change max connections */
	sprintf(buf,"%d",cluster->maxConnections);
	fp = PGRupdate_conf_contents(fp,tmpfp,"max_connections = ", buf);
	fp = PGRupdate_conf_contents(fp,tmpfp, NULL, buf);
	fclose(fp);
	fflush(tmpfp);
	fclose(tmpfp);

	if ((fp = PGRopen_write_file(conf_dir, cluster->workPath, fname, "w")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,fname);
		return STATUS_ERROR;
	}
	if ((tmpfp = fopen(tmp_fname, "r")) == NULL)
	{
		show_error("%s:(%s)open failed",func,fname);
		return STATUS_ERROR;
	}
	fp = PGRcopy_file(tmpfp,fp);
	fclose(fp);
	fclose(tmpfp);
	unlink(tmp_fname);

	return STATUS_OK;
}

FILE * 
PGRupdate_conf_contents(FILE * rfp, FILE * wfp, char * tag, char * data)
{
	char buf[512];
	char hit = 0;
	if ((rfp == NULL) || (wfp == NULL ))
	{
		return NULL;
	}
	while (fgets(buf, sizeof(buf), rfp) != NULL)
	{
		if ((tag != NULL) && (strstr(buf,tag) != NULL))
		{
			sprintf(buf,"%s %s\n",tag, data);
			hit = 1;
		}
		fputs(buf,wfp);
		if (hit)
			break;
	}
	fflush(wfp);
	return rfp;
}

FILE * 
PGRcopy_file(FILE * src_fp, FILE * dest_fp)
{
	char buf[512];
	if ((src_fp == NULL) || (dest_fp == NULL )) 
	{
		return NULL;
	}
	while (fgets(buf, sizeof(buf), src_fp) != NULL)
	{
		fputs(buf,dest_fp);
	}
	fflush(dest_fp);
	return dest_fp;
}

static int
create_each_pg_hba_conf(char * conf_dir, Cluster_Info * cluster )
{
	char * func = "create_each_pg_hba_conf()";
	FILE * fp = NULL;
	int i = 0;
	char * fname = PG_HBA_CONF_FILE;
	Physical_Server_Info * server = PhysicalServerTbl;
	char buf[512];

	if ((fp = PGRopen_write_file(conf_dir, cluster->workPath, fname, "a")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,fname);
		return STATUS_ERROR;
	}

	fputs("# Add pgcluster server address\n",fp);
	for (i = 0 ; i < RecordNumTbl->serverNum ; i ++, server ++)
	{
		sprintf(buf,"host    all         all         %s/32	trust\n",server->ipAddr);
		fputs(buf,fp);
	}

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static int
create_each_cluster_conf(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster )
{
	char * func = "create_each_cluster_conf()";
	FILE * fp = NULL;
	char * fname = CLUSTER_CONF_FILE;

	if ((fp = PGRopen_write_file(conf_dir, cluster->workPath, fname, "w")) == NULL)
	{
		show_error("%s:(%s)PGRopen_write_file failed",func,fname);
		return STATUS_ERROR;
	}

	/* create replication server part of cluster.conf */
	fp = put_replicate_part_of_cluster_conf(0, fp );

	/* create body part of cluster.conf */
	fp = put_cluster_conf_rec (0, fp, server, cluster );

	if (fp != NULL)
	{
		fclose(fp);
	}
	return STATUS_OK;
}

static FILE *
put_replicate_part_of_cluster_conf (int indent, FILE * fp)
{
	char * func = "put_replicate_part_of_cluster_conf()";
	Physical_Server_Info * server = NULL;
	Pgrp_Info * pgrp = PgrpTbl;
	int i;
	char buf[512];

	if (fp == NULL)
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->pgrpNum ; i ++, pgrp++)
	{
		server = PGRget_Physical_Server_Rec(pgrp->physicalServerId);
		if (server == NULL)
		{
			show_error("%s: PGRget_Physical_Server_Rec failed",func);
			continue;
		}

		/* set record header */
		put_conf_header(indent, fp, "Replicator", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, REPLICATION_SERVER_INFO_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* Host_Name */
		put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

		/* Port */
		sprintf(buf,"%d",pgrp->replicationPortNumber );
		put_conf_data_with_tag(indent, fp, PORT_TAG, buf);
	
		/* Recovery_Port */
		sprintf(buf,"%d",pgrp->recoveryPortNumber );
		put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, REPLICATION_SERVER_INFO_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static FILE *
put_cluster_conf_rec (int indent, FILE * fp, Physical_Server_Info * server, Cluster_Info * cluster ) 
{
	char * func = "put_cluster_conf_rec()";
	char buf[512];

	if ((fp == NULL) || (cluster == NULL) || (server == NULL))
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	/* set record header */
	put_conf_header(indent, fp, "Cluster DB", server->physicalServerId);

	/* Host_Name */
	put_conf_data_with_tag(indent, fp, HOST_NAME_TAG, server->hostName);

	/* Recovery_Port */
	sprintf(buf,"%d",cluster->recoveryPortNumber );
	put_conf_data_with_tag(indent, fp, RECOVERY_PORT_TAG, buf);

	/* Rsync_Path */
	put_conf_data_with_tag(indent, fp, RSYNC_PATH_TAG, cluster->rsyncPath);

	/* Rsync_Option */
	put_conf_data_with_tag(indent, fp, RSYNC_OPTION_TAG, cluster->rsyncOption);
	/* Rsync_Compress */
	put_conf_data_with_tag(indent, fp, RSYNC_COMPRESS_TAG, cluster->rsyncCompress);

	/* Rsync_Timeout */
	put_conf_data_with_tag(indent, fp, RSYNC_TIMEOUT_TAG, cluster->rsyncTimeout);

	/* Rsync_Bwlimit */
	put_conf_data_with_tag(indent, fp, RSYNC_BWLIMIT_TAG, cluster->rsyncBwlimit);

	/* Pg_Dump_Path */
	put_conf_data_with_tag(indent, fp, PG_DUMP_PATH_TAG, cluster->pgdumpPath);

	/* Ping_Path */
	put_conf_data_with_tag(indent, fp, PING_PATH_TAG, cluster->pingPath);

	/* When_Stand_Alone */
	put_conf_data_with_tag(indent, fp, STAND_ALONE_TAG, cluster->whenStandAlone);

	/* Replication_Timeout */
	put_conf_data_with_tag(indent, fp, TIMEOUT_TAG, cluster->replicationTimeout);

	/* LifeCheck_Timeout */
	put_conf_data_with_tag(indent, fp, LIFECHECK_TIMEOUT_TAG, cluster->lifecheckTimeout);

	/* LifeCheck_Interval */
	put_conf_data_with_tag(indent, fp, LIFECHECK_INTERVAL_TAG, cluster->lifecheckInterval);

	/* Non Replication part */
	put_not_replication_part_of_cluster_conf (indent, fp);

	return fp;
}

static FILE *
put_not_replication_part_of_cluster_conf (int indent, FILE * fp)
{
	char * func = "put_not_replication_part_of_cluster_conf()";
	Partial_Info * partial = PartialTbl;
	char buf[512];
	int i;

	if (fp == NULL)
	{
		show_error("%s: args is NULL",func);
		return NULL;
	}
	for (i = 0 ; i < RecordNumTbl->partialNum ; i ++, partial++)
	{

		if (strlen(partial->dbName) == 0)
			continue;
		/* set record header */
		put_conf_header(indent, fp, "Not Replicate", i+1);

		/* start tag */
		fputs( begin_tag(indent, buf, NOT_REPLICATE_INFO_TAG) , fp);
		fputs("\n",fp);

		indent ++;
		/* DB_Name */
		put_conf_data_with_tag(indent, fp, DB_NAME_TAG, partial->dbName);

		/* Table_Name */
		put_conf_data_with_tag(indent, fp, TABLE_NAME_TAG, partial->tableName);
	
		/* Query */
		put_conf_data_with_tag(indent, fp, QUERY_TAG, partial->query);

		indent --;
		/* end tag */
		fputs( end_tag(indent, buf, NOT_REPLICATE_INFO_TAG) , fp);
		fputs("\n",fp);
	}

	return fp;
}

static int
create_db_cluster_dir (char * path)
{
	char * func = "create_db_cluster_dir()";
	char conf_dir[512];
	int i;
	Physical_Server_Info * server = NULL;
	Cluster_Info * cluster = ClusterDbTbl;

	if (path == NULL) 
	{
		show_error("%s: args is NULL",func);
		return STATUS_ERROR;
	}
	
	for (i = 0 ; i < RecordNumTbl->clusterNum ; i ++, cluster ++)
	{
		memset(conf_dir, 0, sizeof(conf_dir));
		server = PGRget_Physical_Server_Rec(cluster->physicalServerId);
		if (PGRget_dir_name_with_hostname(conf_dir, path, server) == NULL)
		{
			continue;
		}
		exec_initdb(conf_dir, server, cluster);
	}
	return STATUS_OK;
}

static int
exec_initdb(char * conf_dir, Physical_Server_Info * server, Cluster_Info * cluster )
{
	char * func ="exec_initdb()";
	char * args[8];
	int i = 0;
	char command[128];
	char data_path[512];
	char * p = NULL;

	show_debug("%s:workPath[%s]",func,cluster->workPath);
	p = strrchr(cluster->workPath,'/');
	p++;
	sprintf(data_path,"%s/%s",conf_dir,p);
	sprintf(command,"%s/initdb",cluster->binPath);

	args[i++] = "initdb";
	args[i++] = "-U";
	args[i++] = server->userName;
	args[i++] = "-D";
	args[i++] = data_path;
	args[i++] = NULL;

	return (PGR_execv(command, args));
}

int
PGR_execv(char * command, char *args[])
{
	pid_t pid = 0;
	int status;

	pid = fork();
	if (pid == 0)
	{
		status = execv(command, args);
		exit(0);
	}
	else
	{
		for (;;)
		{
			int result;
			result = wait(&status);
			if (result < 0)
			{
				if (errno == EINTR)
					continue;
				return STATUS_ERROR;
			}

			if (WIFEXITED(status) == 0 || WEXITSTATUS(status) != 0)
				return STATUS_ERROR;
			else
				break;
		}
	}
	return STATUS_OK;
}

