/*
 * dbnames.c: Database connection management
 *
 * Copyright (c) 2009-2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */
 
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "pg_reporter.h"

static Database *getNextDatabase(FILE *fp, int *lineno,
								 const char *id, const char *config);
static int tokenize(char *p, char *tokens[], size_t length[], int max_tokens);

static List *dblist = NIL;	/* database list from pg_reporter.conf */

/* retrieve all database information */
List *
getDatabases(const char *config)
{
	FILE	   *fp;
	int			lineno;
	Database   *db;
	List	   *dbs = NIL;

	if (dblist != NIL)
		return dblist;

	if (config == NULL)
		config = DBNAMES;

	if ((fp = fopen(config, "rt")) ==  NULL)
	{
		elog(ERROR, "could not open file: \"%s\": %s",
			 config, strerror(errno));
	}

	lineno = 1;
	while ((db = getNextDatabase(fp, &lineno, NULL, config)) != NULL)
		dbs = lappend(dbs, db);

	fclose(fp);

	if (dbs == NIL)
	{
		elog(ERROR, "no configuration information: \"%s\"", config);
	}

	dblist = dbs;
	return dbs;
}

/* retrieve one database information */
Database *
getDatabase(const char *id)
{
	Database   *db;
	ListCell *cell;

	if (dblist == NIL)
		elog(ERROR, "Database list is NIL");

	foreach (cell, dblist)
	{
		db = lfirst(cell);
		if (strcmp(id, db->id) == 0)
			return db;
	}

	return NULL;

}

/* retrieve first database information from dblist */
/* Collected database information (getDatabases()) assumed */
Database *
getFirstDatabase(void)
{
	if (dblist == NIL)
	{
		elog(ERROR, "Database list is NIL");
	}

	return list_nth(dblist, 0);
}

/* free one database information */
void
freeDatabase(Database *db)
{
	if (db == NULL)
		return;

	free(db->id);
	free(db->host);
	free(db->port);
	free(db->database);
	free(db->username);
	free(db->password);
	free(db->report_dir);
	free(db);
}

/*
 * Returns the next database.
 * If id is not NULL, find the database that is named as 'id'.
 */
static Database *
getNextDatabase(FILE *fp, int *lineno, const char *id, const char *config)
{
#define LINEBUFSZ		1000
#define COLUMNCOUNT		7
#define COLUMNSIZE      5

	char		line[LINEBUFSZ];

	for (; fgets(line, LINEBUFSZ, fp) != NULL; (*lineno)++)
	{
		size_t		len;
		char	   *tokens[COLUMNCOUNT];
		size_t		length[COLUMNCOUNT];
		int			ntoken;
		Database   *db;
		char       *report_dir;
		struct stat buf;

		len = strlen(line);
		if (line[len - 1] != '\n')
		{
			elog(ERROR, "too long line in file \"%s\": line %d",
				config, *lineno);
		}

		while (len > 0 && IsSpace(line[len - 1]))
		{
			line[len - 1] = '\0';
			len--;
		}

		ntoken = tokenize(line, tokens, length, COLUMNCOUNT);

		if (ntoken == 0)
			continue;	/* empty line */
		else if (ntoken < COLUMNSIZE)
		{
			elog(ERROR, "syntax error in file \"%s\": line %d",
				config, *lineno);
		}

		if (id != NULL &&
			(strncmp(id, tokens[0], length[0]) != 0 || id[length[0]] != '\0'))
			continue;

		report_dir = strdup_with_len(tokens[1], length[1]);
		/* check report_dir */
		if (report_dir[0] != '/')
		{
			elog(ERROR, "specified directory is not a full path: \"%s\": file \"%s\": line %d",
				 report_dir, config, *lineno);
		}
		if (stat(report_dir, &buf) == -1)
		{
			elog(ERROR, "could not access the directory: \"%s\": %s : file \"%s\": line %d",
				 report_dir, strerror(errno), config, *lineno);
		}
		if (S_ISDIR(buf.st_mode) == 0)
		{
			elog(ERROR, "specified path is not a directory: \"%s\": file \"%s\": line %d",
				 report_dir, config, *lineno);
		}
		if (!(((buf.st_mode & (S_IRUSR|S_IXUSR)) == (S_IRUSR|S_IXUSR)
			   && buf.st_uid == getuid())
			  ||((buf.st_mode & (S_IRGRP|S_IXGRP)) == (S_IRGRP|S_IXGRP)
				 && buf.st_gid == getgid())
			  ||((buf.st_mode & (S_IROTH|S_IXOTH)) == (S_IROTH|S_IXOTH))))
		{
			elog(ERROR, "could not access the directory: \"%s\": file \"%s\": line %d",
				 report_dir, config, *lineno);
		}

		db = pgut_new(Database);
		db->id = strdup_with_len(tokens[0], length[0]);
		db->report_dir = report_dir;
		db->host = strdup_with_len(tokens[2], length[2]);
		db->port = strdup_with_len(tokens[3], length[3]);
		db->database = strdup_with_len(tokens[4], length[4]);
		db->username = strdup_with_len(tokens[5], length[5]);	/* nulls ok */
		db->password = strdup_with_len(tokens[6], length[6]);	/* nulls ok */

		(*lineno)++;

		return db;
	}

	return NULL;
}

/*
 * Split text into tokens with white-spaces.
 * Ignore after comment charaster '#'.
 */
static int
tokenize(char *p, char *tokens[], size_t length[], int max_tokens)
{
	int		n;
	char   *end = p + strlen(p);

	for (n = 0; n < max_tokens; n++)
	{
		/* skip white-spaces */
		for (;;)
		{
			if (p >= end || *p == '#')
				return n;
			if (!IsSpace(*p))
				break;
			p++;
		}

		/* get a token */
		tokens[n] = p;
		for (;;)
		{
			if (p >= end || *p == '#')
			{
				length[n] = p - tokens[n];
				++n;
				goto done;
			}
			if (IsSpace(*p))
				break;
			p++;
		}
		length[n] = p - tokens[n];
	}

done:
	/* fill with zeros remaining fields */
	memset(&tokens[n], 0, sizeof(*tokens) * (max_tokens - n));
	memset(&length[n], 0, sizeof(*length) * (max_tokens - n));

	return n;
}
