#!/usr/bin/perl
#############################################################################
#  Copyright (C) 2009 Nippon Telegraph and Telephone Corporation
#############################################################################

#####################################################################
# Function: pg_make_report.pl
#
#
# summary:
# This is a frontend of generate reports function. This function generates
# a report according to specific template file, and output the information. 
#
#
# recital:
# needs following modules.
#
# Text::Template
# DBI
# DBD::Pg
#
#####################################################################


use warnings;
use strict;
use Carp;
use Getopt::Long;
use utf8;
use File::Basename;
use DBI;
use Text::Template;

our $report_resources;

use FindBin;
use lib $FindBin::Bin;
use pg_report_config84;

#-------------------------------------------------------------------------------
# Queries necessary for connecting DB and set collecting target snapshot id 
#
# When we get the snapshot report, we have to set two factors.
# 1. Which DBs we want to analysis ?
#    - We can select "all" or "specified DBID" or "specified connecting string"
# 2. Which snapshots we want to analysis ?
#    - We can select "specified snapshot id range" or "specified timestamp range"
#
# Following queries set above factor, so please be carefull to edit these queries.
#-------------------------------------------------------------------------------

#
# variable: dbconn_all_query
# getting DB connection information (all DB)
#
our $dbconn_all_query = q(
    SELECT dbid, dbname, hostname, port, username FROM t_dbconn WHERE is_snapshot_target = 't' ORDER BY dbid;
);

        
#
# variable: dbconn_targ_query
# getting DB connection information (specified DBID)
#
our $dbconn_targ_query = q(
    SELECT dbid, dbname, hostname, port, username FROM t_dbconn WHERE is_snapshot_target = 't' and dbid = ?
);

#
# variable: dbconn_targ_by_str_query
# getting DB connection information (specified connection-info-string)
#
our $dbconn_targ_by_str_query = q(
    SELECT dbid, dbname, hostname, port, username FROM t_dbconn WHERE is_snapshot_target = 't' and hostname = ? and port = ? and dbname = ? 
);


#
# variable: get_snap_by_time_query
# getting set of snapshot ids between specified begin-time and end-time for each DBID.
#
our $get_snap_by_time_query = q(
    SELECT host_id, dbid, snapshot_id, snapshot_date, level, description FROM t_snapshot_id WHERE snapshot_date between ? and  ? AND dbid IN (%DBID%) ORDER BY host_id, dbid, snapshot_id
);

#
# variable: get_snap_by_id_query
# getting set of snapshot ids between specified begin-id and end-id for each DBID.
#
our $get_snap_by_id_query = q(
    SELECT host_id, dbid, snapshot_id, snapshot_date, level, description  FROM t_snapshot_id WHERE snapshot_id between ? and ? and dbid = (SELECT dbid FROM t_snapshot_id WHERE snapshot_id IN (?,?) GROUP BY dbid)
);

#
# variable: get_epoch_time
# getting elapse time between specied 2 point
#
our $get_epoch_time_query = q( SELECT EXTRACT(EPOCH FROM AGE(?, ?)) AS second );

# For setting search_path 
our $set_search_path_query = q( SET search_path = statsinfo, public );




binmode(STDOUT, ":utf8");


#####################################################################
# Constants: statistic value, message definition
#####################################################################
use constant {
    MODULENAME => basename($0, (".pl")),
    DATASOURCE => 'dbi:Pg:',
    VERSION   => "1.2.0 (For PostgreSQL 8.4)",
    TRUE      => "1",
    FALSE     => "0",
    NO_DATA   => q(none),
    
    
    MSG_OPT_NOOPT               => 'necessary options were not specified. please check pg_make_report options.',
    MSG_OPT_CONFLICT_ID         => 'a option conflicts with other one.',
    MSG_OPT_INVALID_PREFIX      => 'could not generate out put file.',
    MSG_OPT_INVALID_TMPFILE     => 'could not read template file.',
    MSG_OPT_INVALID_ID          => 'start-ID specifies database different from the one being specified end-ID',
    MSG_OPT_CONFLICT_DBID       => '--startid, --endid and --targetdbid cannot be specified together. --targetdbid ignored.',
    MSG_OPT_NO_ID               => 'could not get valid snapshot-id. please check specified snapshot-id, and range of specified time.',
    MSG_DB_CONNFAILED           => 'could not connect database.',
    MSG_DB_SELECTFAILED         => 'collecting information failed.',
    MSG_DB_NO_DBCONN            => 'failed on getting database connection info. If you specify --targetdbid, check the value.',
    MSG_RESOURCE_MISSED         => 'necessary entries were not defined in pg_report_config.pm.',
    MSG_TEMPLATE_FAILED         => 'failed on performing template-engine.',
    
};

# variable: G_db_conn
# DB connection information list
#
my $G_dbinfo;

# variable: G_options
# options information
#
my $G_options;

# variable: G_connattr
# attributes of connection
#
my $G_connattr = {Columns => {}};


#
# variable: G_process
# processes for analysis of data trends
# This module is given set of data values and outputs calculated results.
# -- "diffmean" taken while getting snapshot --
#
# These functions are given refarence of set of data values,
# and must return a value or hash array.
#
my $G_process = {
    mean      => sub { return calcstats(@_)->{mean}; },
    variance  => sub { return calcstats(@_)->{variance}; },
    max       => sub { return calcstats(@_)->{max}; },
    min       => sub { return calcstats(@_)->{min}; },
    latest    => sub { return scalar @{$_[0]} > 1 ? $_[0]->[-1] : scalar @{$_[0]} > 0 ? $_[0]->[0]: NO_DATA; },
    oldest    => sub { return scalar @{$_[0]} > 0 ? $_[0]->[0] : NO_DATA; },
    allvalues => sub { return $_[0] }, 
};

#
# variable: G_none_process
# processes for analysis of data trends (Data not found)
# This varliable is for the case taht we could not get set of data.
# ex) checkpoint did not execute during specified snapshot id range.
#
my $G_none_process = {
    mean      => sub { return NO_DATA; },
    variance  => sub { return NO_DATA; },
    max       => sub { return NO_DATA; },
    min       => sub { return NO_DATA; },
    latest    => sub { return NO_DATA; },
    oldest    => sub { return NO_DATA; },
    allvalues => sub { return [ NO_DATA ]; }, 
};


main();

#####################################################################
# Function: main
#
#
# summary:
# Control generate report execution
#
# parameters:
# ARGV - list of parameters
#
# return:
# 0 - normal exit
# 1 - abnormal exit
#
# execption:
# none
#
# recital:
# none
#####################################################################
sub main {
    
    #
    # option analysis
    #
    $G_options = analyze_option();

    my $datasource = DATASOURCE . "dbname=$G_options->{dbname};host=$G_options->{host};port=$G_options->{port}";
    my $conn;
    
    eval {
        #
        # Create connection to snapshot manager DB.
        # If failed, we prompt password for user (limit 10 sec).
		#

		$SIG{ALRM} = sub { die "timeout" };
        $conn = DBI->connect($datasource, $G_options->{user}, 0, {RaiseError => 0, PrintError => 0});
        if(!$conn) {
            print "Password: ";
			alarm 10;
            my $passwd = <STDIN>;
			alarm 0;
            chomp($passwd);
    
            $conn = DBI->connect($datasource, $G_options->{user}, $passwd, {RaiseError => 0, PrintError => 0})
                or croak(MSG_DB_CONNFAILED . " dbname=$G_options->{dbname}, reason=$DBI::errstr");
        }
   
		# Set search_path
		execute_query($conn, $set_search_path_query);
 
        #
        # Getting connection information about target DB.
        # This information is using for determing target snapshot id.
        # -- If "targetdbid" is specified, we get information only about it.
        #    If not, we get all DB's information.
        #
        if($G_options->{is_target}) {
            
            if($G_options->{target_id}) {
                $G_dbinfo = execute_query($conn, $dbconn_targ_query, [$G_options->{target_id}]);
            } else {
                my @opts = (split('[:/]+', $G_options->{target_string}));
                $G_dbinfo = execute_query($conn, $dbconn_targ_by_str_query, \@opts);
            }
            
        } else {
            $G_dbinfo = execute_query($conn, $dbconn_all_query);
        }
        
        #
        # failed on getting target DB information..
        #
        (scalar @{$G_dbinfo} == 0) and croak(MSG_DB_NO_DBCONN);
        
        #
        # Getting start and end snapshot id.
        #
        my $snapid_hash = get_snapshot_id($conn, $G_dbinfo);


        #
        # OK. We get snapshot using by snapshot-id taken above.
        # Snapshot information are stored in $snapinfo_array.
        # In addtion, $report_resources is set of queries defined in "pg_report_config.pm".
        my $snapinfo_array = [];
        get_snapshot($conn, $snapid_hash, $report_resources, $snapinfo_array);
        $conn->disconnect;
        undef $conn;


        #
        # Read template file and convert line that begins in "|" to
        # $OUT output code.
        #
        open(DATA, $G_options->{tempfile}) or croak(MSG_OPT_INVALID_TMPFILE . " --template ($G_options->{tempfile}) ($!)");
        my $templatestring = do {local $/; <DATA>};
        $templatestring =~ s{^ \s* [|] (.*) $}{\$OUT.="$1\\n";}xmg;
        
        #
        # Execute template engine (Text::Template) to getting report.
        # We execute it for each DB
        #
        my $error = '';
        my $engine = Text::Template->new(
                    TYPE => 'STRING', SOURCE => $templatestring);
        for my $snapinfo_hash (@{$snapinfo_array}) {

	    # Do tempalte engine and fill the hash
            my $text = $engine->fill_in(HASH => $snapinfo_hash, BROKEN=>\&errorHandler, BROKEN_ARG => \$error);
            ($error ne '') and croak($error);
            ($text eq '') and croak(MSG_TEMPLATE_FAILED);
           
            #
            # If we set "prefix" option, outoput to file as "prefix"_hostname_port_dbname.txt 
            # If not, we outoput to STDOUT.
            #
            if(defined $G_options->{prefix}) {

                #
                # If hostname includes '/', we replace it to 'localhost'.
                #
                if ($snapinfo_hash->{hostname} =~ /[\/]/) {
                    $snapinfo_hash->{hostname} = 'localhost';
                }

                my @item = ($G_options->{prefix}, $snapinfo_hash->{hostname}, $snapinfo_hash->{port}, $snapinfo_hash->{dbname});
                my $outputfile = join('_', @item) . '.txt';
                
                open(OUT, ">$outputfile") or croak(MSG_OPT_INVALID_PREFIX . " ($outputfile) ($!)");
                print OUT $text;
                close(OUT);
            }
            else {
                print $text;
            }
        }
    };
    
    if($@) {
		if($@ =~/timeout/) {
			croak ("timeout on connect DB.\n");
		}
		else {
        	$conn and $conn->disconnect;
        	die($@);
		}
    }

    exit(0);
}


#####################################################################
# Function: get_snapshot_id
#
#
# summary:
# Getting snapshot-id for each DBID by using value of specified parameters
#
# parameters:
# conn   - connection
# dbinfo - DB connection information
#
# return:
# snapid - list of snapshot id
#
# execption:
# failed on execute queries or invalid option
#
# recital:
# results is follwing format..
# | $snapid->{Host ID}->{DBID}[1] = {sid => Snapshot ID, date => acquisition timestamp, level => level, desc => description}
# | $snapid->{Host ID}->{DBID}[2] = {sid => Snapshot ID, date => acquisition timestamp, level => level, desc => description}
# | $snapid->{Host ID}->{DBID}[3] = {sid => Snapshot ID, date => acquisition timestamp, level => level, desc => description}
# |...
#
#####################################################################
sub get_snapshot_id {
    my ($conn, $dbinfo) = @_;
    
    my $res;
    
    #
    # Getting snapshot-id in specified timestamp range.
    #
    if($G_options->{is_snapshot_date}) {
        
        #
        # If DBID or connection-string is also specified in option, we narrow
        # the target DB.
        # 
        my $dbidlist;                 # DBID list string (ex 1,2,3)
        if(scalar @{$dbinfo} == 1) {
            $dbidlist = $dbinfo->[0]->{dbid};
        }
        else {
            $dbidlist = join(',' , map {$_->{dbid}} @{$dbinfo});
        }
        my $query = $get_snap_by_time_query;
        
        $query =~ s{%DBID%}{$dbidlist}xms;

        $res = execute_query($conn, $query, 
                    [$G_options->{sdate}, $G_options->{edate}]);   
    } else {

        #
        # Getting snapshot-id in specified snapshot-id.
        #
        eval {
            $res = execute_query($conn, $get_snap_by_id_query,
                        [$G_options->{sid}, $G_options->{eid}, $G_options->{sid}, $G_options->{eid}]);
        };
        if($@) {
            #
            # If specified start and end id have deffrent DBID, we throw error. 
            #
            #
            croak(MSG_OPT_INVALID_ID . " reason=" . $@);    
        }
    }
    
    #
    # can we get valid snapshot-id ?
    #
    (scalar @{$res} <= 1) and croak(MSG_OPT_NO_ID);

    #
    # We convert getting information to hierarchical structure as 
    # "each host-id", "each DBID" and "snapshot-id".
    #
    my $hostinfo = {};
    
    for my $targ (@{$res}) {
        my $hostid = $targ->{host_id};
        my $dbid   = $targ->{dbid};
        my $sid    = $targ->{snapshot_id};
        my $date   = $targ->{snapshot_date};
        my $level  = $targ->{level};
        my $desc   = defined $targ->{description} ? $targ->{description} : 'none';
        
        if(!$hostinfo->{$hostid}->{$dbid}) {
           $hostinfo->{$hostid}->{$dbid} = (); 
        }
        push(@{$hostinfo->{$hostid}->{$dbid}},
                {sid => $sid, date => $date, level => $level, desc => $desc});
    }
    
    #
    # If we got only one  snapshot-id on a DBID, we throw error.
    #
    for my $hostid (keys %{$hostinfo}) {
        my $dbinfo = $hostinfo->{$hostid};
        
        for my $dbid (keys %{$dbinfo}) {
            scalar @{$dbinfo->{$dbid}} < 2 and croak(MSG_OPT_NO_ID . " dbid=" . $dbid);
        }
    }
    
    return $hostinfo;
}


#####################################################################
# Function: get_snapshot
#
#
# summary:
# Getting snapshot and store it in hash array
#
# get_snapshot 
#   -> get_snapshot_by_host(hostA)
#        -> get_snapshot_by_dbid(DB_A)
#        -> get_snapshot_by_dbid(DB_B) ...
#   -> get_snapshot_by_host(hostB)
#        -> get_snapshot_by_dbid(DB_1)
#        -> get_snapshot_by_dbid(DB_2) ...
#   ...
#
#
# parameters:
# conn             - connection
# snapid_hash      - set of snapshot-id for each host
# report_resources - query for getting snapshot(pg_report_config.pm)
# snapinfo_array   - [output] snapshot information array
#
# return:
# none
#
# execption:
# none
#
# recital:
# 
#####################################################################
sub get_snapshot {
    my ($conn, $snapid_hash, $report_resources, $snapinfo_array) = @_;
    
    for my $hostid (keys %{$snapid_hash}) {
        my $targ = $snapid_hash->{$hostid};
        get_snapshot_by_host($conn, $targ, $hostid, $report_resources, $snapinfo_array);
    }
}

#####################################################################
# Function: get_snapshot_by_host
#
#
# summary:
# Getting snapshot for each host, and store it in hash array
#
# parameters:
# conn             - connection
# snapid_hash      - set of snapshot-id for each DBID
# host_id          - host id
# report_resources - query for getting snapshot(pg_report_config.pm)
# snapinfo_array   - [output] snapshot information array
#
# return:
# none
#
# execption:
# none
#
# recital:
# 
#####################################################################
sub get_snapshot_by_host {
    my ($conn, $snapid_hash, $hostid, $report_resources, $snapinfo_array) = @_;
    for my $dbid (keys %{$snapid_hash}) {
        my $targ = $snapid_hash->{$dbid};
        get_snapshot_by_dbid($conn, $targ, $hostid, $dbid, $report_resources, $snapinfo_array);
    }
}


#####################################################################
# Function: get_snapshot_by_dbid
#
#
# summary:
# Getting snapshot for each DBID, and store it in hash array 
#
# parameters:
# conn             - connection
# $snapids         - set of snapshot-id for each DBID
# host_id          - host id
# dbid             - DBID
# report_resources - query for getting snapshot(pg_report_config.pm)
# snapinfo_array   - [output] snpshot information array
#
# return:
# none
#
# execption:
# none
#
# recital:
# 
#####################################################################
sub get_snapshot_by_dbid {
    my ($conn, $snapids, $hostid, $dbid, $report_resources, $snapinfo_array) = @_;
    
    #
    # Getting duration(seconds) between snapshots and calculate average
    # value per seconds. We also get diffrence between latest and oldest
    # snapshot. 
    #
    my $res = execute_query($conn, $get_epoch_time_query,
                    [$snapids->[-1]->{date}, $snapids->[0]->{date}]);
    my $passed = $res->[0]->{'second'};

    # average value per soconds
    $G_process->{diffmean} = 
        eval 'sub {my ($src) = @_;return (defined $src->[0] and scalar @{$src} > 0 and ' . $passed . ' != 0) ? ($src->[-1] - $src->[0]) / ' . $passed . ' : 0 ;}';

    # difference between oldest and latest
    $G_process->{diffv} = 
        eval 'sub {my ($src) = @_;return (defined $src->[0] and scalar @{$src} > 0 ) ? ($src->[-1] - $src->[0]) : 0 ;}';
    
    my $bindoptions = {
        START_ID   => $snapids->[0]->{sid},
        END_ID     => $snapids->[-1]->{sid},
        START_DATE => $snapids->[0]->{date},
        END_DATE   => $snapids->[-1]->{date},
        DBID       => $dbid,
        HOSTID     => $hostid, 
    };
    
    #
    # Create snapshot reference hash.
    # Following information are using for "report header" and so on..
    #
    my $dbinfo = (grep {$_->{dbid} == $dbid } @{$G_dbinfo})[0];
    
    my $snapinfo_hash = {
        start_snapshot_id => $bindoptions->{START_ID},
        end_snapshot_id   => $bindoptions->{END_ID},
        start_date        => $bindoptions->{START_DATE},
        end_date          => $bindoptions->{END_DATE},
        start_level       => $snapids->[0]->{level},
        end_level         => $snapids->[-1]->{level},
        start_desc        => $snapids->[0]->{desc},
        end_desc          => $snapids->[-1]->{desc},
        dbname            => $dbinfo->{dbname},
        hostname          => $dbinfo->{hostname},
        port              => $dbinfo->{port},
        username          => $dbinfo->{username},
    };
    
    get_snapshot_by_pk($conn, $snapids, $bindoptions, $report_resources, $snapinfo_hash, 0, TRUE);
    push(@{$snapinfo_array}, $snapinfo_hash);
}



#####################################################################
# Function: get_snapshot_by_pk
#
#
# summary:
# Getting snapshot from manege DB for each item. These items are 
# stored in $snapinfo_hash by PK(Praimary Key).
# The PK is defined in "pg_report_config.pm".
#
# parameters:
# conn             - connection
# snapids          - set of snapshot-id for each DBID
# bindoptions      - set of values bindable to query
# resources        - query for getting snapshot(pg_report_config.pm)
# snapinfo_hash    - [output] snapshot information hash
# cond             - PK(It's omissible. If you do so, cond is set as PKnone)
# istop            - is Top(1) level invoke?
#
# return:
# none
#
# execption:
# Do not set necessary variable to query information
# 
#
# recital:
# 
#####################################################################
sub get_snapshot_by_pk {
    my ($conn, $snapids, $bindoptions, $resources, $snapinfo_hash, $cond, $istop) = @_;

    #
    # Getting snapshot for each query, and store the information in hash.
    #
    #
    for my $targname (keys %{$resources}) {
        my $data = $resources->{$targname};
        #
        # check query and variables existence
        #
        exists($data->{query}) or croak(MSG_RESOURCE_MISSED . " (target: $targname) (item: query)");
        exists($data->{variables}) or croak(MSG_RESOURCE_MISSED . " (target: $targname) (item: variables)");


        #
        # Execute queries.
        #
        # If bind variables are set, convert them to actual value and execute
        # query. Results of queries are cached.
        #
        if(!exists($data->{snap_cache})) {
            $data->{orderd_pk} = ();         # list of description order of primary key
            $data->{exists_data} = TRUE;     # results of query existence
            my $res;
            if(exists($data->{bind})) {
                $data->{backupbind} = ();
                @{$data->{backupbind}} = @{$data->{bind}};
                for(my $i = 0; $i < scalar @{$data->{bind}}; $i++) {
                    $data->{bind}->[$i] = $bindoptions->{ $data->{bind}->[$i] };
                }
                
                $res = execute_query($conn, $data->{query}, $data->{bind});
            }
            else {
                $res = execute_query($conn, $data->{query});
            }
            
            $data->{snap_cache} = {};
            if(scalar @{$res} == 0) {
                #
                # If we could not get results, return "none".
                #
                $data->{exists_data} = FALSE;
                $data->{snap_cache}->{primarykey} = ();
                my $record = {};
                for my $varname (keys %{$data->{variables}}) {
                    $record->{$varname} = NO_DATA;
                }
                push(@{$data->{orderd_pk}}, 'primarykey');
                push(@{$data->{snap_cache}->{primarykey}}, $record);
            }
            else {
                #
                # We aggregate results by using primary-key. A set of results data 
                # (actually a record) would be unique for each PK.
                #
                for my $record (@{$res}) {
    
                    #
                    # generate PK-string from column specified as primary-key
                    #
                    my $pk;
                    if(exists($data->{primarykey})) {
                        $pk = join('.', map {$record->{$_}} @{$data->{primarykey}} );
                    }
                    else {
                        $pk = 'primarykey';
                    }
                    
                    #
                    # store data in hash for each PK
                    #
                    if(!exists($data->{snap_cache}->{$pk})) {
                        $data->{snap_cache}->{$pk} = ();
                        push(@{$data->{orderd_pk}}, $pk);
                    }
                    push(@{ $data->{snap_cache}->{$pk} }, $record);
                }
            }
        }
        
        #
        # aggregate snapshot information for each pk 
        #
        $snapinfo_hash->{$targname} = ();
        
        for my $pk (@{$data->{orderd_pk}}) {
            my $array = $data->{snap_cache}->{$pk};
            
            #
            # aggregate only the data relative upper primarykey.
            # ex) information of each columns are aggregated for each their table.
            #
            my $cmp_pk = $pk;
            $cmp_pk =~ s{[.][^.]+$}{}xms;
            if($data->{exists_data} and $cond and ($cmp_pk ne $cond)) {
                next;
            }
            
            my $one_array = {};
            while(my ($varname, $varinfo) = each %{$data->{variables}}) {
                
                my $statsname = $varname . '_stats';

                #
                # If "column name" is defined in variable, we get it. 
                # If not, we set variable'name as "column name".
                #
                my $colname = exists($varinfo->{colanme}) ? $varinfo->{colname} : $varname;

                #
                # Calculate data trend and store it in hash.
                # If we could not get any results, we store "none" in hash.
                #
                my $process;
                if($data->{exists_data}) {
                    $process = $G_process;
                } else {
                    $process = $G_none_process;
                }
                
                #
                # If there is some keyword in "default" but not in "stats", we add 
                # the keyword in "default" to "stats".
                # Now we only the keyword as "latest" in "default".
                # 
                #
                my $defaultkey = exists($varinfo->{default})
                    ? $varinfo->{default}
                    : 'latest';
                
                if(exists($varinfo->{stats}) 
                    and scalar(grep {$defaultkey eq $_} @{$varinfo->{stats}}) <= 0) {
                    push(@{$varinfo->{stats}}, $defaultkey);
                }
                $one_array->{$statsname} = create_stats($array, $colname, $varinfo->{stats}, $process);

                $one_array->{$varname} = $one_array->{$statsname}->{$defaultkey};
            }
            
            push(@{ $snapinfo_hash->{$targname} }, $one_array);
            
            
            #
            # Is there child information? If exist it, we aggregate it next.
            #
            #
            if(exists($data->{children})) {
                get_snapshot_by_pk($conn, $snapids, $bindoptions, 
                                        $data->{children}, $one_array, $pk, FALSE);
            }
        }

        #
        # We cached for each Top (1) level information (actually for each DB).
        # So, if istop is false, we clear the cache.
        #
        if($istop) {
            clearCache($data);
        }
    }
}


#####################################################################
# Function: clearCache
#
#
# summary:
# Clear the cache of query results 
#
# parameters:
# data - information for each query
#
# return:
# none
#
# execption:
# none
#
# recital:
# none
#####################################################################
sub clearCache {
    my ($data) = @_;
    exists($data->{bind}) and @{$data->{bind}} = @{$data->{backupbind}};
    exists($data->{snap_cache}) and delete $data->{snap_cache};
    exists($data->{exists_data}) and delete $data->{exists_data};
    exists($data->{orderd_pk}) and delete $data->{orderd_pk};
    
    if(exists($data->{children})) {
        for my $targname (keys %{$data->{children}}) {
            my $childdata = $data->{children}->{$targname};
            clearCache($childdata);
        }
    }
}



#####################################################################
# Function: create_stats
#
#
# summary:
# Return data trend information specified in "stats".
#
# parameters:
# res     - data stored array
# colname - analysis target column
# stats   - keyword list of the kind of data trend (ex. max, min, average..)
# processlist - list of reference of calculate function 
#
# return:
# result set of data trend
#
# execption:
# none
#
# recital:
#
# | data trend are stored as folliwing format
# | result => {
# |     data trend keywaord  => aggregate value
# |     .....
# | }
# 
#####################################################################
sub create_stats {
    my ($res, $colname, $stats, $processlist) = @_;
    
    my @stats;
    if(!$stats or ($stats and ($stats->[0] eq 'all'))) {
        @stats = keys %{$processlist};
    }
    else {
        @stats = @{$stats};
    }
    
    my $result = {};
    my @src = ();
    
    for my $value (@{$res}) {
        defined($value->{$colname}) and push(@src, $value->{$colname});
    }
    
    for my $key (@stats) {
        exists($processlist->{$key}) and $result->{$key} = $processlist->{$key}->(\@src);
    }
    resetcalc();
    
    return $result;
}

#####################################################################
# Function: calcstats, resetcalc
#
#
# summary:
# Calculate average, variance, max, min for given set of data
#
# parameters:
# $src    - set of value
#
# return:
# average, variance, max, min
#
# execption:
# none
#
# recital:
#
# | result is returned as following struct
# | {
# |     mean     => average 
# |     variance => variance
# |     max      => max
# |     min      => min
# | }
#
# This function cache the results, and return the cache value or later.
# If you clear the cache, call resetcalc().
# 
#####################################################################

{
    my $mean = 0; my $variance = 0; my $max = 0; my $min = 0;
    my $iscalc = 0;

    sub calcstats {
        my ($src) = @_;

        if(@{$src} <= 0) {
            return {mean => NO_DATA, variance => NO_DATA, max => NO_DATA, min => NO_DATA};
        }
        if($iscalc) {
            return {mean => $mean, variance => $variance, max => $max, min => $min};
        }
        $iscalc = 1;

        my $s1 = 0; my $s2 = 0; my $n = 0;
        $min = $src->[0], $max = $src->[0];

        for my $value (@{$src}) {
            $n++; $s1 += $value; $s2 += $value * $value;
            $min > $value and $min = $value;
            $max < $value and $max = $value;
        }
        
        $n > 0 and $mean = $s1 / $n;
        $n > 1 and $variance = ($s2 - $n * $mean * $mean) / ($n - 1);
        
        return {mean => $mean, variance => $variance, max => $max, min => $min};
    }
    
    sub resetcalc {
        $iscalc = 0;
        $mean = 0; $variance = 0; $max = 0; $min = 0;
    }
}





#####################################################################
# Function: execute_query
#
#
# summary:
# Execute query(SELECT), and return the result.
#
# parameters:
# conn  - connection
# query - query string
# bind  - list of bind values 
#
# return:
# results of query
#
# execption:
# none
#
# recital:
# result is stored in following struct
# |
# | result[0] => {column name => value [, column name => value]...}
# | result[1] => {column name => value [, column name => value]...}
# | result[2] => {column name => value [, column name => value]...}
# | ...
# | result[n] => {column name => value [, column name => value]...}
# |
#####################################################################
sub execute_query {
    my ($conn, $query, $bind) = @_;
    my $result;
    eval {
        if($bind) {
            $result = $conn->selectall_arrayref($query, $G_connattr, @$bind);
        }
        else {
            $result = $conn->selectall_arrayref($query, $G_connattr);
        }
    };
    if(!$result or $@) {
        if(!$@) {
            $@ = $DBI::errstr;
        }
        croak(MSG_DB_SELECTFAILED . " query=$query dbname=$G_options->{dbname}, reason=$@" );
    }
    
    return $result;
}


#####################################################################
# Function: analyze_option
#
#
# summary:
#
# Analyse of "pg_make_report.pl" option. Existence check of necessary option, and return
# results of analysis. If option is not specified, we use default value and set it. 
#
# parameters:
# none
#
# return:
# result - Hash of results of option abalysis.
#
# execption:
# - necessary option is not specified or failed on getting the option
# 
# recital:
#
#####################################################################
sub analyze_option {
    my %result = ();

    #
    # set default value
    #
    $result{host}      = exists($ENV{PGHOST}) ? $ENV{PGHOST} : undef;
    $result{dbname}    = exists($ENV{PGDATABASE}) ? $ENV{PGDATABASE} : undef;
    $result{port}      = exists($ENV{PGPORT}) ? $ENV{PGPORT} : undef;
    $result{user}      = exists($ENV{PGUSER}) ? $ENV{PGUSER} : undef;
    $result{tempfile}  = './pg_statsreport.tmpl';
    $result{is_target} = FALSE;
    $result{is_snapshot_id} = FALSE;
    $result{is_snapshot_date} = FALSE;
    
    #
    # DO GetOptions and store options in hash
    #
    my $GetOptions_result = GetOptions(
        'host=s' => \$result{host},
        'dbname=s' => \$result{dbname},
        'port=i' => \$result{port},
        'user=s' => \$result{user},
        'output-prefix=s' => \$result{prefix},
        'startid=i' => \$result{sid},
        'endid=i' => \$result{eid},
        'startdate=s' => \$result{sdate},
        'enddate=s' => \$result{edate},
        'template=s' => \$result{tempfile},
        'targetdbid=i' => \$result{target_id},
        'targetstring=s' => \$result{target_string},
        'help' => sub {
            printUsage();
            exit(0);
        },
        'version' => sub {
            printVersion();
            exit(0);    
        },
        'verbose:1' => \$result{verbose}
    );

    $result{is_snapshot_id}   = ($result{sid} and $result{eid});
    $result{is_snapshot_date} = ($result{sdate} and $result{edate});
    $result{is_target}        = ($result{target_id} or $result{target_string});
    
    #
    # Existence check of necessary option, and alos check inavlid specific option 
    #
    die(MSG_OPT_NOOPT . ' --host') if(!$result{host});
    die(MSG_OPT_NOOPT . ' --dbname') if(!$result{dbname});
    die(MSG_OPT_NOOPT . ' --port') if(!$result{port});
    die(MSG_OPT_NOOPT . ' --user') if(!$result{user});
    die(MSG_OPT_CONFLICT_ID . ' --targetdbid, and --targetstring') if($result{target_id} and $result{target_string});
    
    if($result{is_snapshot_id}) {
        if($result{is_snapshot_date}) {
            # Set Snapshot-ID and timestamp... It's invalid
            die(MSG_OPT_CONFLICT_ID . ' --startid, --endid, and --startdate, --enddate');
        }
        if($result{target_id} or $result{target_string}) {
            # Set Snapshot-ID and DBID (or connection-string to DB)... It's invalid
            warn(MSG_OPT_CONFLICT_DBID);
            undef $result{target_id};
            undef $result{target_string};
            $result{is_target} = FALSE;
        }
        #
        # parse and validation the specified option value
        #
        die(MSG_OPT_NO_ID . " --startid($result{sid}) >= --endid($result{eid})") if($result{eid} <= $result{sid});
    }
    elsif(!$result{is_snapshot_date}) {
        # Not set Snapshot-ID and Snapshot-date... It's invalid
        die(MSG_OPT_NOOPT . ' --startid, --endid, or --startdate, --enddate');
    }
    
    
    
    #
    # parse option and file-create-read-permission check.
    #
    my $targetdir = $result{prefix} ? dirname($result{prefix}) : undef; 
    if($targetdir and !(-w $targetdir)) {
        die(MSG_OPT_INVALID_PREFIX . " --output-prefix ($result{prefix}) ($!)");
    }
    
    if($result{tempfile} and !(-r $result{tempfile})) {
        die(MSG_OPT_INVALID_TMPFILE . " --template ($result{tempfile}) ($!)");
    }

    return \%result;    
}

#####################################################################
# Function: errorHandler
#
#
# summary:
#
# errorHandler for template engine
# 
# parameters:
# none
#
# return:
# none
#
# recital:
# none
#
#
#####################################################################
sub errorHandler {
    my %args = @_;
    my $err_ref = $args{arg};
    $$err_ref = MSG_TEMPLATE_FAILED . " reason=$args{error}";
    return undef;
}

#####################################################################
# Function: printVersion
#
#
# summary:
#
# show version
# 
# parameters:
# none
#
# return:
# none
#
# recital:
# none
#
#
#####################################################################
sub printVersion {
    print STDOUT MODULENAME . " " . VERSION . "\n";
}


#####################################################################
# Function: printUsage
#
#
# summary:
#
# show usage
# 
# parameters:
# none
#
# return:
# none
#
# recital:
# none
#
#
#####################################################################
sub printUsage {
    my $modulename = MODULENAME;
	print STDOUT <<_USAGE_;

$modulename - Report the statictics information at target database.

Usage: $modulename.pl [--host=<hostname>] [--dbname=<dbname>] [--port=<port number>]
                         [--user=<username>] [--output-prefix=<prefix>]
                         {
                             --startid=<start snapshot id> --endid=<end snapshot id>
                             | --startdate=<start date> --enddate=<end date>
                         }
                         [--template=<template filename>]
                         [ 
                             { --targetdbid=<dbid>
                              | --targetstring=<conn_string> }
                         ]
                         [--help]
                         [--version]

Options:
    --host=<hostname>               snapshot db server host or IP address (default: \$PGHOST)
    --dbname=<dbname>               snapshot db name (default: \$PGDATABASE)
    --port=<port number>            snapshot db server port (default: \$PGPORT)
    --user=<username>               snapshot db user (default: \$PGUSER)
    --output-prefix=<prefix>        prefix of report-file name as '<prefix>_<hostname>_<port>_<dbname>.txt'(default: print to standard output)
    --startid=<start snapshot id>   start snapshot ID
    --endid=<end snapshot id>       end snapshot ID
    --startdate=<start date>        start snapshot date (format: 'YYYY-MM-DD hh:mm:ss')  (ex. '2008-08-01 22:00:00')
    --enddate=<end date>            end snapshot date (format: 'YYYY-MM-DD hh:mm:ss')  (ex. '2008-08-01 23:00:00')
    --template=<template filename>  template file name (defalut: ./pg_statsreport.tmpl)
    --targetdbid=<dbid>             DBID of target db
    --targetstring=<conn_string>    string to connect target db
    --help                          show this help
    --version                       output version information

_USAGE_

}

sub set_option {
    my ($res) = @_;
    
    while(my ($key, $value) = each %{$res}) {
        $G_options->{$key} = $value;
    }
}
1;



