------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--              A U N I T . R E P O R T E R . G N A T T E S T               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                                                                          --
--                       Copyright (C) 2012-2019, AdaCore                   --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT is maintained by AdaCore (http://www.adacore.com)                   --
--                                                                          --
------------------------------------------------------------------------------

with AUnit.IO;           use AUnit.IO;
with AUnit.Time_Measure; use AUnit.Time_Measure;

--  Reporter intended to be used by test drivers generated by gnattest.

package body  AUnit.Reporter.GNATtest is

   procedure Dump_Result_List (File : File_Type; L : Result_Lists.List);
   --  List failed assertions

   procedure Report_Test (File : File_Type; Test : Test_Result);
   --  Report a single assertion failure or unexpected exception

   procedure Put_Measure is new Gen_Put_Measure_In_Seconds;
   --  Output elapsed time

   procedure Indent (File : File_Type; N : Natural);
   --  Print N indentations to output

   ------------------------
   --  Dump_Result_List  --
   ------------------------

   procedure Dump_Result_List (File : File_Type; L : Result_Lists.List) is

      use Result_Lists;

      C : Cursor := First (L);

   begin

      --  Note: can't use Iterate because it violates restriction
      --  No_Implicit_Dynamic_Code

      while Has_Element (C) loop
         Report_Test (File, Element (C));
         Next (C);
      end loop;
   end Dump_Result_List;

   ------------
   -- Indent --
   ------------

   procedure Indent (File : File_Type; N : Natural) is
   begin
      for J in 1 .. N loop
         Put (File, "    ");
      end loop;
   end Indent;

   ------------
   -- Report --
   ------------

   procedure Report (Engine  : GNATtest_Reporter;
                     R       : in out Result'Class;
                     Options : AUnit_Options := Default_Options)
   is
      File : File_Type renames Engine.File.all;

      Failures_Count : Integer;
      Crashes_Count  : Integer;
      Passed_Count   : Integer;
      Tests_Count    : Integer;
   begin

      Tests_Count := Integer (Test_Count (R));
      Crashes_Count := Integer (Error_Count (R));
      Passed_Count := Integer (Success_Count (R));
      Failures_Count := Tests_Count - (Passed_Count + Crashes_Count);

      if Options.Report_Successes then
         declare
            S : Result_Lists.List;
         begin
            Successes (R, S);
            Dump_Result_List (File, S);
         end;
      end if;

      declare
         F : Result_Lists.List;
      begin
         Failures (R, F);
         Dump_Result_List (File, F);
      end;

      declare
         E : Result_Lists.List;
      begin
         Errors (R, E);
         Dump_Result_List (File, E);
      end;

      Put      (File, Tests_Count, 0);
      Put      (File, " tests run: ");
      Put      (File, Passed_Count, 0);
      Put      (File, " passed; ");
      Put      (File, Failures_Count, 0);
      Put      (File, " failed; ");
      Put      (File, Crashes_Count, 0);
      Put_Line (File, " crashed.");

   end Report;

   ------------------
   -- Report_Test --
   ------------------

   procedure Report_Test (File : File_Type; Test : Test_Result) is
      Is_Assert : Boolean;
      Is_Condition : Boolean := False;

      T : AUnit_Duration;
      N : Integer;
   begin

      Put      (File, Test.Test_Name.all);

      if Test.Failure /= null or else Test.Error /= null then
         if Test.Failure /= null then
            Is_Assert := True;
         else
            Is_Assert := False;
         end if;

         if Is_Assert then
            Put   (File, " error: corresponding test FAILED: ");
         else
            Put   (File, " error: corresponding test CRASHED: ");
         end if;

         if Is_Assert then

            if Test.Failure.Message'Length > 9 then
               N := Test.Failure.Message'First;
               if
                 Test.Failure.Message.all (N .. N + 8) = "req_sloc("
                 or else Test.Failure.Message.all (N .. N + 8) = "ens_sloc("
               then
                  for I in N + 9 .. Test.Failure.Message'Last - 2 loop
                     if Test.Failure.Message.all (I + 1 .. I + 2) = "):" then
                        Put (File, Test.Failure.Message.all
                               (I + 3 .. Test.Failure.Message'Last));

                        Put      (File, " (");
                        Put      (File, Test.Failure.Message.all (N + 9 .. I));
                        Put_Line (File, ")");
                        Is_Condition := True;
                        exit;
                     end if;
                  end loop;
               end if;
            end if;

            if not Is_Condition then
               Put      (File, Test.Failure.Message.all);
               Put      (File, " (");
               Put      (File, Test.Failure.Source_Name.all);
               Put      (File, ":");
               Put      (File, Integer (Test.Failure.Line), 0);
               Put      (File, ")");
            end if;
            if Test.Elapsed /= Time_Measure.Null_Time then
               Put (File, " (");
               T := Get_Measure (Test.Elapsed);
               Put_Measure (File, T);
               Put (File, ")");
            end if;
            New_Line (File);

         else
            Put      (File, Test.Error.Exception_Name.all);

            if Test.Error.Exception_Message /= null then
               Put      (File, " : ");
               Put      (File, Test.Error.Exception_Message.all);
            end if;
            if Test.Elapsed /= Time_Measure.Null_Time then
               Put (File, " (");
               T := Get_Measure (Test.Elapsed);
               Put_Measure (File, T);
               Put (File, ")");
            end if;
            New_Line (File);

            if Test.Error.Traceback /= null then
               Put_Line (File, " Traceback:");

               declare
                  From, To : Natural := Test.Error.Traceback'First;
               begin
                  while From <= Test.Error.Traceback'Last loop
                     To := From;
                     while To <= Test.Error.Traceback'Last
                       and then Test.Error.Traceback (To) /= ASCII.LF
                     loop
                        To := To + 1;
                     end loop;

                     Indent (File, 2);
                     Put_Line (File, Test.Error.Traceback (From .. To - 1));
                     From := To + 1;
                  end loop;
               end;
            end if;

         end if;
      else
         Put (File, " info: corresponding test PASSED");
         if Test.Elapsed /= Time_Measure.Null_Time then
            Put (File, " (");
            T := Get_Measure (Test.Elapsed);
            Put_Measure (File, T);
            Put (File, ")");
         end if;
         New_Line (File);
      end if;

   end Report_Test;
end AUnit.Reporter.GNATtest;
